/* Memory access checker.
   Copyright 1993 Tristan Gingold
		  Written September 1993 by Tristan Gingold

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2 of the
License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 The author may be reached (Email) at the address gingold@amoco.saclay.cea.fr,
 or (US/French mail) as   Tristan Gingold
   			  8 rue Parmentier
   			  F91120 PALAISEAU
   			  FRANCE
*/

#define _MALLOC_INTERNAL
#include "malloc.h"
#include "errlist.h"

/* Emulate sbrk. */
PTR
sbrk (int incr)
{
  struct mdesc *mdp = MD_TO_MDP(NULL_MDESC);
  int round_size = (incr + LITTLE_SPACE-1) & ~(LITTLE_SPACE-1);

  /* initialize all, if necessary */
  if (__malloc_initialized == 0)
    {
      __malloc_initialized = 1;
      chkr_initialize ();	/* which called init_morecore */
    }
  if (incr >= 0)		/* get more memory */
    {
      if (mdp->_lastblock && mdp->_lastblock->state == MDBRK)
	{
	  /* sbrk was already called and the last block is MDBRK.  Just alloc
	   * more memory. */
	  PTR res;
	  res = mdp->morecore (mdp, incr);
	  if (res == (PTR) 0)
	    {
	      errno = ENOMEM;
	      return (PTR) -1;
	    }
	  mdp->_lastblock->info.brk.real_size += incr;
#ifdef CHKR_HEAPBITMAP
          if (incr != 0)
	    chkr_set_right (res, incr, CHKR_WO);
#endif
	  memset (res, 0, incr);
	  return res;
	}
      else if (mdp->_lastblock && mdp->_lastblock->state == MDFREE)
	{
	  /* The last block is a FREE block.  Adjust its size. */
	  if (mdp->morecore (mdp, round_size - mdp->_lastblock->size) == (PTR) 0)
	    {
	      errno = ENOMEM;
	      return (PTR) -1;
	    }
	  mdp->_lastblock->state = MDBRK;
	  mdp->_lastblock->size = round_size;
	  mdp->_lastblock->info.brk.real_size = incr;
	  return (PTR) mdp->_lastblock + HEADER_SIZE;
	}
      else
	{
	  /* Alloc a new block for SBRK */
	  struct malloc_header *res;
	  res = (struct malloc_header *) mdp->morecore (mdp, round_size + HEADER_SIZE);
	  if (res == NULL_HEADER)
	    {
	      errno = ENOMEM;
	      return (PTR) -1;
	    }
	  res->size = round_size;
	  res->prev = mdp->_lastblock;
	  if (mdp->_lastblock)
	    mdp->_lastblock->next = res;
	  res->next = NULL_HEADER;
	  mdp->_lastblock = res;
	  if (!mdp->_firstblock)
	    mdp->_firstblock = res;
	  res->state = MDBRK;
	  res->s_diff = res->size - incr;
	  res->info.brk.real_size = incr;
#ifdef CHKR_HEAPBITMAP
          if (incr != 0)
	    chkr_set_right ((PTR) res + HEADER_SIZE, incr, CHKR_WO);
#endif
	  memset ((PTR) res + HEADER_SIZE, 0, incr);
	  return (PTR) res + HEADER_SIZE;
	}
    }
  else
    {
      /* Realease memory. Be sure the last block is BRK */
      PTR res;
      if (!mdp->_lastblock || mdp->_lastblock->state != MDBRK)
	{
	  /* The last block is not BRK... */
	  chkr_perror (M_M_SBA_BR_ET);
	  errno = ENOMEM;
	  return (PTR) -1;
	}
      res = (PTR) mdp->_lastblock + HEADER_SIZE;
      if (mdp->_lastblock->info.brk.real_size + incr < 0)
	{
	  /* INCR is too big. */
	  chkr_perror (M_M_SBA_BR_ET);
	  incr = -mdp->_lastblock->info.brk.real_size;
	}
      if (mdp->_lastblock->info.brk.real_size + incr == 0)
	{
	  if (mdp->_firstblock == mdp->_lastblock)
	    mdp->_firstblock = NULL_HEADER;
	  mdp->_lastblock = mdp->_lastblock->prev;
	  if (mdp->_lastblock)
	    mdp->_lastblock->next = NULL_HEADER;
	  mdp->morecore (mdp, incr - HEADER_SIZE);
	  return res;
	}
      else
	{
	  mdp->morecore (mdp, round_size);
	  mdp->_lastblock->size += round_size;
	  res += mdp->_lastblock->info.brk.real_size;
	  mdp->_lastblock->info.brk.real_size += incr;
	  return res;
	}
    }
}
