#include <qapp.h>

#include <qdialog.h>
#include <qlabel.h>
#include <qlined.h>
#include <qpushbt.h>
#include <qchkbox.h>
#include <qradiobt.h>
#include <qbttngrp.h>
#include <qlcdnum.h>
#include <qscrbar.h>
#include <qmenubar.h>

#include <qpainter.h>
#include <qfontinf.h>
#include <qpixmap.h>
#include <qbuffer.h>
#include <qimage.h>
#include <qstring.h>

#include <stdlib.h>
#include "aticatac.cpp"

class FontSelect;

class XFormControl : public QFrame
{
    Q_OBJECT
public:
    XFormControl( QWidget *parent=0, const char *name=0 );
   ~XFormControl() {}
signals:
    void newMatrix( Q2DMatrix );
    void newText( char * );
    void newFont( QFont );
    void newMode( bool image );
private slots:
    void newMtx();
    void selectFont();
    void selectFontDestroyed();
    void fontSelected( QFont );
    void toggleMode();
private:
    QScrollBar	*rotS;                 // Rotation angle scroll bar
    QScrollBar	*shearS;               // Shear value scroll bar
    QScrollBar	*magS;                 // Magnification value scroll bar
    QLCDNumber	*rotLCD;               // Rotation angle LCD display
    QLCDNumber	*shearLCD;             // Shear value LCD display
    QLCDNumber	*magLCD;               // Magnification value LCD display
    QCheckBox	*mirror;               // Checkbox for mirror image on/of
    QLineEdit	*textEd;               // Inp[ut field for xForm text
    QPushButton *f;                    // Select font push button
    QCheckBox	*imgCheckBox;          // Check box for image on/off
    FontSelect	*fs;                   // font select dialog box
    bool         fsUp;                 // TRUE if font select is visible
};


/* ShowXForm displays a text or an image (QPixmap) using a coordinate
   transformation matrix (Q2DMatrix)
*/

class ShowXForm : public QWidget
{
    Q_OBJECT
public:
    ShowXForm( QWidget *parent=0, const char *name=0 );
   ~ShowXForm() {}
    void showIt();               // (Re)displays text or image
public slots:
    void setText( char* );
    void setMatrix( Q2DMatrix );
    void setFont( QFont f );
    void setPixmap( QPixmap );
    void setPixmapMode( bool b );
    bool imageMode() { return usingImage; }
private:
    void paintEvent( QPaintEvent * );
    void resizeEvent( QResizeEvent * );
    QString   text;                    // text to be displayed
    Q2DMatrix mtx;                     // coordinate transform matrix
    QPixmap   img;                     // Image to be displayed
    bool      usingImage;              // TRUE if img is being displayed
    QRect     eraseRect;               // covers last displayed text/image
};

class FontSelect : public QDialog
{
    Q_OBJECT
public:
    FontSelect( QView *parent=0, const char *name=0 );
private slots:
    void	 newStrikeOut();
    void	 doFont();
    void	 newUnderline();
    void	 newItalic();
    void	 newWeight( int id );
    void	 newFamily();
    void	 newSize( long );
    void	 slidingSize( long );
    void	 doApply();
    void	 fileActivated( int );
    void	 familyActivated( int );
    void	 pSizeActivated( int );
signals:
    void	 newFont( QFont );
protected:
    void	 updateMetrics();
private:
    QLabel	 *familyL;
    QLabel	 *sizeL;
    QLineEdit	 *family;
    QLineEdit	 *sizeE;
    QCheckBox	 *italic;
    QCheckBox	 *underline;
    QCheckBox	 *strikeOut;
    QPushButton	 *apply;
    QPushButton	 *stat;
    QScrollBar	 *sizeS;
    QRadioButton *rb[5];
    QButtonGroup *weight;
    QGroupBox    *mGroup;
    QFont	  f;
    QMenuBar	 *menu;
    QLabel	 *metrics[4][2];
};


XFormControl::XFormControl( QWidget *parent, const char *name )
	: QFrame( parent, name )
{
    rotLCD      = new QLCDNumber( 4, this, "rotateLCD" );
    rotS        = new QScrollBar( QScrollBar::Horizontal, this, 
			      "rotateScrollBar" );
    shearLCD    = new QLCDNumber( 5,this, "shearLCD" );
    shearS      = new QScrollBar( QScrollBar::Horizontal, this, 
			      "shearScrollBar" );
    mirror      = new QCheckBox( this, "mirrorCheckBox" );
    textEd      = new QLineEdit( this, "text" );
    f	        = new QPushButton( this, "text" );
    imgCheckBox = new QCheckBox( this, "fontOrImage" );

    rotLCD->setGeometry( 10, 10, 100, 60 );
    rotLCD->display( "	0'" );

    rotS->setGeometry( 10, 80, 100, 15 );
    rotS->setRange( -180, 180 );
    rotS->setValue( 0 );
    connect( rotS, SIGNAL(valueChanged(long)), SLOT(newMtx()) );

    shearLCD->setGeometry( 10, 105, 100, 60 );
    shearLCD->display( "0.00" );

    shearS->setGeometry( 10, 175, 100, 15 );
    shearS->setRange( -25, 25 );
    shearS->setValue( 0 );
    connect( shearS, SIGNAL(valueChanged(long)), SLOT(newMtx()) );

    mirror->setGeometry( 10, 200, 100, 15 );
    mirror->setText( "Mirror" );
    connect( mirror, SIGNAL(clicked()), SLOT(newMtx()) );

    imgCheckBox->setGeometry( 10, 220, 100, 15 );
    imgCheckBox->setText( "Image" );
    connect( imgCheckBox, SIGNAL(clicked()), SLOT(toggleMode()) );

    textEd->setGeometry( 10, 245, 100, 20 );
    textEd->setText( "Troll" );
    connect( textEd, SIGNAL(textChanged(char*)), SIGNAL(newText(char*)) );

    f->setGeometry( 10, 275, 100, 20 );
    f->setText( "Select font..." );
    connect( f, SIGNAL(clicked()), SLOT(selectFont()) );

    magS    = 0;
    fs	    = 0;
}

/*
    Called whenever the user has changed one of the matrix parameters
    (i.e. rotate, shear or magnification)
*/
void XFormControl::newMtx()
{
    Q2DMatrix m;
    if ( imgCheckBox->isChecked() ) {
	double magVal = 1.0*magS->value()/100;
	m.scale( magVal, magVal );
    }
    double shearVal = 1.0*shearS->value()/25;
    m.shear( shearVal, shearVal );
    m.rotate( rotS->value() );
    if ( mirror->isChecked() ) {
	m.scale( 1, -1 );
	m.rotate( 180 );
    }

    QString tmp;
    tmp.sprintf( "%1.2f", shearVal  );
    if ( shearVal >= 0 )
	tmp.insert( 0, " " );
    shearLCD->display( tmp );

    int rot = rotS->value();
    if ( rot < 0 )
	rot = rot + 360;
    tmp.sprintf( "%3i'", rot );
    rotLCD->display( tmp );
    emit newMatrix( m );
}

    
void XFormControl::selectFont()
{
    if (!fs) {
	fs   = new FontSelect;
	connect( fs, SIGNAL(destroyed()),    SLOT(selectFontDestroyed()) );
	connect( fs, SIGNAL(newFont(QFont)), SLOT(fontSelected(QFont)) );
	fs->setGeometry( QRect( 100, 200, 380, 260 ) );
	fsUp = FALSE;
    }
    fsUp = !fsUp;
    if ( fsUp )
	fs->show();
    else
	fs->hide();
}

/*
    Called when the user has closed the SelectFont dialog via the
    window manager.
*/
void XFormControl::selectFontDestroyed()
{
    fs = 0;
}

void XFormControl::fontSelected( QFont f )
{
    imgCheckBox->setChecked( FALSE );
    emit newFont( f );
    toggleMode();
}

/*
    Toggles between image and text mode.
*/
void XFormControl::toggleMode()
{
    if ( magS == 0 ) {
	magS = new QScrollBar( QScrollBar::Horizontal, this,
			       "magnifyScrollBar" );
	magS->setGeometry( 10, 375, 100, 15 );
	magS->setRange( 0, 400 );
	magS->setValue( 100 );
	connect( magS, SIGNAL(valueChanged(long)), SLOT(newMtx()) );
	magLCD = new QLCDNumber( 4,this, "shearLCD" );
	magLCD->setGeometry( 10, 305, 100, 60 );
	magLCD->display( "100" );
	connect( magS, SIGNAL(valueChanged(long)), magLCD, SLOT(display(long)) );
    }
    emit newMode( imgCheckBox->isChecked() );
    newMtx();
    if ( imgCheckBox->isChecked() ) {
	magS->show();
	magLCD->show();
    } else {
	magS->hide();
	magLCD->hide();
    }
    qApp->flushX();
}

const int yOff = 35;

ShowXForm::ShowXForm( QWidget *parent, const char *name )
	: QWidget( parent, name )
{
    setFont( QFont( "Charter", 48, QFont::Bold ) );
    setBackgroundColor( white );
    usingImage = FALSE;
}

void ShowXForm::paintEvent( QPaintEvent * )
{
    showIt();
}

void ShowXForm::resizeEvent( QResizeEvent * )
{
    eraseRect = QRect( width()/2, height()/2, 0, 0 );
}

void ShowXForm::setText( char *s )
{
    text = s;
    showIt();
}

void ShowXForm::setMatrix( Q2DMatrix w )
{
    mtx = w;
    showIt();
}

void ShowXForm::setFont( QFont f ) 
{ 
    usingImage = FALSE;
    QWidget::setFont( f );
} 

void ShowXForm::setPixmap( QPixmap pm )
{
    img	       = pm;
    usingImage = TRUE;
    showIt();
}

void ShowXForm::setPixmapMode( bool b )
{
    if ( usingImage == b)
	return;	   
    usingImage = b;
}

void ShowXForm::showIt()
{
    QPainter p;
    QRect rect;   // rectangle covering new text/image in virtual coordinates
    Q2DMatrix m;  // copy user specified transform
    QFontMetrics fm( font() ); // metrics for widget font
    int textYPos = 0; // distance from boundingRect y pos to baseline
    int textXPos = 0; // distance from boundingRect x pos to text start
    if ( imageMode() ) {
        rect = img.rect();
    } else {
        rect     = fm.boundingRect( text );   // rectangle covering text
        textYPos = -rect.y();
        textXPos = -rect.x();
    }
    rect.setTopLeft( QPoint( -rect.width()/2, -rect.height()/2 ) );
          // compute union of new and old rect 
          // the resulting rectangle will cover what is already displayed
          // and have room for the new text/image
    eraseRect = eraseRect.unite( mtx.boundingRect( rect ) ); 
    eraseRect.move( -1, -1 ); // add border for matrix round off
    eraseRect.setSize( QSize( eraseRect.width() + 2,eraseRect.height() + 2 ) );
    QPixmap pix( eraseRect.size() );  // off-screen drawing pixmap
    pix.fill( backgroundColor() );
    p.begin( &pix );
    p.setFont( font() );              // use widget font
    m.translate( eraseRect.width()/2, eraseRect.height()/2 ); //(0,0) in center
    m = mtx * m;
    p.setWorldMatrix( m );
    if ( imageMode() )
        p.drawPixmap( rect.topLeft(), img );
    else
        p.drawText( rect.left() + textXPos, rect.top() + textYPos, text );
    p.end();
    int xpos = width()/2  - eraseRect.width()/2;
    int ypos = height()/2 - eraseRect.height()/2;
    bitBlt( this, xpos, ypos,			// copy pixmap to widget
	    &pix, 0, 0, -1, -1 );
    eraseRect =  mtx.boundingRect( rect );
    return;
}

FontSelect::FontSelect( QView *parent, const char *name)
    : QDialog( parent, name, 0 ), f( "Charter", 48, QFont::Bold )
{
    const char *radios[] = { "Light (25)", "Normal (50)", "DemiBold (63)", 
                             "Bold (75)", "Black (87)" };
    int i;

    familyL    = new QLabel(	 this, "familyLabel" );
    sizeL      = new QLabel(	 this, "sizeLabel" );
    family     = new QLineEdit(	 this, "family" );
    sizeE      = new QLineEdit(	 this, "pointSize" );
    italic     = new QCheckBox(	 this, "italic" );
    underline  = new QCheckBox(	 this, "underline" );
    strikeOut  = new QCheckBox(	 this, "strikeOut" );
    apply      = new QPushButton( this, "apply" );
    sizeS      = new QScrollBar( QScrollBar::Horizontal, this, 
			      "pointSizeScrollBar" );
			      
    familyL->setGeometry( 10, yOff + 10, 100,20 );
    familyL->setText( "Family :" );
	    
    sizeL->setGeometry( 10, yOff + 40, 100, 20 );
    sizeL->setText( "Point size :" );
	    
    family->setGeometry( 110, yOff + 10, 100, 20 );
    family->setText( "Charter" );

    sizeE->setGeometry( 110, yOff + 40, 100, 20 );
    sizeE->setText( "48" );
	    
    sizeS->setGeometry( 220, yOff + 40, 100, 20 );
    sizeS->setRange( 1, 100 );
    sizeS->setValue( 48 );
    sizeS->setTracking( FALSE );
    connect( sizeS, SIGNAL(valueChanged(long)), SLOT(newSize(long)) );
    connect( sizeS, SIGNAL(sliderMoved(long)), SLOT(slidingSize(long)) );

    italic->setGeometry( 10, yOff + 70, 80, 20 );
    italic->setText( "Italic" );
    connect( italic, SIGNAL(clicked()), SLOT(newItalic()) );
	    
    underline->setGeometry( 110, yOff + 70, 80, 20 );
    underline->setText( "Underline" );
    connect( underline, SIGNAL(clicked()), SLOT(newUnderline()) );
	    
    strikeOut->setGeometry( 210, yOff + 70, 80, 20 );
    strikeOut->setText( "StrikeOut" );
    connect( strikeOut, SIGNAL(clicked()), SLOT(newStrikeOut()) );
	    
    apply->setGeometry( 235, yOff + 10, 70, 20);
    apply->setText( "APPLY" );
    apply->setDefault( TRUE );
    connect( apply, SIGNAL(clicked()), SLOT(doApply()) );

    weight = new QButtonGroup( "Weight", this, "weightGroupBox" );
    weight->setGeometry( 10, yOff + 100, 120, 120 );
    connect( weight, SIGNAL(clicked(int)), SLOT(newWeight(int)) );
    QString name;
    for( i = 0 ; i < 5 ; i++ ) {
	name.sprintf("radioButton %i",i);
	rb[i] = new QRadioButton( weight, name );
	rb[i]->setGeometry( 10, 15+i*20 , 95, 20 );
	rb[i]->setText( radios[i] );
    }
    rb[3]->setChecked( TRUE );

    const char *familys[] = { "Charter", "Clean", "Courier", "Fixed",
			    "Gothic", "Helvetica", "Lucida", "Lucidabright",
			    "Lucidatypewriter", "Mincho", 
			    "New century Schoolbook", 
			    "Symbol", "Terminal", "Times", "Utopia", 0 };

    const char *pSizes[]  = {  "8", "10", "12", "14", "18", "24", "36",
			    "48", "72", "96", 0 };



    QPopupMenu *file = new QPopupMenu;
    file->insertItem( "Quit\tC-x" );

    QPopupMenu *family = new QPopupMenu;
    const char **tmp;
    tmp = familys;
    while( *tmp )
	family->insertItem( *tmp++ );

    QPopupMenu *pSize = new QPopupMenu;
    tmp = pSizes;
    while( *tmp )
	pSize->insertItem( *tmp++ );

    menu = new QMenuBar( this );
    menu->move( 0, 0 );
    menu->resize( 350, 30 );
    menu->insertItem( "File", file );
    menu->insertItem( "Family", family );
    menu->insertItem( "Point size", pSize );

    connect( file, SIGNAL(activated(int)), SLOT(fileActivated(int)) );
    connect( family, SIGNAL(activated(int)), SLOT(familyActivated(int)) );
    connect( pSize, SIGNAL(activated(int)), SLOT(pSizeActivated(int)) );

    const char *mLabelStr[] = { "Family:", "Point size:",
				"Weight:", "Italic:" };

    mGroup = new QButtonGroup( this, "metricsGroupBox" );
    mGroup->setTitle( "Actual font" );
    mGroup->setGeometry(140, yOff + 100, 230, 100);
    for( i = 0 ; i < 4 ; i++ ) {
	name.sprintf("MetricsLabel[%i][%i]",i,0);
	metrics[i][0] = new QLabel( mGroup, name);
	metrics[i][0]->setGeometry(10, 15 + 20*i, 70, 20);
	metrics[i][0]->setText( mLabelStr[i] );

	name.sprintf("MetricsLabel[%i][%i]",i,1);
	metrics[i][1] = new QLabel( mGroup, name);
	metrics[i][1]->setGeometry(90, 15 + 20*i, 135, 20);
    }
    updateMetrics();
}

void FontSelect::newStrikeOut()
{
    f.setStrikeOut( strikeOut->isChecked() );
    doFont();
}

void FontSelect::doFont()
{
    updateMetrics();
    emit newFont( f );
}

void FontSelect::newUnderline()
{
    f.setUnderline( underline->isChecked() );
    doFont();
}

void FontSelect::newItalic()
{
    f.setItalic( italic->isChecked() );
    doFont();
}

void FontSelect::newFamily()
{
    f.setFamily( family->text() );
    doFont();
}

void FontSelect::newWeight( int id )
{
    switch( id ) {
	case 0 :
	    f.setWeight( QFont::Light );
	    break;
	case 1 :
	    f.setWeight( QFont::Normal );
	    break;
	case 2 :
	    f.setWeight( QFont::DemiBold );
	    break;
	case 3 :
	    f.setWeight( QFont::Bold );
	    break;
	case 4 :
	    f.setWeight( QFont::Black );
	    break;
	default:
	    return;
    }
    doFont();
}

void FontSelect::newSize( long value )
{
    QString tmp;

    tmp.sprintf("%i", value);
    sizeE->setText( tmp );
    f.setPointSize( value );
    doFont();
}

void FontSelect::slidingSize( long value )
{
    QString tmp;

    tmp.sprintf("%i", value);
    sizeE->setText( tmp );
}

void FontSelect::doApply()
{
    int sz = atoi( sizeE->text() );
    if ( sz > 100) {
	sizeS->blockSignals( TRUE );
	sizeS->setValue( 100 );
	sizeS->blockSignals( FALSE );
	f.setPointSize( sz );
    } else {
	sizeS->setValue( atoi( sizeE->text() ) );
    }
    f.setFamily( family->text() );
    doFont();
}

void FontSelect::fileActivated( int id )
{
    qApp->quit();
}

void FontSelect::familyActivated( int id )
{
    family->setText( ((QPopupMenu*)sender())->string( id ) );
    newFamily();
}

void FontSelect::pSizeActivated( int id )
{
    int value = atoi( ( (QPopupMenu*)sender())->string( id ) );
    sizeS->blockSignals( TRUE );
    sizeS->setValue( value );
    sizeS->blockSignals( FALSE );
    newSize( value );
}

void FontSelect::updateMetrics()
{
    QFontInfo fi( f );
    metrics[0][1]->setText( fi.family() );
    metrics[1][1]->setNum( fi.pointSize() );
    metrics[2][1]->setNum( fi.weight() );
    metrics[3][1]->setNum( fi.italic() );
}

/*
    Grand unifying widget, putting ShowXForm and XFormControl
    together.
*/

class XFormCenter : public QWidget
{
    Q_OBJECT
public:
    XFormCenter( QWidget *parent=0, const char *name=0 );
public slots:
    void setFont( QFont f) { sx->setFont( f ); }
    void newMode( bool );
private:
    void resizeEvent( QResizeEvent* );
    ShowXForm	*sx;
    XFormControl *xc;
};

void XFormCenter::resizeEvent( QResizeEvent* ) 
{ 
    sx->resize( width() - 120, height() );
    xc->resize( 120, height() ); 
}

void XFormCenter::newMode( bool showPix )
{
    static bool first = TRUE;

    if ( sx->imageMode() == showPix )
	return;
    if ( showPix && first ) {
	first = FALSE;
	QPixmap fet;

	QPixmap pm;
	QImageIO io;
        QString tmp; 
        tmp.resize( 5590 + 1);        // taken from the BMP file size
                                      // aticatacStr is defined in aticatac.cpp
        qmemmove( tmp.data(), aticatacStr, 5590 + 1 );
        QBuffer b( tmp );             
	b.open( IO_ReadOnly );
        io.setIODevice( &b );         // read image from string via buffer
	io.read();
	b.close();
	pm.enableImageCache( TRUE );  // Cache data for faster transform
	pm.convertFromImage( io.image() );
	sx->setPixmap( pm );
	return;
    }
    sx->setPixmapMode( showPix );
}

XFormCenter::XFormCenter( QWidget *parent, const char *name )
    : QWidget( parent, name )
{
    sx = new ShowXForm(this);
    sx->move( 120, 0 );             // the size is set by resizeEvent
    xc = new XFormControl(this);
    xc->move( 0, 0 );               // the size is set by resizeEvent
    xc->setFrameStyle( QFrame::Sunken );
    xc->setLineWidth( 2 );
    connect( xc, SIGNAL(newText(char*)), sx, SLOT(setText(char*)) );
    connect( xc, SIGNAL(newMatrix(Q2DMatrix)), 
	     sx, SLOT(setMatrix(Q2DMatrix)) );
    connect( xc, SIGNAL(newFont(QFont)), sx, SLOT(setFont(QFont)) );
    connect( xc, SIGNAL(newMode(bool)), SLOT(newMode(bool)) );
    sx->setText( "Troll" );
}

int main( int argc, char **argv )
{
    QApplication a( argc, argv );

/*  QColor x( 0xaa, 0xbe, 0xff );	// kind of blue
    QColorGroup g( black, x, x.light(), x.dark(), x.dark(120),	black, white );
    QPalette p( g, g, g );
    a.setPalette( p );
*/
    XFormCenter *xfc = new XFormCenter;
    xfc->setGeometry( 0, 0, 500, 400 );
    xfc->show();

    return a.exec( xfc );
}

#include "xform.moc"                  // include metadata generated by the moc
