// -*- C++ -*-


// GIST -- Graphical Interactive Simulation result Tool

//   CNCL interface classes

//   1992,93 Martin Horneffer

#ifndef gist_impl_h
#define gist_impl_h

#include "gist.h"

class CommImpl {
  protected:
    CommImpl() {};
  public:
    virtual ~CommImpl() {};
    virtual int ok() = 0;
    virtual const char *gist_type() = 0;
};

class GISTWorldImpl {
  private:
    char *name_;
  protected:
    GISTWorldImpl(const char *name);
  public:
    virtual ~GISTWorldImpl();
    virtual const char *name() { return name_; };
    virtual void flush(double time) = 0;
    virtual const char *gist_type() = 0;
};

class GISTObjectImpl {
  private:
    char *name_;
  protected:
    GISTObjectImpl(const char *name);
  public:
    virtual ~GISTObjectImpl();
    GISTAttribute *attributes;
    GISTGraphic *graphics;
    virtual const char *name() { return name_; };
    virtual const char *gist_type() = 0;
};

class GISTAttributeImpl {
  private:
    friend class GISTGraphic;
    char *name_;
    GISTObject *parent_;
  protected:
    GISTAttributeImpl(GISTObject *parent,const char *name);
  public:
    GISTAttribute *next;
    virtual ~GISTAttributeImpl();
    virtual const char *name() { return name_; };
    virtual GISTObject *parent() { return parent_; };
};

class GISTTextAttributeImpl: public GISTAttributeImpl {
  public:
    GISTTextAttributeImpl(GISTObject *parent,const char *name): GISTAttributeImpl(parent,name) {};
    virtual void put(double time, const char *data) = 0;
};

class GISTValueAttributeImpl: public GISTAttributeImpl {
  public:
    GISTValueAttributeImpl(GISTObject *parent,const char *name): GISTAttributeImpl(parent,name) {};
    virtual void put(double time, float data) = 0;
};

class GISTCoordAttributeImpl: public GISTAttributeImpl {
  public:
    GISTCoordAttributeImpl(GISTObject *parent,const char *name): GISTAttributeImpl(parent,name) {};
    virtual void put(double time, float x, float y) = 0;
};

class GISTValueArrayAttributeImpl: public GISTAttributeImpl {
  public:
    GISTValueArrayAttributeImpl(GISTObject *parent,const char *name): GISTAttributeImpl(parent,name) {};
    virtual void put(double time, const float data[]) = 0;
};

class GISTCoordArrayAttributeImpl: public GISTAttributeImpl {
  public:
    float *buf;
    unsigned short bufdim;
    
    GISTCoordArrayAttributeImpl(GISTObject *parent,const char *name,unsigned short dim);
    ~GISTCoordArrayAttributeImpl() { delete [] buf; };
    virtual void put(double time, const float data[]) = 0;
};

class GISTGraphicImpl {
  private:
    friend class GISTGraphic;
    char *name_;
  protected:
    GISTObject *parent_;
    GISTGraphicImpl(GISTObject *parent,const char *name);
  public:
    GISTGraphic *next;
    virtual ~GISTGraphicImpl() { delete [] name_; };
    virtual const char *name() { return name_; };
    virtual GISTObject *parent() { return parent_; };
    virtual const char *color_name(float red, float green, float blue) = 0;
    virtual void hint(const char*) {};
};

class GISTPictureImpl: public GISTGraphicImpl {
  public:
    GISTPictureImpl(GISTObject *parent,const char *name);
    virtual void set_color(const char*) = 0;
    virtual void set_color(GISTTextAttributeImpl*) = 0;
    virtual void set_brush(float thickness, unsigned short pattern) = 0;
    virtual void pen_down() = 0;
    virtual void pen_up() = 0;
    virtual void pen_up_fill() = 0;
    virtual void close_path() = 0;
    virtual void set_pos(float x, float y) = 0;
    virtual void set_pos(GISTCoordAttributeImpl*,int) = 0;
    virtual void set_path(GISTCoordArrayAttributeImpl*,int) = 0;
    virtual void move_rel(float dx, float dy) = 0;
    virtual void move_rel(float dx, GISTValueAttributeImpl*,int) = 0;
    virtual void move_rel(GISTValueAttributeImpl*,float dy,int) = 0;
    virtual void move_rel(GISTValueAttributeImpl*,GISTValueAttributeImpl *,int) = 0;
    virtual void move_dir(float r, GISTValueAttributeImpl*,int) = 0;
    virtual void move_dir(GISTValueAttributeImpl*,float a,int) = 0;
    virtual void move_dir(GISTValueAttributeImpl*,GISTValueAttributeImpl *,int) = 0;
    virtual void polyline(GISTCoordArrayAttributeImpl*,int steadiness) = 0;
    virtual void polygon(GISTCoordArrayAttributeImpl*,int steadiness) = 0;
    virtual void text(const char *text,float size,const char *font,const char *style) = 0;
    virtual void text(GISTTextAttributeImpl *text,float size,const char *font,const char *style) = 0;
    virtual void text(GISTValueAttributeImpl *text,float size,int steadiness,int precision,long flags,const char *font,const char *style) = 0;
    virtual void draw_image(const char *name,float x_align,float y_align) = 0;
    virtual void draw_image(GISTTextAttributeImpl *name,float x_align, float y_align) = 0;
};

#endif
