// -*- C++ -*-


// GIST -- Graphical Interactive Simulation result Tool

//   CNCL interface classes

//   1992,93 Martin Horneffer

#include "gist.h"
#include "gist_impl.hh"
#include "gist_ivimpl.hh"

#include <string.h>
#include <fstream.h>
#include <strstream.h>
#include <stdio.h>

extern "C" int unlink(char *);

#define NILCHK(p) if (!p) { char buf[64]; sprintf(buf,"NIL-pointer passed to GIST-function (%s:%d)",__FILE__,__LINE__); CNCL::fatal(buf); };

Comm::Comm(const char *host,
	   bool abort,
	   unsigned short port,
	   const char *server)
{
    char hostname[64];
    strcpy(hostname, host);
    char fn[64], type[16];
    bool do_magic = FALSE;
    long magic = 0;
    pre_filename(fn);
    ifstream pre(fn);
    if ( pre.good() ) {
	do_magic = TRUE;
	pre >> magic;
	pre >> port;
	pre >> hostname;
	pre >> type;
	unlink(fn);
    };
    if ( strcmp(server,"iv")==0 ) {
	impl_ = new CommIVImpl(host,port,do_magic,magic);
	if ( !impl_->ok() ) { 
	    delete impl_; 
	    impl_=0; 
	    strstream msg;
	    msg <<"no iv-gist available on port "<< port << ends;
	    if (abort) fatal(msg.str());
	};
    } else {
	impl_ = 0;
    };
};
Comm::~Comm() { if (impl_) delete impl_; impl_=0; };


GISTWorld::GISTWorld(Comm& comm, 
		     const char *name,
		     float xmin, float xmax, float ymin, float ymax, 
		     double starttime, Flushmode mode)
{
    CommImpl *comimpl = comm.impl();
    if ( comimpl && (strcmp(comimpl->gist_type(),"iv")==0) ) {
	impl_ = new WorldIVImpl(comimpl,name,xmin,xmax,ymin,ymax,starttime,mode);
    } else {
	impl_ = 0;
    };
};
GISTWorld::~GISTWorld() { if (impl_) delete impl_; impl_=0; };
const char *GISTWorld::name() const { return impl_? impl_->name():0; };
void GISTWorld::Flush(double time) { if(impl_) impl_->flush(time); };


GISTObject::GISTObject(GISTWorld &world, const char *name) {
    GISTWorldImpl *wimpl = world.impl();
    if ( wimpl && (strcmp(wimpl->gist_type(),"iv")==0) ) {
	impl_ = new ObjectIVImpl(wimpl,name);
    } else {
	impl_ = 0;
    };
};
GISTObject::~GISTObject() { 
    if (impl_) {
	while (impl_->attributes) {
	    GISTAttribute *next = impl_->attributes->impl_->next;
	    delete impl_->attributes->impl_;
	    impl_->attributes->impl_ = 0;
	    delete impl_->attributes;
	    impl_->attributes = next;
	};
	while (impl_->graphics) {
	    GISTGraphic *next = impl_->graphics->impl_->next;
	    delete impl_->graphics->impl_;
	    impl_->graphics->impl_ = 0;
	    delete impl_->graphics;
	    impl_->graphics = next;
	};
	delete impl_;
	impl_=0;
    };
};
const char *GISTObject::name() const { return impl_? impl_->name():0; };
GISTAttribute *GISTObject::find_attribute(const char *name) {
    GISTAttribute *a = impl_ ? impl_->attributes :0;
    while ( a && a->impl_ && strcmp(name,a->name()) ) a = a->impl_->next;
    return a; 
};
GISTGraphic *GISTObject::find_graphic(const char *name) {
    GISTGraphic *a = impl_ ? impl_->graphics :0;
    while ( a && a->impl_ && strcmp(name,a->name()) ) a = a->impl_->next;
    return a; 
};


GISTAttribute::GISTAttribute() {};
GISTAttribute::~GISTAttribute() { 
    if (impl_) 
      fatal(NIL,"Illegal destructor call to GISTAttribute ",name()); 
};
const char *GISTAttribute::name() const
{ return impl_ ? impl_->name() : 0; };
GISTObject *GISTAttribute::parent() const
{ return impl_ ? impl_->parent() : 0; };

GISTTextAttribute::GISTTextAttribute(GISTObject *object, const char *name, unsigned short size) {
    GISTObjectImpl *objimpl = object->impl();
    if ( objimpl && (strcmp(objimpl->gist_type(),"iv")==0) ) {
	impl_ = new TextAttributeIVImpl(object,name);
	impl_->next = objimpl->attributes;
	objimpl->attributes = this;
    } else {
	impl_ = 0;
    };
};
GISTTextAttribute::~GISTTextAttribute() {};
void GISTTextAttribute::put(double time, const char *data) {
    if (impl_) ((GISTTextAttributeImpl*)impl_)->put(time,data);
};

GISTValueAttribute::GISTValueAttribute(GISTObject *object, const char *name) {
    GISTObjectImpl *objimpl = object->impl();
    if ( objimpl && (strcmp(objimpl->gist_type(),"iv")==0) ) {
	impl_ = new ValueAttributeIVImpl(object,name);
	impl_->next = objimpl->attributes;
	objimpl->attributes = this;
    } else {
	impl_ = 0;
    };
};
GISTValueAttribute::~GISTValueAttribute() {};
void GISTValueAttribute::put(double time, float data) {
    if (impl_) ((GISTValueAttributeImpl*)impl_)->put(time,data);
};

GISTCoordAttribute::GISTCoordAttribute(GISTObject *object, const char *name) {
    GISTObjectImpl *objimpl = object->impl();
    if ( objimpl && (strcmp(objimpl->gist_type(),"iv")==0) ) {
	impl_ = new CoordAttributeIVImpl(object,name);
	impl_->next = objimpl->attributes;
	objimpl->attributes = this;
    } else {
	impl_ = 0;
    };
};
GISTCoordAttribute::~GISTCoordAttribute() {};
void GISTCoordAttribute::put(double time, CNCoord data) {
    if (impl_) ((GISTCoordAttributeImpl*)impl_)->put(time,data.x,data.y);
};
void GISTCoordAttribute::put(double time, float x, float y) {
    if (impl_) ((GISTCoordAttributeImpl*)impl_)->put(time,x,y);
};

GISTValueArrayAttribute::GISTValueArrayAttribute(GISTObject *object, const char *name, unsigned short dim) {
    GISTObjectImpl *objimpl = object->impl();
    if ( objimpl && (strcmp(objimpl->gist_type(),"iv")==0) ) {
	impl_ = new ValueArrayAttributeIVImpl(object,name,dim);
	impl_->next = objimpl->attributes;
	objimpl->attributes = this;
    } else {
	impl_ = 0;
    };
};
GISTValueArrayAttribute::~GISTValueArrayAttribute() {};
void GISTValueArrayAttribute::put(double time, const float data[]) {
    if (impl_) ((GISTValueArrayAttributeImpl*)impl_)->put(time,data);
};

GISTCoordArrayAttribute::GISTCoordArrayAttribute(GISTObject *object, const char *name, unsigned short dim) {
    GISTObjectImpl *objimpl = object->impl();
    if ( objimpl && (strcmp(objimpl->gist_type(),"iv")==0) ) {
	impl_ = new CoordArrayAttributeIVImpl(object,name,dim);
	impl_->next = objimpl->attributes;
	objimpl->attributes = this;
    } else {
	impl_ = 0;
    };
};
GISTCoordArrayAttribute::~GISTCoordArrayAttribute() {};
void GISTCoordArrayAttribute::put(double time, const CNCoord data[]) {
    if (impl_) {
	GISTCoordArrayAttributeImpl *impl = (GISTCoordArrayAttributeImpl*)impl_;
	unsigned short i;
	float *f = impl->buf;
	for (i=0; i<impl->bufdim; i++) {
	    *f++ = data[i].x;
	    *f++ = data[i].y;
	};
	impl->put(time,impl->buf);
    };
};
void GISTCoordArrayAttribute::put(double time, const float data[]) {
    if (impl_) ((GISTCoordArrayAttributeImpl*)impl_)->put(time,data);
};


GISTGraphic::GISTGraphic() {};
GISTGraphic::~GISTGraphic() { 
    if (impl_) 
      fatal(NIL,"Illegal destructor call to GISTGraphic ",name()); 
};
const char *GISTGraphic::name() const { return impl_? impl_->name_ : 0; };
GISTObject *GISTGraphic::parent() const { return impl_? impl_->parent_ : 0; };
const char *GISTGraphic::color_name(float red, float green, float blue)
{ return impl_ ? impl_->color_name(red,green,blue) : 0; };


GISTPicture::GISTPicture(GISTObject *object,
			 const char *name,
			 int prio,
			 GISTTextAttribute *vis,
			 GISTWorld *world)
{
    GISTObjectImpl *objimpl = object->impl();
    if ( objimpl && (strcmp(objimpl->gist_type(),"iv")==0) ) {
	impl_ = new PictureIVImpl(object, name, prio,
				  vis  ? (GISTTextAttributeImpl*) vis->impl() :0,
				  world? (GISTWorldImpl*) world->impl() :0);
	impl_->next = object->impl()->graphics;
	object->impl()->graphics = this;
    } else {
	impl_ = 0;
    };
};
GISTPicture::~GISTPicture() {};
void GISTPicture::set_color(const char *color) { 
    if (impl_) ((GISTPictureImpl*)impl_)->set_color(color);
};
void GISTPicture::set_color(float red, float green, float blue) { 
    if (impl_) ((GISTPictureImpl*)impl_)->set_color(color_name(red,green,blue));
};
void GISTPicture::set_color(GISTTextAttribute *color) {
    NILCHK(color);
    if (impl_ && color->impl()) 
	((GISTPictureImpl*)impl_)->set_color((GISTTextAttributeImpl*)color->impl());
};
void GISTPicture::set_brush(float thickness, unsigned short pattern) {
    if (impl_) ((GISTPictureImpl*)impl_)->set_brush(thickness, pattern);
};
void GISTPicture::pen_down() { 
    if (impl_) ((GISTPictureImpl*)impl_)->pen_down(); 
};
void GISTPicture::pen_up() { 
    if (impl_) ((GISTPictureImpl*)impl_)->pen_up(); 
};
void GISTPicture::pen_up_fill() { 
    if (impl_) ((GISTPictureImpl*)impl_)->pen_up_fill();
};
void GISTPicture::close_path() { 
    if (impl_) ((GISTPictureImpl*)impl_)->close_path();
};
void GISTPicture::set_pos(CNCoord p) {
    if (impl_) ((GISTPictureImpl*)impl_)->set_pos(p.x, p.y);
};
void GISTPicture::set_pos(float x, float y) {
    if (impl_) ((GISTPictureImpl*)impl_)->set_pos(x, y);
};
void GISTPicture::set_pos(GISTCoordAttribute *attr, int steadiness) {
    NILCHK(attr);
    if (impl_ && attr->impl()) 
	((GISTPictureImpl*)impl_)->set_pos((GISTCoordAttributeImpl*)attr->impl(),steadiness);
};
void GISTPicture::set_path(GISTCoordArrayAttribute *attr, int steadiness) {
    NILCHK(attr);
    if (impl_ && attr->impl()) 
	((GISTPictureImpl*)impl_)->set_path((GISTCoordArrayAttributeImpl*)attr->impl(),steadiness);
};
void GISTPicture::move_rel(float dx, float dy) {
    if (impl_) ((GISTPictureImpl*)impl_)->move_rel(dx,dy);
};
void GISTPicture::move_rel(float dx, GISTValueAttribute *dy, int steadiness) {
    NILCHK(dy);
    if (impl_ && dy->impl()) 
	((GISTPictureImpl*)impl_)->move_rel(dx,(GISTValueAttributeImpl*)dy->impl(),steadiness);
};
void GISTPicture::move_rel(GISTValueAttribute *dx, float dy, int steadiness) {
    NILCHK(dx);
    if (impl_ && dx->impl()) 
	((GISTPictureImpl*)impl_)->move_rel((GISTValueAttributeImpl*)dx->impl(),dy,steadiness);
};
void GISTPicture::move_rel(GISTValueAttribute *dx, GISTValueAttribute *dy, int steadiness) {
    NILCHK(dx);
    NILCHK(dy);
    if (impl_ && dx->impl() && dy->impl()) 
	((GISTPictureImpl*)impl_)->move_rel((GISTValueAttributeImpl*)dx->impl(),(GISTValueAttributeImpl*)dy->impl(),steadiness);
};
void GISTPicture::move_dir(float dx, GISTValueAttribute *dy, int steadiness) {
    NILCHK(dy);
    if (impl_ && dy->impl()) 
	((GISTPictureImpl*)impl_)->move_dir(dx,(GISTValueAttributeImpl*)dy->impl(),steadiness);
};
void GISTPicture::move_dir(GISTValueAttribute *dx, float dy, int steadiness) {
    NILCHK(dx);
    if (impl_ && dx->impl()) 
	((GISTPictureImpl*)impl_)->move_dir((GISTValueAttributeImpl*)dx->impl(),dy,steadiness);
};
void GISTPicture::move_dir(GISTValueAttribute *dx, GISTValueAttribute *dy, int steadiness) {
    NILCHK(dx);
    NILCHK(dy);
    if (impl_ && dx->impl() && dy->impl()) 
	((GISTPictureImpl*)impl_)->move_dir((GISTValueAttributeImpl*)dx->impl(),(GISTValueAttributeImpl*)dy->impl(),steadiness);
};
void GISTPicture::polyline(GISTCoordArrayAttribute *a, int steadiness) {
    NILCHK(a);
    if (impl_ && a->impl())
	((GISTPictureImpl*)impl_)->polyline((GISTCoordArrayAttributeImpl*)a->impl(), steadiness);
};
void GISTPicture::polygon(GISTCoordArrayAttribute *a, int steadiness) {
    NILCHK(a);
    if (impl_ && a->impl())
	((GISTPictureImpl*)impl_)->polygon((GISTCoordArrayAttributeImpl*)a->impl(), steadiness);
};
void GISTPicture::text(const char *text,float size,const char *font,const char *style) {
    if (impl_) ((GISTPictureImpl*)impl_)->text(text,size,font,style);
};
void GISTPicture::text(GISTTextAttribute *text,float size,const char *font,const char *style) {
    NILCHK(text);
    if (impl_ && text->impl()) 
	((GISTPictureImpl*)impl_)->text((GISTTextAttributeImpl*)text->impl(),size,font,style);
};
void GISTPicture::text(GISTValueAttribute *text,float size,int steadiness,int precision,long flags,const char *font,const char *style) {
    NILCHK(text);
    if (impl_ && text->impl()) 
	((GISTPictureImpl*)impl_)->text((GISTValueAttributeImpl*)text->impl(),size,steadiness,precision,flags,font,style);
};
void GISTPicture::draw_image(const char *imagename,float x_align,float y_align) {
    if (impl_) ((GISTPictureImpl*)impl_)->draw_image(imagename,x_align,y_align);
};
void GISTPicture::draw_image(GISTTextAttribute *imagename,float x_align,float y_align) {
    NILCHK(imagename);
    if (impl_ && imagename->impl()) 
	((GISTPictureImpl*)impl_)->draw_image((GISTTextAttributeImpl*)imagename->impl(),x_align,y_align);
};

GISTImage::GISTImage(GISTObject *object, 
		     const char *name,	    
		     const char *image, 
		     CNCoord position,
		     float width, float height,
		     float hcenter, float vcenter,
		     int priority,
		     GISTTextAttribute *vis, 
		     GISTWorld *world,
		     float minsize)
{
    GISTObjectImpl *objimpl = object->impl();
    if ( objimpl && (strcmp(objimpl->gist_type(),"iv")==0) ) {
	impl_ = new ImageIVImpl(object,name,image,position.x,position.y,
				width,height,hcenter,vcenter,priority,
				vis  ? (GISTTextAttributeImpl*) vis->impl() :0,
				world? (GISTWorldImpl*) world->impl() :0,
				minsize);
	impl_->next = object->impl()->graphics;
	object->impl()->graphics = this;
    } else {
	impl_ = 0;
    };
};
GISTImage::GISTImage(GISTObject *object, 
		     const char *name,	    
		     const char *image, 
		     float x, float y,
		     float width, float height,
		     float hcenter, float vcenter,
		     int priority,
		     GISTTextAttribute *vis, 
		     GISTWorld *world,
		     float minsize)
{
    GISTObjectImpl *objimpl = object->impl();
    if ( objimpl && (strcmp(objimpl->gist_type(),"iv")==0) ) {
	impl_ = new ImageIVImpl(object,name,image,x,y,
				width,height,hcenter,vcenter,priority,
				vis  ? (GISTTextAttributeImpl*) vis->impl() :0,
				world? (GISTWorldImpl*) world->impl() :0,
				minsize);
	impl_->next = object->impl()->graphics;
	object->impl()->graphics = this;
    } else {
	impl_ = 0;
    };
};
GISTImage::GISTImage(GISTObject *object, 
		     const char *name,	    
		     const char *image, 
		     GISTCoordAttribute *position,
		     float width, float height,
		     float hcenter, float vcenter,
		     int priority,
		     GISTTextAttribute *vis, 
		     GISTWorld *world,
		     int steadiness,
		     float minsize)
{
    GISTObjectImpl *objimpl = object->impl();
    NILCHK(position);
    if ( objimpl && (strcmp(objimpl->gist_type(),"iv")==0) ) {
	impl_ = new ImageIVImpl(object,name,image,
				((GISTCoordAttributeImpl*)position->impl()),
				width,height,hcenter,vcenter,priority,
				vis  ? (GISTTextAttributeImpl*) vis->impl() :0,
				world? (GISTWorldImpl*) world->impl() :0,
				steadiness, minsize);
	impl_->next = object->impl()->graphics;
	object->impl()->graphics = this;
    } else {
	impl_ = 0;
    };
};
GISTImage::GISTImage(GISTObject *object, 
		     const char *name,	    
		     GISTTextAttribute *image, 
		     CNCoord position,
		     float width, float height,
		     float hcenter, float vcenter,
		     int priority,
		     GISTTextAttribute *vis, 
		     GISTWorld *world,
		     float minsize)
{
    GISTObjectImpl *objimpl = object->impl();
    NILCHK(image);
    if ( objimpl && (strcmp(objimpl->gist_type(),"iv")==0) ) {
	impl_ = new ImageIVImpl(object,name,
				(GISTTextAttributeImpl*)image->impl(),
				position.x,position.y,
				width,height,hcenter,vcenter,priority,
				vis  ? (GISTTextAttributeImpl*) vis->impl() :0,
				world? (GISTWorldImpl*) world->impl() :0,
				minsize);
	impl_->next = object->impl()->graphics;
	object->impl()->graphics = this;
    } else {
	impl_ = 0;
    };
};
GISTImage::GISTImage(GISTObject *object, 
		     const char *name,	    
		     GISTTextAttribute *image, 
		     float x, float y,
		     float width, float height,
		     float hcenter, float vcenter,
		     int priority,
		     GISTTextAttribute *vis, 
		     GISTWorld *world,
		     float minsize)
{
    GISTObjectImpl *objimpl = object->impl();
    NILCHK(image);
    if ( objimpl && (strcmp(objimpl->gist_type(),"iv")==0) ) {
	impl_ = new ImageIVImpl(object,name,
				(GISTTextAttributeImpl*)image->impl(),
				x,y,width,height,hcenter,vcenter,priority,
				vis  ? (GISTTextAttributeImpl*) vis->impl() :0,
				world? (GISTWorldImpl*) world->impl() :0,
				minsize);
	impl_->next = object->impl()->graphics;
	object->impl()->graphics = this;
    } else {
	impl_ = 0;
    };
};
GISTImage::GISTImage(GISTObject *object, 
		     const char *name,	    
		     GISTTextAttribute *image, 
		     GISTCoordAttribute *position,
		     float width, float height,
		     float hcenter, float vcenter,
		     int priority,
		     GISTTextAttribute *vis, 
		     GISTWorld *world,
		     int steadiness,
		     float minsize)
{
    GISTObjectImpl *objimpl = object->impl();
    NILCHK(image);
    NILCHK(position);
    if ( objimpl && (strcmp(objimpl->gist_type(),"iv")==0) ) {
	impl_ = new ImageIVImpl(object,name,
				(GISTTextAttributeImpl*)image->impl(),
				(GISTCoordAttributeImpl*)position->impl(),
				width,height,hcenter,vcenter,priority,
				vis  ? (GISTTextAttributeImpl*) vis->impl() :0,
				world? (GISTWorldImpl*) world->impl() :0,
				steadiness, minsize);
	impl_->next = object->impl()->graphics;
	object->impl()->graphics = this;
    } else {
	impl_ = 0;
    };
};
GISTImage::~GISTImage() {};
 
GISTCurve::GISTCurve(GISTObject *object,
		     const char *name, 
		     GISTValueAttribute *value,
		     CNCoord position,
		     Type type,
		     float width, float height,
		     float hcenter, float vcenter,
		     const char *foreground,
		     const char *background,
		     int priority,
		     GISTTextAttribute *vis,
		     GISTWorld *world,
		     float minsize)
{
    GISTObjectImpl *objimpl = object->impl();
    NILCHK(value);
    if ( objimpl && (strcmp(objimpl->gist_type(),"iv")==0) ) {
	impl_ = new CurveIVImpl(object, name,
				(GISTValueAttributeImpl*)value->impl(),
				position.x, position.y, type,
				width, height, hcenter, vcenter,
				foreground, background, priority,
				vis  ? (GISTTextAttributeImpl*) vis->impl() :0,
				world? (GISTWorldImpl*) world->impl() :0,
				minsize);
	impl_->next = object->impl()->graphics;
	object->impl()->graphics = this;
    } else {
	impl_ = 0;
    };
};
GISTCurve::GISTCurve(GISTObject *object,
		     const char *name, 
		     GISTValueAttribute *value,
		     float x, float y,
		     Type type,
		     float width, float height,
		     float hcenter, float vcenter,
		     const char *foreground,
		     const char *background,
		     int priority,
		     GISTTextAttribute *vis,
		     GISTWorld *world,
		     float minsize)
{
    GISTObjectImpl *objimpl = object->impl();
    NILCHK(value);
    if ( objimpl && (strcmp(objimpl->gist_type(),"iv")==0) ) {
	impl_ = new CurveIVImpl(object, name,
				(GISTValueAttributeImpl*)value->impl(),
				x, y, type,
				width, height, hcenter, vcenter,
				foreground, background, priority,
				vis  ? (GISTTextAttributeImpl*) vis->impl() :0,
				world? (GISTWorldImpl*) world->impl() :0,
				minsize);
	impl_->next = object->impl()->graphics;
	object->impl()->graphics = this;
    } else {
	impl_ = 0;
    };
};
GISTCurve::~GISTCurve() {};
void GISTCurve::hint(const char *text) {
    if (impl_) impl_->hint(text);
};

GISTDiagram::GISTDiagram(GISTObject *object,
			 const char *name, 
			 GISTCoordArrayAttribute *valuearray,
			 CNCoord position,
			 Type type,
			 float width, float height,
			 float hcenter, float vcenter,
			 const char *foreg,
			 const char *backg,
			 int priority,
			 GISTTextAttribute *vis,
			 GISTWorld *world,
			 float minsize)
{
    GISTObjectImpl *objimpl = object->impl();
    NILCHK(valuearray);
    if ( objimpl && (strcmp(objimpl->gist_type(),"iv")==0) ) {
	impl_ = new DiagramIVImpl(object, name,
				  (GISTCoordArrayAttributeImpl*)valuearray->impl(),
				  position.x, position.y, type,
				  width, height, hcenter, vcenter,
				  foreg, backg, priority,
				  vis  ? (GISTTextAttributeImpl*) vis->impl() :0,
				  world? (GISTWorldImpl*) world->impl() :0,
				  minsize);
	impl_->next = object->impl()->graphics;
	object->impl()->graphics = this;
    } else {
	impl_ = 0;
    };
};
GISTDiagram::GISTDiagram(GISTObject *object,
			 const char *name, 
			 GISTCoordArrayAttribute *valuearray,
			 float x, float y,
			 Type type,
			 float width, float height,
			 float hcenter, float vcenter,
			 const char *foreg,
			 const char *backg,
			 int priority,
			 GISTTextAttribute *vis,
			 GISTWorld *world,
			 float minsize)
{
    GISTObjectImpl *objimpl = object->impl();
    NILCHK(valuearray);
    if ( objimpl && (strcmp(objimpl->gist_type(),"iv")==0) ) {
	impl_ = new DiagramIVImpl(object, name,
				  (GISTCoordArrayAttributeImpl*)valuearray->impl(),
				  x, y, type,
				  width, height, hcenter, vcenter,
				  foreg, backg, priority,
				  vis  ? (GISTTextAttributeImpl*) vis->impl() :0,
				  world? (GISTWorldImpl*) world->impl() :0,
				  minsize);
	impl_->next = object->impl()->graphics;
	object->impl()->graphics = this;
    } else {
	impl_ = 0;
    };
};
GISTDiagram::~GISTDiagram() {};
void GISTDiagram::hint(const char *text) {
    if (impl_) impl_->hint(text);
};




//  CNCL

static CNClass Comm_desc("Comm", "$Revision: 0.27 $",Comm::new_object);
CNClassDesc CN_COMM = &Comm_desc;

CNObject *Comm::new_object(CNParam *param) { return NIL; };
void Comm::print(ostream &strm) const {
    strm << "..." << endl;
};
void Comm::dump(ostream &strm) const {
    strm << "Comm { $Revision: 0.27 $ ..." << " }" << endl;
};
Comm *Comm::cast_from_object(CNObject *obj) {
    if(!test_flag(type_check)) return (Comm *)obj;
    if(obj->is_a(CN_COMM)) return (Comm *)obj;
    else {
	fatal(NIL,"Invalid type cast: ",obj->class_desc()->name()," -> Comm");
	return NIL;
    };
};

static CNClass GISTWorld_desc("GISTWorld", "$Revision: 0.27 $",GISTWorld::new_object);
CNClassDesc CN_GISTWORLD = &GISTWorld_desc;

CNObject *GISTWorld::new_object(CNParam *param) { return NIL; };
void GISTWorld::print(ostream &strm) const {
    strm << "..." << endl;
};
void GISTWorld::dump(ostream &strm) const {
    strm << "GISTWorld { $Revision: 0.27 $ ..." << " }" << endl;
};
GISTWorld *GISTWorld::cast_from_object(CNObject *obj) {
    if(!test_flag(type_check)) return (GISTWorld *)obj;
    if(obj->is_a(CN_GISTWORLD)) return (GISTWorld *)obj;
    else {
	fatal(NIL,"Invalid type cast: ",obj->class_desc()->name()," -> GISTWorld");
	return NIL;
    };
};

static CNClass GISTObject_desc("GISTObject","$Revision: 0.27 $",GISTObject::new_object);
CNClassDesc CN_GISTOBJECT = &GISTObject_desc;

CNObject *GISTObject::new_object(CNParam *param) { return NIL; };
void GISTObject::print(ostream &strm) const {
    strm << "..." << endl;
};
void GISTObject::dump(ostream &strm) const {
    strm << "GISTObject { $Revision: 0.27 $ ..." << " }" << endl;
};
GISTObject *GISTObject::cast_from_object(CNObject *obj) {
    if(!test_flag(type_check)) return (GISTObject *)obj;
    if(obj->is_a(CN_GISTOBJECT)) return (GISTObject *)obj;
    else {
	fatal(NIL,"Invalid type cast: ",obj->class_desc()->name()," -> GISTObject");
	return NIL;
    };
};

static CNClass GISTAttribute_desc("GISTAttribute", "$Revision: 0.27 $",GISTAttribute::new_object);
CNClassDesc CN_GISTATTRIBUTE = &GISTAttribute_desc;

CNObject *GISTAttribute::new_object(CNParam *param) { return NIL; };
void GISTAttribute::print(ostream &strm) const {
    strm << "..." << endl;
};
void GISTAttribute::dump(ostream &strm) const {
    strm << "GISTAttribute { $Revision: 0.27 $ ..." << " }" << endl;
};
GISTAttribute *GISTAttribute::cast_from_object(CNObject *obj) {
    if(!test_flag(type_check)) return (GISTAttribute *)obj;
    if(obj->is_a(CN_GISTATTRIBUTE)) return (GISTAttribute *)obj;
    else {
	fatal(NIL,"Invalid type cast: ",obj->class_desc()->name()," -> GISTAttribute");
	return NIL;
    };
};

static CNClass GISTTextAttribute_desc("GISTTextAttribute","$Revision: 0.27 $",GISTTextAttribute::new_object);
CNClassDesc CN_GISTTEXTATTRIBUTE = &GISTTextAttribute_desc;

CNObject *GISTTextAttribute::new_object(CNParam *param) { return NIL; };
void GISTTextAttribute::print(ostream &strm) const {
    strm << "..." << endl;
};
void GISTTextAttribute::dump(ostream &strm) const {
    strm << "GISTTextAttribute { $Revision: 0.27 $ ..." << " }" << endl;
};
GISTTextAttribute *GISTTextAttribute::cast_from_object(CNObject *obj) {
    if(!test_flag(type_check)) return (GISTTextAttribute *)obj;
    if(obj->is_a(CN_GISTTEXTATTRIBUTE)) return (GISTTextAttribute *)obj;
    else {
	fatal(NIL,"Invalid type cast: ",obj->class_desc()->name()," -> GISTTextAttribute");
	return NIL;
    };
};

static CNClass GISTValueAttribute_desc("GISTValueAttribute", "$Revision: 0.27 $",GISTValueAttribute::new_object);
CNClassDesc CN_GISTVALUEATTRIBUTE = &GISTValueAttribute_desc;

CNObject *GISTValueAttribute::new_object(CNParam *param) { return NIL; };
void GISTValueAttribute::print(ostream &strm) const {
    strm << "..." << endl;
};
void GISTValueAttribute::dump(ostream &strm) const {
    strm << "GISTValueAttribute { $Revision: 0.27 $ ..." << " }" << endl;
};
GISTValueAttribute *GISTValueAttribute::cast_from_object(CNObject *obj) {
    if(!test_flag(type_check)) return (GISTValueAttribute *)obj;
    if(obj->is_a(CN_GISTVALUEATTRIBUTE)) return (GISTValueAttribute *)obj;
    else {
	fatal(NIL,"Invalid type cast: ",obj->class_desc()->name()," -> GISTValueAttribute");
	return NIL;
    };
};

static CNClass GISTCoordAttribute_desc("GISTCoordAttribute", "$Revision: 0.27 $",GISTCoordAttribute::new_object);
CNClassDesc CN_GISTCOORDATTRIBUTE = &GISTCoordAttribute_desc;

CNObject *GISTCoordAttribute::new_object(CNParam *param) { return NIL; };
void GISTCoordAttribute::print(ostream &strm) const {
    strm << "..." << endl;
};
void GISTCoordAttribute::dump(ostream &strm) const {
    strm << "GISTCoordAttribute { $Revision: 0.27 $ ..." << " }" << endl;
};
GISTCoordAttribute *GISTCoordAttribute::cast_from_object(CNObject *obj) {
    if(!test_flag(type_check)) return (GISTCoordAttribute *)obj;
    if(obj->is_a(CN_GISTCOORDATTRIBUTE)) return (GISTCoordAttribute *)obj;
    else {
	fatal(NIL,"Invalid type cast: ",obj->class_desc()->name()," -> GISTCoordAttribute");
	return NIL;
    };
};

static CNClass GISTValueArrayAttribute_desc("GISTValueArrayAttribute", "$Revision: 0.27 $",GISTValueArrayAttribute::new_object);
CNClassDesc CN_GISTVALUEARRAYATTRIBUTE = &GISTValueArrayAttribute_desc;

CNObject *GISTValueArrayAttribute::new_object(CNParam *param) { return NIL; };
void GISTValueArrayAttribute::print(ostream &strm) const {
    strm << "..." << endl;
};
void GISTValueArrayAttribute::dump(ostream &strm) const {
    strm << "GISTValueArrayAttribute { $Revision: 0.27 $ ..." << " }" << endl;
};
GISTValueArrayAttribute *GISTValueArrayAttribute::cast_from_object(CNObject *obj) {
    if(!test_flag(type_check)) return (GISTValueArrayAttribute *)obj;
    if(obj->is_a(CN_GISTVALUEARRAYATTRIBUTE)) return (GISTValueArrayAttribute *)obj;
    else {
	fatal(NIL,"Invalid type cast: ",obj->class_desc()->name()," -> GISTValueArrayAttribute");
	return NIL;
    };
};

static CNClass GISTCoordArrayAttribute_desc("GISTCoordArrayAttribute", "$Revision: 0.27 $",GISTCoordArrayAttribute::new_object);
CNClassDesc CN_GISTCOORDARRAYATTRIBUTE = &GISTCoordArrayAttribute_desc;

CNObject *GISTCoordArrayAttribute::new_object(CNParam *param) { return NIL; };
void GISTCoordArrayAttribute::print(ostream &strm) const {
    strm << "..." << endl;
};
void GISTCoordArrayAttribute::dump(ostream &strm) const {
    strm << "GISTCoordArrayAttribute { $Revision: 0.27 $ ..." << " }" << endl;
};
GISTCoordArrayAttribute *GISTCoordArrayAttribute::cast_from_object(CNObject *obj) {
    if(!test_flag(type_check)) return (GISTCoordArrayAttribute *)obj;
    if(obj->is_a(CN_GISTCOORDARRAYATTRIBUTE)) return (GISTCoordArrayAttribute *)obj;
    else {
	fatal(NIL,"Invalid type cast: ",obj->class_desc()->name()," -> GISTCoordArrayAttribute");
	return NIL;
    };
};

static CNClass GISTGraphic_desc("GISTGraphic", "$Revision: 0.27 $",GISTGraphic::new_object);
CNClassDesc CN_GISTGRAPHIC = &GISTGraphic_desc;

CNObject *GISTGraphic::new_object(CNParam *param) { return NIL; };
void GISTGraphic::print(ostream &strm) const {
    strm << "..." << endl;
};
void GISTGraphic::dump(ostream &strm) const {
    strm << "GISTGraphic { $Revision: 0.27 $ ..." << " }" << endl;
};
GISTGraphic *GISTGraphic::cast_from_object(CNObject *obj) {
    if(!test_flag(type_check)) return (GISTGraphic *)obj;
    if(obj->is_a(CN_GISTGRAPHIC)) return (GISTGraphic *)obj;
    else {
	fatal(NIL,"Invalid type cast: ",obj->class_desc()->name()," -> GISTGraphic");
	return NIL;
    };
};

static CNClass GISTPicture_desc("GISTPicture", "$Revision: 0.27 $",GISTPicture::new_object);
CNClassDesc CN_GISTPICTURE = &GISTPicture_desc;

CNObject *GISTPicture::new_object(CNParam *param) { return NIL; };
void GISTPicture::print(ostream &strm) const {
    strm << "..." << endl;
};
void GISTPicture::dump(ostream &strm) const {
    strm << "GISTPicture { $Revision: 0.27 $ ..." << " }" << endl;
};
GISTPicture *GISTPicture::cast_from_object(CNObject *obj) {
    if(!test_flag(type_check)) return (GISTPicture *)obj;
    if(obj->is_a(CN_GISTPICTURE)) return (GISTPicture *)obj;
    else {
	fatal(NIL,"Invalid type cast: ",obj->class_desc()->name()," -> GISTPicture");
	return NIL;
    };
};

static CNClass GISTImage_desc("GISTImage", "$Revision: 0.27 $",GISTImage::new_object);
CNClassDesc CN_GISTIMAGE = &GISTImage_desc;

CNObject *GISTImage::new_object(CNParam *param) { return NIL; };
void GISTImage::print(ostream &strm) const {
    strm << "..." << endl;
};
void GISTImage::dump(ostream &strm) const {
    strm << "GISTImage { $Revision: 0.27 $ ..." << " }" << endl;
};
GISTImage *GISTImage::cast_from_object(CNObject *obj) {
    if(!test_flag(type_check)) return (GISTImage *)obj;
    if(obj->is_a(CN_GISTIMAGE)) return (GISTImage *)obj;
    else {
	fatal(NIL,"Invalid type cast: ",obj->class_desc()->name()," -> GISTImage");
	return NIL;
    };
};

static CNClass GISTCurve_desc("GISTCurve", "$Revision: 0.27 $",GISTCurve::new_object);
CNClassDesc CN_GISTCURVE = &GISTCurve_desc;

CNObject *GISTCurve::new_object(CNParam *param) { return NIL; };
void GISTCurve::print(ostream &strm) const {
    strm << "..." << endl;
};
void GISTCurve::dump(ostream &strm) const {
    strm << "GISTCurve { $Revision: 0.27 $ ..." << " }" << endl;
};
GISTCurve *GISTCurve::cast_from_object(CNObject *obj) {
    if(!test_flag(type_check)) return (GISTCurve *)obj;
    if(obj->is_a(CN_GISTCURVE)) return (GISTCurve *)obj;
    else {
	fatal(NIL,"Invalid type cast: ",obj->class_desc()->name()," -> GISTCurve");
	return NIL;
    };
};

static CNClass GISTDiagram_desc("GISTDiagram", "$Revision: 0.27 $",GISTDiagram::new_object);
CNClassDesc CN_GISTDIAGRAM = &GISTDiagram_desc;

CNObject *GISTDiagram::new_object(CNParam *param) { return NIL; };
void GISTDiagram::print(ostream &strm) const {
    strm << "..." << endl;
};
void GISTDiagram::dump(ostream &strm) const {
    strm << "GISTDiagram { $Revision: 0.27 $ ..." << " }" << endl;
};
GISTDiagram *GISTDiagram::cast_from_object(CNObject *obj) {
    if(!test_flag(type_check)) return (GISTDiagram *)obj;
    if(obj->is_a(CN_GISTDIAGRAM)) return (GISTDiagram *)obj;
    else {
	fatal(NIL,"Invalid type cast: ",obj->class_desc()->name()," -> GISTDiagram");
	return NIL;
    };
};
