//*BHEADER* :ts=8  -*- C++ -*-
/*****************************************************************************
 *
 *   |_|_|_  |_|_    |_    |_|_|_  |_		     C O M M U N I C A T I O N
 * |_        |_  |_  |_  |_        |_		               N E T W O R K S
 * |_        |_  |_  |_  |_        |_		                     C L A S S
 *   |_|_|_  |_    |_|_    |_|_|_  |_|_|_|_	                 L I B R A R Y
 *
 * $Id$
 *
 * Class: CNFNumTriangle --- Fuzzy number (triangle)
 *
 *****************************************************************************
 * Copyright (C) 1992-1995   Communication Networks
 *                           Aachen University of Technology
 *                           D-52056 Aachen
 *                           Germany
 *                           Email: cncl-adm@dfv.rwth-aachen.de
 *****************************************************************************
 * This file is part of the CN class library. All files marked with
 * this header are free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.  This library is
 * distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.  You should have received a copy of the GNU
 * Library General Public License along with this library; if not, write
 * to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 **EHEADER********************************************************************/

#include "FNumTriangle.h"



CNFNumTriangle::CNFNumTriangle(const CNFNumTriangle &n)
    : CNFSetTriangle(n.m1, n.alpha, n.beta)
{}



CNFNumTriangle &CNFNumTriangle::operator = (const CNFNumTriangle &n)
{
    m1 = m2 = n.m1;
    alpha   = n.alpha;
    beta    = n.beta;
    return *this;
}



double CNFNumTriangle::center_of_gravity(double min, double max) const
{
    return m1 + (beta - alpha)/3 ;
}



double CNFNumTriangle::defuzzy() const
{
    return center_of_gravity(0, 0);
}



CNFNumTriangle abs(CNFNumTriangle a) 
{
    if (a.m1 < 0.0)
	return -a;
    else
	return a;
}



/*
 * Unary -
 */
CNFNumTriangle CNFNumTriangle::operator - ()
{
    return CNFNumTriangle(-m1, beta, alpha);
}


/*
 * Unary +
 */
CNFNumTriangle CNFNumTriangle::operator + ()
{
    return *this;
}


/*
 * Binary * with double
 */
CNFNumTriangle operator * (double c, CNFNumTriangle a)
{
    if (c >= 0.0)
	return CNFNumTriangle(c*a.m1, c*a.alpha, c*a.beta);
    else
	return CNFNumTriangle(c*a.m1, -c*a.beta, -c*a.alpha);

}

CNFNumTriangle operator * (CNFNumTriangle a, double c)
{
    if (c >= 0.0)
	return CNFNumTriangle(c*a.m1, c*a.alpha, c*a.beta);
    else
	return CNFNumTriangle(c*a.m1, -c*a.beta, -c*a.alpha);

}


/*
 * Binary +
 */
CNFNumTriangle operator + (CNFNumTriangle a, CNFNumTriangle b)
{
    return CNFNumTriangle(a.m1+b.m1, a.alpha+b.alpha, a.beta+b.beta);
}


/*
 * Binary -
 */
CNFNumTriangle operator - (CNFNumTriangle a, CNFNumTriangle b)
{
    return CNFNumTriangle(a.m1-b.m1, a.alpha+b.beta, a.beta+b.alpha);
}



/*
 * Square c*c
 */
CNFNumTriangle sqr(CNFNumTriangle c)
{
    CNFNumTriangle d;
 
    if(c.m1 < 0.0)
    {
	double h;
	c.m1    = -c.m1;
	h       = c.alpha;
	c.alpha = c.beta;
	c.beta  = h;
    }
    
    if (c.alpha>c.m1)
    {
	d.m1  = 0.0;
	d.alpha = 0.0;
	if (c.alpha>c.beta+2.0*c.m1)
	    d.beta = (c.alpha-c.m1)*(c.alpha-c.m1);
	else
	    d.beta = (c.m1+c.beta) * (c.m1+c.beta);
    }
    else
	if ((c.alpha<=0.0)&&(c.beta<=0.0))
	{
	    d.m1  = c.m1 * c.m1;
	    d.alpha = 0.0;
	    d.beta  = 0.0;
	}
	else
	{
	    d.m1 = c.m1 * (c.alpha+c.beta) * (c.m1-c.alpha) * (c.m1+c.beta)
		/ (c.beta * (c.m1+c.beta) + c.alpha * (c.m1-c.alpha));
	    d.alpha = d.m1 - (c.m1-c.alpha) * (c.m1-c.alpha);
	    d.beta  = (c.m1+c.beta) * (c.m1+c.beta) - d.m1;
	}
    
    return(d);
}



/***** Default I/O member function for CNCL classes **************************/

// Normal output
void CNFNumTriangle::print(ostream &strm) const
{
    strm << "m=" << m1 << " a=" << alpha << " b=" << beta
	 << "  defuzzy=" << defuzzy() << endl;
}

// Debug output
void CNFNumTriangle::dump(ostream &strm) const
{
    strm << "CNFNumTriangle { $Revision$  " << this << " }" << endl;
}



/***** CNCL stuff for type information ***************************************/

// Describing object for class CNFNumTriangle
static CNClass CNFNumTriangle_desc("CNFNumTriangle", "$Revision$",
				 CNFNumTriangle::new_object);

// "Type" for type checking functions
CNClassDesc CN_FNUMTRIANGLE = &CNFNumTriangle_desc;
