// -*- C++ -*-

#include <iostream.h>

#include <CNCL/QueueSPT.h>
#include <CNCL/EventScheduler.h>
#include <CNCL/FiboG.h>
#include <CNCL/NegExp.h>
#include <CNCL/Moments.h>

#include <CNCL/Job.h>


enum { NJOBS=100000 };

enum { JOB, TIMER_G, TIMER_S };


class Server : public CNEventHandler
{
private:
    CNJob *job;		// Served job
    CNEventID job_id;	// identification
    CNQueueSPT queue;	// CNQueue
    CNMoments t_w, t_b;	// Evaluation tau_w, tau_b
    enum { WAITING, SERVING };
    
public:
    virtual void event_handler(const CNEvent *ev);

    void print_results();
    void eval_job(CNJob *job);

    Server() : job(NIL), queue(), t_w("tau_w"), t_b("tau_b")
    {
	state(WAITING);
    }
};



class Generator : public CNEventHandler
{
private:
    CNRandom &rnd_a;	// Distribution of arrival time a
    CNRandom &rnd_b;    // Distribution of service time b    
    Server *server;	// Connected queue/server
    long n;
    
public:
    virtual void event_handler(const CNEvent *ev);

    Generator(CNRandom &rnd, CNRandom &rnd2, Server *serv) : rnd_a(rnd), 
    rnd_b(rnd2), server(serv), n(0) {}
};



void Generator::event_handler(const CNEvent *ev)
{
    if(n == NJOBS)
	// Stop simulation
	return;

#ifdef DEBUG
    cout << "----- Generator -----" << endl;
    ev->dump(cout);
#endif    

    // Incoming event -> generate new Job
    send_now(new CNEvent(JOB, server, new CNJob(rnd_b())));
    // CNRandom delay
    send_delay(new CNEvent(TIMER_G), rnd_a());
    n++;
}



void Server::event_handler(const CNEvent *ev)
{
#ifdef DEBUG
    cout << "----- Server -----" << endl;
    ev->dump(cout);
#endif    

    switch(state())
    {
    case SERVING:
	switch(ev->type())
	{
	case JOB:
	    // Incoming job
 	    CNJob *new_job;
    	    new_job = (CNJob *)ev->object();
    	    new_job->in = now();
	    // if new job is longer than remaining length of actual job
	    // put it into queue, otherwise exchange jobs
	    if (new_job->length >= (job->length - now() + job->start))
	        queue.put(new_job);
	    else
	    {
		job->length -= (now() - job->start);
		queue.put(job);
		scheduler()->delete_event(job_id);
		job = new_job;
        job->start = now();
		job_id = send_delay(new CNEvent(TIMER_S), job->length);
	    }
	    break;
	    
	case TIMER_S:
	    // Timer event, service time run down
	    job->out = now();
	    // Evaluate job
	    eval_job(job);
	    delete job;
	    job = NIL;
	    // Get new job from queue
	    if(!queue.empty())
	    {
		job = (CNJob *)queue.get();
		job->start = now();
		// CNRandom service time
		job_id = send_delay(new CNEvent(TIMER_S), job->length);
		state(SERVING);
	    }
	    else
		state(WAITING);
	    break;
	    
	default:
	    error("mm1: ", "illegal event in state SERVING");
	    break;
	}

	break;
	
    case WAITING:
	switch(ev->type())
	{
	case JOB:
	    // Incoming job
	    job = (CNJob *)ev->object();
	    job->in    = now();
	    job->start = now();
	    // CNRandom service time
	    job_id = send_delay(new CNEvent(TIMER_S), job->length);
	    state(SERVING);
	    break;

	default:
	    error("mm1: ", "illegal event in state WAITING");
	    break;
	}
	
	break;
	
    default:
	error("mm1: ", "illegal state");
	break;
	
    }
}



void Server::eval_job(CNJob *job)
{
    t_w.put(job->out - job->orig_length - job->in);
    t_b.put(job->out - job->in);
}


void Server::print_results()
{
    cout << t_w << t_b;
}

 

main()
{
    CNRNG    *rng   = new CNFiboG;
    CNNegExp rnd_a(10, rng);
    CNNegExp rnd_b( 5, rng);

    Server         server;
    Generator      generator(rnd_a, rnd_b, &server);
    CNEventScheduler scheduler;
    
    scheduler.start(new CNEvent(TIMER_G, &generator, 0.0));

    server.print_results();
    
}
