/*
 * $RCSfile: loadscanobjs.c,v $
 * $Revision: 1.10 $
 * $Date: 1993/04/27 22:22:39 $
 *	This example program loads a file with objects and
 *	then scans the file.  The program takes one parameter, the
 *	number of objects to create.
 */


#include "sm_client.h"
#include <stdlib.h>

#ifndef __cplusplus
	extern char		*getenv(char*);
#endif


/* 
 * 'clientvolume' is the name of the file containing the client's
 *  disk volume (created with the 'formatvol' utility).
 */
#define  VOL    getenv("EVOLID") 
VOLID    volid;  /* volume ID */

/*
 *	Store the buffer group and transaction information in
 *	these global variables.
 */
int        bufGroup;     /* buffer group index */
TID        tid;          /* transaction ID */

/*
 * Each dummy object consists of only an integer-valued field.
 */
typedef struct { 
    int          value;        /* the object's value */
} DUMMYOBJ;


/*
 *	Declare the error handling function
 */
int ErrorCheck (int, char *);


/* 
 * This example routine creates a file and then creates 100 dummy data 
 * objects in the file. The ID of the file is stored as a root entry
 * in the volume and is used by the routine Scan( )
 * to determine which file to scan.
 */
void
Load (int numObjs) {
    int        e;            /* error return */
    int        i;            /* loop index */
    FID        fid;          /* file ID */
    OID        oid;          /* object ID */
    OBJHDR     objHdr;       /* object header */
    DUMMYOBJ   dummyObj;     /* dummy object */

    /*
     * Create the file.
     */
    e = sm_CreateFile(bufGroup, volid, &fid);
    ErrorCheck(e, "sm_CreateFile");
    
    /*
     * Fill the file with 100 objects.
     */
    for (i = 0; i < numObjs; i++) {
        dummyObj.value = i;
        e = sm_CreateObject(bufGroup, &fid, NEAR_LAST, NULL,
                &objHdr, sizeof(DUMMYOBJ), (char*) &dummyObj, &oid);
        ErrorCheck(e, "sm_CreateObject");
    }

    /*
     * Remember the file's ID in a root entry so Scan( )
     * can find it.
     */
    e = sm_SetRootEntry(volid, "fid", (char*)&fid, sizeof(FID));
    ErrorCheck(e, "sm_SetRootEntry");
}


/* 
 * This routine scans and prints the file created with 
 * the Load( ) routine.
 */
 void
Scan ( )
{
    int        e;             /* error return */
    int        i;             /* index */
    FID        fid;           /* file ID */
    BOOL       eof;           /* end of file flag */
    USERDESC   *userDesc;     /* user descriptor */
    DUMMYOBJ   **dummyObj;    /* pointer to dummy object */
    int        rootDataSize;  /* size of fid from root entry */
	SCANDESC   *scanDesc;	  /* scan descriptor */	

    /*
     * Get the file's ID from the root page entry.
     */
    rootDataSize = sizeof(fid);
    e = sm_GetRootEntry(volid, "fid", &fid, &rootDataSize);
    ErrorCheck(e, "sm_GetRootEntry");
    if (rootDataSize != sizeof(fid)) {
        fprintf(stderr, "Error: Bad root entry\n");
        exit(1);
    }

	/*
	 *	Start-up the scan 
	 */
	e = sm_OpenScan(&fid, 0, 10/*page buf group*/, &scanDesc, NULL);
    ErrorCheck(e, "sm_OpenScan");
	e = sm_ScanNextObject(scanDesc, 0, READ_ALL, &userDesc, &eof);
    ErrorCheck(e, "sm_ScanNextObject");

    for (i = 0; eof == FALSE; i++) {

        /* 
         * Cast an object pointer onto the basePtr of
         * the user descriptor.
         *
         * CAUTION: Double indirection via a user descriptor
         * should always be used to access objects in the 
         * buffer pool! The storage manager may move objects 
         * in the buffer pool, and therefore double indirection 
         * is always necessary.
         */
        dummyObj = (DUMMYOBJ **) &userDesc->basePtr;

        /*
         * Print out the object's value
         */
        printf("object[%d].value = %d\n", i, (*dummyObj)->value);

        /*
         * Get the next object.
         */
		e = sm_ScanNextObject(scanDesc, 0, READ_ALL, &userDesc, &eof);
        ErrorCheck(e, "sm_ScanNextObject");
    }
}


/*
 * ErrorCheck( ) is a simple routine for checking
 * Storage Manager return codes.
 */
 int
ErrorCheck (
	int    e,              /* error return code */
	char   *routine        /* the routine that caused an error */
)
{
    if (e < 0) {
        fprintf(stderr, "Got a Storage Manager error from %s\n", routine);
        fprintf(stderr, "Error = %s\n", sm_Error(sm_errno));
		if (sm_errno == esmTRANSABORTED) {
			fprintf(stderr, "Transaction  aborted by server due to: %s\n", sm_Error(sm_reason));
		}
        exit(1);
    }
	return(0);
}


/*
 *	Main program:
 */
main(int argc, char** argv)
{

	int		objCount;
	int		e;
	FILE	*errorStream;
	char	*errorMsg;
    int		rootDataSize;  /* size of fid from root entry */
	FID		fid;

	/*
	 *  Read the default configuration files to have the bufpages
	 *  option set.
	 */
	e = sm_ReadConfigFile(NULL, argv[0], &errorMsg);
	if (e != esmNOERROR) {
		fprintf(stderr, "Configuration file error: %s\n", errorMsg);
		ErrorCheck(e, "sm_ReadConfigFile");
		exit(0);
	}

	/*
	 *	Set any options from the command line
	 */
	e = sm_ParseCommandLine(&argc, argv, &errorMsg);
	if (e != esmNOERROR) {
		fprintf(stderr, "command line error: %s\n", errorMsg);
		ErrorCheck(e, "sm_PargeCommandLine");
		exit(0);
	}

    /* check for the proper number of arguments */
    if (argc !=2 ) {
        fprintf(stderr,"usage: %s count\n", argv[0]);
        exit(-1);
    }
    /* get the parameters of the test */
    objCount = atoi(argv[1]);

    /*
     * Initialize the storage manager.
     */
    e = sm_Initialize();
    ErrorCheck(e, "sm_Initialize");
	tid = 0;

	/*
	 *	Send all error output to /dev/null since any "real errors" 
	 *	will be seen in an sm_* call.
	 */
	errorStream = fopen("/dev/null", "a");
	if (errorStream == 0) {
		perror("Cannot open /dev/null");
	}
	sm_ErrorStream = errorStream;

    /*
     * Get the ID of the volume to use
     */
	if (VOL == NULL) {
		fprintf(stderr, "Error: you must setenv EVOLID.\n");
		exit(1);
	}
	volid = atoi(VOL);

    /*
     * Open an LRU buffer group of size 100.
     */
    e = sm_OpenBufferGroup(100, BF_LRU, &bufGroup, NOFLAGS);
    ErrorCheck(e, "sm_OpenBufferGroup");
    
    /*
     * Begin the transaction.
     */
    e = sm_BeginTransaction(&tid);
    ErrorCheck(e, "sm_BeginTransaction");

    /*
     * If a file exists from a previous Load() call, then destroy it.
     */
    rootDataSize = sizeof(fid);
    e = sm_GetRootEntry(volid, "fid", &fid, &rootDataSize);
	if (e == esmNOERROR) {
		if (rootDataSize != sizeof(fid)) {
			fprintf(stderr, "Error: Bad root entry\n");
			exit(1);
		}
		e = sm_DestroyFile(bufGroup, &fid);
		ErrorCheck(e, "sm_DestroyFile");
	} else if (sm_errno == esmBADROOTNAME) {
		/* no previous file */
	} else {
		ErrorCheck(e, "sm_GetRootEntry");
	}

	/*
	 *	Load some objects and scan them
	 */
	printf("Loading %d objects... \n", objCount);
	Load(objCount);
	printf("Scanning %d objects... \n", objCount);
	Scan();

    e = sm_CommitTransaction(tid);
    ErrorCheck(e, "sm_EndTransaction");

    e = sm_CloseBufferGroup(bufGroup);
    ErrorCheck(e, "sm_CloseBufferGroup");
    
	e = sm_ShutDown( );
	ErrorCheck(e, "sm_ShutDown");
}

