#ifndef __PAGEHASH_H__
#define __PAGEHASH_H__

/*
 *   $RCSfile: PAGEHASH.h,v $  
 *   $Revision: 1.1 $  
 *   $Date: 1993/04/21 17:20:46 $      
 */ 

/**********************************************************************
* EXODUS Database Toolkit Software
* Copyright (c) 1991 Computer Sciences Department, University of
*                    Wisconsin -- Madison
* All Rights Reserved.
*
* Permission to use, copy, modify and distribute this software and its
* documentation is hereby granted, provided that both the copyright
* notice and this permission notice appear in all copies of the
* software, derivative works or modified versions, and any portions
* thereof, and that both notices appear in supporting documentation.
*
* THE COMPUTER SCIENCES DEPARTMENT OF THE UNIVERSITY OF WISCONSIN --
* MADISON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION.  
* THE DEPARTMENT DISCLAIMS ANY LIABILITY OF ANY KIND FOR ANY DAMAGES
* WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
*
* The EXODUS Project Group requests users of this software to return 
* any improvements or extensions that they make to:
*
*   EXODUS Project Group 
*     c/o David J. DeWitt and Michael J. Carey
*   Computer Sciences Department
*   University of Wisconsin -- Madison
*   Madison, WI 53706
*
*	 or exodus@cs.wisc.edu
*
* In addition, the EXODUS Project Group requests that users grant the 
* Computer Sciences Department rights to redistribute these changes.
**********************************************************************/
#include "resources.h"
#include "error.h"
#include "io.h"
#include "object.h"
#include "tid.h"
#include "bf_macro.h"
#include "trace.h"

class PAGEHASH {
public:

	BOOL 		Dump(FILE *f);
	PID 		*keylocation (int unique, PID **unused);
	LISTELEMENT *listlocation (int unique);
	void 		ReInit();
	void 		Init();

#ifdef SERVER_MAKE
	/* stuff specific to SERVER side */

	/*
	 *	This structure is used to represent a page in the buffer pool
	 *	There is one of these per page that is resident
	 */
	UONE		flags;			/* flags for this page				*/
#define H_FREE			0x0		/* The page hash structure is free	*/
#define H_DIRTY			0x1		/* The page is dirty				*/
#define H_LOG			0x2		/* The page is a log page			*/
#define H_LRC_REPAIR	0x4		/* The page having its lrc repaired.
								   In this case the page has firstLRC
								   and firstLSN's set, but it is not
								   marked H_DIRTY					*/
#define H_INDEX			0x5
#define H_TEMP			0x6		/* diff meaning fromn cli side */
#define H_NEW			0x7		/* used on client side only */
#define H_INVALID   	0x0   /* USED ONLY ON CLIENT SIDE */

	UONE		pageType;		/* the type of page					*/
								/* 	values are PAGE_xxx	io.h		*/
#define PPAGE_TYPE(ph) ((ph)->pageType)
#define PSET_PAGE_TYPE(ph,typ) ((ph)->pageType = (typ))

	UONE		blockCount;		/* number of blocks in this page	*/
    UTWO		bufIndex;		/* index of page in BufTable		*/
	char		*bufFrame;		/* pointer to the buffer frame		*/
#define PBUFFRAME(ph) ((ph)->bufFrame)
#define BUFFRAME(ph) ((ph).bufFrame)

	BUFINFO		*bufInfo;		/* pointer to buf info entry		*/
	SEMAPHORE	semaphore;
	LATCH		latch;
	LISTELEMENT	clientDirtyList;/* page received from client		*/
	int			pendingOpCount;	/* number of pending operations		*/
	LSN			lsn;			/* LSN of first log record to dirty */

#else
	/* stuff specific to CLIENT side */
	/*
	 * If the hash entry is for a page that belongs to a chunk, then the 
	 * entry will have a non-NULL chunkDesc field, pointing to the CHUNKDESC
	 * structure that the page is a part of.
	 * 
	 * If the hash entry is for a slotted page, then the entry
	 * will have a non-NULL pageInfo field, pointing to the PAGEINFO 
	 * structure associated with the page.
	 * Hanging off the PAGEINFO structure are page user descriptors, which
	 * are used much like chunk user descriptors, except they are used to
	 * reference small objects within the page.
	 */

	/*
	 *  Page type is embedded in flags on client side for historic reasons
	 *  see client's bf.h
	 */
    UTWO   		flags;          /* type of structure & flags(e.g., dirty)   */
/*
 *  Values for PAGEFLAGS type (flags in the PAGEHASH struct)
 *  Page type is embedded in flags for historic reasons... mucho code
 *  checks flags to get the page type.
 *  Henceforth code should check page type by using the macro
 *  PAGE_TYPE(pageHash), which inspects only the lower byte of flags.
 *  Some day, when all the code is converted, we can remove the
 *  types to a u_char of their own.
 */
#define H_FREE      PAGE_GENERIC	/* The page hash structure is free  */
#define H_SLOTTED   PAGE_SLOTTED 	/* a slotted (small object) page    */
#define H_LARGE     PAGE_LARGENODE	/* a control page of a large object */
#define H_CHUNK     PAGE_LARGEDATA	/* a data page of a large object    */
#define H_LOG       PAGE_LOG		/* The page is a log page           */
#define H_INDEX     PAGE_INDEX		/* btree index page                 */
#define H_INDEXDESC	PAGE_INDEXDESC	/* an index descriptor page         */
#define H_DIRTY     0x100           /* The page is dirty                */
#define H_INVALID   0x200           /* The page is here but perhaps out of
										date*/
#define H_TEMP   	0x400           /* temporarily put in hash table; 
									a kludge on server side; 
									page on temp volume on client side  */
#define H_NEW       0x800           /* The page is newly allocated - used
									 * on the client side only */

#define PPAGE_TYPE(ph) ((ph)->flags & 0xff)
#define PSET_PAGE_TYPE(ph,typ) ((ph)->flags = (typ))

    PAGEINFO    *pageInfo;      /* ptr to page's info                       */
#define PBUFFRAME(ph) ((ph)->pageInfo->bufFrame)
#define BUFFRAME(ph) ((ph).pageInfo->bufFrame)

    LISTELEMENT dirtyList;      /* list of dirty pages in the transaction   */
    TID         readLock;
    TID         writeLock;

#define IS_VALID(__phash) \
((__phash->readLock!=NULL_TID)&&(LIST_MEMBER(&(__phash->hashList.hashList))))

#define IS_INVALID(__phash) \
((__phash->readLock==NULL_TID)||(!LIST_MEMBER(&(__phash->hashList.hashList))))

    /* added for client operation logging - MJF */
    LSN			firstLSN; 	/* approximate firstLSN for the page	    */
    LRC			firstLRC; 	/* the LRC for the first opeation applied
																to the page	    */
	int			releaseCnt;
	int			releaseLimit;

#endif


	/* stuff common to SERVER and CLIENT */
	LIST		memberList;		/* list of chunks or groups			*/
    PAGEHASHQ   hashList;       /* hash queue list element                  */
    PAGE2SIZE   page2size;      /* size of the page in blocks               */
    FORCEMARK   forceMark;
    FORCEMARK   firstMark;
    TWO         memberCount;    /* number of chunks or groups               */
	LRC			lrc;			/* LRC of first log record to dirty */
    MAGIC       magic;          /* magic number for verification            */
/*
 *  define magic number
 */
#define PAGEHASH_MAGIC      0x76129ad5

}; /* end class PAGEHASH */

#endif __PAGEHASH_H__

#ifdef __BF_PageHash_C__

#ifndef __BF_PageHash_C__Defined
#define __BF_PageHash_C__Defined

BOOL 
PAGEHASH:: Dump(FILE *f) {
	fprintf(f,  "DUMP OF PAGEHASH 0x%x\n", this);
	fprintf(f, 
		"PAGEHASH flags 0x%x, page2size %d, magic 0x%x\n",
		this->flags, this->page2size, this->magic);
	fprintf(f, "\t pageType %d, lrc %d.%d pid %d.%d\n",
		PPAGE_TYPE(this), 
		this->lrc.wrapCount, this->lrc.count,
		this->hashList.hashPid.page,
		this->hashList.hashPid.volid);
	return FALSE;
} 

PID *
PAGEHASH:: keylocation (int unique, PID **unused) {
	switch(unique) {
	case PAGEHASH_TABLE:
		return &(hashList.hashPid); 
	default:
		TRPRINT(TR_BF, TR_LEVEL_1, ("bad unique value: %d\n", unique));
		SM_ERROR(TYPE_FATAL, esmINTERNAL);
		/* to prevent the compiler from complaining 
			about no return value: */
		return &(hashList.hashPid); 
	}
}

LISTELEMENT *
PAGEHASH:: listlocation (int unique) {
	switch (unique) {
	case PAGEHASH_TABLE:
	case PAGEHASH_POOL:
		return &(hashList.hashList); 
	case DIRTY_LIST:
#ifdef SERVER_MAKE
		return &(clientDirtyList); 
#else SERVER_MAKE
		return &(dirtyList); 
#endif SERVER_MAKE
	case MEMBER_LIST:
		return &(memberList); 
	default:
		TRPRINT(TR_BF, TR_LEVEL_1, ("bad unique value: %d\n", unique));
		SM_ERROR(TYPE_FATAL, esmINTERNAL);
		/* to prevent the compiler from 
		complaining about no return value: */
		return &(hashList.hashList); 
	}
}

void 
PAGEHASH:: ReInit() {
	this->flags = H_FREE;
	this->page2size = MIN_PAGE2SIZE;
	PSET_PAGE_TYPE(this, PAGE_GENERIC);
	this->lrc = LrcNone;
	this->hashList.hashPid = PidNone;
	this->memberCount = 0;
#ifndef SERVER_MAKE
	this->readLock = 
		this->writeLock = NULL_TID;
	this->releaseCnt = 
		this->releaseLimit = 0;
#endif SERVER_MAKE
}

void 
PAGEHASH:: Init() {
	initializeListElement(this->listlocation(PAGEHASH_POOL),
										(char *)this);
	initializeListElement(this->listlocation(DIRTY_LIST), 
										(char *)this);
	initializeList(this->listlocation(MEMBER_LIST));
	INIT_PAGEHASH_MAGIC( this );

#ifdef SERVER_MAKE
	initializeLatch( &(this->latch) );
	initializeSemaphore( &(this->semaphore), FALSE );
#endif SERVER_MAKE

	this->ReInit();
}
#endif __BF_PageHash_C__Defined
#endif __BF_PageHash_C__

