#ifndef __LHINDPAGE_H__
#define __LHINDPAGE_H__

/*
 *	$RCSfile: LHINDPAGE.h,v $
 *	$Revision: 1.8 $
 *	$Date: 1993/05/10 15:28:52 $
 */
/**********************************************************************
* EXODUS Database Toolkit Software
* Copyright (c) 1991 Computer Sciences Department, University of
*                    Wisconsin -- Madison
* All Rights Reserved.
*
* Permission to use, copy, modify and distribute this software and its
* documentation is hereby granted, provided that both the copyright
* notice and this permission notice appear in all copies of the
* software, derivative works or modified versions, and any portions
* thereof, and that both notices appear in supporting documentation.
*
* THE COMPUTER SCIENCES DEPARTMENT OF THE UNIVERSITY OF WISCONSIN --
* MADISON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION.  
* THE DEPARTMENT DISCLAIMS ANY LIABILITY OF ANY KIND FOR ANY DAMAGES
* WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
*
* The EXODUS Project Group requests users of this software to return 
* any improvements or extensions that they make to:
*
*   EXODUS Project Group 
*     c/o David J. DeWitt and Michael J. Carey
*   Computer Sciences Department
*   University of Wisconsin -- Madison
*   Madison, WI 53706
*
*	 or exodus@cs.wisc.edu
*
* In addition, the EXODUS Project Group requests that users grant the 
* Computer Sciences Department rights to redistribute these changes.
**********************************************************************/
#include "lh_extfuncs.h"  

/*
 *	Size of a LHINDPAGE defined in object.h
 */

#ifdef __cplusplus			/* LHINDPAGE defs for C++ only */

//
//	LH_TRACE macro                   
//
#ifdef LH_DEBUG
#define LH_TRACE(x)	printf x
#define BT_TRACE(x)	printf x
#else
#define LH_TRACE(x)
#endif

enum LH_OPER  {
	LH_READ = 1,
	LH_INSERT = 2,
	LH_DELETE = 4,
	LH_INSERT_DELETE = 6
	};


// 
//	LHINDEX page control structure 
//
struct LHINDCONTROL { 
	LRC			lrc;			// for recovery (MUST BE 1st FIELD (cheated))
	PID			selfID;			// id of this page
	FOUR        selfInd;        // index value of this page 
	TWO         endData;        // offset to the end of the data section
	TWO         numOffsets;     // number of offsets
	TWO         numFree;        // number of free bytes
	ONE         selfOv;			// flag indicating this page is overflow bucket
	ONE         nextOv;			// flag indicating next chained overflow bucket
	SHORTPID    ovbId;          // id of next overflow bucket
	float       load;			// load of current index page
	ONE         type;           // SMDATATYPE
	};



const LHIND_MAGIC = 14021967;   // change

//
// 	LHINDEX page definition
//
const LH_DATASIZE = BTREE_PAGESIZE - sizeof(LHINDCONTROL);

class LHINDPAGE {
	
	// 
	//	Physical undos and redos must access page directly
	//
	friend void undoLHash_fr(LOGRECORDHDR*);
	friend void redoLHash_fr(LOGRECORDHDR*);

		LHINDCONTROL	lhCtrl;						// control information
		char			data[LH_DATASIZE - sizeof(SLOT)];
		SLOT			slot[1];					// first slot on page

		//
		//	Compress to minimize fragmentation
		//
		void		Compress();

	public:
		//
		//	General queries
		//
		PID&		SelfID()		{ return lhCtrl.selfID; }
		int			Volume()		{ return lhCtrl.selfID.volid; }
		int			NumSlots()		{ return lhCtrl.numOffsets; }
		SMDATATYPE	KeyType()		{ return (SMDATATYPE) lhCtrl.type; }
		FOUR        SelfInd()		{ return lhCtrl.selfInd; }
		int			OvLabel()		{ return lhCtrl.selfOv; }
		int			HaveOv()		{ return lhCtrl.nextOv; }
		SHORTPID	OvPage()		{ return lhCtrl.ovbId; }
		float		Load()			{ return lhCtrl.load; }

		//
		//	set overflow flag
		//
		void		SetPageOverflow(SHORTPID& ovPid)	{
						lhCtrl.nextOv = TRUE;
						lhCtrl.ovbId = ovPid;
						}
		//
		//	reset overflow flag
		//
		void		ResetPageOverflow(int keepFlag)	{
						lhCtrl.nextOv = FALSE;
						if (!keepFlag)
							lhCtrl.ovbId = NULLPID;
						}
		
		//
		//	Space usage 
		//
		int			UsableSpace()	{ 
						return LH_DATASIZE - lhCtrl.endData 
								- (lhCtrl.numOffsets * sizeof(slot[0]));
						}
		int			UsedSpace()		{ return LH_DATASIZE - lhCtrl.numFree; }
		int			FreeSpace()		{ return lhCtrl.numFree; }

		//
		//	LRC 
		//
		LRC&		lrc()			{ return lhCtrl.lrc; }

		//
		//	Return TRUE if this page is safe
		//
		int			IsSafe(LH_OPER oper, int maxKeyLen);

		//
		//	Return a reference to tuple in slot sNum
		//
		BT_Tuple&	Tuple(int sNum);

		//
		//	Add/Remove an element to/from slot sNum
		//
        int         AddElem(int sNum, void* const elem);
		int         RemoveElem(int sNum, void* const elem);


		//
		//	Search for keyVal and, if found, return the slot in retSlot.
		//	Returns TRUE if found, FALSE otherwise.
		//
		int 		Search(int keyLen, void* keyVal, 
										PFC compFunc, TWO& retSlot);

		//
		//	Insert/Remove a tuple into/from slot "slotNum"
		//
		void		InsertTuple(BT_Tuple& const tuple, int slotNum);
		void		RemoveTuple(int slotNum);

		//
		//	Move numTuples starting at slotNum to destSlot in destPage
		//
		void		ShiftTuple(int slotNum, LHINDPAGE* destPage, 
										int destSlot, int numTuples);

		//
		//	Set tuple to overflow (reclaim space used by oidList)
		//
		void		SetOverflow(int slotNum, PID& const ovPid);
		
		//
		//	Set tuple to underflow (reinsert oidList)
		//
        void        SetUnderflow(int slotNum, int oidCnt, void* const elList);

		//
		//	Copy data and data usage indicators to target page
		//
		void 		CopyData(LHINDPAGE* target)	{
								bcopy(data, target->data, LH_DATASIZE);
								target->lhCtrl.endData = lhCtrl.endData;
								target->lhCtrl.numOffsets = lhCtrl.numOffsets;
								target->lhCtrl.numFree = lhCtrl.numFree;
								target->lhCtrl.ovbId = lhCtrl.ovbId;
								target->lhCtrl.load = lhCtrl.load;
								}

		//
		//	Initialize a new page
		//
		void 		Init(PID& const pid, FOUR hashval, SMDATATYPE keyType,
									int ovFlag) {
								lhCtrl.endData = lhCtrl.numOffsets = 0;
								lhCtrl.numFree = LH_DATASIZE;
								lhCtrl.selfID = pid;
								lhCtrl.selfInd = hashval;
								lhCtrl.type = keyType;
								lhCtrl.ovbId = NULLPID;
								lhCtrl.nextOv = FALSE;
								lhCtrl.selfOv = ovFlag;
								lhCtrl.load = 0.0;
								}

		//
		//	calculate current load of this page
		//
		float 		LoadComp();

		//
		//	set current load of this page
		//
		void 		SetLoad(float load) {
							lhCtrl.load = load;
							}
		
		//
		//	convert overlow pages
		//
		void 	OvConvert(FOUR hashVal, BOOL convert) {
							lhCtrl.selfInd = hashVal;
							lhCtrl.selfOv = !convert;
							}
		//
		//	Debugging utilities
		//
		int  		CheckContent();
		void		Print(BUFGROUP* bufGroup);
	};

#if (defined DEBUG && defined lhash_MODULE) || ! defined DEBUG

INLINE BT_Tuple& LHINDPAGE::Tuple(int sNum)
{
    return *((BT_Tuple*)(data+slot[-sNum]));
}

#endif /* (DEBUG && lhash_MODULE) || !DEBUG */



class LHashStack;
class LH_SMOPages;


extern "C"  {
	extern int LH_AllocDirPage(TID tid, int volume, BUFGROUP* bufGroup,
							PID& returnPid, int rootFlag, SMDATATYPE keyType);
	extern int LH_AllocPage(TID tid, int volume, BUFGROUP* bufGroup,
							PID& returnPid, FOUR hashVal, SMDATATYPE keyType,
							int ovFlag);
	extern int LH_DirSearch(PID& const rootPid, int keyLen, void* keyVal,
							LHashStack& dirPl, SHORTPID& indPid, LH_OPER oper);
	extern int LH_DirSplit(TID tid, int volume, BUFGROUP* bufGroup,
							LHashStack& dirPl, LH_SMOPages& smo,
							PageDesc& dirPd, SHORTPID& const indPid);
	extern int LH_DirSplitUpdate(TID tid, int volume, BUFGROUP* bufGroup,
							LHashStack& dirPl, SHORTPID& const indPid,
							LH_SMOPages& smo, float oldFromLoad,
							float fromLoad, float toLoad);
	extern int LH_IndSearch(PID& const indPid, int keyLen, void* keyVal, 
							PageDesc& curPd, LHashStack& curPl, int& found,
							TWO& returnSlot, LH_OPER oper, PFC compFunc);
	extern int LH_SplitPage(TID tid, LHashStack& dirPl, BUFGROUP* bufGroup, 
							LH_SMOPages& smo, LHashStack& fromPl, 
							LHashStack& toPl, PFC CompFunc);
	extern int LH_InsertNewTuple(TID tid, BUFGROUP* bufGroup, LHashStack& curPl,
							PageDesc& curPd, TWO& slot, BT_Tuple& const tuple,
							PFC compFunc);
	extern int LH_NewElem(TID tid, BUFGROUP* bufGroup, PID& const dirPid,
							LHashStack& curPl, int keyLen, void* keyVal, 
							TWO elSize, void* const newElem, 
							int maxKeyLen, PFC compFunc);
	extern int LH_DecrElCnt(TID tid, PID& const rootPid, BUFGROUP* bufGroup,
							void* key, int keyLen,
							int maxKeyLen, PFC compFunc);
    extern int LH_IncrElCnt(TID tid, PID& const rootPid, BUFGROUP* bufGroup,
						   void* key, int keyLen,
						   int maxKeyLen, PFC compFunc);
	extern int LH_SetOverflow(TID tid, PID& const rootPid,
							BUFGROUP* bufGroup,
						   	void* key, int keyLen,
						   	int maxKeyLen, PFC compFunc,
						   	void* const elList, int elSize, int numEl);
	extern int LH_ResetOverflow(TID tid, PID& const rootPid,
							BUFGROUP* bufGroup,
							void* key, int keyLen,
							int maxKeyLen, PFC compFunc,
						   	void* const elList, int numOid);

	extern void LH_LogDirPageInit(PAGEHASH* pHash, int rootFlag,
									SMDATATYPE keyType);
	extern void LH_LogIndexPageInit(PAGEHASH* pHash, SMDATATYPE keyType,
							FOUR hashVal, int ovFlag);
	extern void LH_LogLogicInsert(PID& pid, PAGEHASH* pHash, LRC& lrc,
							PID& const rootPid, void* const key, 
							TWO keyLen, void* const el, TWO elSize, 
							TWO maxKeyLen, TWO unique, SMDATATYPE keyType,
							BOOL primary);
	extern void LH_LogLogicDelete(PID& pid, PAGEHASH* pHash, LRC& lrc,
							PID& const rootPid, void* const key, 
							TWO keyLen, void* const el, TWO elSize, 
							TWO maxKeyLen, TWO unique, SMDATATYPE keyType,
							BOOL primary);
	extern void LH_LogSetIndexOverflow(PID& pid, PAGEHASH* pHash, LRC& lrc,
							SHORTPID newOvPage, 
							SHORTPID oldOvPage);
	extern void LH_LogResetIndexOverflow(PID& pid, PAGEHASH* pHash, LRC& lrc,
							SHORTPID oldOvPage, int keepFlag);
	extern void LH_LogDirUpdate(PID& pid, PAGEHASH* pHash, LRC& lrc,
							 float oldFromLoad, float fromLoad, 
							 float toLoad, BOOL growFlag);
	extern void LH_LogDirSetNextEntry(PID& pid, PAGEHASH* pHash, LRC& lrc,
							 SHORTPID newPage, FOUR hashVal);
	extern void LH_LogDirRootCtrl(PID& pid, PAGEHASH* pHash, LRC& lrc, 
							 int undoFlag);
	extern void LH_LogDirSplitCopy(PID& pid, PAGEHASH* pHash, LRC& lrc,
							 int height, int curPos);
	extern void LH_LogDirSplitReset(PID& pid, PAGEHASH* pHash, LRC& lrc,
							 SHORTPID page1, SHORTPID page2,
							 LHDIRPAGE* newp);
	extern void LH_LogLogicIncrElCnt(PID& pid, PAGEHASH* pHash, LRC& lrc,
				        	 PID& const rootPid, void* const key,
							 TWO keyLen, TWO maxKeyLen, SMDATATYPE keyType);
	extern void LH_LogLogicDecrElCnt(PID& pid, PAGEHASH* pHash, LRC& lrc,
				       		 PID& const rootPid, void* const key,
							 TWO keyLen, TWO maxKeyLen, SMDATATYPE keyType);
	extern void LH_LogLogicSetOverflow(PID& pid, PAGEHASH* pHash, LRC& lrc,
				PID& const rootPid, void* const key, TWO keyLen, TWO maxKeyLen,
				SMDATATYPE keyType, int numEl, int elSize, void* const elList,
				PID& const ovPid);
	extern void LH_LogLogicResetOverflow(PID& pid, PAGEHASH* pHash, LRC& lrc,
				PID& const rootPid, void* const key, TWO keyLen, TWO maxKeyLen,
				SMDATATYPE keyType, int numEl, int elSize, void* const elList,
				PID& const ovPid);
	extern void LH_LogOvConvert(PID& pid, PAGEHASH* pHash, LRC& lrc,
				FOUR oldHashVal, FOUR newHashVal, BOOL cvtFlag);
	extern void LH_LogSetThreshold(PID& pid, PAGEHASH* pHash, LRC& lrc,
							 float oldThreshold, float newThreshold);

#ifdef SERVER_MAKE
	extern void LH_LogPureCLR(LSNOFFSET);
#endif
}


#ifdef SERVER_MAKE
	extern void LH_LogPhysicInsert(PID&, PAGEHASH*, LRC&,
						TWO, TWO, int, void*);
	extern void LH_LogPhysicDelete(PID&, PAGEHASH*, LRC&, 
						TWO, TWO, int, void*);
	extern void LH_LogPhyLogInsert(PID&, PAGEHASH*, LRC&,
						TWO, int, void*);
	extern void LH_LogPhyLogDelete(PID&, PAGEHASH*, LRC&, 
						TWO, int, void*);
#endif
	extern void LH_LogPhysicInsert(PID& pid, PAGEHASH* pHash, LRC& lrc,
									TWO slotNum, TWO numSlot);
	extern void LH_LogPhyLogInsert(PID& pid, PAGEHASH* pHash, LHINDPAGE* indp,
									LRC& lrc, TWO numSlot);
	extern void LH_LogPhysicDelete(PID& pid, PAGEHASH* pHash, LRC& lrc,
									TWO slotNum, TWO numSlot);
	extern void LH_LogPhyLogDelete(PID& pid, PAGEHASH* pHash, LHINDPAGE* indp,
									LRC& lrc, TWO numSlot);
	extern void LH_LogDirLoadUpdate(PID& pid, PAGEHASH* pHash, LRC& lrc,
							 float load1, float load2);
	extern void LH_LogDirLoadUpdate(PID&, PAGEHASH*, LRC&,
						float);
#endif		/* #ifdef C++ */

#endif __LHINDPAGE_H__


