#ifndef _HASHTABLE_C_
#define _HASHTABLE_C_
/*
 *   $RCSfile: HashTable.c,v $  
 *   $Revision: 1.3 $  
 *   $Date: 1993/04/21 17:21:46 $      
 */ 
/**********************************************************************
* EXODUS Database Toolkit Software
* Copyright (c) 1991 Computer Sciences Department, University of
*                    Wisconsin -- Madison
* All Rights Reserved.
*
* Permission to use, copy, modify and distribute this software and its
* documentation is hereby granted, provided that both the copyright
* notice and this permission notice appear in all copies of the
* software, derivative works or modified versions, and any portions
* thereof, and that both notices appear in supporting documentation.
*
* THE COMPUTER SCIENCES DEPARTMENT OF THE UNIVERSITY OF WISCONSIN --
* MADISON ALLOWS FREE USE OF THIS SOFTWARE IN ITS "AS IS" CONDITION.  
* THE DEPARTMENT DISCLAIMS ANY LIABILITY OF ANY KIND FOR ANY DAMAGES
* WHATSOEVER RESULTING FROM THE USE OF THIS SOFTWARE.
*
* The EXODUS Project Group requests users of this software to return 
* any improvements or extensions that they make to:
*
*   EXODUS Project Group 
*     c/o David J. DeWitt and Michael J. Carey
*   Computer Sciences Department
*   University of Wisconsin -- Madison
*   Madison, WI 53706
*
*	 or exodus@cs.wisc.edu
*
* In addition, the EXODUS Project Group requests that users grant the 
* Computer Sciences Department rights to redistribute these changes.
**********************************************************************/


#ifndef  __GNUC__
#include "sysdefs.h"
#include "ess.h"
#include "checking.h"
#include "list.h"
#include "trace.h"
#include "error.h"
#include "HashTable.h"
#endif  __GNUC__

#define HASH(key) (key->hashFunc() & this->mask)

template < class HASHTKEY,class CONTENTS > 
void
HashTable<HASHTKEY,CONTENTS>::Stats( 
	FILE *f 
)
{
	fprintf(f, "HashTable %s, mask 0x%x, %d buckets", name, mask, numBuckets);
	fprintf(f, "\t%d items, %d buckets are empty\n",
		 numItems,  numEmptyBuckets);
	fprintf(f, "\tbucket size (highwater)=%d\n", bucketSizeHighWater);
}

template < class HASHTKEY,class CONTENTS > 
HashTable<HASHTKEY,CONTENTS>::HashTable (
	char *name, 
	int which,
	int tablesize
)

{
	register int		i;
	register unsigned	numbytes;

	TRACE(TR_BF, TR_LEVEL_1);
	numbytes = tablesize * sizeof(LIST);

	if ((buckets = (LIST *) malloc(numbytes)) == NULL)	{
		SM_ERROR(TYPE_FATAL, esmMALLOCFAILED);
	}
	this->spaceRequirement = numbytes;

	this->numEmptyBuckets = this->numBuckets = tablesize;
	this->mask = (unsigned) (tablesize - 1);
	this->bucketSizeHighWater = this->numItems = 0;

	(void) bzero( this->name, sizeof(this->name));
	{ 	int length;
		length = strlen(name);
		if(length >= sizeof(this->name)) 
			length = sizeof(this->name)-1;

		(void) strncpy(this->name, name, length);
	}

	this->unique = which;

	for (i = 0; i < tablesize; i++)	{
		initializeList( &(this->buckets[i]) );
	}
}

template < class HASHTKEY,class CONTENTS > 
HashTable<HASHTKEY,CONTENTS>::~HashTable()
{
	LISTELEMENT			*listelement;
	register int		slot;

	TRACE(TR_BF, TR_LEVEL_1);
	/* 
	 * kind of like ForEach, except that we
	 * remove the listelement from the table as we go.
	 */

	for (slot = 0; slot < this->numBuckets; slot++ ) {
		if(LIST_EMPTY(&(this->buckets[slot])) )
			continue;

		TRPRINT(TR_BF, TR_LEVEL_1, ("~HashTable %s: slot %d not empty", 
			this->name, slot));

		LISTDEQ( &(this->buckets[slot]), listelement );
		while ( listelement != NULL) {
			LISTDEQ( &(this->buckets[slot]), listelement );
		}
	}
	free(this->buckets);
}

template < class HASHTKEY,class CONTENTS > 
CONTENTS *
HashTable<HASHTKEY,CONTENTS>::Find(
	const HASHTKEY  *key
)
{
	int 		slot;
	LISTELEMENT *listelement;
	CONTENTS	*contents;

	TRPRINT(TR_BF, TR_LEVEL_1, ("Find"));

	slot = HASH(key);
	TRPRINT(TR_BF, TR_LEVEL_3, ("slot:%d", slot));


	if(LIST_EMPTY( &(this->buckets[slot]) ) )
		return NULL;

    for(listelement = FIRST_LIST_ELEMENT_ITSELF( &(this->buckets[slot]) );
			listelement != &(this->buckets[slot]); 
			listelement = NEXT_LIST_ELEMENT_ITSELF( listelement ) ) {

		contents = (CONTENTS *)(listelement->item);
		if (key->equalFunc(  contents->keylocation(this->unique,&key) )) {
	    	return(contents);
		} 
	}

	TRPRINT(TR_BF, TR_LEVEL_2, ("item not found"));
    return (CONTENTS *)NULL;
}

template < class HASHTKEY,class CONTENTS > 
void     
HashTable<HASHTKEY,CONTENTS>::Insert(
	CONTENTS 	*contents
)
{
	LISTELEMENT 	*listelement = contents->listlocation(this->unique);
	HASHTKEY		*key;
	int slot;

	key = contents->keylocation(this->unique,&key);

	slot = HASH(key);
	TRPRINT(TR_BF, TR_LEVEL_1, ("Insert in %s, slot %d\n", this->name, slot));

#ifdef DEBUG
	if( this->Find(key) ) {
		TRPRINT(TR_BF, TR_LEVEL_3, 
			("double insert into %s hash table; key %x ", this->name, key));
		SM_ERROR(TYPE_CRASH, esmINTERNAL);
	}
#endif DEBUG

    listPush( &(this->buckets[slot]), (LIST *)listelement);
	this->numItems++;
#ifdef DEBUG
	{
		register int i = listLength( &(this->buckets[slot]) );
		TRPRINT(TR_BF, TR_LEVEL_2, ("bucket %d size %d",
			slot, i));
		if(i>this->bucketSizeHighWater)
			this->bucketSizeHighWater = i;
	}
#endif DEBUG
}

template < class HASHTKEY,class CONTENTS > 
void     
HashTable<HASHTKEY,CONTENTS>::Remove(
	CONTENTS *contents
)
{
	LISTELEMENT 	*listelement = contents->listlocation(this->unique);

	if (LIST_MEMBER( listelement ))  {

		LISTREMOVE (listelement); /* fatal error if not a member */
	}
	this->numItems--;
}



template < class HASHTKEY,class CONTENTS > 
void     
HashTable<HASHTKEY,CONTENTS>::ForEach(
	int				nargs, /* TOTAL # args, including func */
#ifndef __GNUC__
	FOREACHFUNC foreachfunc,
#endif __GNUC__
	...
)
{
	va_list 		ap;
#define MAXNARGS 	10
	void			*arglist[MAXNARGS];
	int 			slot;
	LISTELEMENT 	*listelement ;
	LISTELEMENT 	*next ;
#ifdef __GNUC__
	FOREACHFUNC foreachfunc;
#endif __GNUC__

	TRPRINT(TR_BF, TR_LEVEL_3, ("ForEach"));
#ifdef __GNUC__
	SM_ASSERT(LEVEL_1, (nargs <= MAXNARGS));
#endif __GNUC__

	SM_ASSERT(LEVEL_1, (nargs >= 1)); /* must include func */
	{
		register int i;

#ifdef __GNUC__
		va_start(ap,nargs);	
		foreachfunc = va_arg(ap,FOREACHFUNC);
#else
		va_start(ap,foreachfunc);	
#endif __GNUC__

		for(i=0; i<nargs; i++) {
			arglist[i] = va_arg(ap,void *);
			TRPRINT(TR_BF, TR_LEVEL_1, 
				("arglist[%d]=0x%x", i, arglist[i]));
		}
		va_end(ap);
	}

	for (slot = 0; slot < this->numBuckets; slot++ ) {
		if(LIST_EMPTY( &(this->buckets[slot]) ) )
			continue;

		TRPRINT(TR_BF, TR_LEVEL_3, ("slot:%d", slot));

		for( listelement = FIRST_LIST_ELEMENT_ITSELF( &(this->buckets[slot]) );
			listelement != &(this->buckets[slot]); listelement = next) {

			/* Get the next list element before calling the function.
			 * That way, if the function removes the list element from
			 * the list, we can still function properly.
			 * We will not catch new elements, should the function
			 * cause new elements to be added to the list.
			 */
			next = NEXT_LIST_ELEMENT_ITSELF( listelement );
			if((*foreachfunc)(listelement->item, 
				arglist[0], arglist[1], arglist[2], arglist[3],
				arglist[4], arglist[5], arglist[6], arglist[7],
				arglist[8], arglist[9]))
				return;
		}
	}
}


#ifdef DEBUG
template < class HASHTKEY,class CONTENTS > 
void
HashTable<HASHTKEY,CONTENTS>::Dump(
	FILE *f,
	int	 line,
	char *file
)
{
	/* Like ForEach, except that we dump the table info too */
	int 			slot;
	LISTELEMENT 	*listelement ;
	CONTENTS		*contents;

	fprintf(f, 
		"Dump of hash table %s (line %d, file %s):\n", Name(), line, file);

	for (slot = 0; slot < this->numBuckets; slot++ ) {
		if(LIST_EMPTY( &(this->buckets[slot]) ) )
			continue;

		fprintf(f, "----- slot %d ----------->>>\n", slot);

		for( listelement = FIRST_LIST_ELEMENT_ITSELF( &(this->buckets[slot]) );
					listelement != &(this->buckets[slot]);
					listelement = NEXT_LIST_ELEMENT_ITSELF( listelement )) { 

			contents = (CONTENTS *)listelement->item;
			if(contents == NULL) {
				fprintf(f, "NULL CONTENTS: BARF\n");
				exit(1);
			}
			contents->Dump(f);
		}
		fprintf(f, "<<<-- slot %d ----------- \n", slot);
	}
	fprintf(f, 
		"End of Dump of %s (line %d, file %s):\n", Name(), line, file);
}
#endif DEBUG
#endif /* _HASHTABLE_C_ */
