//  ximage.c: functions to handle XImages, save & load images, palettes
//  a part of PMS-grabber package
// the "GNU Public Lincense" policy applies to all programs of this package
// (c) Wolfgang Koehler, wolf@first.gmd.de, Dec. 1994
//     Kiefernring 15
//     14478 Potsdam, Germany
//     0331-863238

#include "window.h"
#include <X11/Xutil.h>
#include "evaluation.h"
#include "view.h"
#include "files.h"

#ifdef SPARC

// define swapping big-endian -> little-endian
#define S_SWAB(x) (((x & 0xFF) << 8) | ((x & 0xFF00) >> 8))
#define L_SWAB(x) (((x & 0xFF) << 24) | ((x & 0xFF00) << 8) | \
		   ((x & 0xFF0000) >> 8) | ((x & 0xFF000000) >> 24))

#define L_XCH(x) x = L_SWAB((long) x)  // in place swapping
#define S_XCH(x) x = S_SWAB((short) x)  // in place swapping

// getw and putw are not defined on Suns
int getw(FILE* f) { 
  int w; fread(&w,4,1,f); 
  return (L_SWAB(w));
}

int putw(int w, FILE *f) { int wx = L_SWAB(w); fwrite(&wx,4,1,f); }

void swap_xcolor(XColor &xc) { // swaps the members of XColor struct
  L_XCH(xc.pixel); 
  S_XCH(xc.red); S_XCH(xc.green); S_XCH(xc.blue);
}

#else

#define L_XCH(x) // do nothing
#define S_XCH(x)
void swap_xcolor(XColor &xc) {};

#endif // SPARC 

// load XColor structure from file, if on SPARC swap the bytes
XColor fread_color(FILE *f) {
  XColor xc;
  fread(&xc, sizeof(XColor), 1, f);
  swap_xcolor(xc);
  //  printf(" %x %x %x %x\n",xc.pixel,xc.red,xc.green,xc.blue);
  return xc;
}

void fwrite_color(XColor xc, FILE *f) {
  swap_xcolor(xc);
  fwrite(&xc, sizeof(XColor), 1, f);
}

// erzeugt "image_buf" und assoziert es mit XImage
XImage * create_image(int width ,int height) {
  // printf("create image size = %d %d \n",width,height);
  char *image_buf = new char[width * height];
  if (image_buf == NULL) error("no room");
  return (XCreateImage(display,DefaultVisual(display,screen),
		       DefaultDepth(display,screen),ZPixmap,0,
		       image_buf, width, height, 8,0) );
}

// save captured picture in palettized format (1 byte per pixel = [0..ncols-1])
// together with the pixtab-entries of the XColor palette into the newly 
// (quite similiar to xwd/xwud)

void save_image(XImage *ximage, FILE *fsave,int ncols,unsigned long *pixtab) { 
  if (fsave == NULL) return; // possibly not confirmed
  putw(ncols, fsave);  // = used palette size
  int i;
  for (i=0; i < ncols; i++) { 
    XColor xc = { pixtab[i] }; 
    XQueryColor(display, def_cmap, &xc); // get rgb-values for pixtab-value
    fwrite_color(xc, fsave);
  }
  int ww = ximage->width, hh = ximage->height;
  putw(ww,fsave); putw(hh,fsave);
  fwrite(ximage->data, ww * hh, 1, fsave); // the direct image
  fclose(fsave);
 
}

// loads image to newly created XImage structure, transforms to color cells
// pointed to from pixtab vector, returns ximage
// set color cells ; returns also "ncols" = number of used color cells
XImage* load_image(FILE *fload, int &ncols, unsigned long *pixtab) {
  ncols = getw(fload); 
  if (ncols > max_cols) error("not enough color cells");
  int i; // trafo between color index from file -> pixtab entries
  unsigned long col_trans[256]; 
  for (i=0; i< 256; i++) col_trans[i] = 0;
  for (i=0; i < ncols; i++) { 
    XColor xc = fread_color(fload);
    if (xc.pixel > 256) error("wrong pixel");
    col_trans[xc.pixel] = pixtab[i]; xc.pixel = pixtab[i];
    XStoreColor(display, def_cmap, &xc);
    
  }
  int ww = getw(fload), hh = getw(fload), nn = ww*hh;
  XImage *ximage = create_image(ww,hh); 
  unsigned char *ibuf =  (unsigned char *) ximage->data; // usigned !!
  fread(ibuf, nn, 1, fload); 
  // transform pixel values to actual pixtab-values
  for (i = 0; i < nn; i++) ibuf[i] = col_trans[ibuf[i]];
  fclose(fload);
  return (ximage);
}

// this simple loading function serves only as example 
// intermediate between load_pix & load_image
void load_pix_file(char *fina) {
  
  FILE *fload = fopen(fina,"r");
  if (fload == NULL) { /* printf("error loading file %s\n",fina); */ return; }

  XImage *xi = load_image(fload,ncols,pixtab);
  
  if (xi == NULL) return; // error reading file
  printf("loaded file %s with %d colors\n",fina,ncols);

  if (ximage)  XDestroyImage(ximage);

  ximage = xi; 
  int ww = ximage->width, hh = ximage->height;
  // this simple resizing conflicts with create_buffer
  // it should be disentangled for better usage
  // XResizeWindow(display,mw->Win,ww,hh+41); 
  grab_wi->redraw();
}

// ****** simple callback functions for pulldown menus *******

void save_pix() {   
  char *defname = "default.pix", fina[80];
  FILE *fsave = open_file_interactive(defname,fina,MODE_WRITE);
  if (fsave == NULL) return;
  save_image(ximage,fsave,ncols,pixtab); 
  int ww = ximage->width, hh = ximage->height;
  printf("saved image to %s with %d colors, size = (%d x %d) %d byte\n",
       fina,ncols,ww,hh,ww*hh);
}

void load_pix() {  
  char *defname = "*.pix", fina[80]; 
  FILE *fload = open_file_interactive(defname,fina,MODE_READ);
  if (fload == NULL) return;

  XImage *xi = load_image(fload,ncols,pixtab);
  if (xi == NULL) return; // error reading file
  printf("loaded file %s with %d colors\n",fina,ncols);
  if (ximage)  XDestroyImage(ximage);
  ximage = xi; 
  int ww = ximage->width, hh = ximage->height;
  printf("loaded image from %s with %d colors, size = (%d x %d) %d byte\n",
	 fina,ncols,ww,hh,ww*hh);

  grab_wi->redraw();
}

// save captured image in direct color rgb555 format
void save_capture() {
  char *defname = "default.cap555", fina[80];
  FILE *fsave = open_file_interactive(defname,fina,MODE_WRITE);
  if (fsave == NULL) return;
  putw(biWidth,fsave); putw(biHeight,fsave);
  fwrite(cap_buf, 2, biWidth*biHeight, fsave);
  fclose(fsave);  
  printf("saved capture to %s size = %d byte = %d x %d\n",
	 fina,2*biWidth*biHeight,biWidth,biHeight);
}

// save captured image in direct color rgb555 format
void load_capture() {
  char *defname = "*.cap555", fina[80]; 
  FILE *fload = open_file_interactive(defname,fina,MODE_READ);
  if (fload == NULL) return;
  int ww  = getw(fload), hh = getw(fload), nn = ww*hh;
  if ((ww != biWidth) || (hh != biHeight)) adjust_size(ww,hh);  
  fread(cap_buf, 2, nn, fload);
  fclose(fload);  

  map_and_redraw();

  printf("loaded capture from %s size = %d byte = %d x %d\n",
	 fina,2*nn,biWidth,biHeight);
}

// save coltab55
void save_palette() {  
  char *defname = "default.pal555", fina[80];
  FILE *fsave = open_file_interactive(defname,fina,MODE_WRITE);
  if (fsave == NULL) return;

  putw(ncols, fsave); int i;  
  for (i=0; i < ncols; i++) { 
    XColor xc = { pixtab[i] }; 
    XQueryColor(display, def_cmap, &xc); // get rgb-values for pixtab-value
    fwrite_color(xc, fsave);
  }
  fwrite(coltab555, 32*32*32, 1, fsave);
  fclose(fsave);   
  printf("palette with %d colors saved to file %s\n",ncols,fina);
}

// load saved palette, the only problem is that the pixel values of the saved
// palette are not nessecarily free on the display they are loaded to
void load_palette() {
  char *defname = "*.pal555", fina[80]; 
  FILE *fload = open_file_interactive(defname,fina,MODE_READ);
  if (fload == NULL) return;

  ncols = getw(fload); 
  if (ncols > max_cols) error("not enough color cells");
  unsigned long col_trans[256]; 
  int i;
  // compute translation of old -> new pixel values
  for (i=0; i< 256; i++) col_trans[i] = 0;
  for (i=0; i < ncols; i++) { 
    XColor xc = fread_color(fload);
    if (xc.pixel > 256) error("wrong pixel");
    col_trans[xc.pixel] = pixtab[i]; xc.pixel = pixtab[i];
    XStoreColor(display, def_cmap, &xc);    
  }
  fread (coltab555, 32*32*32, 1, fload);
  fclose(fload); 
  for (i=0; i < 32*32*32; i++) coltab555[i] = col_trans[coltab555[i]];
  printf("palette with %d colors loaded from file %s\n",ncols,fina);
  map_and_redraw();
}

