// SetMVV.c : setting functions for the PMS-board
//  wolf@first.gmd.de, Dec. 1994
// partly adopted from MediaVision

typedef  unsigned char  BYTE;
typedef unsigned short  WORD;

int FALSE =  0;
int TRUE  = 1;

#define  MVVMEMORYWIDTH    0x40    //512

#define  UNDEFINED  0x0000
#define  MOTOROLA  0x0001    //Early debug board
#define  PHILIPS2  0x0002
#define  PHILIPS1  0x0003

typedef struct {
    BYTE  bSlave;
    BYTE  bSubAddr;
    BYTE  bData;
    BYTE  bHits;
} I2CInfoType;

static  int   iI2CCount = 0;
static  I2CInfoType i2cInfo[64];

WORD  wMVVDecoder = 2; //1=Motorola, 2=Philips 2-chip,  3=Philips 3-chip
WORD  wVideoStandard = 2; //0=Auto, 1=NTSC, 2=PAL, 3=SECAM
int   iMVVBoardID = 0;
int   iMVVID = 0;

int biWidth = 320;
int biHeight = 240;

//  ***********************************************************************
int I2CWrite(WORD bSlaveAddr, WORD bSubAddr,WORD bData)
{
    int iSkipWrite = FALSE;

    int  i;

    for(i=0;i<iI2CCount;i++) {
      if((i2cInfo[i].bSlave==bSlaveAddr) &&(i2cInfo[i].bSubAddr==bSubAddr)) {
        iSkipWrite = (i2cInfo[i].bData==bData);
        i2cInfo[i].bData = (BYTE) bData;
        i = iI2CCount+1;
      }
    }
    if((i==iI2CCount) && (iI2CCount<64)) {
      i2cInfo[iI2CCount].bSlave   = (BYTE) bSlaveAddr;
      i2cInfo[iI2CCount].bSubAddr = (BYTE) bSubAddr;
      i2cInfo[iI2CCount].bData    = (BYTE) bData;
      iI2CCount++;
    }

    if(iSkipWrite) return(0);

    mvv_write (0x29, bSubAddr );
    mvv_write (0x2A, bData );
    mvv_write (0x28, bSlaveAddr );

    OUT_MVV(0x28); 
    unsigned short cx = 0;
    while ((IN_MVV & 1) == 0) if (--cx == 0) break;
    while ((IN_MVV & 1) != 0) if (--cx == 0) break;
    
    char ah = IN_MVV;
    if (ah & 2) return(-1); else return(ah);

}

int I2CRead(WORD bSlaveAddr, WORD bSubAddr)
{
  int i;
  for(i=0;i<iI2CCount;i++)
    if((i2cInfo[i].bSlave==bSlaveAddr) &&(i2cInfo[i].bSubAddr==bSubAddr))
      return(i2cInfo[i].bData);
  return(0);
}

//  ***********************************************************************

void I2CAndOr(WORD bSlaveAddr, WORD bSubAddr,WORD bAND,WORD bOR)
{
    BYTE  bTmp;
    bTmp = I2CRead(bSlaveAddr,bSubAddr);
    bTmp = (bTmp & bAND) | bOR;
    I2CWrite(bSlaveAddr,bSubAddr,bTmp);
}

//  ***********************************************************************
//   Testing the iic- read/write mechanism

void print_iinfo() {
  int i;
  printf("I2C regs \n i slave addr val\n");
  for (i=0; i< iI2CCount; i++) {
    printf("%2d   %2x   %2x   %2x\n",i,
	   i2cInfo[i].bSlave,i2cInfo[i].bSubAddr,i2cInfo[i].bData);
  }
}

int test_i2c() {
  unsigned char st;
  int i;
  do { 
    st = mvv_read(0x28);
    printf(" st = %x",st);
    for (i=0x29; i < 0x2c; i++) printf(" %2x",mvv_read(i));
  } while ((st & 7) != 0);
  printf("\n");
}

char i2c_read(char slave, char addr) {
  printf("reading %x %x\n",(unsigned char) slave,(unsigned char) addr);
  mvv_write (0x29, addr);
  test_i2c();  	
  mvv_write (0x28, slave | 1); // set read bit
  // while (mvv_read(0x28) != (slave | 1));
  test_i2c(); 		
  return (mvv_read (0x2A));
}

//  ***********************************************************************

void SetMVVVideoSource(WORD wData)
{
    mvv_write(0x2E,0x30 | ((~wData) & 0x01));
}

void SetMVVHue(WORD wData)
{
    switch(wMVVDecoder) {
  case MOTOROLA :
      I2CWrite(0x8A,0x00,wData);
      break;
  case PHILIPS2 :
      I2CWrite(0x8A,0x07,wData);
      break;
  case PHILIPS1 :
      I2CWrite(0x42,0x07,wData);
      break;
    }
}

void SetMVVColor(WORD wData)
{
    switch(wMVVDecoder) {
  case MOTOROLA :
      I2CWrite(0x8A,0x00,wData);
      break;
  case PHILIPS1 :
      I2CWrite(0x42,0x12,wData);
      break;
    }
}

void SetMVVContrast(WORD wData)
{
    switch(wMVVDecoder) {
  case MOTOROLA :
      I2CWrite(0x8A,0x00,wData);
      break;
  case PHILIPS1 :
      I2CWrite(0x42,0x13,wData);
      break;
    }
}

void SetMVVBrightness(WORD wData)
{
    switch(wMVVDecoder) {
  case MOTOROLA :
      I2CWrite(0x8A,0x00,wData);  //Red
      I2CWrite(0x8A,0x00,wData);  //Green
      I2CWrite(0x8A,0x00,wData);  //Blue
      break;
  case PHILIPS1 :
      I2CWrite(0x42,0x19,wData);
      break;
    }
}

void SetMVVHStart(WORD wData)
{
  switch(wMVVDecoder) {
  case PHILIPS2 :
      I2CWrite(0x8A,0x05,wData);  //PAL/SECAM (+-118)
      I2CWrite(0x8A,0x18,wData);  //NTSC (+- 97)
      break;
  case PHILIPS1 :
      I2CWrite(0x42,0x05,wData);
      I2CWrite(0x42,0x18,wData);
      break;
    }
}

void SetMVVVStart(WORD wData)
{
    mvv_write(0x16,wData);
    mvv_write(0x17,(wData >> 8) & 0x01);
}

void SetMVVVideoStd(WORD wData)
{
    wVideoStandard = wData;

    switch(wMVVDecoder) {
  case PHILIPS2 :
      switch(wData) {
        case 0 :    //Auto
            I2CAndOr(0x8A,0x0D,0xFE,0x00);
            I2CAndOr(0x8A,0x0F,0x3F,0x80);
        break;
        case 1 :    //NTSC
            I2CAndOr(0x8A,0x0D,0xFE,0x00);
            I2CAndOr(0x8A,0x0F,0x3F,0x40);
        break;
        case 2 :    //PAL
            I2CAndOr(0x8A,0x0D,0xFE,0x00);
            I2CAndOr(0x8A,0x0F,0x3F,0x00);
        break;
        case 3 :    //SECAM
            I2CAndOr(0x8A,0x0D,0xFE,0x01);
            I2CAndOr(0x8A,0x0F,0x3F,0x00);
        break;
      } break;
  case PHILIPS1 :
      switch(wData) {
        case 0 :    //Auto
            I2CAndOr(0x42,0x0D,0xFE,0x00);
            I2CAndOr(0x42,0x0F,0x3F,0x80);
        break;
        case 1 :    //NTSC
            I2CAndOr(0x42,0x0D,0xFE,0x00);
            I2CAndOr(0x42,0x0F,0x3F,0x40);
        break;
        case 2 :    //PAL
            I2CAndOr(0x42,0x0D,0xFE,0x00);
            I2CAndOr(0x42,0x0F,0x3F,0x00);
        break;
        case 3 :    //SECAM
            I2CAndOr(0x42,0x0D,0xFE,0x01);
            I2CAndOr(0x42,0x0F,0x3F,0x00);
        break;
      }
  default :
      break;
    }
}

void SetMVVVCRInput(WORD wData)
{
    switch(wMVVDecoder) {
  case PHILIPS2 :
      I2CAndOr(0x8A,0x0D,0x7F,(wData & 0x01) <<7);
      break;
  case PHILIPS1 :
      I2CAndOr(0x42,0x0D,0x7F,(wData & 0x01) <<7);
      break;
  default :
      break;
    }
}

void SetMVVBandpass(WORD wData)
{
    switch(wMVVDecoder) {
  case PHILIPS2 :
      I2CAndOr(0x8A,0x06,0xCF,((wData & 0x03) << 4));
      break;
  case PHILIPS1 :
      I2CAndOr(0x42,0x06,0xCF,((wData & 0x03) << 4));
      break;
    }
}

void SetMVVAntiSnow(WORD wData)
{
    switch(wMVVDecoder) {
  case PHILIPS2 :
      I2CAndOr(0x8A,0x06,0xF3,((wData & 0x03) << 2));
      break;
  case PHILIPS1 :
      I2CAndOr(0x42,0x06,0xF3,((wData & 0x03) << 2));
      break;
    }
}

void SetMVVSharpness(WORD wData)
{
    switch(wMVVDecoder) {
  case PHILIPS2 :
      I2CAndOr(0x8A,0x06,0xFC,wData & 0x03);
      break;
  case PHILIPS1 :
      I2CAndOr(0x42,0x06,0xFC,wData & 0x03);
      break;
    }
}

void SetMVVChromaAGC(WORD wData)
{
    switch(wMVVDecoder) {
  case PHILIPS2 :
      I2CAndOr(0x8A,0x0C,0x9F,((wData & 0x03) << 5));
      break;
  case PHILIPS1 :
      I2CAndOr(0x42,0x0C,0x9F,((wData & 0x03) << 5));
      break;
    }
}

void SetMVVVertNoise(WORD wData)
{
    switch(wMVVDecoder) {
  case PHILIPS2 :
      I2CAndOr(0x8A,0x10,0xFC,wData & 0x03);
      break;
  case PHILIPS1 :
      I2CAndOr(0x42,0x10,0xFC,wData & 0x03);
      break;
    }
}

void SetMVVSECAMCross(WORD wData)
{
    switch(wMVVDecoder) {
  case PHILIPS2 :
      I2CAndOr(0x8A,0x0F,0xDF,((wData & 0x01) << 5));
      break;
  case PHILIPS1 :
      I2CAndOr(0x42,0x0F,0xDF,((wData & 0x01) << 5));
      break;
    }
}

void SetMVVForceColor(WORD wData) {
    switch(wMVVDecoder) {
  case PHILIPS2 :
      I2CAndOr(0x8A,0x0C,0x7F,((wData & 0x01) << 7));
      break;
  case PHILIPS1 :
      I2CAndOr(0x42,0x0C,0x7F,((wData & 0x01) << 7));
      break;
    }
}

void SetMVVAntiGamma(WORD wData)
{
    switch(wMVVDecoder) {
  case PHILIPS2 :
      I2CAndOr(0xB8,0x00,0x7F,((wData & 0x01) << 7));
      break;
  case PHILIPS1 :
      I2CAndOr(0x42,0x20,0x7F,((wData & 0x01) << 7));
      break;
    }
}

void SetMVVPrefilter(WORD wData)
{
    switch(wMVVDecoder) {
  case PHILIPS2 :
      I2CAndOr(0x8A,0x06,0xBF,((wData & 0x01) << 6));
      break;
  case PHILIPS1 :
      I2CAndOr(0x42,0x06,0xBF,((wData & 0x01) << 6));
      break;
    }
}

void SetMVVHFilter(WORD wData)
{
    switch(wMVVDecoder) {
  case PHILIPS2 :
      I2CAndOr(0xB8,0x04,0x1F,((wData & 0x07) << 5));
      break;
  case PHILIPS1 :
      I2CAndOr(0x42,0x24,0x1F,((wData & 0x07) << 5));
      break;
    }
}

void SetMVVVFilter(WORD wData)
{
    switch(wMVVDecoder) {
  case PHILIPS2 :
      I2CAndOr(0xB8,0x08,0x9F,((wData & 0x03) << 5));
      break;
  case PHILIPS1 :
      I2CAndOr(0x42,0x28,0x9F,((wData & 0x03) << 5));
      break;
    }
}

void SetMVVColorKill(WORD wData)
{
    switch(wMVVDecoder) {
  case PHILIPS2 :
      I2CAndOr(0x8A,0x08,0x07,((wData & 0x1F) << 3));  //QUAM
      I2CAndOr(0x8A,0x09,0x07,((wData & 0x1F) << 3));  //SECAM
      break;
  case PHILIPS1 :
      I2CAndOr(0x42,0x08,0x07,((wData & 0x1F) << 3));  //QUAM
      I2CAndOr(0x42,0x09,0x07,((wData & 0x1F) << 3));  //SECAM
      break;
    }
}

void SetMVVSwSense(WORD wData)
{
    switch(wMVVDecoder) {
  case PHILIPS2 :
      I2CWrite(0x8A,0x0A,wData);  //PAL
      I2CWrite(0x8A,0x0B,wData);  //SECAM
      break;
  case PHILIPS1 :
      I2CWrite(0x42,0x0A,wData);  //PAL
      I2CWrite(0x42,0x0B,wData);  //SECAM
      break;
    }
}

void SetMVVChromaGain(WORD wData)
{
    switch(wMVVDecoder) {
  case PHILIPS2 :
      I2CWrite(0x8A,0x11,wData);  //QUAM
      break;
  case PHILIPS1 :
      I2CWrite(0x42,0x11,wData);  //QUAM
      break;
    }
}

void SetMVVSpacialCompL(WORD wData)
{
    mvv_write(0x3B,wData);
}

void SetMVVSpacialCompH(WORD wData)
{
    mvv_write(0x3A,wData);
}

void SetMVVFrameRate(WORD wData)
{
    if((wVideoStandard==1)) {
      mvv_write(0x14,0x80 | (30/wData));
    } else mvv_write(0x14,0x80 | (25/wData));

    mvv_write(0x15,0x01);   //Field Deci Num
}
//  ***********************************************************************
//        MVV Specific Handling
//  ***********************************************************************

void MVVSetVertDeci(WORD wDeciNum,WORD wDeciDen) {
  if((wDeciNum%5)==0) {
    wDeciDen /= 5; wDeciNum /= 5;
  }
  while(((wDeciNum%3)==0)&&((wDeciDen%3)==0)) {
    wDeciDen /= 3; wDeciNum /= 3;
  }
  while((((wDeciNum&0x0001)|(wDeciDen&0x0001))==0x0000)) {
    wDeciDen >>= 1;
    wDeciNum >>= 1;
  }
  while((wDeciDen>32)) {
    wDeciDen >>= 1;
    wDeciNum = (wDeciNum+1)>>1;
  }
  if(wDeciDen==32) wDeciDen--;
  // printf("vert %d %d\n", wDeciDen, wDeciNum);
  mvv_write(0x1C,wDeciDen); //Vert Decimation Denominator
  mvv_write(0x1D,wDeciNum); //Vert Decimation Numerator
}

void MVVSetHorzDeci(WORD wDeciNum,WORD wDeciDen) {
  if(wDeciNum<=512) {
    if((wDeciNum%5)==0) {
      wDeciDen /= 5;
      wDeciNum /= 5;
    }
  } else {
    wDeciNum = 512;
    wDeciDen = 640;     //++Ideally we would do 768
  }

  while ( ((wDeciNum&0x0001) | (wDeciDen&0x0001)) == 0x0000) {
    wDeciDen >>= 1;
    wDeciNum >>= 1;
  }
  while((wDeciDen>32)) {
    wDeciDen >>= 1;
    wDeciNum = (wDeciNum+1)>>1;
  }
  if(wDeciDen==32) wDeciDen--;
  // printf("horz %d %d\n", wDeciDen, wDeciNum);
  mvv_write(0x24,0x80 | wDeciDen);  //Horz Decimation Denominator
  mvv_write(0x25,wDeciNum);   //Horz Decimation Numerator

}

void SetMVVResolution(WORD wWidth, WORD wHeight) {
  if(wHeight<=256) {
    mvv_write(0x18,wHeight);    //Cap Vert Len (L)
    mvv_write(0x19,(wHeight)>>8); //Cap Vert Len (H)
  } else {
    mvv_write(0x18,wHeight/2);  //Cap Vert Len (L)
    mvv_write(0x19,(wHeight/2)>>8); //Cap Vert Len (H)
  }
    
  if((wVideoStandard==1)) {
    
    mvv_write(0x1A,0xFC);   //Cap Vert Comp (L)
    mvv_write(0x1B,0x00);   //Cap Vert Comp (H)

    if(wHeight>256) MVVSetVertDeci(wHeight/2,240);
    else MVVSetVertDeci(wHeight,240);
  } else {
    mvv_write(0x1A,0x1A);   //Cap Vert Comp (L)
    mvv_write(0x1B,0x01);   //Cap Vert Comp (H)

    if(wHeight>256) MVVSetVertDeci(wHeight/2,270);
    else MVVSetVertDeci(wHeight,270);
  }

  if(iMVVID==0) {
    mvv_write(0x12,0x00);     //Cap Mem Page Ctrl: 0000h
    mvv_write(0x13,MVVMEMORYWIDTH);  //Cap Mem Page Slct: 512 words

    mvv_write(0x42,0x00);     //Disp Mem Page Ctrl: 0000h
    mvv_write(0x43,0x00);     //Disp Mem Page Ctrl: 0000h
    mvv_write(0x44,MVVMEMORYWIDTH);  //Disp Mem Page Slct: 512 words
  } else {
    mvv_write(0x12,(wHeight-1)*2);  //Cap Mem Page Ctrl: bottom
    mvv_write(0x13,(((wHeight-1)*2)>>8) | MVVMEMORYWIDTH); //Cap Mem Page Slct: 512 words

    mvv_write(0x42,0x00);     //Disp Mem Page Ctrl: 0000h
    mvv_write(0x43,(wHeight-1)*2);  //Cap Mem Page Ctrl: bottom
    mvv_write(0x44,(((wHeight-1)*2)>>8) | MVVMEMORYWIDTH); //Cap Mem Page Slct: 512 words
  }

  if(wWidth<=512) {
    mvv_write(0x22,wWidth+8);   //6/3: Hack around left edge
    mvv_write(0x23,(wWidth+8)>>8);
  } else {
    mvv_write(0x22,512);
    mvv_write(0x23,(512)>>8);
  }

  if((wVideoStandard==1)) {
    MVVSetHorzDeci(wWidth,640);
  } else {
    MVVSetHorzDeci(wWidth,768);
  }

  mvv_write(0x30,mvv_read(0x30) & 0xFE);
  mvv_write(0x08,mvv_read(0x08) | 0x01);
  mvv_write(0x01,mvv_read(0x01) & 0xFD);  //6/1:Hack to get compressed capture working
  mvv_write(0x32,0x00);
  mvv_write(0x33,MVVMEMORYWIDTH);

  if(wWidth<=512) {
    mvv_write(0x36,(wWidth+8)/4);   //6/3: Hack around left edge
    mvv_write(0x37,0x80 | ((wWidth+8)>>10));
  } else {
    mvv_write(0x36,512/4);
    mvv_write(0x37,0x80 | (512>>10));
  }
  // only for compression counter
  if(wHeight<=256) {
    mvv_write(0x38,wHeight/4);
    mvv_write(0x39,0x80 | (wHeight>>10));
  } else {
    mvv_write(0x38,wHeight/4);
    mvv_write(0x39,0x80 | (wHeight>>11));
  }
}

#include "grabber.h"

char	szFrameRate[]   = "Frame Rate";
char	szVideoSrc[]	= "Video Source";
char	szHStart[]	= "Horz Start";
char	szVStart[]	= "Vert Start";
char	szHue[]		= "Hue";
char	szColor[]	= "Color";
char	szContrast[]	= "Contrast";
char	szBrightness[]	= "Brightness";
char	szSaturation[]	= "Saturation";
char	szVideoStd[]	= "Video Standard";
char	szVCRInput[]	= "VCR Input";
char	szBandpass[]	= "Bandpass Filter";
char	szAntiSnow[]	= "Anti-snow Filter";
char	szSharpness[]	= "Sharpness";
char	szChromaAGC[]	= "Chroma AGC";
char	szVertNoise[]	= "Vert Noise";
char	szSECAMCross[]	= "SECAM Cross Color";
char	szForceColor[]	= "Force Color";
char	szAntiGamma[]	= "Anti-Gamma";
char	szPreFilter[]	= "Prefilter";
char	szHFilter[]	= "Horz Filter";
char	szVFilter[]	= "Vert Filter";
char	szColorKill[]	= "Color Killer";
char	szSwSense[]	= "Switch Sensitivity";
char	szChromaGain[]	= "Chroma Gain";
char	szSpacialComp[]	= "Spacial Compression";
char	szSpacialCompL[] = "Spacial Compression (L)";
char	szSpacialCompH[] = "Spacial Compression (H)";

char * OptListEnable[] = { "Disable","Enable",0};
char * OptListVideoSrc[] = { "Composite","S-video",0};
char * OptListVideoStd[] = { "Auto-detect","NTSC","PAL","Secam",0};

char * OptListBandpass[] = { "Lowest Detail","Low Detail",
			     "High Detail","Highest Detail",0};

char * OptListCoring[] = {"Disabled","1 LSB of 8-bit",
			  "2 LSB of 8-bit","3 LSB of 8-bit",0};

char * OptListAperture[] = {"Weighting Factor 0","Weighting Factor 0.25",
			    "Weighting Factor 0.5","Weighting Factor 1",0};

char * OptListChromaAGC[] = {"Slow","Medium","Fast","Fixed",0};

char * OptListVertNoise[] = {"Normal","Searching","Free-running","Bypass",0};

char * OptListHFilter[] = {"2-tap filter","3-tap filter","5-tap filter",
			   "9-tap filter","Bypass","Bypass with delay",
			   "8-tap filter","4-tap filter",0};

char * OptListVFilter[] = { "Bypass","Delay","2-tap filter","3-tap filter",0};

char * toggle_switch; // to have an unique pointer

option_struct option_vector[] = {
  { szFrameRate ,0,   1, 30,  15, NULL, &SetMVVFrameRate },
  { szVideoSrc  ,0,   0,  1,   0, OptListVideoSrc, &SetMVVVideoSource },
  { szHStart    ,0,-128,127, -41, NULL, &SetMVVHStart },
  { szVStart    ,0,   0,255,  15, NULL, &SetMVVVStart }, //NTSC vs PAL/SECAM?
  { szHue       ,0,   0,255,   0, NULL, &SetMVVHue },
  { szVideoStd  ,0,   0,  3,   0, OptListVideoStd, &SetMVVVideoStd },
  { szVCRInput  ,0,   0,  1,   1, &toggle_switch, &SetMVVVCRInput },
  { szBandpass  ,0,   0,  3,   3, OptListBandpass, &SetMVVBandpass },
  { szAntiSnow  ,0,   0,  3,   0, OptListCoring, &SetMVVAntiSnow },
  { szSharpness ,0,   0,  3,   0, OptListAperture, &SetMVVSharpness },
  { szChromaAGC ,0,   0,  3,   0, OptListChromaAGC, &SetMVVChromaAGC },
  { szVertNoise ,0,   0,  3,   0, OptListVertNoise, &SetMVVVertNoise },
  { szSECAMCross,0,   0,  1,   0, &toggle_switch, &SetMVVSECAMCross },
  { szForceColor,0,   0,  1,   0, &toggle_switch, &SetMVVForceColor },
  { szAntiGamma ,0,   0,  1,   1, &toggle_switch, &SetMVVAntiGamma },
  { szPreFilter ,0,   0,  1,   0, &toggle_switch, &SetMVVPrefilter },
  { szHFilter   ,0,   0,  7,   4, OptListHFilter, &SetMVVHFilter },
  { szVFilter   ,0,   0,  3,   0, OptListVFilter, &SetMVVVFilter },
  { szColorKill ,0,   0, 31,  31, NULL, &SetMVVColorKill }, //++QUAM vs SECAM?
  { szSwSense   ,0,   0,255,  64, NULL, &SetMVVSwSense },    //++PAL vs SECAM?
  { szChromaGain,0,   0,255,  89, NULL, &SetMVVChromaGain },  //++NTSC vs PAL
  { szSpacialCompL,0, 0,254,  16, NULL, &SetMVVSpacialCompL },
  { szSpacialCompH,0, 1,255, 128, NULL, &SetMVVSpacialCompH }
};

int noptions = sizeof(option_vector)/sizeof(option_struct); 
