// Aaron's silly little waveform-viewer...hopefully one day to be an
// editor

#include <vga.h>
#include <vgagl.h>
#include <vgamouse.h>
#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <linux/soundcard.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include "wavpic.h"
#include "waveform.h"
#include "format.h"


// global data
// The clipboard
wavedata clipboard;

// pointer to wave file headers
WaveHeader *wave_info = NULL;


int main(int argc, char *argv[])
{
ms_wav sound(argv[1]);
int c;

init_screen();
init_mouse();

gl_setwritemode(WRITEMODE_MASKED);
gl_write(1,0,"Aaron's Wave Editor v0.1");

DrawWave(&sound);
MouseShadow(1);
DrawMouse();
while (1)
  {
  usleep(20000);
  if (mouse_update())
    {
    Selection(&sound);
    DrawMouse();
    }
  if (c = vga_getkey())
    {
    switch(c)
      {
    case 'p': // Play waveform
      PlayWave(&sound);
      break;
    case 's': // Save waveform to disk
      sound.WriteWave(vga_gettext("Filename: "));
      break;
    case 'q': // quit
      goto out;
      break;
    case ' ': // deselect current selection
      if (sound.select_valid())
	{
	sound.Select(-1,-1);
	Refresh(&sound);
	}
      break;
    case 'z': // trim to selection
      clipboard = sound.Cut();
      sound.waveform(clipboard);
      clipboard.size = 0;
      sound.Select(-1,-1);
      Refresh(&sound);
      break;
    case 'x': // cut selection
      clipboard = sound.Cut();
      Refresh(&sound);
      break;
    case 'c': // copy selection
      clipboard = sound.Copy();
      break;
    case 'v': // paste selection
      sound.Paste(clipboard, (sound.wave_size()/WIDTH)*mouse_getx());
      Refresh(&sound);
      break;
    default:
      break;
      }
    }
  }

 out:
// clean-up for the mouse
mouse_close();
// clean-up for VGA stuff
vga_setmode(CHARS);
exit(0);
}


void DrawMouse(void)
// Draw the cursor...this should use very little processor time
{
int i,j;

MouseShadow(1);
MouseShadow(2);
DRAWCURSOR(CURSOR);
}


void MouseShadow(int action)
// save/restore the mouse shadow
{
static int shadow[2][16];
int i,j;
int x = mouse_getx();
int y = mouse_gety();
static int oldx, oldy;

switch (action)
  {
 case 2:
  oldx = x;
  oldy = y;
  for (i = 0; i < 2; i++)
    for (j = 0; j < 16; j++)
      shadow[i][j] = gl_getpixel(x+i,y-8+j);
  break;

 case 3:
  for (i = 0; i < 2; i++)
    for (j = 0; j < 16; j++)
      shadow[i][j] = gl_getpixel(oldx+i,oldy-8+j);
  break;

 case 1:
  for (i = 0; i < 2; i++)
    for (j = 0; j < 16; j++)
      gl_setpixel(oldx+i,oldy-8+j,shadow[i][j]);
  break;

default:
  fprintf(stderr, "Invalid Mouse shadow operation.\n");
  exit(-1);
  break;
  }
}


void init_screen(void)
// init the SVGA screen
{
GraphicsContext physicalscreen;
int VGAMODE = G1024x768x256;
void *font;

vga_init();

if (!vga_hasmode(VGAMODE))
  {
  fprintf(stderr, "Mode not available.\n");
  exit(-1);
  }
vga_setmode(VGAMODE);
gl_setcontextvga(VGAMODE);      /* Physical screen context. */
gl_getcontext(&physicalscreen);
gl_setrgbpalette();
gl_fillbox(0,0,WIDTH,HEIGHT,BACKGROUND);
gl_enableclipping();

font = malloc(256 * 8 * 8 * BYTESPERPIXEL);
gl_expandfont(8, 8, CHARS, gl_font8x8, font);
gl_setfont(8, 8, font);
}


void init_mouse(void)
// init the svgalib mouse
{
if (mouse_init("/dev/mouse", MOUSE_MICROSOFT, MOUSE_DEFAULTSAMPLERATE))
  {
  fprintf(stderr, "Could not initialize mouse.\n");
  exit(-1);
  }
mouse_setdefaulteventhandler();
mouse_setposition(WIDTH/2,HEIGHT/2);
mouse_setxrange(0,WIDTH-1);
mouse_setyrange(0,HEIGHT-1);
}


void Refresh(waveform *sound)
{
gl_fillbox(0,HEIGHT/2,WIDTH,256,BACKGROUND);
SelectBox(sound, SELECT);
DrawWave(sound);
DrawMouse();
}


void Selection(waveform *sound)
// Draw the wave-form...now we've got selections...yay!
{
static int select = 0;
static unsigned long start, end;
int x = mouse_getx();
int inc = (int) (sound->wave_size() / WIDTH);

if (mouse_getbutton() == MOUSE_LEFTBUTTON)
  {
  SelectBox(sound, BACKGROUND);
  if (select)
    end = x;
  else
    {
    select = 1;
    start = end = x;
    }
  if (x < start)
    start = x;
  sound->Select(inc*start, inc*end);
  SelectBox(sound, SELECT);
  DrawWave(sound);
  }
else
  select = 0;
}


void SelectBox(waveform *sound, int color)
{
int inc = (int) (sound->wave_size() / WIDTH);
int start = sound->select_start()/inc;
int end = sound->select_end()/inc;

if (start < end)
  {
  MouseShadow(1);
  gl_fillbox(start,HEIGHT/2,end-start,256,color);
  MouseShadow(3);
  }
}


void DrawWave(waveform *sample)
// Draw the waveform
{
unsigned long index = 0;
const unsigned char *data = sample->wave_data();
unsigned long size = sample->wave_size();
int inc = (int) (size / WIDTH);
int i;

if (size < WIDTH)
  inc = 1;

for (i = 0; i < WIDTH; i++)
  {
  if (index < size)
    {
    gl_setpixel(i, data[index] + HEIGHT/2, WAVE);
    index += inc;
    }
  else
    gl_setpixel(i, 128, WAVE);
  }
}


void PlayWave(ms_wav *sound)
{
wavedata temp;
int speed = sound->get_freq();
int wave_out; // file descriptor

if (sound->select_valid())
  {
  temp = sound->Copy();
  }
else
  {
  temp.size = sound->wave_size();
  temp.data = sound->wave_data();
  }

if ((wave_out = open("/dev/dsp", O_WRONLY)) == -1)
  {
  fprintf(stderr, "%cCould not open wave output device\n", 7);
  return;
  }

if (ioctl(wave_out,SNDCTL_DSP_SPEED, &speed)==-1 )
  {
  fprintf(stderr, "%cCould not set playback speed\n", 7);
  close(wave_out);
  return;
  }

if (write(wave_out, temp.data, temp.size) == -1)
  {
  close(wave_out);
  fprintf(stderr, "Error playing waveform\n");
  return;
  }
close(wave_out);

if (sound->select_valid())
  delete temp.data;
}


int vga_getkey(void)
{
struct termio zap, original;
char c;
int e;

ioctl(fileno(stdin), TCGETA, &original);        /* Get termio */
zap = original;
zap.c_cc[VMIN] = 0;                             /* Modify termio  */
zap.c_cc[VTIME] = 0;
zap.c_lflag = 0;
ioctl(fileno(stdin), TCSETA, &zap);             /* Set new termio */
e = read(fileno(stdin), &c, 1);                 /* Read one char */
ioctl(fileno(stdin), TCSETA, &original);        /* Restore termio */
if (e != 1)
  return 0;                               /* No key pressed. */
return c;                                       /* Return key. */
}


char *vga_gettext(char *prompt)
{
int x = 1;
int y = 20;
int c; // current user input character
static char input[128]; // 127 character input buffer
int i = 0; // index variable into temp

gl_setwritemode(WRITEMODE_OVERWRITE);
gl_write(x,y,prompt);

y = 30;

// input line is currently null
*input = 0;
while ((c = vga_getkey()) != '\n')
  {
  switch (c)
    {
  case 0:
    break;
  case 127:
    input[--i] = 0;
    break;
  default:
    input[i++] = c;
    input[i] = 0;
    if (i == 127)
      i--;
    break;
    }
  if (c)
    gl_fillbox(x,y,WIDTH,8,BACKGROUND);
  gl_write(x,y,input);
  }

gl_fillbox(0,20,WIDTH,40,BACKGROUND);
return input;
}
