/************************************************************************
 * patchload.c  --  loads patches for playmidi package
 *
 * This code was written by by Nathan Laredo (laredo@gnu.ai.mit.edu)
 * Source code may be freely distributed in unmodified form.
 * Some of this code was adapted from code written by Hannu Solovainen
 *************************************************************************/

#include "playmidi.h"
#include <sys/ultrasound.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include "gmvoices.h"

SEQ_USE_EXTBUF();
extern int play_gus, play_sb, play_ext, playing, verbose, force8bit, reverb;
extern int gus_dev, sb_dev, ext_dev, seqfd, patchloaded[256], wantopl3;

static int use8bit = 0;

void load_sysex(length, data)
int length;
char *data;
{
    struct sysex_info *sysex;

    if (!play_ext)
	return;
    sysex = (struct sysex_info *) malloc(length + sizeof(*sysex));
    if (sysex == NULL) {
	fprintf(stderr, "No more memory for sysex, ignoring\n");
	return;
    }
    sysex->key = SYSEX_PATCH;
    sysex->device_no = ext_dev;
    sysex->len = length;
    memcpy(sysex->data, data, length);
#ifdef USE_SEQUENCER2
    SEQ_WRPATCH(sysex, sizeof(*sysex) + sysex->len);
#else
    for (length = 0; length < sysex->len; length++)
	SEQ_MIDIOUT(ext_dev, sysex->data[length]);
#endif
    if (verbose > 3) {
	printf("Sent SysEx: ");
	for (length = 0; length < sysex->len; length++)
	    printf("%02x", sysex->data[length]);
	putchar('\n');
    }
    free(sysex);
}

struct pat_header {
    char magic[12];
    char version[10];
    char description[60];
    unsigned char instruments;
    char voices;
    char channels;
    unsigned short nr_waveforms;
    unsigned short master_volume;
    unsigned int data_size;
};

struct sample_header {
    char name[7];
    unsigned char fractions;
    int len;
    int loop_start;
    int loop_end;
    unsigned short base_freq;
    int low_note;
    int high_note;
    int base_note;
    short detune;
    unsigned char panning;

    unsigned char envelope_rate[6];
    unsigned char envelope_offset[6];

    unsigned char tremolo_sweep;
    unsigned char tremolo_rate;
    unsigned char tremolo_depth;

    unsigned char vibrato_sweep;
    unsigned char vibrato_rate;
    unsigned char vibrato_depth;

    char modes;

    short scale_frequency;
    unsigned short scale_factor;
};

struct patch_info *patch;
int spaceleft, totalspace;

void gus_reload_8_bit();

int gus_load(pgm)
int pgm;
{
    int i, j, patfd, offset;
    struct pat_header header;
    struct sample_header sample;
    char buf[256], name[256];
    struct stat info;

    if (pgm < 0) {
	use8bit = force8bit;
	GUS_NUMVOICES(gus_dev, 32);
	SEQ_DUMPBUF();
	for (i = 0; i < 256; i++)
	    patchloaded[i] = 0;
	if (ioctl(seqfd, SNDCTL_SEQ_RESETSAMPLES, &gus_dev) == -1) {
	    perror("Sample reset");
	    exit(-1);
	}
	spaceleft = gus_dev;
	ioctl(seqfd, SNDCTL_SYNTH_MEMAVL, &spaceleft);
	totalspace = spaceleft;
	return 0;
    }
    if (patchloaded[pgm] < 0)
	return -1;

    if (patchloaded[pgm] == 1)
	return 0;

    if (gmvoice[pgm] == NULL) {
	if (verbose > 1)
	    printf("No GUS patch for program %d\n", pgm);
	patchloaded[pgm] = -1;
	return -1;
    }
    sprintf(name, PATCH_PATH1 "/%s.pat", gmvoice[pgm]);

    if (stat(name, &info) == -1) {
	sprintf(name, PATCH_PATH2 "/%s.pat", gmvoice[pgm]);
	if (stat(name, &info) == -1) {
	    perror(name);
	    return -1;
	}
    }
    if ((patfd = open(name, O_RDONLY, 0)) == -1) {
	perror(name);
	return -1;
    }
    if (spaceleft < info.st_size) {
	if (!use8bit)
	    gus_reload_8_bit();
	if (use8bit)
	    if (spaceleft < info.st_size / 2) {
		fprintf(stderr, "Patch %s won't fit\n", name);
		close(patfd);
		patchloaded[pgm] = -1;	/* mark no space for that patch */
		return -1;
	    }
    }
    if (read(patfd, buf, 0xef) != 0xef) {
	fprintf(stderr, "%s: Short file\n", name);
	close(patfd);
	return -1;
    }
    memcpy((char *) &header, buf, sizeof(header));

    if (strncmp(header.magic, "GF1PATCH110", 12)) {
	fprintf(stderr, "%s: Not a patch file\n", name);
	close(patfd);
	return -1;
    }
    if (strncmp(header.version, "ID#000002", 10)) {
	fprintf(stderr, "%s: Incompatible patch file version\n", name);
	close(patfd);
	return -1;
    }
    header.nr_waveforms = *(unsigned short *) &buf[85];
    header.master_volume = *(unsigned short *) &buf[87];

    offset = 0xef;

    for (i = 0; i < header.nr_waveforms; i++) {

	if (lseek(patfd, offset, 0) == -1) {
	    perror(name);
	    close(patfd);
	    return -1;
	}
	if (read(patfd, &buf, sizeof(sample)) != sizeof(sample)) {
	    fprintf(stderr, "%s: Short file\n", name);
	    close(patfd);
	    return -1;
	}
	memcpy((char *) &sample, buf, sizeof(sample));

	/*
	 * Since some fields of the patch record are not 32bit aligned, we must
	 * handle them specially.
	 */
	sample.low_note = *(int *) &buf[22];
	sample.high_note = *(int *) &buf[26];
	sample.base_note = *(int *) &buf[30];
	sample.detune = *(short *) &buf[34];
	sample.panning = (unsigned char) buf[36];

	memcpy(sample.envelope_rate, &buf[37], 6);
	memcpy(sample.envelope_offset, &buf[43], 6);

	sample.tremolo_sweep = (unsigned char) buf[49];
	sample.tremolo_rate = (unsigned char) buf[50];
	sample.tremolo_depth = (unsigned char) buf[51];

	sample.vibrato_sweep = (unsigned char) buf[52];
	sample.vibrato_rate = (unsigned char) buf[53];
	sample.vibrato_depth = (unsigned char) buf[54];
	sample.modes = (unsigned char) buf[55];
	sample.scale_frequency = *(short *) &buf[56];
	sample.scale_factor = *(unsigned short *) &buf[58];

	offset = offset + 96;
	patch = (struct patch_info *) malloc(sizeof(*patch) + sample.len);

	if (patch == NULL) {
	    fprintf(stderr, "no memory for loading patch, ignoring\n");
	    close(patfd);
	    return -1;
	}
	patch->key = GUS_PATCH;
	patch->device_no = gus_dev;
	patch->instr_no = pgm;
	patch->mode = sample.modes | WAVE_TREMOLO |
	    WAVE_VIBRATO | WAVE_SCALE;
	patch->len = (use8bit ? sample.len / 2 : sample.len);
	patch->loop_start =
	    (use8bit ? sample.loop_start / 2 : sample.loop_start);
	patch->loop_end = (use8bit ? sample.loop_end / 2 : sample.loop_end);
	patch->base_note = sample.base_note;
	patch->high_note = sample.high_note;
	patch->low_note = sample.low_note;
	patch->base_freq = sample.base_freq;
	patch->detuning = sample.detune;
	patch->panning = (sample.panning - 7) * 16;

	memcpy(patch->env_rate, sample.envelope_rate, 6);
	for (j = 0; j < 6; j++)	/* tone things down slightly */
	    patch->env_offset[j] =
		(736 * sample.envelope_offset[j] + 384) / 768;

	if (reverb)
	    if (pgm < 120)
		patch->env_rate[3] = (2 << 6) | (12 - (reverb >> 4));
	    else if (pgm > 127)
		patch->env_rate[1] = (3 << 6) | (63 - (reverb >> 1));

	patch->tremolo_sweep = sample.tremolo_sweep;
	patch->tremolo_rate = sample.tremolo_rate;
	patch->tremolo_depth = sample.tremolo_depth;

	patch->vibrato_sweep = sample.vibrato_sweep;
	patch->vibrato_rate = sample.vibrato_rate;
	patch->vibrato_depth = sample.vibrato_depth;

	patch->scale_frequency = sample.scale_frequency;
	patch->scale_factor = sample.scale_factor;

	patch->volume = header.master_volume;

	if (lseek(patfd, offset, 0) == -1) {
	    perror(name);
	    close(patfd);
	    return -1;
	}
	if (read(patfd, patch->data, sample.len) != sample.len) {
	    fprintf(stderr, "%s: Short file\n", name);
	    close(patfd);
	    return -1;
	}
	if (patch->mode & WAVE_16_BITS && use8bit) {
	    patch->mode &= ~WAVE_16_BITS;
	    /* cut out every other byte to make 8-bit data from 16-bit */
	    for (j = 0; j < patch->len; j++)
		patch->data[j] = patch->data[1 + j * 2];
	}
	SEQ_WRPATCH(patch, sizeof(*patch) + patch->len);
	free(patch);
	offset = offset + sample.len;
    }
    close(patfd);
    spaceleft = gus_dev;
    ioctl(seqfd, SNDCTL_SYNTH_MEMAVL, &spaceleft);
    patchloaded[pgm] = 1;
    if (verbose > 1)
	printf("Patch %03d = %8s.pat (%d bytes used, %d free)\n", pgm,
	       gmvoice[pgm], totalspace - spaceleft, spaceleft);
    return 0;
}

void gus_reload_8_bit()
{
    int i;

    fprintf(stderr,"8-bit patches forced by lack of memory\n");
    if (ioctl(seqfd, SNDCTL_SEQ_RESETSAMPLES, &gus_dev) == -1) {
	perror("Sample reset");
	exit(-1);
    }
    spaceleft = gus_dev;
    ioctl(seqfd, SNDCTL_SYNTH_MEMAVL, &spaceleft);
    totalspace = spaceleft;
    use8bit = 1;
    for (i = 0; i < 256; i++)
	if (patchloaded[i] > 0) {
	    patchloaded[i] = 0;
	    gus_load(i);
	}
}
    
char fm_volume_table[128] =
{-63, -48, -40, -35, -32, -29, -27, -26,	/* 0 -   7 */
 -24, -23, -21, -20, -19, -18, -18, -17,	/* 8 -  15 */
 -16, -15, -15, -14, -13, -13, -12, -12,	/* 16 -  23 */
 -11, -11, -10, -10, -10, -9, -9, -8,	/* 24 -  31 */
 -8, -8, -7, -7, -7, -6, -6, -6,/* 32 -  39 */
 -5, -5, -5, -5, -4, -4, -4, -4,/* 40 -  47 */
 -3, -3, -3, -3, -2, -2, -2, -2,/* 48 -  55 */
 -2, -1, -1, -1, -1, 0, 0, 0,	/* 56 -  63 */
 0, 0, 0, 1, 1, 1, 1, 1,	/* 64 -  71 */
 1, 2, 2, 2, 2, 2, 2, 2,	/* 72 -  79 */
 3, 3, 3, 3, 3, 3, 3, 4,	/* 80 -  87 */
 4, 4, 4, 4, 4, 4, 4, 5,	/* 88 -  95 */
 5, 5, 5, 5, 5, 5, 5, 5,	/* 96 - 103 */
 6, 6, 6, 6, 6, 6, 6, 6,	/* 104 - 111 */
 6, 7, 7, 7, 7, 7, 7, 7,	/* 112 - 119 */
 7, 7, 7, 8, 8, 8, 8, 8};	/* 120 - 127 */

int new_fm_vol(int volbyte, int mainvol)
{
    int oldvol, newvol, n;

    oldvol = 0x3f - (volbyte & 0x3f);
    newvol = fm_volume_table[mainvol] + oldvol;

    if (newvol > 0x3f) newvol = 0x3f;
    else if (newvol < 0) newvol = 0;
    n = 0x3f - (newvol & 0x3f);
    return( (volbyte & 0xc0) | (n & 0x3f) );
}

void loadfm()
{
    int sbfd, i, n, voice_size, data_size;
    char buf[60];
    struct sbi_instrument instr;
    int mainvol = 127;	/* note: change this to taste */

    if (wantopl3) {
	voice_size = 60;
	if (ioctl(seqfd, SNDCTL_FM_4OP_ENABLE, &sb_dev) == -1) {
	    fprintf(stderr, "Could not set 4-op fm mode\n");
	    exit(-1);
	}
	sbfd = open(O3MELODIC, O_RDONLY, 0);
    } else {
	voice_size = 52;
	sbfd = open(SBMELODIC, O_RDONLY, 0);
    }
    if (sbfd == -1) {
	perror("open fm library");
	exit(-1);
    }
    instr.device = sb_dev;

    for (i = 0; i < 128; i++) {
	if (read(sbfd, buf, voice_size) != voice_size) {
	    fprintf(stderr, "Short fm library file\n");
	    exit(-1);
	}
	instr.channel = i;

	if (voice_size == 52) {
	    instr.key = FM_PATCH;
	    data_size = 11;
	} else if ((buf[49] & 0x3f) == 0x3f && (buf[50] & 0x3f) == 0x3f) {
	    instr.key = FM_PATCH;
	    data_size = 11;
	} else {
	    instr.key = OPL3_PATCH;
	    data_size = 22;
	}

	if (instr.key == FM_PATCH) {
	    buf[39] = new_fm_vol(buf[39], mainvol);
	    if (buf[46] & 1) buf[38] = new_fm_vol(buf[38], mainvol);
	    buf[46] = (buf[46] & 0xcf) | 0x30;
/*
		((i << 4) & 0x30 ? (i << 4) & 0x30 : 0x30);
*/
	    if (reverb) {
		unsigned val;
		val = buf[43] & 0x0f;
		if (val > 0) val--;
		buf[43] = (buf[43]&0xf0) | val;
	    }
	}
	else {
	    int mode;
	    if (buf[46]&1) mode = 2; else mode = 0;
	    if (buf[57]&1) mode++;
	    buf[50] = new_fm_vol(buf[50], mainvol);
	    if (mode == 3) buf[49] = new_fm_vol(buf[49], mainvol);
	    if (mode == 1) buf[39] = new_fm_vol(buf[39], mainvol);
	    if (mode == 2 || mode == 3) buf[38] = new_fm_vol(buf[38], mainvol);
	    buf[46] = (buf[46] & 0xcf) | 0x30;
/*
		((i << 4) & 0x30 ? (i << 4) & 0x30 : 0x30);
*/
	    buf[57] = (buf[57] & 0xcf) | 0x30;
/*
		((i << 4) & 0x30 ? (i << 4) & 0x30 : 0x30);
*/
	    if (mode == 1 && reverb) {
		unsigned val;
		val = buf[43] & 0x0f;
		if (val > 0) val--;
		buf[43] = (buf[43]&0xf0) | val;
		val = buf[54] & 0x0f;
		if (val > 0) val--;
		buf[54] = (buf[54]&0xf0) | val;
	    }
	}

	for (n = 0; n < 32; n++)
	    instr.operators[n] = (n < data_size) ? buf[36 + n] : 0;

	SEQ_WRPATCH(&instr, sizeof(instr));
    }
    close(sbfd);

    if (wantopl3)
	sbfd = open(O3DRUMS, O_RDONLY, 0);
    else
	sbfd = open(SBDRUMS, O_RDONLY, 0);

    for (i = 128; i < 175; i++) {
	if (read(sbfd, buf, voice_size) != voice_size) {
	    fprintf(stderr, "Short fm library file\n");
	    exit(-1);
	}
	instr.channel = i;

	if (voice_size == 52) {
	    instr.key = FM_PATCH;
	    data_size = 11;
	} else if ((buf[49] & 0x3f) == 0x3f && (buf[50] & 0x3f) == 0x3f) {
	    instr.key = FM_PATCH;
	    data_size = 11;
	} else {
	    instr.key = OPL3_PATCH;
	    data_size = 22;
	}

	if (instr.key == FM_PATCH) {
	    buf[39] = new_fm_vol(buf[39], mainvol);
	    if (buf[46] & 1) buf[38] = new_fm_vol(buf[38], mainvol);
	    buf[46] = (buf[46] & 0xcf) |
		((i << 4) & 0x30 ? (i << 4) & 0x30 : 0x30);
	    if (reverb) {
		unsigned val;
		val = buf[43] & 0x0f;
		if (val > 0) val--;
		buf[43] = (buf[43]&0xf0) | val;
	    }
	}
	else {
	    int mode;
	    if (buf[46]&1) mode = 2; else mode = 0;
	    if (buf[57]&1) mode++;
	    buf[50] = new_fm_vol(buf[50], mainvol);
	    if (mode == 3) buf[49] = new_fm_vol(buf[49], mainvol);
	    if (mode == 1) buf[39] = new_fm_vol(buf[39], mainvol);
	    if (mode == 2 || mode == 3) buf[38] = new_fm_vol(buf[38], mainvol);
	    buf[46] = (buf[46] & 0xcf) |
		((i << 4) & 0x30 ? (i << 4) & 0x30 : 0x30);
	    buf[57] = (buf[57] & 0xcf) |
		((i << 4) & 0x30 ? (i << 4) & 0x30 : 0x30);
	    if (mode == 1 && reverb) {
		unsigned val;
		val = buf[43] & 0x0f;
		if (val > 0) val--;
		buf[43] = (buf[43]&0xf0) | val;
		val = buf[54] & 0x0f;
		if (val > 0) val--;
		buf[54] = (buf[54]&0xf0) | val;
	    }
	}

	for (n = 0; n < 32; n++)
	    instr.operators[n] = (n < data_size) ? buf[36 + n] : 0;

	SEQ_WRPATCH(&instr, sizeof(instr));
    }
    close(sbfd);
}
