// trackwin.cc,v 2.3 1995/06/22 18:41:16 andreas Exp

/*
**  jazz - a midi sequencer for Linux
**
**  Copyright (C) 1994 Andreas Voss (andreas@avix.rhein-neckar.de)
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "trackwin.h"
#include "song.h"
#include "mstdfile.h"
#include "filter.h"
#include "pianowin.h"
#include "command.h"
#include "dialogs.h"
#include "player.h"
#include <stdlib.h>

tTrackWin *TrackWin = 0;


tTrackWin::tTrackWin(wxFrame *frame, char *title, tSong *song)
  : tEventWin(frame, title, song)
{
  NextWin = new tPianoWin(frame, "Piano Roll", Song);
  NextWin->Create();
  nBars = 0;
  RecInfo.Track = 0;
  RecInfo.Muted = 0;

  CounterMode = CmProgram;
  NumberMode = NmMidiChannel;
}


Bool tTrackWin::OnClose()
{
  if (Midi->Playing)
  {
    Midi->StopPlay();
#ifndef wx_msw
    sleep(1);
#endif
  }
  delete Midi;
  delete NextWin;
  return TRUE;
}




void tTrackWin::Setup()
{
  float x, y;

  tEventWin::Setup();

  Canvas->GetTextExtent("H", &x, &y);
  LittleBit = (int)(x/2);

  Canvas->GetTextExtent("HXWjgi", &x, &y);
  hLine = (int)y + LittleBit;
  hTop = hFixedFont + 2 * LittleBit;

  Canvas->GetTextExtent("99", &x, &y);
  wNumber = (int)x + LittleBit;

  Canvas->GetTextExtent("Normal Trackname", &x, &y);
  wName = (int)x + LittleBit;

  Canvas->GetTextExtent("m", &x, &y);
  wState = (int)x + LittleBit;

  Canvas->GetTextExtent("999", &x, &y);
  wPatch = (int)x + 2 * LittleBit;

  wLeft = wNumber + wName + wState + wPatch + 1;

  UnMark();
}



// ************************************************************************
// Menubar
// ************************************************************************

#define MEN_LOAD 	1
#define MEN_SAVE 	2
#define MEN_QUIT 	3
#define MEN_ABOUT 	4
#define MEN_TWSETTING	5
#define MEN_FILTER	6
#define MEN_METERCH	7
#define MEN_MERGE	8
#define MEN_NEW		9
#define MEN_COPY	10
#define MEN_DELETE	11
#define MEN_QUANTIZE	12
#define MEN_UNDO	13
#define MEN_DEBUG	14
#define MEN_SETCHAN	15
#define MEN_TRANSP	16
#define MEN_VELOC	17
#define MEN_LENGTH	18
#define MEN_RESET       19
#define MEN_SONG	20
#define MEN_TMERGE	21
#define MEN_TSPLIT	22
#define MEN_MIXER	23
#define MEN_SOUND	24
#define MEN_VIBRATO	25
#define MEN_ENVELOPE	26
#define MEN_WHEEL	27
#define MEN_EFFECTS	28
#define MEN_CLOCKSRC	29


void tTrackWin::CreateMenu()
{
  wxMenuBar *menu_bar = NULL;
  wxMenu *file_menu = new wxMenu;
  file_menu->Append(MEN_LOAD,   "&Load ...");
  file_menu->Append(MEN_MERGE,  "&Merge ...");
  file_menu->Append(MEN_NEW,    "&New ...");
  file_menu->Append(MEN_SAVE,   "&Save ...");
  file_menu->Append(MEN_QUIT,   "&Quit");

  wxMenu *edit_menu = new wxMenu;
  edit_menu->Append(MEN_COPY,      "&Replicate ...");
  edit_menu->Append(MEN_DELETE,    "&Delete ...");
  edit_menu->Append(MEN_QUANTIZE,  "&Quantize ...");
  edit_menu->Append(MEN_SETCHAN,   "&Set MIDI Channel ...");
  edit_menu->Append(MEN_TRANSP,    "&Transpose ...");
  edit_menu->Append(MEN_VELOC,     "&Velocity ...");
  edit_menu->Append(MEN_LENGTH,    "&Length ...");
  // edit_menu->Append(MEN_DEBUG,  "&Debug...");

  wxMenu *parts_menu = new wxMenu;
  parts_menu->Append(MEN_MIXER,    "&Mixer ...");
  parts_menu->Append(MEN_SOUND,    "&Sound ...");
  parts_menu->Append(MEN_VIBRATO,  "&Vibrato ...");
  parts_menu->Append(MEN_ENVELOPE,    "&Envelope ...");
  parts_menu->Append(MEN_WHEEL,    "&Wheel ...");

  wxMenu *setting_menu = new wxMenu;
  setting_menu->Append(MEN_FILTER,    "&Filter ...");
  setting_menu->Append(MEN_TWSETTING, "&Window ...");
  setting_menu->Append(MEN_SONG,      "&Song ...");
  setting_menu->Append(MEN_EFFECTS,   "&Effects ...");
  setting_menu->Append(MEN_CLOCKSRC,  "&Clock source ...");

  wxMenu *misc_menu = new wxMenu;
  misc_menu->Append(MEN_UNDO,     "&Undo ...");
  misc_menu->Append(MEN_TMERGE,   "&Merge Tracks ...");
  misc_menu->Append(MEN_TSPLIT,   "&Split Tracks ...");
  misc_menu->Append(MEN_METERCH,  "&Meterchange ...");
  misc_menu->Append(MEN_RESET,    "&Reset Midi (GS)");

  wxMenu *help_menu = new wxMenu;
  help_menu->Append(MEN_ABOUT, "&About");

  menu_bar = new wxMenuBar;
  menu_bar->Append(file_menu,    "&File");
  menu_bar->Append(edit_menu,    "&Edit");
  menu_bar->Append(parts_menu,   "&Parts");
  menu_bar->Append(setting_menu, "&Settings");
  menu_bar->Append(misc_menu,    "&Misc");
  menu_bar->Append(help_menu,    "&About");

  SetMenuBar(menu_bar);
}



void tTrackWin::OnMenuCommand(int id)
{
  char *s;
  static char *lasts = "noname.mid";

  switch (id)
  {
    case MEN_MERGE:
      if (MixerForm) {
	wxMessageBox("Quit sounds dialog first.");
	break;
      }
      s = wxFileSelector("Merge File", NULL, NULL, NULL, "*.mid");
      if (s)
      {
      	if ( strcmp( lasts, "noname.mid" ) ) {
		char *savefile = new char[ strlen("Save file '' before merge ?") + strlen(lasts) + 1 ];
		sprintf( savefile, "Save file '%s' before merge ?", lasts );
      		if (wxMessageBox(savefile, "Save ?", wxYES_NO) == wxYES) {
       		 	tStdWrite io;
		        Song->Write(io, lasts);
		}
		delete[] savefile;
      	}
        tStdRead io;
        Song->Read(io, s);
        Redraw();
      }
      break;

    case MEN_NEW:
      if (MixerForm) {
	wxMessageBox("Quit sounds dialog first.");
	break;
      }
      if (wxMessageBox("Clear Song?", "Shure?", wxOK | wxCANCEL) == wxOK)
      {
	Song->Clear();
	Redraw();
	if ( strcmp( lasts, "noname.mid" ) ) {
		free( lasts );
		lasts = "noname.mid";
	}
        SetTitle(lasts);
      }
      break;

    case MEN_LOAD:
      if (MixerForm) {
	wxMessageBox("Quit sounds dialog first.");
	break;
      }
      s = wxFileSelector("Load File", NULL, NULL, NULL, "*.mid");
      if (s)
      {
      	if ( strcmp( lasts, "noname.mid" ) ) {
		char *savefile = new char[ strlen("Save file '' before load ?") + strlen(lasts) + 1 ];
		sprintf( savefile, "Save file '%s' before load ?", lasts );
      		if (wxMessageBox(savefile, "Save ?", wxYES_NO) == wxYES) {
       		 	tStdWrite io;
		        Song->Write(io, lasts);
		}
		delete[] savefile;
      	}
        tStdRead io;
        Song->Clear();
        Song->Read(io, s);
	if ( strcmp( lasts, "noname.mid" ) ) free( lasts );
	lasts = strdup(s);
        SetTitle(s);
        Redraw();
      }
      break;

    case MEN_SAVE:
      {
        char *s;
        if (strcmp (lasts, "noname.mid"))
        {
	  char file[256], path[256];
	  strcpy(file, wxFileNameFromPath(lasts));
	  strcpy(path, wxPathOnly(lasts));
	  s = wxFileSelector("Save File", path, file, NULL, "*.mid");
	}
	else
	  s = wxFileSelector("Save File", NULL, NULL, NULL, "*.mid");
	if (s)
	{
	  tStdWrite io;
	  Song->Write(io, s);
	}
      }
      break;

#if 0
      int saved = 0;
      if ( strcmp( lasts, "noname.mid" ) ) {
	char *savefile = new char[ strlen("Save in '' ? (Press No for browser)") + strlen(lasts) + 1 ];
	sprintf( savefile, "Save in '%s' ? (Press No for browser)", lasts );
      	if (wxMessageBox(savefile, "Save ?", wxYES_NO) == wxYES) {
        	tStdWrite io;
	        Song->Write(io, lasts);
		saved = 1;
	}
	delete[] savefile;
      }
      if (!saved) {
      	s = wxFileSelector("Save File", NULL, NULL, NULL, "*.mid");
      	if (s)
      	{
       	 tStdWrite io;
       	 Song->Write(io, s);
	 if ( strcmp( lasts, "noname.mid" ) ) free( lasts );
	 lasts = strdup(s);
         SetTitle(s);
      	}
      }
      break;
#endif

    case MEN_QUIT:
        switch (wxMessageBox("Save before Quit?", "Save ?", wxYES_NO | wxCANCEL)) {
          case wxCANCEL:
            return;

          case wxYES:
            {
              char *s;
              if (strcmp (lasts, "noname.mid"))
              {
		char file[256], path[256];
		strcpy(file, wxFileNameFromPath(lasts));
		strcpy(path, wxPathOnly(lasts));
		s = wxFileSelector("Save File", path, file, NULL, "*.mid");
	      }
	      else
	        s = wxFileSelector("Save File", NULL, NULL, NULL, "*.mid");
	      if (s)
	      {
	       tStdWrite io;
	       Song->Write(io, s);
	      }
	    }
	}
      OnClose();
      delete this;
      break;

#if 0

      int answer = wxOK;
      char *savefile = new char[ strlen("Save in '' ? (YES saves and quits, OK gives browser)") + strlen(lasts) + 1 ];
      sprintf( savefile, "Save in '%s' ? (YES saves and quits, OK gives browser)", lasts );
      if ((answer = wxMessageBox(savefile, "Save ?", wxYES_NO | wxOK | wxCANCEL)) == wxYES) {
	tStdWrite io;
	Song->Write(io, lasts);
      }
      delete[] savefile;
      if (answer == wxCANCEL) break;
      if (answer == wxOK) {
      	s = wxFileSelector("Save File", NULL, NULL, NULL, "*.mid");
      	if (s)
      	{
       	 tStdWrite io;
       	 Song->Write(io, s);
      	}
	else {
         if ( strcmp( lasts, "noname.mid" ) ) break;
        }
      }
      OnClose();
      delete this;
      break;
#endif

    case MEN_TWSETTING: SettingsDialog(0); break;
    case MEN_COPY: 	MenCopy(); break;
    case MEN_QUANTIZE: 	MenQuantize(); break;
    case MEN_DELETE: 	MenDelete(); break;
    case MEN_SETCHAN: 	MenSetChannel(); break;
    case MEN_TRANSP: 	MenTranspose(); break;
    case MEN_VELOC: 	MenVelocity(); break;
    case MEN_LENGTH: 	MenLength(); break;
    case MEN_METERCH:	MenMeterChange(); break;
    case MEN_RESET:	Midi->AllNotesOff(1); break;

    case MEN_SONG: 	MenSongSettings(); break;
    case MEN_TSPLIT: 	
      if (MixerForm) {
	wxMessageBox("Quit sounds dialog first.");
	break;
      }
      MenSplitTracks(); 
      break;
    case MEN_TMERGE: 	
      if (MixerForm) {
	wxMessageBox("Quit sounds dialog first.");
	break;
      }
      MenMergeTracks(); 
      break;

    case MEN_UNDO:
      Song->Undo();
      Redraw();
      break;

    case MEN_DEBUG:
            break;

    case MEN_MIXER:
      MenMixer();
      break;

    case MEN_VIBRATO:
      MenVibrato();
      break;

    case MEN_SOUND:
      MenSound();
      break;

    case MEN_ENVELOPE:
      MenEnvelope();
      break;

    case MEN_WHEEL:
      MenWheel();
      break;

    case MEN_FILTER:
      Filter->Dialog(0);
      break;

    case MEN_EFFECTS:
      MenEffects();
      break;

    case MEN_CLOCKSRC:
      MenClockSource(); 
      break;

    case MEN_ABOUT:
      extern const char *about_text;
      wxMessageBox( (char *) about_text, "About");

      break;
  }
}


// ************************************************************************
// Painting
// ************************************************************************



void tTrackWin::DrawCounters()
{
  char *str;
  int i;
  switch (CounterMode)
  {
    case CmProgram: str = "Prg"; break;
    case CmBank	  : str = "Bnk"; break;
    case CmVolume : str = "Vol"; break;
    case CmPan    : str = "Pan"; break;
    case CmReverb : str = "Rev"; break;
    case CmChorus : str = "Cho"; break;
    default       : str = "???"; break;
  }
  LineText(xPatch, yy-1, wPatch, str, hTop);

  dc->SetClippingRegion(xPatch, yEvents, xPatch + wPatch, yEvents + hEvents);
  for (i = FromLine; i < ToLine; i++)
  {
    tTrack *t = Song->GetTrack(i);
    if (t)
    {
      char buf[20];
      int  val;
      switch (CounterMode)
      {
        case CmProgram: val = t->GetPatch(); break;
        case CmBank   : val = t->GetBank(); break;
        case CmVolume : val = t->GetVolume(); break;
        case CmPan    : val = t->GetPan(); break;
        case CmReverb : val = t->GetReverb(); break;
        case CmChorus : val = t->GetChorus(); break;
        default : val = 0; break;
      }
      sprintf(buf, "%3d", val);
      LineText(xPatch, Line2y(i), wPatch, buf);
    }
    else
      LineText(xPatch, Line2y(i), wPatch, "?");
  }
  dc->DestroyClippingRegion();
}




void tTrackWin::DrawNumbers()
{
  char *str;
  int i;
  switch (NumberMode)
  {
    case NmTrackNr     : str = "T"; break;
    case NmMidiChannel : str = "M"; break;
    default            : str = "?"; break;
  }
  LineText(xNumber, yy-1, wNumber, str, hTop);

  dc->SetClippingRegion(xNumber, yEvents, xNumber + wNumber, yEvents + hEvents);
  for (i = FromLine; i < ToLine; i++)
  {
    tTrack *t = Song->GetTrack(i);
    if (t)
    {
      char buf[20];
      int  val;
      switch (NumberMode)
      {
        case NmTrackNr: val = i; break;
        case NmMidiChannel : val = t->Channel; break;
        default : val = 0; break;
      }
      sprintf(buf, "%02d", val);
      LineText(xNumber, Line2y(i), wNumber, buf);
    }
    else
      LineText(xNumber, Line2y(i), wNumber, "?");
  }
  dc->DestroyClippingRegion();
}



void tTrackWin::DrawSpeed(int Value)
{
  char buf[50];
  if (Value < 0)
    Value = Song->GetTrack(0)->GetSpeed();
  sprintf(buf, "speed: %3d", Value);
  LineText(xName, yy-1, wName, buf, hTop);
}


void tTrackWin::OnPaint(int What, long x, long y, long w, long h)
{
  tEventWin::OnPaint(What, x, y, w, h);

  xNumber  = xx;
  xName    = xNumber  + wNumber;
  xState   = xName    + wName;
  xPatch   = xState   + wState;

  long StopClk;
  tBarInfo BarInfo(Song);
  char buf[20];

  dc->BeginDrawing();
  dc->Clear();

  #define VLine(x) DrawLine(x,  yy, x, yEvents+hEvents)
  #define HLine(y) DrawLine(xx, y, xx + ww, y)

  dc->SetPen(wxBLACK_PEN);

  // vertikale Linien
  dc->VLine(xNumber);
  dc->VLine(xName);
  dc->VLine(xState);
  dc->VLine(xPatch);
  dc->VLine(xEvents);
  dc->VLine(xEvents-1);
  dc->HLine(yEvents);
  dc->HLine(yEvents-1);


  // Taktstriche und -nummern

  BarInfo.SetClock(FromClock);
  StopClk = x2Clock(xx + ww);
  nBars = 0;
  dc->SetPen(wxGREY_PEN);
  while (1)
  {
    x = Clock2x(BarInfo.Clock);
    if (x > xx + ww)
      break;
    if (x >= xEvents)	// so ne Art clipping
    {
      if ((BarInfo.BarNr % 4) == 0)
      {
	dc->SetPen(wxBLACK_PEN);
	sprintf(buf, "%d", BarInfo.BarNr + 1);
	dc->DrawText(buf, x + LittleBit, yEvents - hLine);
	dc->DrawLine(x, yEvents - hLine, x, yEvents + hEvents);
	dc->SetPen(wxGREY_PEN);
      }
      else
	dc->DrawLine(x, yEvents, x, yEvents+hEvents);

      if (nBars < MaxBars)	// x-Koordinate fuer MouseAction->Snap()
	xBars[nBars++] = x;

    }
    BarInfo.Next();
  }
  dc->SetPen(wxBLACK_PEN);


  DrawNumbers();
  DrawSpeed();
  DrawCounters();


  // je Track Nummer, Name, State, Patch anzeigen

  dc->SetClippingRegion(xx, yEvents, xx+ww, yEvents + hEvents);
  int TrackNr = FromLine;
  for (y = Line2y(TrackNr); y < yEvents + hEvents; y += hLine)
  {
    dc->HLine(y);
    tTrack *Track = Song->GetTrack(TrackNr);
    if (Track)
    {
      // TrackName
      dc->DrawText(Track->GetName(), xName + LittleBit, y + LittleBit);
      // TrackStatus
      dc->DrawText(Track->GetStateChar(), xState + LittleBit, y + LittleBit);
    }
    ++ TrackNr;
  }
  dc->DestroyClippingRegion();

  // Events zeichnen
  dc->SetClippingRegion(xEvents, yEvents, wEvents, hEvents);
  TrackNr = FromLine;
  for (y = Line2y(TrackNr); y < yEvents + hEvents; y += hLine)
  {
    tTrack *Track = Song->GetTrack(TrackNr);
    if (Track)
    {
      tEventIterator Iterator(Track);
      tEvent *e = Iterator.Range(FromClock, StopClk);
      float y0 = y + LittleBit;
      float y1 = y + hLine - LittleBit;

      if (UseColors)
      {
	while (e)	// slow!
	{
	  float x = Clock2x(e->Clock);
	  dc->SetPen(e->GetPen());
	  dc->DrawLine(x, y0, x, y1);
	  e = Iterator.Next();
	}
	dc->SetPen(wxBLACK_PEN);
      }
      else
      {
        float xblack = -1.0;
	while (e)
	{
	  float x = Clock2x(e->Clock);
#ifdef SLOW_MACHINE
          // Avoid painting events ON the bar
          if ( !(e->Clock % BarInfo.TicksPerBar) ) x = x + 1;
#endif
	  if (x > xblack)
	  {
	    dc->DrawLine(x, y0, x, y1);
	    xblack = x;
#ifdef SLOW_MACHINE
	    xblack = x + 4;
#endif
	  }
	  e = Iterator.Next();
	}
      }
    }

    ++ TrackNr;
  }

  // strange bug? after SetPen(GREY_PEN) XOR-Painting doesnt work anymore
  // unless SetBackground(WHITE_BRUSH) is called
  dc->SetBackground(wxWHITE_BRUSH);

  SnapSel->Draw(xEvents, yEvents, wEvents, hEvents);

  if (Marked.x > 0)
    LineText((long)Marked.x, (long)Marked.y, (long)Marked.w, ">");

  dc->DestroyClippingRegion();
  DrawPlayPosition();
  dc->EndDrawing();
}

// ************************************************************************
// Utilities
// ************************************************************************

long tTrackWin::x2xBar(long x)
{
  for (int i = 1; i < nBars; i++)
    if (x < xBars[i])
      return xBars[i - 1];
  return -1;
}


long tTrackWin::x2wBar(long x)
{
  for (int i = 1; i < nBars; i++)
    if (x < xBars[i])
      return xBars[i] - xBars[i - 1];
  return 0;
}


tTrack *tTrackWin::y2Track(long y)
{
  return Song->GetTrack(y2Line(y));
}

void tTrackWin::Mark(long x, long y)
{
  Marked.x = x2xBar(x);
  Marked.y = y2yLine(y);
  Marked.w = x2wBar(x);
  Marked.h = hLine;
  LineText((int)Marked.x, (int)Marked.y, (int)Marked.w, ">");
}

void tTrackWin::UnMark()
{
  Marked.x = -1;
}


// ********************************************************************
// PlayPosition
// ********************************************************************

void tTrackWin::NewPlayPosition(long Clock)
{
  if (!SnapSel->Active && ((Clock > (FromClock + 5 * ToClock) / 6L) || (Clock < FromClock)) && (Clock >= 0L) )
  {
    long x = Clock2x(Clock);
    Canvas->SetScrollPosition(x - wLeft, yy);
  }
  tEventWin::NewPlayPosition(Clock);
}

// ********************************************************************
// Snapper
// ********************************************************************

void tTrackWin::SnapSelStart(wxMouseEvent &e)
{
  SnapSel->SetXSnap(nBars, xBars);
  SnapSel->SetYSnap(Line2y(FromLine), yEvents + hEvents, hLine);
}


void tTrackWin::SnapSelStop(wxMouseEvent &e)
{
  if (SnapSel->Selected)
  {
    Filter->FromTrack = y2Line((long)SnapSel->r.y);
    Filter->ToTrack   = y2Line((long)(SnapSel->r.y + SnapSel->r.h - 1));
    Filter->FromClock = x2BarClock((long)SnapSel->r.x + 1);
    Filter->ToClock   = x2BarClock((long)(SnapSel->r.x + SnapSel->r.w + 1));
  }
}



// --------------------------------------------------------------------
// Tracknummer
// --------------------------------------------------------------------

void tTrackWin::MouseNumber(wxMouseEvent &e)
{
  if (e.LeftDown())
  {
    float x, y;
    e.Position(&x, &y);
    tTrack *t = y2Track((long)y);
    if (t != 0)
    {
      tRect r;
      r.x = 0;
      r.y = y2yLine((long)y);
      r.w = Clock2x(t->GetLastClock()) + 1000;
      r.h = hLine;
      SnapSel->Select(r, xEvents, yEvents, wEvents, hEvents);
      SnapSelStop(e);
    }
  }
}

// ******************************************************************
// Mixer Dialog
// ******************************************************************

class tMixerDlg : public wxForm
{
 public:
  tEventWin *EventWin;
  tMixerDlg(tEventWin *w);
  void EditForm(wxPanel *panel);

  static tSliderArray SliderArray[16][MxParams];
  static void GetSliderVal( wxSlider& s );
  static void SetSliderVal( int chan, int type, int val );
  static void SliderChange( wxSlider& slider, wxCommandEvent& event );
  static tSliderArray MasterSliders[2];
  static void MasterChange( wxSlider& slider, wxCommandEvent& event );
  static void Dismiss( wxButton& button, wxCommandEvent& event );
};

tSliderArray tMixerDlg::SliderArray[16][MxParams];
tSliderArray tMixerDlg::MasterSliders[2];

tMixerDlg::tMixerDlg(tEventWin *w)
{
  EventWin = w;
  SliderArray[0][0].tw = 0;
}

void tMixerDlg::Dismiss( wxButton& button, wxCommandEvent& event )
{
  tEventWin *w = (tEventWin*) SliderArray[0][0].tw;

  w->MixerForm->Show(FALSE);
  delete w->MixerForm;
  w->MixerForm = 0;
}


void tMixerDlg::SetSliderVal( int chan, int type, int val ) {
	if (!SliderArray[0][0].tw || !SliderArray[0][0].tw->MixerForm) return;
	if (type == MxPan)
		SliderArray[chan][type].slider->SetValue( val - 65 );
	else
		SliderArray[chan][type].slider->SetValue( val );
}


void tMixerDlg::GetSliderVal( wxSlider& s ) {
	for (int chan = 0; chan < 16; chan++) {
		for (int type = MxVol; type < MxParams; type++) {
			if (SliderArray[chan][type].track == NULL) continue;
			switch (type) {
			case MxVol:
				if (SliderArray[chan][type].slider == &s) {
					SliderArray[chan][type].track->SetVolume(s.GetValue());
	  				SliderArray[0][0].tw->DrawCounters();
					return;
				}
			case MxPan:
				if (SliderArray[chan][type].slider == &s) {
					SliderArray[chan][type].track->SetPan(s.GetValue() + 65);
	  				SliderArray[0][0].tw->DrawCounters();
					return;
				}
			case MxRev:
				if (SliderArray[chan][type].slider == &s) {
					SliderArray[chan][type].track->SetReverb(s.GetValue());
	  				SliderArray[0][0].tw->DrawCounters();
					return;
				}
			case MxCho:
				if (SliderArray[chan][type].slider == &s) {
					SliderArray[chan][type].track->SetChorus(s.GetValue());
	  				SliderArray[0][0].tw->DrawCounters();
					return;
				}
			}
			
		}
	}
}

void tMixerDlg::SliderChange( wxSlider& slider, wxCommandEvent& event ) {
	GetSliderVal( slider );
}

void tMixerDlg::MasterChange( wxSlider& slider, wxCommandEvent& event ) {
	if (MasterSliders[MxVol].slider == &slider) {
		MasterSliders[MxVol].track->SetMasterVol(slider.GetValue());
		return;
	}
	if (MasterSliders[MxPan].slider == &slider) {
		MasterSliders[MxPan].track->SetMasterPan(slider.GetValue() + 65);
		return;
	}
}


void tMixerDlg::EditForm(wxPanel *panel)
{
const int slider_width = 100;
const int ypos_text = 5;
const int xpos_label = 5;
const int xstart_sliders = xpos_label + 120;
const int xstart_text = xstart_sliders + 105;
const int xcol_interval = 210;
const int ystart_sliders = 30;
const int yline_interval = 30;

const int def_vol = 100;
const int def_pan = 65;
const int def_rev = 41;
const int def_cho = 1;

	(void) new wxMessage( panel, "Volume", xstart_text, ypos_text);
	(void) new wxMessage( panel, "Pan", xstart_text + xcol_interval, ypos_text);
	(void) new wxMessage( panel, "Reverb", xstart_text + 2*xcol_interval, ypos_text);
	(void) new wxMessage( panel, "Chorus", xstart_text + 3*xcol_interval, ypos_text);

	int vol, pan, rev, cho;
	wxSlider *vslider, *pslider, *rslider, *cslider;
	tTrack *track;

	SliderArray[0][0].tw = (tTrackWin*) EventWin;
	for (int chan = 0; chan < 16; chan++) {
		char labelstr[40];

		track = NULL;
  		for (int j = EventWin->FromLine; j < EventWin->ToLine; j++) {
			tTrack *t = EventWin->Song->GetTrack(j);
			if (t && (t->Channel == (chan + 1)) ) {
				if (t->GetName() && strlen(t->GetName())) {
					track = t;
					break;
				}
			}
		}


		if (track) {
			sprintf( labelstr, "%d %s", chan + 1, track->GetName() );

			vol = track->GetVolume(); 
			if (!vol) { vol = def_vol; track->SetVolume( vol ); }
			pan = track->GetPan();
			if (!pan) { pan = def_pan; track->SetPan( pan ); }
			rev = track->GetReverb(); 
			if (!rev) { rev = def_rev; track->SetReverb( rev ); }
			cho = track->GetChorus();
			if (!cho) { cho = def_cho; track->SetChorus( cho ); }

			vslider = SliderArray[chan][MxVol].slider = new wxSlider( panel, (wxFunction) SliderChange, " ", vol, 1, 128, slider_width, xstart_sliders, ystart_sliders + chan*yline_interval );
			pslider = SliderArray[chan][MxPan].slider = new wxSlider( panel, (wxFunction) SliderChange, " ", pan - 65, -63, 63, slider_width, xstart_sliders + 1*xcol_interval, ystart_sliders + chan*yline_interval );
			rslider = SliderArray[chan][MxRev].slider = new wxSlider( panel, (wxFunction) SliderChange, " ", rev, 1, 128, slider_width, xstart_sliders + 2*xcol_interval, ystart_sliders + chan*yline_interval );
			cslider = SliderArray[chan][MxCho].slider = new wxSlider( panel, (wxFunction) SliderChange, " ", cho, 1, 128, slider_width, xstart_sliders + 3*xcol_interval, ystart_sliders + chan*yline_interval );
		}
		else {
			sprintf( labelstr, "%d", chan + 1 );
			vslider = pslider = rslider = cslider = NULL;
		}
		SliderArray[chan][MxVol].track = track;
		SliderArray[chan][MxPan].track = track;
		SliderArray[chan][MxRev].track = track;
		SliderArray[chan][MxCho].track = track;

		(void) new wxMessage( panel, labelstr, xpos_label, ystart_sliders + chan*yline_interval);
		(void) new wxMessage( panel, labelstr, xstart_sliders + 4*xcol_interval + 10, ystart_sliders + chan*yline_interval);
	}

	chan++;
	const int def_mastervol = 128;
	const int def_masterpan = 65;
	int mastervol, masterpan;
	tTrack *mt = EventWin->Song->GetTrack(0);
	wxSlider *mvslider, *mpslider;
	if (mt) {
		mastervol = mt->GetMasterVol();
		if (!mastervol) { mastervol = def_mastervol; mt->SetMasterVol( mastervol ); }
		masterpan = mt->GetMasterPan();
		if (!masterpan) { masterpan = def_masterpan; mt->SetMasterPan( masterpan ); }
		mvslider = MasterSliders[MxVol].slider = new wxSlider( panel, (wxFunction) MasterChange, " ", mastervol, 1, 128, slider_width, xstart_sliders, ystart_sliders + chan*yline_interval );
		mpslider = MasterSliders[MxPan].slider = new wxSlider( panel, (wxFunction) MasterChange, " ", masterpan - 65, -63, 63, slider_width, xstart_sliders + 1*xcol_interval, ystart_sliders + chan*yline_interval );
	}
	else {
		mvslider = mpslider = 0;
	}
	MasterSliders[MxVol].track = mt;
	MasterSliders[MxPan].track = mt;
	(void) new wxMessage( panel, "Master", xpos_label, ystart_sliders + chan*yline_interval);
	chan++;
	(void) new wxButton( panel, (wxFunction) Dismiss, "Dismiss", xpos_label, ystart_sliders + chan*yline_interval);
	((tTrackWin*) EventWin)->DrawCounters();
}


void tTrackWin::MenMixer()
{
  tMixerDlg *dlg;
  if (MixerForm) {
    MixerForm->Show(TRUE);
    return;
  }
  for (int i = FromLine; i < ToLine; i++) {
    tTrack *t = Song->GetTrack(i);
    if (t && t->DialogBox) {
	return;
    }
  }
  MixerForm = new wxDialogBox(this, "Mixer", FALSE);
  dlg = new tMixerDlg(this);
  dlg->EditForm(MixerForm);
  MixerForm->Fit();
  MixerForm->Show(TRUE);
}

// ******************************************************************
// Sound Dialog
// ******************************************************************


class tSoundDlg : public wxForm
{
 public:
  tEventWin *EventWin;
  tSoundDlg(tEventWin *w);
  void EditForm(wxPanel *panel);

  static tSliderArray SliderArray[16][NrpnSoundParams];
  static void GetSliderVal( wxSlider& s );
  static void SetSliderVal( int chan, int type, int val );
  static void SliderChange( wxSlider& slider, wxCommandEvent& event );
  static void Dismiss( wxButton& button, wxCommandEvent& event );
};

tSliderArray tSoundDlg::SliderArray[16][NrpnSoundParams];

tSoundDlg::tSoundDlg(tEventWin *w)
{
  EventWin = w;
  SliderArray[0][0].tw = 0;
}

void tSoundDlg::Dismiss( wxButton& button, wxCommandEvent& event )
{
  tEventWin *w = (tEventWin*) SliderArray[0][0].tw;
  w->MixerForm->Show(FALSE);
  delete w->MixerForm;
  w->MixerForm = 0;
}

void tSoundDlg::SetSliderVal( int chan, int type, int val ) {
	if (!SliderArray[0][0].tw || !SliderArray[0][0].tw->MixerForm) return;
	SliderArray[chan][type].slider->SetValue( val - 65 );
}


void tSoundDlg::GetSliderVal( wxSlider& s ) {
	for (int chan = 0; chan < 16; chan++) {
		for (int type = NrpnCutoff; type < NrpnSoundParams; type++) {
			if (SliderArray[chan][type].track == NULL) continue;
			switch (type) {
			case NrpnCutoff:
				if (SliderArray[chan][type].slider == &s) {
					SliderArray[chan][type].track->SetCutoff(s.GetValue() + 65);
					return;
				}
			case NrpnResonance:
				if (SliderArray[chan][type].slider == &s) {
					SliderArray[chan][type].track->SetResonance(s.GetValue() + 65);
					return;
				}
			}
		}
	}
}

void tSoundDlg::SliderChange( wxSlider& slider, wxCommandEvent& event ) {
	GetSliderVal( slider );
}


void tSoundDlg::EditForm(wxPanel *panel)
{
const int slider_width = 100;
const int ypos_text = 5;
const int xpos_label = 5;
const int xstart_sliders = xpos_label + 120;
const int xstart_text = xstart_sliders + 95;
const int xcol_interval = 210;
const int ystart_sliders = 30;
const int yline_interval = 30;

const int def_cut = 65;
const int def_res = 65;

	(void) new wxMessage( panel, "Cutoff frequency", xstart_text, ypos_text);
	(void) new wxMessage( panel, "Resonance", xstart_text + xcol_interval, ypos_text);

	int cut, res;
	wxSlider *cslider, *rslider;
	tTrack *track;

	SliderArray[0][0].tw = (tTrackWin*) EventWin;
	for (int chan = 0; chan < 16; chan++) {
		char labelstr[40];

		track = NULL;
  		for (int j = EventWin->FromLine; j < EventWin->ToLine; j++) {
			tTrack *t = EventWin->Song->GetTrack(j);
			if (t && (t->Channel == (chan + 1)) ) {
				if (t->GetName() && strlen(t->GetName())) {
					track = t;
					break;
				}
			}
		}


		if (track) {
			sprintf( labelstr, "%d %s", chan + 1, track->GetName() );

			cut = track->GetCutoff(); 
			if (!cut) { cut = def_cut; track->SetCutoff( cut ); }
			res = track->GetResonance();
			if (!res) { res = def_res; track->SetResonance( res ); }

			cslider = SliderArray[chan][NrpnCutoff].slider = new wxSlider( panel, (wxFunction) SliderChange, " ", cut - 65, -50, 16, slider_width, xstart_sliders, ystart_sliders + chan*yline_interval );
			rslider = SliderArray[chan][NrpnResonance].slider = new wxSlider( panel, (wxFunction) SliderChange, " ", res - 65, -50, 50, slider_width, xstart_sliders + 1*xcol_interval, ystart_sliders + chan*yline_interval );
		}
		else {
			sprintf( labelstr, "%d", chan + 1 );
			cslider = rslider = NULL;
		}
		SliderArray[chan][NrpnCutoff].track = track;
		SliderArray[chan][NrpnResonance].track = track;

		(void) new wxMessage( panel, labelstr, xpos_label, ystart_sliders + chan*yline_interval);
		(void) new wxMessage( panel, labelstr, xstart_sliders + 2*xcol_interval + 10, ystart_sliders + chan*yline_interval);
	}

	(void) new wxButton( panel, (wxFunction) Dismiss, "Dismiss", xpos_label, ystart_sliders + chan*yline_interval);
}


void tTrackWin::MenSound()
{
  tSoundDlg *dlg;
  if (MixerForm) {
    MixerForm->Show(TRUE);
    return;
  }
  for (int i = FromLine; i < ToLine; i++) {
    tTrack *t = Song->GetTrack(i);
    if (t && t->DialogBox) {
	return;
    }
  }
  MixerForm = new wxDialogBox(this, "Sound", FALSE);
  dlg = new tSoundDlg(this);
  dlg->EditForm(MixerForm);
  MixerForm->Fit();
  MixerForm->Show(TRUE);
}

// ******************************************************************
// Vibrato Dialog
// ******************************************************************


class tVibratoDlg : public wxForm
{
 public:
  tEventWin *EventWin;
  tVibratoDlg(tEventWin *w);
  void EditForm(wxPanel *panel);

  static tSliderArray SliderArray[16][NrpnVibParams];
  static void GetSliderVal( wxSlider& s );
  static void SetSliderVal( int chan, int type, int val );
  static void SliderChange( wxSlider& slider, wxCommandEvent& event );
  static void Dismiss( wxButton& button, wxCommandEvent& event );
};

tSliderArray tVibratoDlg::SliderArray[16][NrpnVibParams];

tVibratoDlg::tVibratoDlg(tEventWin *w)
{
  EventWin = w;
  SliderArray[0][0].tw = 0;
}

void tVibratoDlg::Dismiss( wxButton& button, wxCommandEvent& event )
{
  tEventWin *w = (tEventWin*) SliderArray[0][0].tw;
  w->MixerForm->Show(FALSE);
  delete w->MixerForm;
  w->MixerForm = 0;
}


void tVibratoDlg::SetSliderVal( int chan, int type, int val ) {
	if (!SliderArray[0][0].tw || !SliderArray[0][0].tw->MixerForm) return;
	SliderArray[chan][type].slider->SetValue( val - 65 );
}


void tVibratoDlg::GetSliderVal( wxSlider& s ) {
	for (int chan = 0; chan < 16; chan++) {
		for (int type = NrpnVibRate; type < NrpnVibParams; type++) {
			if (SliderArray[chan][type].track == NULL) continue;
			switch (type) {
			case NrpnVibRate:
				if (SliderArray[chan][type].slider == &s) {
					SliderArray[chan][type].track->SetVibRate(s.GetValue() + 65);
					return;
				}
			case NrpnVibDepth:
				if (SliderArray[chan][type].slider == &s) {
					SliderArray[chan][type].track->SetVibDepth(s.GetValue() + 65);
					return;
				}
			case NrpnVibDelay:
				if (SliderArray[chan][type].slider == &s) {
					SliderArray[chan][type].track->SetVibDelay(s.GetValue() + 65);
					return;
				}
			}
		}
	}
}

void tVibratoDlg::SliderChange( wxSlider& slider, wxCommandEvent& event ) {
	GetSliderVal( slider );
}


void tVibratoDlg::EditForm(wxPanel *panel)
{
const int slider_width = 100;
const int ypos_text = 5;
const int xpos_label = 5;
const int xstart_sliders = xpos_label + 120;
const int xstart_text = xstart_sliders + 95;
const int xcol_interval = 210;
const int ystart_sliders = 30;
const int yline_interval = 30;

const int def_rate = 65;
const int def_depth = 65;
const int def_delay = 65;

	(void) new wxMessage( panel, "Vib. Rate", xstart_text, ypos_text);
	(void) new wxMessage( panel, "Vib. Depth", xstart_text + xcol_interval, ypos_text);
	(void) new wxMessage( panel, "Vib. Delay", xstart_text + 2*xcol_interval, ypos_text);

	int rate, depth, delay;
	wxSlider *rslider, *delslider, *depslider;
	tTrack *track;

	SliderArray[0][0].tw = (tTrackWin*) EventWin;
	for (int chan = 0; chan < 16; chan++) {
		char labelstr[40];

		track = NULL;
  		for (int j = EventWin->FromLine; j < EventWin->ToLine; j++) {
			tTrack *t = EventWin->Song->GetTrack(j);
			if (t && (t->Channel == (chan + 1)) ) {
				if (t->GetName() && strlen(t->GetName())) {
					track = t;
					break;
				}
			}
		}


		if (track) {
			sprintf( labelstr, "%d %s", chan + 1, track->GetName() );

			rate = track->GetVibRate(); 
			if (!rate) { rate = def_rate; track->SetVibRate( rate ); }
			depth = track->GetVibDepth();
			if (!depth) { depth = def_depth; track->SetVibDepth( depth ); }
			delay = track->GetVibDelay();
			if (!delay) { delay = def_delay; track->SetVibDelay( delay ); }

			rslider = SliderArray[chan][NrpnVibRate].slider = new wxSlider( panel, (wxFunction) SliderChange, " ", rate - 65, -50, 50, slider_width, xstart_sliders, ystart_sliders + chan*yline_interval );
			depslider = SliderArray[chan][NrpnVibDepth].slider = new wxSlider( panel, (wxFunction) SliderChange, " ", depth - 65, -50, 50, slider_width, xstart_sliders + 1*xcol_interval, ystart_sliders + chan*yline_interval );
			delslider = SliderArray[chan][NrpnVibDelay].slider = new wxSlider( panel, (wxFunction) SliderChange, " ", delay - 65, -50, 50, slider_width, xstart_sliders + 2*xcol_interval, ystart_sliders + chan*yline_interval );
		}
		else {
			sprintf( labelstr, "%d", chan + 1 );
			rslider = depslider = delslider = NULL;
		}
		SliderArray[chan][NrpnVibRate].track = track;
		SliderArray[chan][NrpnVibDepth].track = track;
		SliderArray[chan][NrpnVibDelay].track = track;

		(void) new wxMessage( panel, labelstr, xpos_label, ystart_sliders + chan*yline_interval);
		(void) new wxMessage( panel, labelstr, xstart_sliders + 3*xcol_interval + 10, ystart_sliders + chan*yline_interval);
	}

	(void) new wxButton( panel, (wxFunction) Dismiss, "Dismiss", xpos_label, ystart_sliders + chan*yline_interval);
}


void tTrackWin::MenVibrato()
{
  tVibratoDlg *dlg;
  if (MixerForm) {
    MixerForm->Show(TRUE);
    return;
  }
  for (int i = FromLine; i < ToLine; i++) {
    tTrack *t = Song->GetTrack(i);
    if (t && t->DialogBox) {
	return;
    }
  }
  MixerForm = new wxDialogBox(this, "Vibrato", FALSE);
  dlg = new tVibratoDlg(this);
  dlg->EditForm(MixerForm);
  MixerForm->Fit();
  MixerForm->Show(TRUE);
}

// ******************************************************************
// Envelope Dialog
// ******************************************************************


class tEnvelopeDlg : public wxForm
{
 public:
  tEventWin *EventWin;
  tEnvelopeDlg(tEventWin *w);
  void EditForm(wxPanel *panel);

  static tSliderArray SliderArray[16][NrpnEnvParams];
  static void GetSliderVal( wxSlider& s );
  static void SetSliderVal( int chan, int type, int val );
  static void SliderChange( wxSlider& slider, wxCommandEvent& event );
  static void Dismiss( wxButton& button, wxCommandEvent& event );
};

tSliderArray tEnvelopeDlg::SliderArray[16][NrpnEnvParams];

tEnvelopeDlg::tEnvelopeDlg(tEventWin *w)
{
  EventWin = w;
  SliderArray[0][0].tw = 0;
}


void tEnvelopeDlg::Dismiss( wxButton& button, wxCommandEvent& event )
{
  tEventWin *w = (tEventWin*) SliderArray[0][0].tw;
  w->MixerForm->Show(FALSE);
  delete w->MixerForm;
  w->MixerForm = 0;
}


void tEnvelopeDlg::SetSliderVal( int chan, int type, int val ) {
	if (!SliderArray[0][0].tw || !SliderArray[0][0].tw->MixerForm) return;
	SliderArray[chan][type].slider->SetValue( val - 65 );
}


void tEnvelopeDlg::GetSliderVal( wxSlider& s ) {
	for (int chan = 0; chan < 16; chan++) {
		for (int type = NrpnEnvAttack; type < NrpnEnvParams; type++) {
			if (SliderArray[chan][type].track == NULL) continue;
			switch (type) {
			case NrpnEnvAttack:
				if (SliderArray[chan][type].slider == &s) {
					SliderArray[chan][type].track->SetEnvAttack(s.GetValue() + 65);
					return;
				}
			case NrpnEnvDecay:
				if (SliderArray[chan][type].slider == &s) {
					SliderArray[chan][type].track->SetEnvDecay(s.GetValue() + 65);
					return;
				}
			case NrpnEnvRelease:
				if (SliderArray[chan][type].slider == &s) {
					SliderArray[chan][type].track->SetEnvRelease(s.GetValue() + 65);
					return;
				}
			}
		}
	}
}

void tEnvelopeDlg::SliderChange( wxSlider& slider, wxCommandEvent& event ) {
	GetSliderVal( slider );
}


void tEnvelopeDlg::EditForm(wxPanel *panel)
{
const int slider_width = 100;
const int ypos_text = 5;
const int xpos_label = 5;
const int xstart_sliders = xpos_label + 120;
const int xstart_text = xstart_sliders + 95;
const int xcol_interval = 210;
const int ystart_sliders = 30;
const int yline_interval = 30;

const int def_att = 65;
const int def_dec = 65;
const int def_rel = 65;

	(void) new wxMessage( panel, "Env. Attack", xstart_text, ypos_text);
	(void) new wxMessage( panel, "Env. Decay", xstart_text + xcol_interval, ypos_text);
	(void) new wxMessage( panel, "Env. Release", xstart_text + 2*xcol_interval, ypos_text);

	int att, dec, rel;
	wxSlider *aslider, *dslider, *rslider;
	tTrack *track;

	SliderArray[0][0].tw = (tTrackWin*) EventWin;
	for (int chan = 0; chan < 16; chan++) {
		char labelstr[40];

		track = NULL;
  		for (int j = EventWin->FromLine; j < EventWin->ToLine; j++) {
			tTrack *t = EventWin->Song->GetTrack(j);
			if (t && (t->Channel == (chan + 1)) ) {
				if (t->GetName() && strlen(t->GetName())) {
					track = t;
					break;
				}
			}
		}


		if (track) {
			sprintf( labelstr, "%d %s", chan + 1, track->GetName() );

			att = track->GetEnvAttack(); 
			if (!att) { att = def_att; track->SetEnvAttack( att ); }
			dec = track->GetEnvDecay();
			if (!dec) { dec = def_dec; track->SetEnvDecay( dec ); }
			rel = track->GetEnvRelease();
			if (!rel) { rel = def_rel; track->SetEnvRelease( rel ); }

			aslider = SliderArray[chan][NrpnEnvAttack].slider = new wxSlider( panel, (wxFunction) SliderChange, " ", att - 65, -50, 50, slider_width, xstart_sliders, ystart_sliders + chan*yline_interval );
			dslider = SliderArray[chan][NrpnEnvDecay].slider = new wxSlider( panel, (wxFunction) SliderChange, " ", dec - 65, -50, 50, slider_width, xstart_sliders + 1*xcol_interval, ystart_sliders + chan*yline_interval );
			rslider = SliderArray[chan][NrpnEnvRelease].slider = new wxSlider( panel, (wxFunction) SliderChange, " ", rel - 65, -50, 50, slider_width, xstart_sliders + 2*xcol_interval, ystart_sliders + chan*yline_interval );
		}
		else {
			sprintf( labelstr, "%d", chan + 1 );
			aslider = dslider = rslider = NULL;
		}
		SliderArray[chan][NrpnEnvAttack].track = track;
		SliderArray[chan][NrpnEnvDecay].track = track;
		SliderArray[chan][NrpnEnvRelease].track = track;

		(void) new wxMessage( panel, labelstr, xpos_label, ystart_sliders + chan*yline_interval);
		(void) new wxMessage( panel, labelstr, xstart_sliders + 3*xcol_interval + 10, ystart_sliders + chan*yline_interval);
	}

	(void) new wxButton( panel, (wxFunction) Dismiss, "Dismiss", xpos_label, ystart_sliders + chan*yline_interval);
}


void tTrackWin::MenEnvelope()
{
  tEnvelopeDlg *dlg;
  if (MixerForm) {
    MixerForm->Show(TRUE);
    return;
  }
  for (int i = FromLine; i < ToLine; i++) {
    tTrack *t = Song->GetTrack(i);
    if (t && t->DialogBox) {
	return;
    }
  }
  MixerForm = new wxDialogBox(this, "Envelope", FALSE);
  dlg = new tEnvelopeDlg(this);
  dlg->EditForm(MixerForm);
  MixerForm->Fit();
  MixerForm->Show(TRUE);
}

// ******************************************************************
// Wheel Dialog
// ******************************************************************

enum { WheelBend = 0, WheelMod, WheelParams };

class tWheelDlg : public wxForm
{
 public:
  tEventWin *EventWin;
  tWheelDlg(tEventWin *w);
  void EditForm(wxPanel *panel);

  static tSliderArray SliderArray[16][WheelParams];
  static void GetSliderVal( wxSlider& s );
  static void SetSliderVal( int chan, int type, int val );
  static void SliderChange( wxSlider& slider, wxCommandEvent& event );
  static void Dismiss( wxButton& button, wxCommandEvent& event );
};

tSliderArray tWheelDlg::SliderArray[16][WheelParams];

tWheelDlg::tWheelDlg(tEventWin *w)
{
  EventWin = w;
  SliderArray[0][0].tw = 0;
}


void tWheelDlg::Dismiss( wxButton& button, wxCommandEvent& event )
{
  tEventWin *w = (tEventWin*) SliderArray[0][0].tw;
  w->MixerForm->Show(FALSE);
  delete w->MixerForm;
  w->MixerForm = 0;
}


void tWheelDlg::SetSliderVal( int chan, int type, int val ) {
	if (!SliderArray[0][0].tw || !SliderArray[0][0].tw->MixerForm) return;
	SliderArray[chan][type].slider->SetValue( val - 1 );
}


void tWheelDlg::GetSliderVal( wxSlider& s ) {
	for (int chan = 0; chan < 16; chan++) {
		for (int type = WheelBend; type < WheelParams; type++) {
			if (SliderArray[chan][type].track == NULL) continue;
			switch (type) {
			case WheelBend:
				if (SliderArray[chan][type].slider == &s) {
					SliderArray[chan][type].track->SetPitchBendSens(s.GetValue() + 1);
					return;
				}
			case WheelMod:
				if (SliderArray[chan][type].slider == &s) {
					SliderArray[chan][type].track->SetModDepth(s.GetValue() + 1);
					return;
				}
			}
		}
	}
}

void tWheelDlg::SliderChange( wxSlider& slider, wxCommandEvent& event ) {
	GetSliderVal( slider );
}


void tWheelDlg::EditForm(wxPanel *panel)
{
const int slider_width = 100;
const int ypos_text = 5;
const int xpos_label = 5;
const int xstart_sliders = xpos_label + 120;
const int xstart_text = xstart_sliders + 85;
const int xcol_interval = 210;
const int ystart_sliders = 30;
const int yline_interval = 30;

const int def_bend = 3;
const int def_mod = 11;

	(void) new wxMessage( panel, "Pitch Bend Sens.", xstart_text, ypos_text);
	(void) new wxMessage( panel, "Modulation Depth", xstart_text + xcol_interval, ypos_text);

	int bend, mod;
	wxSlider *bslider, *mslider;
	tTrack *track;

	SliderArray[0][0].tw = (tTrackWin*) EventWin;
	for (int chan = 0; chan < 16; chan++) {
		char labelstr[40];

		track = NULL;
  		for (int j = EventWin->FromLine; j < EventWin->ToLine; j++) {
			tTrack *t = EventWin->Song->GetTrack(j);
			if (t && (t->Channel == (chan + 1)) ) {
				if (t->GetName() && strlen(t->GetName())) {
					track = t;
					break;
				}
			}
		}


		if (track) {
			sprintf( labelstr, "%d %s", chan + 1, track->GetName() );

			bend = track->GetPitchBendSens(); 
			if (!bend) { bend = def_bend; track->SetPitchBendSens( bend ); }
			mod = track->GetModDepth();
			if (!mod) { mod = def_mod; track->SetModDepth( mod ); }

			bslider = SliderArray[chan][WheelBend].slider = new wxSlider( panel, (wxFunction) SliderChange, " ", bend - 1, 0, 24, slider_width, xstart_sliders, ystart_sliders + chan*yline_interval );
			mslider = SliderArray[chan][WheelMod].slider = new wxSlider( panel, (wxFunction) SliderChange, " ", mod - 1, 0, 127, slider_width, xstart_sliders + 1*xcol_interval, ystart_sliders + chan*yline_interval );
		}
		else {
			sprintf( labelstr, "%d", chan + 1 );
			bslider = mslider = NULL;
		}
		SliderArray[chan][WheelBend].track = track;
		SliderArray[chan][WheelMod].track = track;

		(void) new wxMessage( panel, labelstr, xpos_label, ystart_sliders + chan*yline_interval);
		(void) new wxMessage( panel, labelstr, xstart_sliders + 2*xcol_interval + 10, ystart_sliders + chan*yline_interval);
	}

	(void) new wxButton( panel, (wxFunction) Dismiss, "Dismiss", xpos_label, ystart_sliders + chan*yline_interval);
}


void tTrackWin::MenWheel()
{
  tWheelDlg *dlg;
  if (MixerForm) {
    MixerForm->Show(TRUE);
    return;
  }
  for (int i = FromLine; i < ToLine; i++) {
    tTrack *t = Song->GetTrack(i);
    if (t && t->DialogBox) {
	return;
    }
  }
  MixerForm = new wxDialogBox(this, "Wheel", FALSE);
  dlg = new tWheelDlg(this);
  dlg->EditForm(MixerForm);
  MixerForm->Fit();
  MixerForm->Show(TRUE);
}

// ******************************************************************
// Effects Dialog
// ******************************************************************


class tEffectsDlg : public wxForm
{
 public:
  tEventWin *EventWin;
  tEffectsDlg(tEventWin *w);
  char *ReverbSelection;
  char *ChorusSelection;
  void EditForm(wxPanel *panel);
  void OnOk();
  void OnCancel();
  char *RevSelArray[8];
  char *ChoSelArray[8];
};

tEffectsDlg::tEffectsDlg(tEventWin *w)
{
  EventWin = w;

  RevSelArray[0] = "Room 1";
  RevSelArray[1] = "Room 2";
  RevSelArray[2] = "Room 3";
  RevSelArray[3] = "Hall 1";
  RevSelArray[4] = "Hall 2";
  RevSelArray[5] = "Plate";
  RevSelArray[6] = "Delay";
  RevSelArray[7] = "Panning Delay";

  ChoSelArray[0] = "Chorus 1";
  ChoSelArray[1] = "Chorus 2";
  ChoSelArray[2] = "Chorus 3";
  ChoSelArray[3] = "Chorus 4";
  ChoSelArray[4] = "Feedback Chorus";
  ChoSelArray[5] = "Flanger";
  ChoSelArray[6] = "Short Delay";
  ChoSelArray[7] = "Short Delay FB";
}

void tEffectsDlg::OnCancel()
{
  EventWin->DialogBox = 0;
  wxForm::OnCancel();
}


void tEffectsDlg::OnOk()
{
  tTrack *t = EventWin->Song->GetTrack(0);
  for (int i = 0; i < 8; i++) {
	if (!strcmp(ReverbSelection,RevSelArray[i])) {
		t->SetReverbType( i + 1 );
		break;
	}
  }
  for (i = 0; i < 8; i++) {
	if (!strcmp(ChorusSelection,ChoSelArray[i])) {
		t->SetChorusType( i + 1 );
		break;
	}
  }
  EventWin->Redraw();
  EventWin->DialogBox = 0;
  wxForm::OnOk();
}

void tEffectsDlg::EditForm(wxPanel *panel)
{
  tTrack *t = EventWin->Song->GetTrack(0);
  int reverb_type;
  int chorus_type;
  const int reverb_type_def = 5;
  const int chorus_type_def = 3;

  reverb_type = t->GetReverbType();
  if (!reverb_type) reverb_type = reverb_type_def;
  chorus_type = t->GetChorusType();
  if (!chorus_type) chorus_type = chorus_type_def;
	
  ReverbSelection = copystring( RevSelArray[ reverb_type - 1 ] );
  ChorusSelection = copystring( ChoSelArray[ chorus_type - 1 ] );

  Add(wxMakeFormString("Reverb Type", &ReverbSelection, wxFORM_CHOICE,
                       new wxList(wxMakeConstraintStrings(	RevSelArray[0], 
		       						RevSelArray[1], 
								RevSelArray[2], 
								RevSelArray[3], 
								RevSelArray[4],
								RevSelArray[5],
								RevSelArray[6],
								RevSelArray[7],
								0), 0)));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormString("Chorus Type", &ChorusSelection, wxFORM_CHOICE,
                       new wxList(wxMakeConstraintStrings(	ChoSelArray[0], 
		       						ChoSelArray[1], 
								ChoSelArray[2], 
								ChoSelArray[3], 
								ChoSelArray[4],
								ChoSelArray[5],
								ChoSelArray[6],
								ChoSelArray[7],
								0), 0)));
  AssociatePanel(panel);
}

void tTrackWin::MenEffects()
{
  tEffectsDlg *dlg;
  if (DialogBox)
  {
    DialogBox->Show(TRUE);
    return;
  }
  DialogBox = new wxDialogBox(this, "Effects Settings", FALSE);
  dlg = new tEffectsDlg(this);
  dlg->EditForm(DialogBox);
  DialogBox->Fit();
  DialogBox->Show(TRUE);
}

// --------------------------------------------------------------------
// PatchCounter
// --------------------------------------------------------------------

class tPatchCounter : public tMouseCounter
{
    tTrackWin *tw;
  public:
    int Event(wxMouseEvent &e);
    tPatchCounter(tTrackWin *t, tRect *r, int v, int min, int max)
      : tMouseCounter(t->dc, r, v, min, max)
    { 
      tw = t; 
    }
};


int tPatchCounter::Event(wxMouseEvent &e)
{
  if (tMouseCounter::Event(e))
  {
    tTrack *t = tw->y2Track((long)r.y);
    if (t)
    {
      switch (tw->CounterMode)
      {
        case CmProgram : t->SetBank( t->GetBank() ); t->SetPatch(Value); break;
        case CmBank    : t->SetBank(Value); t->SetPatch( t->GetPatch() ); break;
        case CmVolume  : t->SetVolume(Value); 
			 tMixerDlg::SetSliderVal( t->Channel - 1, MxVol, Value );
			 break;
        case CmPan     : t->SetPan(Value);
			 tMixerDlg::SetSliderVal( t->Channel - 1, MxPan, Value );
			 break;
        case CmReverb  : t->SetReverb(Value);
			 tMixerDlg::SetSliderVal( t->Channel - 1, MxRev, Value );
			 break;
        case CmChorus  : t->SetChorus(Value);
			 tMixerDlg::SetSliderVal( t->Channel - 1, MxCho, Value );
			 break;
        default: break;
      }
    }
    tw->MouseAction = 0;
    delete this;
  }
  return 0;
}


void tTrackWin::MousePatch(wxMouseEvent &e)
{
  float x, y;

  if (!e.LeftDown() && !e.RightDown())
    return;

  e.Position(&x, &y);
  tTrack *t = y2Track((long)y);
  if (t)
  {
    tRect r;
    int Value;
    switch (CounterMode)
    {
      case CmProgram: Value = t->GetPatch(); break;
      case CmBank   : Value = t->GetBank(); break;
      case CmVolume : Value = t->GetVolume(); break;
      case CmPan    : Value = t->GetPan(); break;
      case CmReverb : Value = t->GetReverb(); break;
      case CmChorus : Value = t->GetChorus(); break;
      default       : Value = 0; break;
    }
    r.x = xPatch + LittleBit;
    r.y = y2yLine((long)y) + LittleBit;
    r.w = wPatch - LittleBit;
    r.h = hLine - LittleBit;

    tPatchCounter *PatchCounter = new tPatchCounter(this, &r, Value, 0, 128);
    PatchCounter->Event(e);
    MouseAction = PatchCounter;
  }
}

// --------------------------------------------------------------------
// SpeedCounter
// --------------------------------------------------------------------

class tSpeedCounter : public tMouseCounter
{
    tTrackWin *tw;
  public:
    int Event(wxMouseEvent &e);
    tSpeedCounter(tTrackWin *t, tRect *r, int v, int min, int max)
      : tMouseCounter(t->dc, r, v, min, max)
    { 
      tw = t; 
    }
    void ShowValue();
};


int tSpeedCounter::Event(wxMouseEvent &e)
{
  if (tMouseCounter::Event(e))
  {
    tTrack *t = tw->Song->GetTrack(0);
    t->SetSpeed(Value);
    tw->MouseAction = 0;
    delete this;
  }
  return 0;
}

void tSpeedCounter::ShowValue()
{
  tw->DrawSpeed(Value);
}

void tTrackWin::MouseSpeed(wxMouseEvent &e)
{
  tRect r;
  int Value;

  if (!e.LeftDown() && !e.RightDown())
    return;

  Value = Song->GetTrack(0)->GetSpeed();
  tSpeedCounter *SpeedCounter = new tSpeedCounter(this, &r, Value, 20, 250);
  SpeedCounter->Event(e);
  MouseAction = SpeedCounter;
}

// -----------------------------------------------------------------------
// Track-Status
// -----------------------------------------------------------------------


void tTrackWin::MouseState(wxMouseEvent &e)
{
  if (e.LeftDown() || e.RightDown())
  {
    float x, y;
    e.Position(&x, &y);
    tTrack *t = y2Track((long)y);
    if (t)
    {
      t->ToggleState(e.LeftDown() ? 1 : -1); // toggle
      LineText(xState, (long)y, wState, t->GetStateChar());
    }
  }
}


// --------------------------------------------------------------------------
// Name
// --------------------------------------------------------------------------

void tTrackWin::MouseName(wxMouseEvent &e)
{
  if (e.LeftDown())
  {
    float x, y;
    e.Position(&x, &y);
    tTrack *t = y2Track((long)y);
    if (t && !MixerForm)
      t->Dialog(this);
  }
}

// --------------------------------------------------------------------------
// Event-Bereich
// --------------------------------------------------------------------------

void tTrackWin::MouseEvents(wxMouseEvent &e)
{
  if (e.RightDown())
  {
    float x, y;
    e.Position(&x, &y);
    int TrackNr = y2Line((long)y);
    long Clock  = x2BarClock((long)x);
    NextWin->NewPosition(TrackNr, Clock);
  }
  else
    tEventWin::OnMouseEvent(e);
}

// --------------------------------------------------------------------------
// Playbar
// --------------------------------------------------------------------------

/*
 * not playing:
 *  events selected:
 *    left : start rec/play
 *    right: mute + start rec/play
 *  no events selected:
 *    left+right: start play
 * playing:
 *  left+right: stop
 */

void tTrackWin::MousePlay(wxMouseEvent &e)
{
  if (e.LeftDown() || e.RightDown())
  {
    float x, y;
    e.Position(&x, &y);

    if (!Midi->Playing)		// Start Record/Play
    {
      if (SnapSel->Selected)
      {
        RecInfo.TrackNr   = Filter->FromTrack;
	RecInfo.Track     = Song->GetTrack(RecInfo.TrackNr);
	RecInfo.FromClock = Filter->FromClock;
	RecInfo.ToClock   = Filter->ToClock;
	if (e.RightDown())
	{
	  RecInfo.Muted = 1;
	  RecInfo.Track->SetState(tsMute);
	  LineText(xState, Line2y(RecInfo.TrackNr), wState, RecInfo.Track->GetStateChar());
	}
	else
	  RecInfo.Muted = 0;
      }
      else
	RecInfo.Track = 0;
      long LoopClock = 0;
      long Clock = x2BarClock((long)x);
      if (RecInfo.Track && e.ShiftDown())
      {
        Clock     = RecInfo.FromClock;
        LoopClock = RecInfo.ToClock;
      }
      Midi->StartPlay(Clock, LoopClock);
    }

    else			// Stop Record/Play
    {
      Midi->StopPlay();
      if (RecInfo.Track)
      {
        if (RecInfo.Muted)
        {
	  RecInfo.Track->SetState(tsPlay);
	  LineText(xState, Line2y(RecInfo.TrackNr), wState, RecInfo.Track->GetStateChar());
	}
	if (!Midi->RecdBuffer.IsEmpty())
	{
	  int choice = wxMessageBox("Keep recorded events?", "You played", wxOK | wxCANCEL);
	  if (choice == wxOK)
	  {
	    Song->NewUndoBuffer();
	    RecInfo.Track->MergeRange(&Midi->RecdBuffer, RecInfo.FromClock, RecInfo.ToClock, RecInfo.Muted);
	  }
	}
      }
    }
  }
}



// -----------------------------------------------------------------------
// Event-Verteiler 
// -----------------------------------------------------------------------


int tTrackWin::OnMouseEvent(wxMouseEvent &e)
{

  if (!MouseAction)
  {
    float x, y;
    e.Position(&x, &y);
    if (y > yEvents)
    {
      // Events
      if (xNumber < x && x < xNumber + wNumber)
        MouseNumber(e);
      else if (xName < x && x < xName + wName)
	MouseName(e);
      else if(xState < x && x < xState + wState)
	MouseState(e);
      else if (xPatch < x && x < xPatch + wPatch)
	MousePatch(e);
      else if (xEvents < x && x < xEvents + wEvents)
	MouseEvents(e);
      else 
        tEventWin::OnMouseEvent(e);
    }
    else
    {
      // Playbar

      if (xNumber < x && x < xNumber + wNumber)
      {
        if (e.LeftDown())
        {
          NumberMode = (tNumberModes)(((int)NumberMode + 1) % (int)NmModes);
          DrawNumbers();
	}
      }
      else if (xName < x && x < xName + wName)
	MouseSpeed(e);
      else if(xState < x && x < xState + wState)
	;
      else if (xPatch < x && x < xPatch + wPatch)
      {
        if (e.LeftDown())
        {
          CounterMode = (tCounterModes)(((int)CounterMode + 1) % (int)CmModes);
	  DrawCounters();
	}
        else if (e.RightDown())
        {
          CounterMode = (tCounterModes)(((int)CounterMode + (int)CmModes - 1) % (int)CmModes);
	  DrawCounters();
	}
      }
      else if (xEvents < x && x < xEvents + wEvents)
	MousePlay(e);
      else 
        tEventWin::OnMouseEvent(e);
    }
  }

  else
    tEventWin::OnMouseEvent(e);

  return 0;
}



// **************************************************************************
// Copy
// **************************************************************************


class tCopyDlg;

class tCopyCommand : public wxObject, public tMouseAction
{
    tTrackWin *tw;
    int MarkRepeat;
    float StartX, StartY, StopX, StopY;

    tMarkDestin *Mouse;
    tCopyDlg *CopyDlg;

  public:
    static Bool RepeatCopy;
    static Bool EraseSource;
    static Bool EraseDestin;
    static Bool InsertSpace;

    int Event(wxMouseEvent &e);
    tCopyCommand(tTrackWin *t);
    void EditForm(wxPanel *panel);
    void OnOk();
    void OnCancel();
    void Execute(int doit);
};    

Bool tCopyCommand::EraseDestin = 1;
Bool tCopyCommand::RepeatCopy = 0;
Bool tCopyCommand::EraseSource = 0;
Bool tCopyCommand::InsertSpace = 0;


class tCopyDlg : public wxForm
{
  tCopyCommand *cp;

 public:
  tCopyDlg::tCopyDlg(tCopyCommand *c)   { cp = c; }
  void OnOk()				{ cp->OnOk(); wxForm::OnOk(); }
  void OnCancel()			{ cp->OnCancel(); wxForm::OnCancel(); }
  void EditForm(wxPanel *panel);
};


void tCopyDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormBool("Erase Destin", &cp->EraseDestin));
  Add(wxMakeFormBool("Repeat Copy", &cp->RepeatCopy));
  Add(wxMakeFormBool("Erase Source", &cp->EraseSource));
  Add(wxMakeFormBool("Insert Space", &cp->InsertSpace));
  AssociatePanel(panel);
}



tCopyCommand::tCopyCommand(tTrackWin *t)
{
  tw = t;
  MarkRepeat = 0;
  Mouse = new tMarkDestin(tw->Canvas, tw, 0);
  CopyDlg = 0;
}


int tCopyCommand::Event(wxMouseEvent &e)
{
  if (!CopyDlg && Mouse && Mouse->Event(e))
  {
    if (Mouse->Aborted)
    {
      if (CopyDlg)
        CopyDlg->OnCancel();
      else
        Execute(0);
    }
    else if (!MarkRepeat)
    {
      e.Position(&StartX, &StartY);
      tw->Mark((long)StartX, (long)StartY);
      wxDialogBox *panel = new wxDialogBox(tw, "Replicate", FALSE);
      CopyDlg = new tCopyDlg(this);
      CopyDlg->EditForm(panel);
      panel->Fit();
      panel->Show(TRUE);
    }
    else
    {
      e.Position(&StopX, &StopY);
      Execute(1);
    }
  }
  return 0;
}


void tCopyCommand::OnOk()
{
  CopyDlg = 0;
  if (RepeatCopy)
  {
    delete Mouse;
    Mouse = new tMarkDestin(tw->Canvas, tw, 1);
    MarkRepeat = 1;
  }
  else
    Execute(1);
}


void tCopyCommand::OnCancel()
{
  CopyDlg = 0;
  Execute(0);
}


void tCopyCommand::Execute(int doit)
{

  if (doit)
  {
    long DestTrack = tw->y2Line((long)StartY);
    long DestClock = tw->x2BarClock((long)StartX);
    tCmdCopy cpy(tw->Filter, DestTrack, DestClock);

    if (RepeatCopy)
      cpy.RepeatClock = tw->x2BarClock((long)StopX, 1);
    cpy.EraseSource = EraseSource;
    cpy.EraseDestin = EraseDestin;
    cpy.InsertSpace = InsertSpace;
    cpy.Execute();
  }

  tw->UnMark();
  tw->MouseAction = 0;
  tw->Redraw(-1);
  delete Mouse;
  delete this;
}


void tTrackWin::MenCopy()
{
  if (!EventsSelected())
    return;
  MouseAction = new tCopyCommand(this);
}

// ******************************************************************
// Song-Settings Dialog
// ******************************************************************



class tSongSettingsDlg : public wxForm
{
 public:
  tEventWin *EventWin;
  int TicksPerQuarter;
  tSongSettingsDlg(tEventWin *w);
  void EditForm(wxPanel *panel);
  virtual void OnOk();
  virtual void OnCancel();
};



tSongSettingsDlg::tSongSettingsDlg(tEventWin *w)
{
  EventWin = w;
  TicksPerQuarter = EventWin->Song->TicksPerQuarter;
}



void tSongSettingsDlg::OnCancel()
{
  EventWin->DialogBox = 0;
  wxForm::OnCancel();
}


void tSongSettingsDlg::OnOk()
{
  EventWin->Song->SetTicksPerQuarter(TicksPerQuarter);
  EventWin->Redraw();
  EventWin->DialogBox = 0;
  wxForm::OnOk();
}



void tSongSettingsDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormMessage("Supported: 48, 72, 96, 120, 144, 168, 192"));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Ticks per Quarter:", &TicksPerQuarter, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(48.0, 192), 0)));
  AssociatePanel(panel);
}


void tTrackWin::MenSongSettings()
{
  tSongSettingsDlg *dlg;
  if (DialogBox)
  {
    DialogBox->Show(TRUE);
    return;
  }
  DialogBox = new wxDialogBox(this, "Song Settings", FALSE);
  dlg = new tSongSettingsDlg(this);
  dlg->EditForm(DialogBox);
  DialogBox->Fit();
  DialogBox->Show(TRUE);
}


// *************************************************************************************
// Split / Merge Tracks
// *************************************************************************************

void tTrackWin::MenMergeTracks()
{
  int choice = wxMessageBox("Merge all Tracks to Track 0?", "Shure?", wxOK | wxCANCEL);
  if (choice == wxOK)
  {
    int tn;
    Song->NewUndoBuffer();
    tTrack *dst = Song->GetTrack(0);
    for (tn = 1; tn < Song->nTracks; tn++)
    {
      tTrack *src = Song->GetTrack(tn);
      tEventIterator Iterator(src);
      tEvent *e = Iterator.First();
      while (e)
      {
        tEvent *c = e->Copy();
        src->Kill(e);
        dst->Put(c);
        e = Iterator.Next();
      }
      src->Cleanup();
    }
    dst->Cleanup();
  }
}


void tTrackWin::MenSplitTracks()
{
  int choice = wxMessageBox("Split Track 0 by Midi-Channel to Track 1..16?", "Shure?", wxOK | wxCANCEL);
  if (choice == wxOK)
  {
    int ch;
    Song->NewUndoBuffer();
    tTrack *src = Song->GetTrack(0);
    tEventIterator Iterator(src);
    tEvent *e = Iterator.First();
    while (e)
    {
      tChannelEvent *ce = e->IsChannelEvent();
      if (ce)
      {
	int cn = ce->Channel;
	tTrack *dst = Song->GetTrack(cn + 1);
	if (dst)
	{
	  tEvent *cp = ce->Copy();
	  src->Kill(ce);
	  dst->Put(cp);
	}
      }
      e = Iterator.Next();
    }

    for (ch = 0; ch <= 16; ch++)
    {
      src = Song->GetTrack(ch);
      if (src)
        src->Cleanup();
    }
  }
}

// ******************************************************************
// Clock-source Dialog
// ******************************************************************


class tClockSourceDlg : public wxForm
{
 public:
  tEventWin *EventWin;
  tClockSourceDlg(tEventWin *w);
  void EditForm(wxPanel *panel);
  virtual void OnOk();
  virtual void OnCancel();
  char *ClkSrcSelection;
  char *ClkSrcArray[3];
};

tClockSourceDlg::tClockSourceDlg(tEventWin *w)
{
  EventWin = w;
  ClkSrcArray[0] = "INTERNAL";
  ClkSrcArray[1] = "FSK";
  ClkSrcArray[2] = "MIDI IN";
}



void tClockSourceDlg::OnCancel()
{
  EventWin->DialogBox = 0;
  wxForm::OnCancel();
}


void tClockSourceDlg::OnOk()
{
  for (int i = 0; i < 3; i++) {
	if (!strcmp(ClkSrcSelection,ClkSrcArray[i])) {
		Midi->ClockSource = (tClockSource) i;
		break;
	}
  }
  EventWin->Redraw();
  EventWin->DialogBox = 0;
  wxForm::OnOk();
}



void tClockSourceDlg::EditForm(wxPanel *panel)
{
  ClkSrcSelection = copystring( ClkSrcArray[ Midi->ClockSource ] );

  Add(wxMakeFormString("Clock Source", &ClkSrcSelection, wxFORM_CHOICE,
                       new wxList(wxMakeConstraintStrings(	ClkSrcArray[0], 
		       						ClkSrcArray[1], 
								ClkSrcArray[2], 
								0), 0)));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormBool("Send realtime messages to MIDI Out", &Midi->RealTimeOut));
  Add(wxMakeFormNewLine());
  AssociatePanel(panel);
}


void tTrackWin::MenClockSource()
{
  tClockSourceDlg *dlg;
  if (DialogBox)
  {
    DialogBox->Show(TRUE);
    return;
  }
  DialogBox = new wxDialogBox(this, "Clock Source", FALSE);
  dlg = new tClockSourceDlg(this);
  dlg->EditForm(DialogBox);
  DialogBox->Fit();
  DialogBox->Show(TRUE);
}


