// track.h,v 2.2 1995/06/19 19:04:47 andreas Exp

/*
**  jazz - a midi sequencer for Linux
**
**  Copyright (C) 1994 Andreas Voss (andreas@avix.rhein-neckar.de)
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#ifndef track_h
#define track_h

#ifndef events_h
#include "events.h"
#endif

#ifndef wx_wxh
#include "wx.h"
#endif

#ifndef util_h
#include "util.h"
#endif


// ************************************************************************

// ************************************************************************


// Track-States

#define tsPlay 0
#define tsMute 1
#define tsSolo 2

// Mixer-defs
enum { MxVol = 0, MxPan, MxRev, MxCho, MxParams };


// Param (Nrpn / Rpn) things
enum { NrpnVibRate = 0, NrpnVibDepth, NrpnVibDelay, NrpnVibParams };
enum { NrpnCutoff = 0, NrpnResonance, NrpnSoundParams };
enum { NrpnEnvAttack = 0, NrpnEnvDecay, NrpnEnvRelease, NrpnEnvParams };

class tParam {
public:
	tControl Msb;
	tControl Lsb;
	tControl Data;
	tParam( int cha, uchar id1, uchar msb, uchar id2, uchar lsb, uchar val ) :
		Msb( 0, cha, id1, msb ),
		Lsb( 0, cha, id2, lsb ),
		Data( 0, cha, 0x06, val ) {
	}
	virtual int Write(tWriteBase &io);
	virtual void SetCha( uchar cha );
	virtual int GetVal() { return( Data.Value ); }
};

class tNrpn : public tParam {
public:
	tNrpn( int cha, uchar msb, uchar lsb, uchar val ) : 
		tParam( cha, 0x63, msb, 0x62, lsb, val ) {
	}
};

class tRpn : public tParam {
public:
	tRpn( int cha, uchar msb, uchar lsb, uchar val ) : 
		tParam( cha, 0x65, msb, 0x64, lsb, val ) {
	}
};

uchar *SysExDT1( uchar aa, uchar bb, uchar cc, uchar dd );

class tEventArray : public wxObject
{
    friend class tEventIterator;
    friend class tTrackDlg;
    friend class tTrack;
    friend class tUndoBuffer;

  public:

    long nEvents;	// act Number of Events in **Events
    long MaxEvents;	// Memory allocated in **Events
    tEvent **Events;
    void Resize();	// resize **Events

    // Events set by Cleanup
    tTrackName *Name;
    tProgram   *Patch;
    tSetTempo  *Speed;
    tControl   *Volume;
    tControl   *Pan;
    tControl   *Reverb;
    tControl   *Chorus;
    tControl   *Bank;

    tSysEx  *ModDepth;
    tSysEx  *ReverbType;
    tSysEx  *ChorusType;

    tSysEx  *MasterVol;
    tSysEx  *MasterPan;

    tNrpn   *VibRate;
    tNrpn   *VibDepth;
    tNrpn   *VibDelay;
    tNrpn   *Cutoff;
    tNrpn   *Resonance;
    tNrpn   *EnvAttack;
    tNrpn   *EnvDecay;
    tNrpn   *EnvRelease;
    tRpn    *PitchBendSens;

    void Sort();

    int Channel;	// 1..16, set from first ChannelEvent, 0 = multichannel/nochannel
    int ForceChannel;


    void Put(tEvent *e);
    virtual void Clear();
    void Cleanup(tUndoBuffer *UndoBuffer);	

    void Keyoff2Length();
    void Length2Keyoff();

    tEventArray();
    virtual ~tEventArray();

    void Read(tReadBase &io);
    void Write(tWriteBase &io);

    long GetLastClock();
    int  IsEmpty();
    int  GetFirstClock();

};

/*
 * UndoBuffer enthaelt geloeschte Events (IsKilled) und
 * Referenzen(!) auf kopierte Events
 */

class tUndoBuffer 
{
    friend class tTrack;

    tEventArray Killed;	// copies of Killed events
    tEventArray Copies; // references! to copied events

  public:

    tUndoBuffer();
    void Clear();	// deleted nur IsKilled() Events
    void Put(tEvent *e);
    void Debug();
};



#define MaxUndo 10

class tTrack : public tEventArray
{
  public:

    int iUndo;
    tUndoBuffer UndoBuffers[MaxUndo];

    int State;		// tsXXX

    wxDialogBox *DialogBox;
    tNamedValue *PatchNames;

    tTrack();
    void Dialog(wxFrame *parent);

    void Put(tEvent *e)
    {
      tEventArray::Put(e);
      UndoBuffers[iUndo].Put(e);
    }

    void Kill(tEvent *e)
    {
      e->Kill();
    }

    void Merge(tEventArray *other);
    void MergeRange(tEventArray *other, long FromClock, long ToClock, int Replace = 0);
    void Undo();
    void NewUndoBuffer();
    void Clear();
    void Cleanup();

    char *GetName();
    void SetName(char *Name);

    char *GetStateChar();
    void SetState(int NewState);
    void ToggleState(int Direction); 	// +1 = next, -1 = prev

    int  GetChannel() { return Channel; }
    void SetChannel(int NewChannel);

    int  GetPatch();
    void SetPatch(int PatchNr);

    int  GetVolume();
    void SetVolume(int Volume);

    int  GetPan();
    void SetPan(int Pan);

    int  GetReverb();
    void SetReverb(int Reverb);

    int  GetChorus();
    void SetChorus(int Chorus);

    int  GetVibRate();
    void SetVibRate(int VibRate);

    int  GetVibDepth();
    void SetVibDepth(int VibDepth);

    int  GetVibDelay();
    void SetVibDelay(int VibDelay);

    int  GetCutoff();
    void SetCutoff(int Cutoff);

    int  GetResonance();
    void SetResonance(int Resonance);

    int  GetEnvAttack();
    void SetEnvAttack(int EnvAttack);

    int  GetEnvDecay();
    void SetEnvDecay(int EnvDecay);

    int  GetEnvRelease();
    void SetEnvRelease(int EnvRelease);

    int  GetPitchBendSens();
    void SetPitchBendSens(int PitchBendSens);

    int  GetModDepth();
    void SetModDepth(int ModDepth);

    int  GetReverbType();
    void SetReverbType(int ReverbType);

    int  GetChorusType();
    void SetChorusType(int ChorusType);

    int  GetBank();
    void SetBank(int Bank);

    int  GetSpeed();	// beats per minute
    void SetSpeed(int bpm);

    int  GetMasterVol();
    void SetMasterVol(int MasterVol);

    int  GetMasterPan();
    void SetMasterPan(int MasterPan);

};


// ***********************************************************************
// tEventIterator
// *********************************************************************

class tEventIterator
{
    tEventArray *Track;
    long Start, Stop, Actual;

  public:

    tEventIterator(tEventArray *t)
    {
      Track  = t;
      Start  = 0;
      Stop   = Track->nEvents;
      Actual = Start;
    }


    tEvent *GreaterEqual(long Clock)
    {
      long lo = Start;
      long hi = Stop;
      long clk = 0;
      while (lo < hi)
      {
	Actual  = (hi + lo) / 2;
        clk = Track->Events[Actual]->Clock;
        if (clk < Clock)
          lo = Actual + 1;
	else 
	  hi = Actual;
      }
      if (Actual < Stop-1 && clk < Clock)
	clk = Track->Events[++Actual]->Clock;
      if (Actual < Stop && clk >= Clock)
	return Track->Events[Actual];
      return 0;
    }


    tEvent *First(long Clock = 0)
    {
      Actual = Start;
      return GreaterEqual(Clock);
    }


    tEvent *Range(long frClock, ulong toClock)
    {
      Start = Actual = 0;
      Stop  = Track->nEvents;

      if (!GreaterEqual(frClock))
        return 0;
      Start = Actual;
      if (GreaterEqual(toClock))
        Stop = Actual;
      Actual = Start;
      return (Actual < Stop ? Track->Events[Actual] : 0);
    }


    tEvent *Next()
    {
      if (Actual < Stop)
	++ Actual;
      return (Actual < Stop ? Track->Events[Actual] : 0);
    }

    long EventsLeft()
    {
      return Stop - Actual;
    }

};

#endif

