// dialogs.cc,v 2.1 1995/06/19 19:04:39 andreas Exp

/*
**  jazz - a midi sequencer for Linux
**
**  Copyright (C) 1994 Andreas Voss (andreas@avix.rhein-neckar.de)
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include "dialogs.h"
#include "song.h"
#include "command.h"
#include "eventwin.h"
#include "track.h"
#include "events.h"
#include "util.h"
#include "jazz.h"

// **************************************************************************
// Quantize
// *************************************************************************

Bool tQuantizeDlg::NoteStart = 1;
Bool tQuantizeDlg::NoteLength = 0;
int  tQuantizeDlg::QntStep = 30;

tQuantizeDlg::tQuantizeDlg(tFilter *f)
{
  Filter = f;
  Song = f->Song;
}



void tQuantizeDlg::OnOk()
{
  tCmdQuantize qnt(Filter, QntStep);
  qnt.NoteStart = NoteStart;
  qnt.NoteLength = NoteLength;
  qnt.Execute();
  wxForm::OnOk();
}


void tQuantizeDlg::EditForm(wxPanel *panel)
{
  char buf[200];
  sprintf(buf, "Ticks per Quarter: %d", (int)Song->TicksPerQuarter);
  Add(wxMakeFormMessage(buf));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Ticks", &QntStep, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(1.0, (double)Song->TicksPerQuarter), 0)));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormBool("Note start", &NoteStart));
  Add(wxMakeFormBool("Note length", &NoteLength));
  AssociatePanel(panel);
}

// **************************************************************************
// Shift
// *************************************************************************

int tShiftDlg::Steps = 0;

tShiftDlg::tShiftDlg(tFilter *f, long unit)
{
  Filter = f;
  Song = f->Song;
  Unit  = unit;
}



void tShiftDlg::OnOk()
{
  tCmdShift cmd(Filter, Steps * Unit);
  cmd.Execute();
  wxForm::OnOk();
}


void tShiftDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormShort("Snaps", &Steps, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(-16.0, 16.0), 0)));
  Add(wxMakeFormNewLine());
  AssociatePanel(panel);
}


// **************************************************************************
// Transpose
// *************************************************************************

static tNamedValue ScaleNames[] = 
{
  {"None",     ScaleChromatic},
  {"Selected", ScaleSelected},
  {"C",   0},
  {"C#",  1},
  {"D",   2},
  {"D#",  3},
  {"E",   4},
  {"F",   5},
  {"F#",  6},
  {"G",   7},
  {"G#",  8},
  {"A",   9},
  {"A#", 10},
  {"B",  11},
  { 0,   ScaleChromatic}
};


int tTransposeDlg::Notes  = 0;
long tTransposeDlg::Scale  = ScaleChromatic;
Bool tTransposeDlg::FitIntoScale = 0;

tTransposeDlg::tTransposeDlg(tFilter *f)
   :ScaleDlg("scale", ScaleNames, &Scale)
{
  Filter = f;
  Song   = f->Song;
}


void tTransposeDlg::OnOk()
{
  ScaleDlg.GetValue();
  tCmdTranspose trn(Filter, Notes, Scale, FitIntoScale);
  trn.Execute();
  wxForm::OnOk();
}


void tTransposeDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormShort("Amount", &Notes, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(-24.0, 24.0), 0)));
  Add(wxMakeFormNewLine());
  Add(ScaleDlg.mkFormItem());
  Add(wxMakeFormBool("Fit into Scale", &FitIntoScale));
  Add(wxMakeFormNewLine());
  AssociatePanel(panel);
}


// **************************************************************************
// SetChannel
// *************************************************************************

int tSetChannelDlg::NewChannel = 0;

tSetChannelDlg::tSetChannelDlg(tFilter *f)
{
  Filter = f;
  Song = f->Song;
}



void tSetChannelDlg::OnOk()
{
  if (NewChannel)
  {
    tCmdSetChannel exe(Filter, NewChannel - 1);
    exe.Execute();
  }
  wxForm::OnOk();
}


void tSetChannelDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormShort("new Channel", &NewChannel, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(1.0, 16.0), 0)));
  Add(wxMakeFormNewLine());
  AssociatePanel(panel);
}


// **************************************************************************
// Velocity
// *************************************************************************

int tVelocityDlg::FromValue = 64;
int tVelocityDlg::ToValue = -1;
int tVelocityDlg::Mode = 0;
char *tVelocityDlg::mode_str = 0;

tVelocityDlg::tVelocityDlg(tFilter *f)
{
  Filter = f;
  Song = f->Song;
}


void tVelocityDlg::OnOk()
{
  if (!strcmp(mode_str, "Set")) Mode = 0;
  if (!strcmp(mode_str, "Add")) Mode = 1;
  if (!strcmp(mode_str, "Sub")) Mode = 2;
  tCmdVelocity cmd(Filter, FromValue, ToValue, Mode);
  cmd.Execute();
  wxForm::OnOk();
}


void tVelocityDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormShort("Start", &FromValue, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(0.0, 127.0), 0)));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Stop ", &ToValue, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(0.0, 127.0), 0)));
  Add(wxMakeFormNewLine());

  mode_str = 0;
  Add(wxMakeFormString("Mode", &mode_str, wxFORM_CHOICE,
                       new wxList(wxMakeConstraintStrings("Set", "Add", "Sub", 0) , 0), NULL, wxHORIZONTAL));
  AssociatePanel(panel);
}


// **************************************************************************
// Length
// *************************************************************************

int tLengthDlg::FromValue = 30;
int tLengthDlg::ToValue = -1;
int tLengthDlg::Mode = 0;
char *tLengthDlg::mode_str = 0;

tLengthDlg::tLengthDlg(tFilter *f)
{
  Filter = f;
  Song = f->Song;
}


void tLengthDlg::OnOk()
{
  if (!strcmp(mode_str, "Set")) Mode = 0;
  if (!strcmp(mode_str, "Add")) Mode = 1;
  if (!strcmp(mode_str, "Sub")) Mode = 2;
  tCmdLength cmd(Filter, FromValue, ToValue, Mode);
  cmd.Execute();
  wxForm::OnOk();
}


void tLengthDlg::EditForm(wxPanel *panel)
{
  char buf[200];
  sprintf(buf, "Ticks per Quarter: %d", (int)Song->TicksPerQuarter);
  Add(wxMakeFormMessage(buf));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Start", &FromValue, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(0.1, Song->TicksPerQuarter), 0)));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Stop ", &ToValue, wxFORM_DEFAULT,
                       new wxList(wxMakeConstraintRange(0.0, Song->TicksPerQuarter), 0)));
  Add(wxMakeFormNewLine());

  mode_str = 0;
  Add(wxMakeFormString("Mode", &mode_str, wxFORM_CHOICE,
                       new wxList(wxMakeConstraintStrings("Set", "Add", "Sub", 0) , 0), NULL, wxHORIZONTAL));
  AssociatePanel(panel);
}



// *************************************************************************
// Delete
// *************************************************************************

Bool tDeleteDlg::LeaveSpace = 1;

tDeleteDlg::tDeleteDlg(tFilter *f)
{
  Filter = f;
}


void tDeleteDlg::OnOk()
{
  tCmdErase cmd(Filter, LeaveSpace);
  cmd.Execute();
  wxForm::OnOk();
}

void tDeleteDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormBool("Leave Space", &LeaveSpace));
  AssociatePanel(panel);
}


// ***********************************************************************
// Event-Dialoge
// ***********************************************************************

class tEventDlg : public wxForm
{
  public:

    tTrack    *Track;
    tClockDlg ClockDlg;
    tEventWin *Win;

    tEvent    *Event;
    tEvent    *Copy;

    tEventDlg(tEvent *e, tEventWin *w, tTrack *t);
    virtual void EditForm(wxPanel *panel);
    virtual void OnOk();
    virtual void OnCancel();
};


tEventDlg::tEventDlg(tEvent *e, tEventWin *w, tTrack *t)
  : ClockDlg(w->Song, "Time: ", e->Clock)
{
  Win   = w;
  Track = t;
  Event = e;
  Copy  = e->Copy();
}

void tEventDlg::EditForm(wxPanel *panel)
{
  Add(ClockDlg.mkFormItem());
  Add(wxMakeFormNewLine());
  AssociatePanel(panel);
}

void tEventDlg::OnCancel()
{
  delete Copy;
  wxForm::OnCancel();
}

void tEventDlg::OnOk()
{
  Copy->Clock = ClockDlg.GetClock();
  Track->Kill(Event);
  Track->Put(Copy);
  Track->Cleanup();
  wxForm::OnOk();
}

// --------------------------- ChannelEvent ----------------------------


class tChEventDlg : public tEventDlg
{
  public:

    int Channel;

    tChEventDlg(tChannelEvent *e, tEventWin *w, tTrack *t)
      : tEventDlg(e, w, t)
    {
      Channel = e->Channel + 1;		// 1..16
    }
    void EditForm(wxPanel *panel);
    void OnOk();
};

void tChEventDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormShort("Channel:", &Channel, wxFORM_DEFAULT, new wxList(wxMakeConstraintRange(1.0, 16.0), 0)));
  Add(wxMakeFormNewLine());
  tEventDlg::EditForm(panel);
}


void tChEventDlg::OnOk()
{
  ((tChannelEvent *)Copy)->Channel = Channel - 1;
  tEventDlg::OnOk();
}

// -------------------------------- Note On -------------------------------

class tKeyOnDlg : public tChEventDlg
{
 public:

  tKeyDlg PitchDlg;
  int Pitch;
  int Veloc;
  int Length;

  tKeyOnDlg(tKeyOn *e, tEventWin *w, tTrack *t);

  void EditForm(wxPanel *panel);
  void OnOk();
};


tKeyOnDlg::tKeyOnDlg(tKeyOn *e, tEventWin *w, tTrack *t)
  : tChEventDlg(e, w, t),
    PitchDlg("Pitch:", e->Key)
{ 
  Event = e;
  Veloc = e->Veloc;
  Pitch = e->Key;
  Length = e->Length;
}


void tKeyOnDlg::OnOk()
{
  tKeyOn *k = (tKeyOn *)Copy;
  k->Key = PitchDlg.GetKey();
  k->Veloc = Veloc;
  k->Length = Length;
  tChEventDlg::OnOk();
}

void tKeyOnDlg::EditForm(wxPanel *panel)
{
  Add(PitchDlg.mkFormItem());
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Veloc:", &Veloc, wxFORM_DEFAULT, new wxList(wxMakeConstraintRange(1.0, 127.0), 0)));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Length:", &Length, wxFORM_DEFAULT));
  Add(wxMakeFormNewLine());
  tChEventDlg::EditForm(panel);
}


// -------------------------------- Pitch -------------------------------

class tPitchDlg : public tChEventDlg
{
 public:

  int Value;

  tPitchDlg(tPitch *e, tEventWin *w, tTrack *t);

  void EditForm(wxPanel *panel);
  void OnOk();
};


tPitchDlg::tPitchDlg(tPitch *e, tEventWin *w, tTrack *t)
  : tChEventDlg(e, w, t)
{ 
  Event = e;
  Value = e->Value;
}


void tPitchDlg::OnOk()
{
  ((tPitch *)Copy)->Value = Value;
  tChEventDlg::OnOk();
}

void tPitchDlg::EditForm(wxPanel *panel)
{
  Add(wxMakeFormShort("Pitch:", &Value, wxFORM_DEFAULT, new wxList(wxMakeConstraintRange(-8191, 8191), 0)));
  Add(wxMakeFormNewLine());
  tChEventDlg::EditForm(panel);
}

// -------------------------------- Controller ---------------------------

class tControlDlg : public tChEventDlg
{
 public:

  int Value; 
  long Control;
  tNamedChoice Choice;

  tControlDlg(tControl *e, tEventWin *w, tTrack *t);

  void EditForm(wxPanel *panel);
  void OnOk();
};


tControlDlg::tControlDlg(tControl *e, tEventWin *w, tTrack *t)
  : tChEventDlg(e, w, t),
    Choice("Controller", ControlNames, &Control)
{ 
  Event = e;
  Value = e->Value;
  Control = e->Control + 1;
}


void tControlDlg::OnOk()
{
  ((tControl *)Copy)->Value = Value;
  Choice.GetValue();
  ((tControl *)Copy)->Control = Control - 1;
  tChEventDlg::OnOk();
}

void tControlDlg::EditForm(wxPanel *panel)
{
  Add(Choice.mkFormItem(300, 300));
  Add(wxMakeFormNewLine());
  Add(wxMakeFormShort("Value:", &Value, wxFORM_DEFAULT, new wxList(wxMakeConstraintRange(0, 127), 0)));
  Add(wxMakeFormNewLine());
  tChEventDlg::EditForm(panel);
}

// -------------------------------- Program ---------------------------

class tProgramDlg : public tChEventDlg
{
 public:

  long Program; 
  tNamedChoice Choice;

  tProgramDlg(tProgram *e, tEventWin *w, tTrack *t);

  void EditForm(wxPanel *panel);
  void OnOk();
};


tProgramDlg::tProgramDlg(tProgram *e, tEventWin *w, tTrack *t)
  : tChEventDlg(e, w, t),
    Choice("Program", VoiceNames, &Program)
{ 
  Event = e;
  Program = e->Program + 1;
}


void tProgramDlg::OnOk()
{
  Choice.GetValue();
  if (Program <= 0)
    Program = 1;
  ((tProgram *)Copy)->Program = Program - 1;
  tChEventDlg::OnOk();
}

void tProgramDlg::EditForm(wxPanel *panel)
{
  Add(Choice.mkFormItem(300, 300));
  Add(wxMakeFormNewLine());
  tChEventDlg::EditForm(panel);
}

// --------------------------------------------------------------------------
// Verteiler
// --------------------------------------------------------------------------

static tEvent *CreateEventDialog(long Clock, int Channel, int Pitch)
{
  static char *Names[] = 
  { "Note On", "Pitch", "Controller", "Program Change", (char *)0 };
  static long Values[] =
  { StatKeyOn, StatPitch, StatControl, StatProgram, -1 };
  tEvent *e = 0;
  int i = wxGetSingleChoiceIndex("Select Eventtype", "", 4, Names);
  if (i >= 0)
  {
    switch (Values[i])
    {
      case StatKeyOn:
        e = new tKeyOn(Clock, Channel, Pitch, 64, 64);
        break;
      case StatPitch:
        e = new tPitch(Clock, Channel, 0);
        e->SetPitch(Pitch);
        break;
      case StatControl:
        e = new tControl(Clock, Channel, Pitch, 64);
        break;
      case StatProgram:
        e = new tProgram(Clock, Channel, Pitch);
        break;
    }
  }
  return e;
}




void EventDialog(tEvent *e, tEventWin *w, tTrack *t, long Clock, int Channel, int Pitch)
{
  if (!e)
    e = CreateEventDialog(Clock, Channel, Pitch);

  tEventDlg *dlg = 0;
  char *str = 0;
  switch (e->Stat)
  {
    case StatKeyOn:
      str = "Key On";
      dlg = new tKeyOnDlg(e->IsKeyOn(), w, t);
      break;

    case StatPitch:
      str = "Pitch Wheel";
      dlg = new tPitchDlg(e->IsPitch(), w, t);
      break;

    case StatControl:
      str = "Controller";
      dlg = new tControlDlg(e->IsControl(), w, t);
      break;

    case StatProgram:
      str = "Program Change";
      dlg = new tProgramDlg(e->IsProgram(), w, t);
      break;

    default:
      break;
  }
  if (dlg)
  {
    wxDialogBox *panel = new wxDialogBox(w, str, FALSE);
    dlg->EditForm(panel);
    panel->Fit();
    panel->Show(TRUE);
  }
}
