/*
**  jazz - a midi sequencer for Linux
**
**  Copyright (C) 1994 Andreas Voss (andreas@avix.rhein-neckar.de)
**
**  Portions Copyright (C) 1995 Per Sigmond (Per.Sigmond@hiagder.no)
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* midinetd.c using TCP protocol - Written by Per Sigmond, HiA			*/
/* Version 2.0f */

#include <stdio.h>
#include <string.h>
#ifdef linux
#include <linux/time.h>
#endif
#ifdef sun386
#include <memory.h>
#endif
#include <sys/file.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <netdb.h>
#include <netinet/in.h>
#include <errno.h>
#include <fcntl.h>
#ifndef linux
#include <netinet/tcp.h>
#endif
#include <assert.h>
#include <stdlib.h>
#include <unistd.h>

#define MPUDEVICE "/dev/mpu401"
#define MIDINETSERVICE "midinet"

#define MAXTRIES 100000

#define MAX(x,y) ((x) > (y) ? (x) : (y))

#define CMD (2<<6)
#define DAT (1<<6)
#define RES (3<<6)

#define CLOCK_TO_HOST 0xfd

#define START_PLAY_COMMAND 0xfa
#define STOP_PLAY_COMMAND 0xfb
#define START_OF_RECORD_BUFFER 0xfb

int printerror( char *str ) {

	char timestr[25];
	time_t now;

	now = time( NULL );
	strncpy( timestr, ctime( &now ), 24 );
	timestr[24] = '\0';

	fprintf( stderr, "%s midinetd: ", timestr );
	fprintf( stderr, "errno=%d ", errno );
	perror( str );
	return(0);
}


int main(int argc, char **argv)
{
struct sockaddr_in remote_addr, my_addr;
struct sockaddr_in *remote_addr_ptr = &remote_addr;
int sd1 = 0,sd2;
int fd;
unsigned char sockbuf[BUFSIZ];
unsigned char *sockbuf_ptr;
int sockbytes, sockbytes_read, sockbytes_to_write;
unsigned char filebuf[BUFSIZ];
unsigned char *recbuf = NULL;
int bytes_in_recbuf = 0;
int size_of_recbuf = 0;
int recbuf_bytes_sent, bytes_sent_now;
int filebytes;
struct protoent *pent;
struct servent *sent;
int addlen;
int *addlen_ptr = &addlen;
int setval;

unsigned char clock_to_host_byte = CLOCK_TO_HOST;
unsigned char start_play_command = START_PLAY_COMMAND;
int start_play_command_received = 0;
unsigned char stop_play_command = STOP_PLAY_COMMAND;
int stop_play_command_received = 0;
unsigned char start_marker = START_OF_RECORD_BUFFER;

int nfds;
fd_set rfds;

int inetd = 0;
int i, j;
int k;
unsigned char ch;

int logfd;

/* argv[0] == "in.midinetd" means started by inetd */
if (!strcmp( argv[0], "in.midinetd") ) inetd = 1;

if (argc > 1) {
	logfd = open( argv[1], O_APPEND | O_CREAT | O_WRONLY, 00666 );
	if (logfd == -1) {
		fprintf( stderr, "Could not open logfile %s\n", argv[1] );
		exit(1);
	}
	if (dup2( logfd, 2 ) == -1) {
		fprintf( stderr, "Could not dup2 stderr to logfile\n" );
		exit(1);
	}
}

if (!inetd) {
	/* Set adress-type and address */
	memset((char *)&my_addr, 0,  sizeof my_addr);
	my_addr.sin_family      = AF_INET;
	my_addr.sin_addr.s_addr = INADDR_ANY;

	/* Set TCP-port to listen on */
	sent = getservbyname( MIDINETSERVICE, "tcp" );
	if (!sent) {
		printerror("getservbyname");
		exit(1);
	}
	my_addr.sin_port        = sent->s_port;


	/* Open/bind socket */
	sd1 = socket(PF_INET,SOCK_STREAM,0);
	if (bind(sd1,(struct sockaddr*) &my_addr,sizeof(my_addr)) < 0) {
		printerror("bind");
       		exit(1);
	}
 
	/* Issue listen-command */
	if (listen(sd1,5) < 0) {
		printerror("listen");
		exit(1);
	}

	/* Wait for incoming call by issuing accept-call */
	/* (hangs until call arrives) */
	fprintf(stderr,"Start accepting\n");
	sd2 = accept(sd1,(struct sockaddr*) remote_addr_ptr,addlen_ptr);
  
	if (sd2 < 0) {
		printerror("accept");
    		exit(1);
	}
	else {
		fprintf( stderr, "Connected...\n");
	}
}
else {
	/* Started by inetd; the socket is at stdin/stdout ! */
	sd2 = 0;
}

/* Open device for reading/writing */
if ((fd = open( MPUDEVICE, O_RDWR )) < 0) {
	printerror("Open file failed");
	if (!inetd) {
		close(sd1);
	}
	close(sd2);
	exit(1);
}

/* Set TCP options (send bytes immediately) */
pent = getprotobyname( "tcp" );
setval = 1;
if (setsockopt( sd2, pent->p_proto, TCP_NODELAY, &setval, sizeof(setval) )) {
	printerror("setsockopt");
	if (!inetd) {
		close(sd1);
	}
	close(sd2);
	close (fd);
	exit(0);
}

/* Prepare for select */
nfds = MAX( fd, sd2 ) + 1;
FD_ZERO( &rfds );

/* Initialize some numbers */
sockbytes = 0;

/* Select-loop */
for (;;) {
	FD_SET( fd, &rfds );
	FD_SET( sd2, &rfds );

	/* Hang here until data is ready for reading */
	if (select( nfds, &rfds, (fd_set*) 0, (fd_set*) 0, (struct timeval*) 0 ) < 0) {
		printerror("select");
		if (!inetd) {
			close(sd1);
		}
		close(sd2);
		close( fd );
		exit(0);
	}

	/* File ready ? */
	if (FD_ISSET( fd, &rfds )) {
		/* Read from file */
		filebytes = read( fd, filebuf, BUFSIZ );
		if ( filebytes == -1 ) {
			printerror("read file");
			if (!inetd) {
				close(sd1);
			}
			close(sd2);
			close( fd );
			exit(1);
		}

		/* Write to recbuffer and send clock_to_host_byte to socket */
		if (filebytes > 0) {
			for (k = 0; k < filebytes; k++) {
				ch = filebuf[k];
				if (ch == 0xff) continue; 
				/* I sometimes get these 0xff bytes, 
				   don't know why but they are of no use */

				/* clock-to-host received ? */
				if (ch == 0xfd) {
					/* Send clock-to-host to jazz */
					j = write(sd2, &clock_to_host_byte, 1);
					if (j <= 0) {
						printerror("write socket (1)");
						if (!inetd) {
							close(sd1);
						}
						close(sd2);
						close( fd );
						exit(1);
					}
				}
				else {
					/* record byte received; store in record buffer */
					if (bytes_in_recbuf >= size_of_recbuf) {
						recbuf = (unsigned char*) realloc( recbuf, size_of_recbuf + BUFSIZ );
						assert( recbuf );
						size_of_recbuf = size_of_recbuf + BUFSIZ;
					}
					recbuf[bytes_in_recbuf++] = ch;
				}
				
			}
		}
	}

	/* Socket ready ? */
	if (FD_ISSET( sd2, &rfds )) {
		/* Read from socket */
		sockbytes_read = read( sd2, sockbuf + sockbytes, BUFSIZ - sockbytes );
		if (sockbytes_read == -1) {
			printerror("read socket");
			if (!inetd) {
				close(sd1);
			}
			close(sd2);
			close( fd );
			exit(1);
		}

		sockbytes = sockbytes + sockbytes_read;
		if (sockbytes == 0) break;

		/* Did jazz request start playing ? */
		sockbuf_ptr = memchr( sockbuf, start_play_command, sockbytes );
		if ( sockbuf_ptr ) { /* Yes */
			/* Take the command byte away and adjust the sockbuf data */
			while (sockbuf_ptr < &sockbuf[sockbytes]) {
				*sockbuf_ptr = *(sockbuf_ptr + 1);
				sockbuf_ptr++;
			}
			sockbytes--;
			start_play_command_received = 1;
		}
		else {
			start_play_command_received = 0;
		}

		/* Did jazz request stop playing ? */
		sockbuf_ptr = memchr( sockbuf, stop_play_command, sockbytes );
		if ( sockbuf_ptr ) { /* Yes */
			/* Take the command byte away and adjust the sockbuf data */
			while (sockbuf_ptr < &sockbuf[sockbytes]) {
				*sockbuf_ptr = *(sockbuf_ptr + 1);
				sockbuf_ptr++;
			}
			sockbytes--;
			recbuf_bytes_sent = 0;
			stop_play_command_received = 1;
		}
		else {
			stop_play_command_received = 0;
		}

		/* Write socket data to file */
		if (sockbytes > 0) {
			volatile int ii;
			/* Check for incomplete commands */
			/* (TCP does not guarantee block boundaries) */
			{
				char *ptr, *beg, *end, *cmd;

				/* Set "ptr" at beginning, set "end" at
				   first "unused" byte:
				 */
				beg = sockbuf; end = &sockbuf[sockbytes];
				ptr = beg; cmd = beg;
				while (ptr < end) {
					char n;
					cmd = ptr;
					n = *ptr & 0x3f;
					ptr = ptr + n + 1;
				}
				/* "ptr" will now point one byte after last
				   command. If complete commands, this will be
				   same as "end".
				   "cmd" points at beginning of last command.
				*/
				if (ptr > end) {
					/* Last command incomplete.
					   Remember how many bytes.
					*/
					sockbytes_to_write = cmd - beg;
				}
				else {
					/* Complete commands. Write all bytes */
					sockbytes_to_write = sockbytes;
				}

			}
			j = 0;
			for (ii = 0; ii < MAXTRIES; ii++) {
				k = write(fd, sockbuf + j, sockbytes_to_write - j);
				if (k > 0) j = j + k;
				if (j >= sockbytes_to_write) break;
			}
			if (ii >= MAXTRIES) {
				printerror("write file (1)");
				if (!inetd) {
					close(sd1);
				}
				close(sd2);
				close( fd );
				exit(1);
			}
			/* Move remaining bytes to beginning of buffer */
			for (i = 0, j = sockbytes_to_write; j < sockbytes; i++, j++) {
				sockbuf[i] = sockbuf[j];
			}
			/* Remember how many to write next time */
			sockbytes = i;
		}

		if ( start_play_command_received ) {
			if (recbuf) free( recbuf );
			recbuf = NULL;
			bytes_in_recbuf = 0;
			recbuf = (unsigned char*) malloc( BUFSIZ );
			assert( recbuf );
			size_of_recbuf = BUFSIZ;
		}

		/* Did we receive request for sending record buffer data? */
		if ( stop_play_command_received ) {
			/* Yes, send start marker first (START_OF_RECORD_BUFFER) */
			if (write( sd2, &start_marker, 1 ) < 1 ) {
				printerror("write socket (2)");
				if (!inetd) {
					close(sd1);
				}
				close(sd2);
				close( fd );
				exit(1);
			}
			/* Send number of bytes in record buffer */
			j = htonl( bytes_in_recbuf );
			if ( write( sd2, &j, sizeof(int)) < sizeof(int) ) {
				printerror("write socket (3)");
				if (!inetd) {
					close(sd1);
				}
				close(sd2);
				close( fd );
				exit(1);
			}

			/* Send contents of record buffer */
			recbuf_bytes_sent = 0;
			while (recbuf_bytes_sent < bytes_in_recbuf) {
				if ( (bytes_in_recbuf - recbuf_bytes_sent) >= BUFSIZ)
					k = BUFSIZ;
				else
					k = bytes_in_recbuf - recbuf_bytes_sent;
				bytes_sent_now = write(sd2, recbuf + recbuf_bytes_sent, k );
				if (bytes_sent_now < 0) {
					printerror("write socket (4)");
					if (!inetd) {
						close(sd1);
					}
					close(sd2);
					close( fd );
					exit(1);
				}
				recbuf_bytes_sent = recbuf_bytes_sent + bytes_sent_now;
			}
		}

	}


} /* end for */


/* Close sockets and files */
if (!inetd) {
	close(sd1);
}
close(sd2);
close( fd );

exit(0);
} /* end main */
  
