/*
 * Roland MPU-401 MIDI-Driver
 * 
 * Andreas Voss (andreas@avix.rni.sub.org)
 *
 * mpu_write():
 *
 *   writes Data to MPU's Registers in MPU-Format with 1 additional leading Byte:
 *     Bits 0..5: length of MPU-Data following.
 *     Bits 6..7:
 *       00 : send later on Track Data Request Interrupt 
 *       01 : write immediately to Data Register
 *       10 : write immediately to Command Register
 *       11 : driver command: 0xc0 = reset everything
 *
 * mpu_read()
 *    returns recorded data in mpu-format
 */

#include <dos.h>
#include "driver.h"
#include "mpu.h"

#define EINVAL 1

struct mpu_queue
{
  int rd, wr;
  unsigned char buffer[MPU_BUFFER_SIZE];
};

struct mpu_status
{
  struct mpu_queue recvq;
  struct mpu_queue sendq[ACTIVE_TRACKS];
  int running_status;
};

static struct mpu_status mpu;


/* get char from queue */
static int mpu_readq(struct mpu_queue *q)
{
  unsigned char c;
  if (q->wr == q->rd)
    return -1;
  c = q->buffer[q->rd];
  q->rd = (q->rd + 1) % MPU_BUFFER_SIZE;
  return c;
}

/* put char into queue */
static int mpu_writeq(struct mpu_queue *q, unsigned char c)
{
  q->buffer[q->wr] = c;
  q->wr = (q->wr + 1) % MPU_BUFFER_SIZE;
  return 0;
}

/* true, if queue not empty */
static int mpu_statq(struct mpu_queue *q)
{
  return q->rd != q->wr;
}


/* # bytes free in queue */
static int mpu_leftq(struct mpu_queue *q)
{
  int w = q->wr;
  if (q->wr < q->rd)
    w += MPU_BUFFER_SIZE;
  return MPU_BUFFER_SIZE-1 - (w - q->rd);
}


static void mpu_clearq(struct mpu_queue *q)
{
  q->rd = q->wr = 0;
}

/*
 * MPU-Hardware
 */

#define MPU_DSR 0x80
#define MPU_DRR 0x40
#define MPU_ACK 0xfe

#define mpu_write(a) send_data(a)
#define mpu_read() get_data()

static int mpu_reset(void)
{
  send_command(0xff);
  return 0;	/* always ok */
}

void far mpu_interrupt( unsigned char c ) {

int n;
int track;

  if ( (c >= 0xf0) && (c < 0xf8) ) {
    /* 
     * play track data request 
     */
   track = c - 0xf0;

   if (!mpu_statq(&mpu.sendq[track]))	/* nothing to send?? */
   {
    	mpu_write(0xf8);		/* wait 240 ticks */
   	return;
    }
    n = mpu_readq(&mpu.sendq[track]);
    while (n-- > 0)
    {
    	c = mpu_readq(&mpu.sendq[track]);
        mpu_write(c);
    }
  } 

}


void mpu_received(int c)
{
  int n;

  /* length of midi-event             8  9  A  B  C  D  E */
  static const int three_bytes[8] = { 1, 1, 1, 1, 0, 0, 1 };

  if (c < 0xf0)	
  {
    /*
     * found timing byte. store it and get midi-message or mpu-mark
     */

    mpu_writeq(&mpu.recvq, c);	/* save timing byte to record-buffer */
    c = mpu_read();		/* get statusbyte (or 1st data, if running status) */
    mpu_writeq(&mpu.recvq, c);	/* store it, so we know what happend at this time */

    if (c < 0xf0)
    {
      /*
       * it's a midi-voice-message
       */

      if (c & 0x80)			/* new running status */
      {
	mpu.running_status = c;
	c = mpu_read();			/* get 1st data-byte */
	mpu_writeq(&mpu.recvq, c);
      }
      if (three_bytes[((unsigned char)mpu.running_status >> 4) - 8])
      {
	c = mpu_read();
	mpu_writeq(&mpu.recvq, c);
      }
    }

    /* 
     * else: mpu mark's are already stored
     */

  }

  else if ( c < 0xf8 ) {
    /* 
     * play track data request 
     */
    /* Already handled by mpu_interrupt() */
  }
  else
  {
    /* 
     * got mpu-stuff
     */

    switch (c)
    {
      case 0xfe:	/* ignore ack's */
        break;

      /*
       * implemented:
       *   f8: timer overflow
       *   fc: end of data
       *   fd: clock to host
       *
       * not implemented:
       *   all others (real time messages, conductor etc). record them
       *   so the application might discover an error.
       */

      default:
	mpu_writeq(&mpu.recvq, c);
	break;
    }
  }
  return;
}


#define mpu_command(cmd) send_command(cmd)

void mpu_init() {
int i;
  mpu_clearq(&mpu.recvq);
  for (i = 0; i < ACTIVE_TRACKS; i++) mpu_clearq(&mpu.sendq[i]);
  mpu.running_status = 0x90;

  return;
}


int write_mpu(char * buffer, int count)
{
  char c, *temp;
  int err, n, i;
  int track;

  temp = buffer;

  while (count > 0)
  {
    c = *temp;
    temp++;
    count--;
    n = c & 0x3f;
    if (n > count)
      return -EINVAL;

    switch ((unsigned char)c >> 6)
    {

      case 0: /* write into send-queue */

	/* Get track number */
	c = *temp;
	temp++;
	count--;
	track = c;
	if ( (track < 0) || (track >= ACTIVE_TRACKS) ) return( -EINVAL );

        while (n + 1 > mpu_leftq(&mpu.sendq[track]))	
        {
	  return temp - buffer - 2;	/* resend command-byte */
        }

	n--; /* Skip track number byte */
	mpu_writeq(&mpu.sendq[track], n);
	while (n-- > 0)
	{
	  c = *temp;
	  temp++;
	  count--;
	  mpu_writeq(&mpu.sendq[track], c);
	}
	break;


      case 1: /* write to data-register */ 

	while (n-- > 0)
	{
	  c = *temp;
	  temp++;
	  count--;
	  if ((err = mpu_write(c)) < 0)
	    return err;
	}
	break;


      case 2: /* write to command-register */

	while (n-- > 0)
	{
	  c = *temp;
	  temp++;
	  count--;
	  if ((err = mpu_command(c)) < 0)
	    return err;
	}
	break;


      default:	/* reset everything */
      {
	unsigned long flags;
	int i;

	mpu_reset();

	disable();
	mpu_clearq(&mpu.recvq);
	for (i = 0; i < ACTIVE_TRACKS; i++ ) mpu_clearq(&mpu.sendq[i]);
	mpu.running_status = 0x90;
	enable();
      }
      break;

    }
  }
  return temp - buffer;
}


int read_mpu(char * buffer, int count)
{
  char c, *temp = buffer;
  int d;

  d = mpu_read();
  if (d != -1) mpu_received( d );

  while (count > 0 && mpu_statq(&mpu.recvq))
  {
    c = mpu_readq(&mpu.recvq);
    *temp = c;
    temp++;
    count --;
  }
  return temp - buffer;
}
