/*
 *  Routines for sampling from Soundblaster 8-bit soundcards
 *  These routines require the SB functions and DMA functions written
 *  by Heath I. Hunnicutt.  The required functions are included here.
 *  For the full copy of his routines, refer to:
 *  oak.oakland.edu:/simtel/msdos/c/????
 *
 *  Copyright (C) 1995  Philip VanBaren
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "freq.h"

#ifdef SC_SB8

#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <graphics.h>
#include "sb.h"
#include "extern.h"

int atox(char *ptr)
{
   // Convert ascii hex values to integer
   int v=0;

   while( ((*ptr>='0') && (*ptr<='9')) || ( ((*ptr|0x20)>='a') && ((*ptr|0x20) <= 'f') ) )
   {
      v=v*16;
      if(*ptr<='9')
	 v=v+*ptr-'0';
      else
	 v=v+(*ptr|0x20)-'a'+10;
      ptr++;
   }
   return v;
}

int is_blaster(char *var)
{
   // Return true if the string begins with "BLASTER"
   return((var[0]=='B') && (var[1]=='L') && (var[2]=='A') && (var[3]=='S') && (var[4]=='T') && (var[5]=='E') && (var[6]=='R'));
}


void far interrupt (*OldIRQ)();

/* Interrupt handler for DMA complete IRQ from Soundblaster */
void far interrupt SBHandler()
{
   flag[record_buffer]=1;
   if(++record_buffer>=BUFFERS)
      record_buffer=0;
   inportb(DSP_DATA_AVAIL);
   outportb(0x20,0x20);
}

void init_sb8(char **environ)
{
   int i;
   unsigned char tm,im;
   // Scan the environment variables for BLASTER=Axxx Ix Dx
   for(i=0;environ[i]!=NULL;i++)
   {
      if(is_blaster(environ[i]))
      {
         int j;
         for(j=8;environ[i][j]!=0;j++)
         {
	    if(environ[i][j]=='A')
	       sb_addr=atox(&environ[i][j+1]);
	    if(environ[i][j]=='D')
	       sb_dma=atoi(&environ[i][j+1]);
	    if(environ[i][j]=='I')
	       sb_irq=atoi(&environ[i][j+1]);
	    // Skip to the next parameter
	    while((environ[i][j] != ' ') && (environ[i][j+1] != 0))
	       j++;
	 }
	 break;
      }
   }
   if(sb_dma>4)
   {
      puts("Only SB DMA channels 1-3 are supported.");
      exit(1);
   }
   if(sb_irq>7)
   {
      puts("Only SB IRQs up to 7 are supported.");
      exit(1);
   }
   reset_soundcard=reset_sb8;
   halt_soundcard=halt_sb8;
   cleanup_soundcard=cleanup_sb8;
   recordblock=recordblock_sb8;
   sample_size=8;
   mixers=0;

   set_SB_address(sb_addr);
   if(!dsp_reset())
   {
      closegraph();
      printf("Soundblaster not found at 0x%04x\n",sb_addr);
      exit(1);
   }
   dsp_voice(0);

   /*
    *  Add the SB DMA interrupt handler in the interrupt chain
    */
   disable();
   OldIRQ = getvect(0x08 + sb_irq);
   setvect(0x08 + sb_irq,SBHandler);
   im=inportb(0x21);
   tm=~(1 << sb_irq);
   outportb(0x21,im & tm);
   enable();
}

void reset_sb8(void)
{
   /*
    *  Initialize Soundblaster stuff
    */
   int i;
   int dsp_tc=256-1000000L/(long)SampleRate;

   dsp_time(dsp_tc);

   /*
    *  Initialize the SB DMA channel
    */
   if(dma_reset(sb_dma))
   {
      closegraph();
      puts("Error resetting SB DMA channel.");
      puts(dma_errlist[dma_errno]);
      cleanup_sb8();
      exit(1);
   }
   // Reset the buffer pointers
   queue_buffer=0;     // Pointer to next buffer to be queued
   record_buffer=0;    // Pointer to next buffer to be filled
   process_buffer=0;   // Pointer to next buffer to be FFTed

   for(i=0;i<BUFFERS;i++)
      flag[i]=0;
   /*
    *  This function starts the DMA process.
    */
   recordblock_sb8(buffer[queue_buffer]);
   if(++queue_buffer>=BUFFERS)
      queue_buffer=0;
}


void halt_sb8(void)
{
   /* Shut down the DMA transfers */
   disable();
   dma_reset(sb_dma);
   dsp_reset();
   enable();
}


void cleanup_sb8(void)
{
   // Clean up the soundcard routines

   unsigned char im,tm;

   disable();
   dma_reset(sb_dma);
   dsp_reset();
   /* Restore old IRQ vector */
   setvect(0x08 + sb_irq,OldIRQ);
   im=inportb(0x21);
   tm=1 << sb_irq;
   outportb(0x21,im | tm);
   enable();
}

void recordblock_sb8(void far *buffer)
{
   /*
    *  Start recording a new buffer.
    *  For now we don't have queueing capabilities for the SB
    */
   if(dma_setup(sb_dma,buffer,fftlen,0))
   {
      closegraph();
      printf("Error in dma_setup(): %d\n",dma_errno);
      puts(dma_errlist[dma_errno]);
      cleanup_sb8();
      exit(1);
   }
   if(dma_errno!=0)
   {
      closegraph();
      puts("DMA error");
      puts(dma_errlist[dma_errno]);
      cleanup_sb8();
      exit(1);
   }
   dsp_dma_prepare(0,fftlen);
}

#endif
