/*
 *  Routine for processing the keyboard input.
 *
 *  Copyright (C) 1995  Philip VanBaren
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <ctype.h>
#include <string.h>
#include <math.h>

#include "freq.h"
#include "fft.h"
#include "extern.h"
#include "display.h"

int display_peak=0;     /* Flag for displaying the peak information */
int decay_mode=0;       /* Flag indicating decay mode on/off */
int help_mode=0;        /* Flag indicating help mode 0 (off), 1, or 2 */
int bw_mode=0;          /* Flag indicating a Black/White palette */
int freeze=0;           /* Flag indicating a paused display */
float freq_scalefactor=1.0;  /* Scalefactor for increasing the horizonal scale */
float freq_base=0.0;    /* Base frequency for the display */
int bin=0;              /* Bin number for cursor display */
int mic_level=0;        /* Storage for mic input mixer level (0-100) */
int ext_level=0;        /* Storage for ext input mixer level (0-100) */
int int_level=0;        /* Storage for int input mixer level (0-100) */

#ifdef DOS
void cleanup_vga(void)
{
   /*
    *  Clean up VGA registers so Borland libraries work again.
    */
   outportb(0x3c4,2);
   outportb(0x3c5,0x0f);   /* Use all bit planes */
}

void setup_vga(void)
{
   /*
    *  Set up VGA registers for fast graphics display
    */
   outportb(0x3ce,1);
   outportb(0x3cf,0);   /* Disable set/reset for all planes */
   outportb(0x3ce,3);
   outportb(0x3cf,0);   /* No rotate, just write data */
   outportb(0x3ce,4);
   outportb(0x3cf,3);   /* Read from bit plane 3 */
   outportb(0x3ce,5);
   outportb(0x3cf,inportb(0x3cf)&0x70); /* Read and write direct to memory */
   outportb(0x3ce,8);
   outportb(0x3cf,0xff);  /* Allow modification of all bits */
   outportb(0x3c4,2);
   outportb(0x3c5,12);   /* Use bit planes 2 and 3 */
}
#endif


void input_text(char *text,int maxlen,int x,int y)
{
   /*
    *  Input a line of text using graphical display at (x,y)
    *  Maximum length allowed for input is given by maxlen
    *  The input begins with the current contents of text, so this must
    *  be initialized before calling the routine.
    */
   int c=0;
   int count;
   char ach[3];
   ach[1]='_';
   ach[2]=0;

   draw_text_left(x,y,text);
   count=strlen(text);
   x+=count*_font_width;
   draw_text_left(x,y,"_");

   while((c != 0x0d) && (c != 0x0a))
   {
      c=draw_getch();

      if(((c==0x08)||(c==0x7f)) && count)
      {
	 count--;
	 x-=_font_width;
	 draw_bar(x,y-1,x+2*_font_width,y+_font_height,0);
	 draw_text_left(x,y,"_");
      }
      else if(count<(maxlen-1))
      {
	 if(((c>='0') && (c<='9')) || ((c>='A') && (c<='Z')) || ((c>='a') && (c<='z')) || (c=='.') || (c=='\\') || (c=='/') || (c==':'))
	 {
	    draw_bar(x,y-1,x+_font_width,y+_font_height,0);

	    ach[0]=c;
	    text[count]=c;
	    draw_text_left(x,y,ach);
	    count++;
	    x+=_font_width;
	 }
      }
      if(c==0x1b)
      {
	 count=0;
	 break;
      }
   }
   text[count]=0;
}

void update_display(void)
{
   /*  Update the frequency and amplitude information displayed while */
   /*  the input is frozen. */

   char ach[100];
   int bri=BitReversed[bin];
   double re=fftdata[bri];
   double im=fftdata[bri+1];
   double amp,db,phase;

   amp=sqrt(re*re+im*im)/32768.0;

   if(gain3db)
      amp*=sqrt((double)bin*(double)SampleRate/fftlen/(double)ref_freq);

   if(deriv==1)
      amp*=(double)SampleRate/(2*M_PI*(double)ref_freq);

   if(deriv==2)
      amp*= (double)SampleRate/(2*M_PI*(double)ref_freq)
	   *(double)SampleRate/(2*M_PI*(double)ref_freq);

   if(amp!=0)
   {
      phase=atan2(im,re)*2*M_PI;
      db=20*log10(amp);
   }
   else
   {
      phase=0;
      db=-100;
   }

   draw_bar(0,21,160,71,0);
   sprintf(ach," Freq: %7.5g Hz ",(double)bin*SampleRate/fftlen);
   draw_text_left(0,22,ach);
   sprintf(ach,"  Amp: %7.5g   ",amp);
   draw_text_left(0,34,ach);
   sprintf(ach,"       %7.5g db ",db);
   draw_text_left(0,46,ach);
   sprintf(ach,"Phase: %7.4g deg. ",phase);
   draw_text_left(0,58,ach);
}


void highlight(int on)
{
   /*  Turn highlighting of a specified bin on the display on (1) or off (0) */

   int pos;
   double freq=(double)bin*(double)SampleRate/fftlen;
   if(logfreq)
      pos=floor(log(freq/freq_base)/log(fftlen/2)*freq_scalefactor*(double)(WINDOW_RIGHT-WINDOW_LEFT)+0.5);
   else
      pos=floor((freq-freq_base)*(double)(WINDOW_RIGHT-WINDOW_LEFT)/(double)SampleRate*freq_scalefactor*2.0+0.5);

   if(on)
      draw_line(WINDOW_LEFT+pos,WINDOW_TOP,WINDOW_LEFT+pos,lasty[pos],DARK_HIGHLIGHT);
   else
      draw_line(WINDOW_LEFT+pos,WINDOW_TOP,WINDOW_LEFT+pos,lasty[pos],0);
   if(on)
      draw_line(WINDOW_LEFT+pos,lasty[pos],WINDOW_LEFT+pos,WINDOW_BOTTOM,LIGHT_HIGHLIGHT);
   else
      draw_line(WINDOW_LEFT+pos,lasty[pos],WINDOW_LEFT+pos,WINDOW_BOTTOM,GRAPH_COLOR);
}

int process_input(int c)
{
   /*  Process any keyboard input received by the program */
   char ach[50];

   if(c==0) return(0);

   if(freeze) highlight(0);

   if(c==' ')  /* Toggle freeze mode on/off */
   {
      freeze=!freeze;
      if(freeze)
      {
	 if(help_mode)
	 {
	    help_mode=0;
	    update_header();
	 }
	 if((double)bin<freq_base*fftlen/SampleRate)
	    bin=ceil(freq_base*fftlen/SampleRate);
	 if((double)bin>maxfreq*fftlen/SampleRate)
	    bin=floor(maxfreq*fftlen/SampleRate);

	 draw_text_centered(MGX,MGY,"Use H,J,K,L to move, Enter to save to disk, Space to continue.");

	 update_display();
      }
      else
      {
	 draw_bar(0,21,160,71,0);
	 draw_bar(WINDOW_LEFT-5,MGY-1,639,MGY+_font_height,0);
      }
   }
   if(freeze)
   {
      /* Keys only valid in freeze-display */
      if((c==0x0d)||(c==0x0a))  /* <CR> saves the spectrum data to a file */
      {
	 char filename[30];
	 draw_bar(WINDOW_LEFT-5,MGY-1,639,MGY+_font_height,0);
	 draw_text_left(MGX-156,MGY,"File in which to save this spectrum:");

	 filename[0]=0;
	 input_text(filename,sizeof(filename),MGX+140,MGY);
	 if(filename[0]!=0)
	 {
	    clock_t clk;
	    FILE *fp=fopen(filename,"w");
	    if(fp)
	    {
	       int i;
	       for(i=0;i<fftlen/2;i++)
		  fprintf(fp,"%f\t%d\t%d\n",(double)i/(double)fftlen*(double)SampleRate,fftdata[BitReversed[i]],fftdata[BitReversed[i]+1]);
	       fclose(fp);
	       sprintf(ach,"Spectrum data saved in %s",filename);
	    }
	    else
	       sprintf(ach,"Unable to open (%s)",filename);

	    draw_bar(WINDOW_LEFT-5,MGY-1,639,MGY+_font_height,0);
	    draw_text_centered(MGX,MGY,ach);

	    clk=clock();
	    while(!draw_getkey() && ((clock()-clk)<CLOCKS_PER_SEC*3));
	 }
	 draw_bar(WINDOW_LEFT-5,MGY-1,639,MGY+_font_height,0);
	 draw_text_centered(MGX,MGY,"Use H,J,K,L to move, Enter to save to disk, Space to continue.");
      }
      if(c=='J' || c=='j')  /* Move the cursor one bin left */
      {
	 double current;
	 if(bin>logfreq)
	    bin--;
	 current=(double)bin*(double)SampleRate/(double)fftlen;
	 if(current<freq_base)
	 {
	    freq_base=current;
	    /* Re-initialize the display */
	    setup_xscale();
	 }
	 update_display();
      }
      else if(c=='K' || c=='k') /* Move the cursor one bin right */
      {
	 double current;
	 if(bin<fftlen/2-1)
	    bin++;
	 current=(double)bin*(double)SampleRate/(double)fftlen;
	 if(current>maxfreq)
	 {
	    if(logfreq)
	       freq_base=current/exp(log(fftlen/2)/freq_scalefactor);
	    else
	       freq_base=current-(double)SampleRate/(freq_scalefactor*2.0);
	    /* Re-initialize the display */
	    setup_xscale();
	 }
	 update_display();
      }
      else if(c=='H' || c=='h') /* Move the cursor 10 bins left */
      {
	 double current;
	 bin-=10;
	 if(bin<logfreq)
	    bin=logfreq;
	 current=(double)bin*(double)SampleRate/(double)fftlen;
	 if(current<freq_base)
	 {
	    freq_base=current;
	    /* Re-initialize the display */
	    setup_xscale();
	 }
	 update_display();
      }
      else if(c=='L' || c=='l') /* Move the cursor 10 bins right */
      {
	 double current;
	 bin+=10;
	 if(bin>=fftlen/2)
	    bin=fftlen/2-1;
	 current=(double)bin*(double)SampleRate/(double)fftlen;
	 if(current>maxfreq)
	 {
	    if(logfreq)
	       freq_base=current/exp(log(fftlen/2)/freq_scalefactor);
	    else
	       freq_base=current-(double)SampleRate/(freq_scalefactor*2.0);
	    /* Re-initialize the display */
	    setup_xscale();
	 }
	 update_display();
      }
      else if(c==HOME) /* Move the cursor to the lowest frequency */
      {
	 bin=logfreq;
	 update_display();
	 if(freq_base>((double)SampleRate/(double)fftlen*(double)logfreq))
	 {
	    freq_base=0;
	    /* Re-initialize the display */
	    setup_xscale();
	 }
      }
      else if(c==END) /* Move the cursor to the highest frequency */
      {
	 bin=fftlen/2-1;
	 update_display();
	 if(maxfreq<(double)SampleRate/2)
	 {
	    freq_base=SampleRate/2;
	    /* Re-initialize the display */
	    setup_xscale();
	 }
      }
   }
   else
   {
      /* Keys only valid in non-freeze display */
      if(c==HOME) /* Move the cursor to the lowest frequency */
      {
	 if(freq_base>((double)SampleRate/(double)fftlen*(double)logfreq))
	 {
	    freq_base=0;
	    /* Re-initialize the display */
	    setup_xscale();
	 }
      }
      else if(c==END) /* Move the cursor to the highest frequency */
      {
	 if(maxfreq<(double)SampleRate/2)
	 {
	    freq_base=SampleRate/2;
	    /* Re-initialize the display */
	    setup_xscale();
	 }
      }
      else if(c=='H' || c=='h' || c=='?' || c=='/') /* Toggle help mode */
      {
	 help_mode++;
	 if(help_mode>2) help_mode=0;

	 if(help_mode)
	    show_help();
	 else
	    update_header();
      }
      else if(c=='F' || c== 'f') /* Change the FFT length */
      {
	 int i;
	 halt_soundcard();
	 EndFFT();
	 if(c=='f')
	 {
	    fftlen*=2;
	    if(fftlen>MAX_LEN) fftlen=8;
	 }
	 else
	 {
	    fftlen/=2;
	    if(fftlen<8) fftlen=MAX_LEN;
	 }
	 InitializeFFT(fftlen);
	 compute_window_function();
	 /* Flush the buffers */
	 for(i=0;i<fftlen/2;i++)
	    lasta2[i]=0;
	 for(i=0;i<fftlen;i++)
	    fftdata[i]=0;

	 /* Re-initialize the display */
	 setup_xscale();
	 if(help_mode)
	    show_help();
	 else
	    update_header();
	 reset_soundcard();
      }
      else if(c=='R' || c=='r') /* Change the sampling rate */
      {
	 char in[6];
	 draw_bar(WINDOW_LEFT-5,MGY-1,639,MGY+_font_height,0);
	 draw_text_left(MGX-80,MGY,"Sampling rate: ");

	 in[0]=0;
	 input_text(in,sizeof(in),MGX+40,MGY);
	 if(in[0]!=0)
	 {
	    halt_soundcard();

	    SampleRate=atol(in);
	    if(SampleRate<5000L) SampleRate=5000L;
	    if(SampleRate>88200L) SampleRate=88200L;

	    reset_soundcard();

	    /* Re-initialize the display */
	    setup_xscale();
	 }
	 if(help_mode)
	    show_help();
	 else
	    update_header();
      }
   }

   /* Keys valid in both freeze and non-freeze mode */

   if(c=='=') /* Increase the vertical scale factor */
   {
      if(logamp)
      {
	 if(log_base<10)
	 {
	    logs++;
	    log_base++;
	    setup_logscales();
	 }
      }
      else
      {
	 if(ys>0.15)
	 {
	    ys-=0.1;
	    setup_linscales();
	 }
	 else if(ys>0.01)
	 {
	    ys-=0.01;
	    if(ys<0.01) ys=0.01;
	    setup_linscales();
	 }
      }
      amplitude_scale();
   }
   else if(c=='-') /* Decrease the vertical scale factor */
   {
      if(logamp)
      {
	 if(logs>0)
	 {
	    logs--;
	    log_base--;
	    setup_logscales();
	 }
      }
      else
      {
	 if(ys<0.095)
	 {
	    ys+=0.01;
	    setup_linscales();
	 }
	 else if(ys<2.00)
	 {
	    ys+=0.1;
	    if(ys>2.0) ys=2.0;
	    setup_linscales();
	 }
      }

      amplitude_scale();
   }
   else if(c=='+') /* Shift the vertical scales up */
   {
      if(logamp)
      {
	 if(log_base<10)
	 {
	    log_base++;
	    setup_logscales();
	 }
	 else if(logs>0)
	 {
	    logs--;
	    setup_logscales();
	 }
      }
      else
      {
	 if(ys>0.15)
	 {
	    ys-=0.1;
	    setup_linscales();
	 }
	 else if(ys>0.01)
	 {
	    ys-=0.01;
	    if(ys<0.01) ys=0.01;
	    setup_linscales();
	 }
      }
      amplitude_scale();
   }
   else if(c=='_') /* Shift the vertical scales down */
   {
      if(logamp)
      {
	 if(log_base>logs+3)
	 {
	    log_base--;
	    setup_logscales();
	 }
      }
      else
      {
	 if(ys<0.095)
	 {
	    ys+=0.01;
	    setup_linscales();
	 }
	 else if(ys<2.00)
	 {
	    ys+=0.1;
	    if(ys>2.0) ys=2.0;
	    setup_linscales();
	 }
      }
      amplitude_scale();
   }
   else if(c=='<') /* Decrease the horizontal scale factor */
   {
      if(freq_scalefactor>1.0)
      {
	 freq_scalefactor*=0.84089641525371;  /* 1/(2^0.25) */
	 if(logfreq)
	    freq_base*=exp(log(fftlen/2)*0.07955179237314/freq_scalefactor);
	 else
	    freq_base-=(double)SampleRate*0.03977589618657/freq_scalefactor;
	 /* Re-initialize the display */
	 if(freeze)
	 {
	    double current=(double)bin*(double)SampleRate/(double)fftlen;
	    xrange_check();
	    if(current<freq_base)
	    {
	       freq_base=current;
	    }
	    if(current>maxfreq)
	    {
	       if(logfreq)
		  freq_base=current/exp(log(fftlen/2)/freq_scalefactor);
	       else
		  freq_base=current-(double)SampleRate/2.0/freq_scalefactor;
	    }
	 }
	 setup_xscale();
      }
   }
   else if(c=='>') /* Increase the horizontal scale factor */
   {
      if(freq_scalefactor<16.0)
      {
	 if(logfreq)
	    freq_base*=exp(log(fftlen/2)*0.07955179237314/freq_scalefactor);
	 else
	    freq_base+=(double)SampleRate*0.03977589618657/freq_scalefactor;
	 freq_scalefactor*=1.18920711500272;  /* 2^0.25 */

	 /* Re-initialize the display */
	 if(freeze)
	 {
	    double current=(double)bin*(double)SampleRate/(double)fftlen;
	    xrange_check();
	    if(current<freq_base)
	    {
	       freq_base=current;
	    }
	    if(current>maxfreq)
	    {
	       if(logfreq)
		  freq_base=current/exp(log(fftlen/2)/freq_scalefactor);
	       else
		  freq_base=current-(double)SampleRate/2.0/freq_scalefactor;
	    }
	 }
	 setup_xscale();
      }
   }
   else if(c=='.') /* Shift the horizontal display to the right */
   {
      if(maxfreq<(double)SampleRate/2)
      {
	 if(logfreq)
	    freq_base*=exp(log(fftlen/2)/(8.0*freq_scalefactor));
	 else
	    freq_base+=(double)SampleRate/(16.0*freq_scalefactor);
	 /* Re-initialize the display */
	 if(freeze)
	 {
	    double current=(double)bin*(double)SampleRate/(double)fftlen;
	    xrange_check();
	    if(current<freq_base)
	    {
	       bin=ceil(freq_base/(double)SampleRate*(double)fftlen);
	       update_display();
	    }
	 }
	 setup_xscale();
      }
   }
   else if(c==',') /* Shift the horizontal display to the left */
   {
      if(freq_base>((double)logfreq*(double)SampleRate/(double)fftlen))
      {
	 if(logfreq)
	    freq_base/=exp(log(fftlen/2)/(8.0*freq_scalefactor));
	 else
	    freq_base-=(double)SampleRate/(16.0*freq_scalefactor);
	 /* Re-initialize the display */
	 if(freeze)
	 {
	    double current=(double)bin*(double)SampleRate/(double)fftlen;
	    xrange_check();
	    if(current>maxfreq)
	    {
	       bin=floor(maxfreq/(double)SampleRate*(double)fftlen);
	       if(bin>=fftlen/2) bin=fftlen/2-1;
	       update_display();
	    }
	 }
	 setup_xscale();
      }
   }
   else if(c=='P' || c=='p') /* Toggle peak display mode on/off */
   {
      display_peak=!display_peak;
      if(display_peak)
      {
	 draw_text_left(PKX-64,PKY,"Peak at         Hz");
      }
      else
      {
	 draw_bar(PKX-64,PKY-1,PKX+79,PKY+_font_height,0);
      }
   }
   else if(c=='V' || c=='v') /* Redraw the video display */
   {
      int i;
      draw_bar(0,0,639,479,0);

      /* Refresh the video display */

      draw_rectangle(WINDOW_LEFT-2,WINDOW_TOP-2,WINDOW_RIGHT+2,WINDOW_BOTTOM+2,BORDER_COLOR);
      setup_xscale();
      amplitude_scale();
      if(help_mode)
	 show_help();
      else
	 update_header();
      if(display_peak)
      {
	 draw_text_left(PKX-64,PKY,"Peak at         Hz");
      }
      if(freeze)
      {
	 draw_text_centered(MGX,MGY,"Use H,J,K,L to move, Enter to save to disk, Space to continue.");
	 update_display();
      }
      /* Reset the last-value contents */
      for(i=0;i<WINDOW_RIGHT-WINDOW_LEFT;i++)
	 lasty[i]=WINDOW_BOTTOM;
   }
   else if(c=='S' || c=='s') /* Save the current state to an INI file */
   {
      draw_bar(WINDOW_LEFT,MGY-1,639,MGY+_font_height,0);
      draw_text_left(MGX-130,MGY,"Save the current state to: ");
      strncpy(ach,ini_file,20);
      input_text(ach,20,MGX+86,MGY);
      if(ach[0]!=0)
      {
	 FILE *fp;
	 strcpy(ini_file,ach);
	 fp=fopen(ini_file,"w");
	 if(fp)
	 {
	    fprintf(fp,"Soundcard: %d\n",Soundcard);
            #ifdef LINUX
	       fprintf(fp,"Audio device: %s\n",audio_device);
            #endif /* LINUX */
            #ifdef LINUX_MIXER
	       fprintf(fp,"Mixer device: %s\n",mixer_device);
            #endif /* LINUX_MIXER */
	    fprintf(fp,"Sample rate: %ld\n",SampleRate);
	    fprintf(fp,"FFT length: %d\n",fftlen);
	    fprintf(fp,"Window function: %d\n",windfunc);
	    fprintf(fp,"Log freq scale: %d\n",logfreq);
	    fprintf(fp,"Log amp scale: %d\n",logamp);
	    fprintf(fp,"Base db: %d\n",log_base);
	    fprintf(fp,"Top db: %d\n",logs);
	    fprintf(fp,"Max amp: %g\n",ys);
	    fprintf(fp,"DB/octave gain: %d\n",gain);
	    fprintf(fp,"Reference frequency: %ld\n",ref_freq);
	    fprintf(fp,"Base frequency: %g\n",freq_base);
	    fprintf(fp,"Frequency factor: %g\n",freq_scalefactor);
	    fprintf(fp,"Decay mode: %d\n",decay_mode);
	    fprintf(fp,"Decay factor: %g\n\n",decay_factor);
	    fprintf(fp,"Background color: %d,%d,%d\n",background.red,background.green,background.blue);
	    fprintf(fp,"Clipping warning color: %d,%d,%d\n",warn.red,warn.green,warn.blue);
	    fprintf(fp,"Graph color: %d,%d,%d\n",graph.red,graph.green,graph.blue);
	    fprintf(fp,"Tick mark color: %d,%d,%d\n",tick.red,tick.green,tick.blue);
	    fprintf(fp,"Axis label color: %d,%d,%d\n",label.red,label.green,label.blue);
	    fprintf(fp,"Border color: %d,%d,%d\n",border.red,border.green,border.blue);
	    fprintf(fp,"Text color: %d,%d,%d\n",text.red,text.green,text.blue);
	    fprintf(fp,"Cursor upper color: %d,%d,%d\n",darkhl.red,darkhl.green,darkhl.blue);
	    fprintf(fp,"Cursor lower color: %d,%d,%d\n",lighthl.red,lighthl.green,lighthl.blue);
	    fclose(fp);
	 }
	 else
	 {
	    clock_t clk=clock();
	    sprintf(ach,"Unable to open %s",ini_file);
	    draw_bar(WINDOW_LEFT-5,MGY-1,639,MGY+_font_height,0);
	    draw_text_centered(MGX,MGY,ach);
	    while(!draw_getkey() && ((clock()-clk)<3*CLOCKS_PER_SEC));
	 }
      }
      if(help_mode)
	 show_help();
      else
	 update_header();
      if(freeze)
	 draw_text_centered(MGX,MGY,"Use arrows to move, Space to continue.");
   }
   else if((c=='+') || (c=='=')) /* Increase decay rate */
   {
      decay_factor=decay_factor*decay_factor;
   }
   else if((c=='-') || (c=='_')) /* Decrease decay rate */
   {
      decay_factor=sqrt(decay_factor);
   }
   else if(c=='D' || c=='d') /* Toggle decay mode on/off */
   {
      decay_mode=!decay_mode;
      if(decay_mode)
      {
	 int i;
	 for(i=0;i<fftlen/2;i++)
	    lasta2[i]=0;
      }
   }
   else if(c=='B' || c=='b') /* Toggle Black/White palette on/off */
   {
      bw_mode=!bw_mode;
      if(bw_mode)
	 setbwpalette();
      else
	 setnormalpalette();
   }
   else if(c=='X' || c=='x') /* Toggle linear/logarithmic frequency axis */
   {
      /*
       *  Toggle between linear and logarithmic frequency scale
       */
      logfreq=!logfreq;
      if(freeze)
      {
	 double current;
	 if(logfreq && (bin==0)) bin=1;
	 current=(double)bin*(double)SampleRate/(double)fftlen;
	 xrange_check();
	 if(current<freq_base)
	 {
	    freq_base=current;
	 }
	 if(current>maxfreq)
	 {
	    if(logfreq)
	       freq_base=current/exp(log(fftlen/2)/freq_scalefactor);
	    else
	       freq_base=current-(double)SampleRate/2.0/freq_scalefactor;
	 }
      }
      setup_xscale();
   }
   else if(c=='Y' || c=='y') /* Toggle linear/logarithmic amplitude axis */
   {
      /*
       *  Toggle between linear and logarithmic amplitude scale
       */
      logamp=!logamp;
      
      if(logamp)
	 setup_logscales();
      else
	 setup_linscales();

      amplitude_scale();
   }
   else if(c=='W' || c=='w') /* Advance to the next windowing function */
   {
      /*
       *  Change the windowing function
       */
      if(c=='w')
      {
	 windfunc++;
	 if(windfunc>6) windfunc=0;
      }
      else
      {
	 windfunc--;
	 if(windfunc<0) windfunc=6;
      }
      compute_window_function();

      if(!help_mode)
      {
	 sprintf(ach,"%s window function",window_name[windfunc]);
	 draw_bar(WFX-150,WFY-1,WFX+150,WFY+_font_height,0);
	 draw_text_centered(WFX,WFY,ach);
      }
   }
   else if(c=='0') /* Set 0db/octave gain */
   {
      /*
       *  0 dB/octave gain
       */
      gain=0;
      gain3db=0;
      deriv=0;

      if(freeze)
	 update_display();

      if(logamp)
	 setup_logscales();
      else
	 setup_linscales();

      if(!help_mode)
      {
	 draw_bar(DGX-150,DGY-1,DGX+150,DGY+_font_height,0);
	 draw_text_centered(DGX,DGY,"Gain of 0 dB per octave");
      }
   }
   else if(c=='3') /* Set 3dB/octave gain */
   {
      /*
       *  3 dB/octave gain
       */
      gain=3;
      gain3db=1;
      deriv=0;

      if(freeze)
	 update_display();

      if(logamp)
	 setup_logscales();
      else
	 setup_linscales();

      if(!help_mode)
      {
	 draw_bar(DGX-150,DGY-1,DGX+150,DGY+_font_height,0);
	 draw_text_centered(DGX,DGY,"Gain of 3 dB per octave");
      }
   }
   else if(c=='6') /* Set 6dB/octave gain */
   {
      /*
       *  6 dB/octave gain
       */
      gain=6;
      gain3db=0;
      deriv=1;

      if(freeze)
	 update_display();

      if(logamp)
	 setup_logscales();
      else
	 setup_linscales();

      if(!help_mode)
      {
	 draw_bar(DGX-150,DGY-1,DGX+150,DGY+_font_height,0);
	 draw_text_centered(DGX,DGY,"Gain of 6 dB per octave");
      }
   }
   else if(c=='9') /* Set 9dB/octave gain */
   {
      /*
       *  9 dB/octave gain
       */
      gain=9;
      gain3db=1;
      deriv=1;

      if(freeze)
	 update_display();

      if(logamp)
	 setup_logscales();
      else
	 setup_linscales();

      if(!help_mode)
      {
	 draw_bar(DGX-150,DGY-1,DGX+150,DGY+_font_height,0);
	 draw_text_centered(DGX,DGY,"Gain of 9 dB per octave");
      }
   }
   else if(c=='1' || c=='2') /* Set 12dB/octave gain */
   {
      /*
       *  12 dB/octave gain
       */
      gain=12;
      gain3db=0;
      deriv=2;

      if(freeze)
	 update_display();

      if(logamp)
	 setup_logscales();
      else
	 setup_linscales();

      if(!help_mode)
      {
	 draw_bar(DGX-150,DGY-1,DGX+150,DGY+_font_height,0);
	 draw_text_centered(DGX,DGY,"Gain of 12 dB per octave");
      }
   }
   else if(mixers)
   {
      if(c=='[') /* External jack down */
      {
	 ext_level-=2;
	 if(ext_level<0) ext_level=0;
	 if(!help_mode)
	 {
	    sprintf(ach,"%3d",ext_level);
	    draw_bar(LVX+104,LVY-1,LVX+127,LVY+_font_height,0);
	    draw_text_left(LVX+104,LVY,ach);
	 }
	 set_mixer(MIXER_EXT,ext_level);
      }
      else if(c==']') /* External jack up */
      {
	 ext_level+=2;
	 if(ext_level>100) ext_level=100;
	 if(!help_mode)
	 {
	    sprintf(ach,"%3d",ext_level);
	    draw_bar(LVX+104,LVY-1,LVX+127,LVY+_font_height,0);
	    draw_text_left(LVX+104,LVY,ach);
	 }
	 set_mixer(MIXER_EXT,ext_level);
      }
      else if(c=='{') /* CD input down */
      {
	 int_level-=2;
	 if(int_level<0) int_level=0;
	 if(!help_mode)
	 {
	    sprintf(ach,"%3d",int_level);
	    draw_bar(LVX+168,LVY-1,LVX+191,LVY+_font_height,0);
	    draw_text_left(LVX+168,LVY,ach);
	 }
	 set_mixer(MIXER_INT,int_level);
      }
      else if(c=='}') /* CD input up */
      {
	 int_level+=2;
	 if(int_level>100) int_level=100;
	 if(!help_mode)
	 {
	    sprintf(ach,"%3d",int_level);
	    draw_bar(LVX+168,LVY-1,LVX+191,LVY+_font_height,0);
	    draw_text_left(LVX+168,LVY,ach);
	 }
	 set_mixer(MIXER_INT,int_level);
      }
      else if(c=='(') /* Mic input down */
      {
	 mic_level-=2;
	 if(mic_level<0) mic_level=0;
	 if(!help_mode)
	 {
	    sprintf(ach,"%3d",mic_level);
	    draw_bar(LVX+32,LVY-1,LVX+55,LVY+_font_height,0);
	    draw_text_left(LVX+32,LVY,ach);
	 }
	 set_mixer(MIXER_MIC,mic_level);
      }
      else if(c==')') /* Mic input up */
      {
	 mic_level+=2;
	 if(mic_level>100) mic_level=100;
	 if(!help_mode)
	 {
	    sprintf(ach,"%3d",mic_level);
	    draw_bar(LVX+32,LVY-1,LVX+55,LVY+_font_height,0);
	    draw_text_left(LVX+32,LVY,ach);
	 }
	 set_mixer(MIXER_MIC,mic_level);
      }
   }

   if(freeze) highlight(1);

   return(c=='Q' || c=='q' || c==0x03);
}
