/*
 *     Program: FREQ.C
 *      Author: Philip VanBaren
 *        Date: 15 August 1993
 *
 * Description: This program samples data from a sound card, performs an FFT,
 *              and displays the result.
 *              Can handle up to 2048 points (actually any size is possible
 *              with a little fiddling of buffers to get around 64k limits).
 *              (This restriction is given freq.h, and may be changed.)
 *              On a 486/33 this code can perform and plot 1024-point and
 *              below in nearly real-time at 44100kHz.  (1024 FFT=31ms)
 *
 *  The DOS portion of this code was written for Borland C, but should work 
 *  with other compilers if the graphics and console-io base functions are
 *  changed appropriately.  The UNIX specific stuff has been compiled using
 *  GCC on Linux and on Sun Workstations.
 *
 *  The source for specific graphics environments and sound cards may require
 *  other packages.  Refer to the specific files for more details.
 *
 *  Most changes are required only in the sc_*.c and gr_*.c files.
 *
 *  Copyright (C) 1995  Philip VanBaren
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "freq.h"
#include "fft.h"
#include "extern.h"
#include "display.h"

/*
 *  Table for approximating the logarithm.
 */
int ln[]={ -10000, -4096, -3072,-2473,-2048,-1718,-1449,-1221,-1024,-850,-694,-554,-425,-307,-197,-95,
	   0,90,174,254,330,402,470,536,599,659,717,773,827,879,929,977 };

int flag[BUFFERS];      /* Array of flags indicating fullness of buffers */
int queue_buffer;       /* Pointer to next buffer to be queued */
int record_buffer;      /* Pointer to next buffer to be filled */
int process_buffer;     /* Pointer to next buffer to be FFTed */

short *fftdata;           /* Array for FFT data */
short *wind;              /* Array storing windowing function */

int x[WINDOW_RIGHT-WINDOW_LEFT+1];     /* Array of bin #'s displayed */
int lasty[WINDOW_RIGHT-WINDOW_LEFT+1]; /* Last y position for FFT bins */
unsigned int yscale[WINDOW_RIGHT-WINDOW_LEFT+1]; /* Scaling factors */
int ybase[WINDOW_RIGHT-WINDOW_LEFT+1];           /* Scaling offset for log calculations */

long *lasta2;
int shift=0;            /* Number of bits for gain shift */

void FAR *buffer[BUFFERS]; /* Buffers for gathering data */

short *p1,*p2;        /* Various indexing pointers */
int *p3,*p4,*p5;
unsigned int *p6;

unsigned char FAR *sample8;
short FAR *sample16;

long a2,root,mask;      /* Variables for computing Sqrt/Log of Amplitude^2 */
long peak_amp;          /* Peak amplitude found */
int peak_index;         /* Bin number of the peak amplitude */
float decay_factor=0.95; /* Geometric decay factor */
long back1,back2;       /* Variables for differencing */
char ini_file[100];     /* Filename for the ini file */

int done=0;             /* Flag indicating program should exit*/

int main(int argc,char *argv[],char *environ[])
{
   int i,y;
   int first;
   int key=0;

   draw_init();

   DOUT("Getting the command line arguments");

   /*
    *  Check if the first parameter is an ini file name
    */
   if(argc>1 && argv[1][0]!='-' && argv[1][0]!='/')
   {
      strncpy(ini_file,argv[1],sizeof(ini_file));
      first=2;
   }
   else
   {
      strncpy(ini_file,"freq.ini",sizeof(ini_file));
      first=1;
   }

   /*
    *  Parse the ini file and command line
    */
   DOUT("Parsing the ini file");
   parse_ini_file();

   DOUT("Parsing the command line");
   parse_command((argc-first),&argv[first],environ);

   /*
    *  Initialize the buffer info for the maximum size we will encounter
    */
   DOUT("Allocating the buffer space");
   setup_buffers(MAX_LEN);

   DOUT("Computing the window functions");
   compute_window_function();

   /*
    *  Set up the required arrays in the FFT code.
    */
   DOUT("Initializing the FFT code");
   InitializeFFT(fftlen);

   /*
    *  Initialize the graphics to 640x480 VGA mode
    */
   DOUT("Setting the graphics mode");
   setup_graphics();

   setnormalpalette();
   draw_fontcolor(TEXT_COLOR);
   draw_rectangle(WINDOW_LEFT-2,WINDOW_TOP-2,
		  WINDOW_RIGHT+2,WINDOW_BOTTOM+2,BORDER_COLOR);

   DOUT("Resetting the sound card");
   reset_soundcard();

   /*
    *  Initalize the graph scales
    */
   setup_xscale();
   amplitude_scale();

   DOUT("Drawing the header information");
   update_header();

   /*
    *  Keep getting data and plotting it.
    *  A space will pause, a second space will continue.
    *  Any other key will quit.
    */
   DOUT("Entering data loop");
   while(!done)
   {
      /*
       *  Wait for current buffer to fill up
       */
      key=draw_getkey();
      while((!flag[process_buffer] || freeze) && !key)
	 key=draw_getkey();

      if(key)
	 done |= process_input(key);

      if(!key && !freeze)
      {
	 int clip=0;
	 /*
	  *  Perform windowing on the data
	  */
	 p1=fftdata;
	 p2=wind;

	 if(deriv==0)
	 {
	    if(sample_size==8)
	    {
	       sample8=(unsigned char FAR *)buffer[process_buffer];
	       for(i=0;i<fftlen;i++)
	       {
		  if((*sample8==0) || (*sample8==255))
		     clip=1;
		  *p1=( ((long)(*sample8)-128L)  * (long)(*p2)) >> 7;
		  sample8++;
		  p1++;
		  p2++;
	       }
	    }
	    else
	    {
	       sample16=(short FAR *)buffer[process_buffer];

	       for(i=0;i<fftlen;i++)
	       {
		  if((*sample16==32767) || (*sample16==-32768))
		     clip=1;
		  *p1=((long)(*sample16) * (long)(*p2)) >> 15;
		  sample16++;
		  p1++;
		  p2++;
	       }
	    }
	 }
	 else if(deriv==1)
	 {
	    if(sample_size==8)
	    {
	       sample8=(unsigned char FAR *)buffer[process_buffer];
	       back1=*sample8;
	       for(i=0;i<fftlen;i++)
	       {
		  if((*sample8==0) || (*sample8==255))
		     clip=1;
		  *p1=(((long)(*sample8)-back1) * (long)(*p2)) >> 7;
		  back1=*sample8;
		  sample8++;
		  p1++;
		  p2++;
	       }
	    }
	    else
	    {
	       sample16=(short FAR *)buffer[process_buffer];
	       back1=*sample16;
	       for(i=0;i<fftlen;i++)
	       {
		  if((*sample16==32767) || (*sample16==-32768))
		     clip=1;
		  *p1=(((long)(*sample16)-back1) * (long)(*p2)) >> 15;
		  back1=*sample16;
		  sample16++;
		  p1++;
		  p2++;
	       }
	    }
	 }
	 else   /* deriv==2 */
	 {
	    if(sample_size==8)
	    {
	       sample8=(unsigned char FAR *)buffer[process_buffer];
	       back1=back2=*sample8;

	       for(i=0;i<fftlen;i++)
	       {
		  if((*sample8==0) || (*sample8==255))
		     clip=1;
		  *p1=(((long)(*sample8)-2*back1+back2) * (long)(*p2)) >> 7;
		  back2=back1;
		  back1=*sample8;
		  sample8++;
		  p1++;
		  p2++;
	       }
	    }
	    else
	    {
	       sample16=(short FAR *)buffer[process_buffer];
	       back1=back2=*sample16;
	       if((*sample16==32767) || (*sample16==-32768))
		  clip=1;
	       *p1=(((long)(*sample16)-2*back1+back2) * (long)(*p2)) >> 15;
	       back2=back1;
	       back1=*sample16;
	       sample16++;
	       p1++;
	       p2++;
	    }
	 }

	 if(clip)
	    draw_setpalette(0,warn.red,warn.green,warn.blue);
	 else
	    draw_setpalette(0,background.red,background.green,background.blue);

	 /*
	  *  Free up the buffer we just processed.
	  */
	 flag[process_buffer]=0;
	 if(++process_buffer>=BUFFERS)
	    process_buffer=0;

	 /* Now that we have processed the buffer, queue it up again. */
	 recordblock(buffer[queue_buffer]);
	 if(++queue_buffer>=BUFFERS)
	    queue_buffer=0;

	 /*
	  *  The real meat of the code lies elsewhere!
	  */
	 RealFFT(fftdata);
      }
      if(freeze)   /* Unhighlight the bin in freeze mode */
	 highlight(0);

      /*
       *  Use pointers for indexing to speed things up a bit.
       */
      p3=lasty;
      p4=x;
      p5=ybase;
      p6=yscale;
      peak_amp=0;
      peak_index=0;
      y=0;
      
      #ifdef DOS
	 setup_vga();   /* Prepare VGA for video update */
      #endif
      
      for(i=WINDOW_LEFT;i<WINDOW_RIGHT+1;i++)
      {
	 /*
	  *  If this line is the same as the previous one,
	  *  just use the previous y value.
	  *  Else go ahead and compute the value.
	  */
	 if(*p4!=-1)
	 {
	   {
	    register int bri=BitReversed[*p4];
	    register long re=fftdata[bri];
	    register long im=fftdata[bri+1];
	    /* Compute the squared amplitude */
	    a2=re*re+im*im;
	    if(decay_mode)
	    {
	       register long l=lasta2[*p4];
	       if(!freeze)
		  l=floor(l*decay_factor);
	       if(a2<l) a2=l;
	       lasta2[*p4]=a2;
	    }
	    if(a2>peak_amp)
	    {
	       peak_amp=a2;
	       peak_index=*p4;
	    }
	   }

	    if(logamp)
	    {
	       /* Logarithm code */
	       root=4096;

	       while(a2>=32)
	       {
		  root+=1024;
		  a2>>=1;
	       }

	       if((root+=ln[a2]-*p5)<0)
		  root=0;
	    }
	    else
	    {
	       /* Square root code */
	       root=32;
	       do
	       {
		  mask=a2/root;
		  root=(root+mask)>>1;
	       } while(abs(root-mask)>1);
	    }

	    y=(WINDOW_BOTTOM) - ((unsigned long)((unsigned long)root
			       * (unsigned long)*p6) >> shift) ;
	    if(y<WINDOW_TOP) y=WINDOW_TOP;
	 }

	 if(y>*p3)
	 {
	    /*
	     *  Draw a black line
	     */
	    #ifdef DOS
	       unsigned char bit=~(0x80 >> (i&0x07));
	       unsigned int endbase=y*80;
	       unsigned int base=*p3*80+(i>>3);
	       while(base<endbase)
	       {
		  screen(base)&=bit;
		  base+=80;
	       }
	    #else
	       draw_line(i,y-1,i,*p3,0);
	    #endif /* DOS */
	 }
	 else
	 {
	    /*
	     *  Draw a blue line.
	     */
	    #ifdef DOS
	       unsigned char bit=0x80 >> (i&0x07);
	       unsigned int endbase=(*p3+1)*80;
	       unsigned base=y*80+(i>>3);

	       while(base<endbase)
	       {
		  screen(base)|=bit;
		  base+=80;
	       }
	    #else 
	       draw_line(i,*p3,i,y,GRAPH_COLOR);
	    #endif /* DOS */
	 }
	 
	 *p3=y;
	 p3++;
	 p4++;
	 p5++;
	 p6++;
      }
      #ifdef DOS
	 cleanup_vga();  /* Reset VGA for normal functions */
      #endif

      if(display_peak)
      {
	 char ach[20];
	 sprintf(ach,"%7.1f",(double)SampleRate*peak_index/fftlen);
	 draw_bar(PKX,PKY-1,PKX+63,PKY+_font_height,0);
	 draw_text_left(PKX,PKY,ach);
      }
      if(freeze)     /* Highlight the bin in freeze mode */
	 highlight(1);
   }
   /*
    *  Shut down the DMA system.
    */
   cleanup_soundcard();

   cleanup_graphics();

   printf("You have been using Freq " VERSION);
#ifdef SC_PAS16
   if(Soundcard==SC_PAS16) printf(" in ProAudio Spectrum 16-bit mode.");
#endif
#ifdef SC_SB8
   if(Soundcard==SC_SB8) printf(" in Soundblaster 8-bit mode.");
#endif
#ifdef SC_VESA
   if(Soundcard==SC_VESA) printf(" in VESA audio driver %d-bit mode.",sample_size);
#endif
#ifdef SC_LINUX
   if(Soundcard==SC_LINUX) printf(" in Linux audio driver %d-bit mode.",sample_size);
#endif
#ifdef SC_MULAW
   if(Soundcard==SC_MULAW) printf(" in Linux mu-law driver 8-bit mode.");
#endif
   puts("\nCopyright (C) " COPYRIGHT " Philip VanBaren");
   return(0);
}


