/*
   OSTERM.C
   Platform dependent terminal and keyboard functions.

   $Id$
 */
/*    Copyright (c) 1994.  The Regents of the University of California.
                    All rights reserved.  */

#include "events.h"

/* Functions defined in this file -- used only in events.c.  */
extern char *os_get_line(void);
extern void os_free_line(void);
extern void *os_key_context(void);
extern int os_key_type(void);

/* Functions defined in this file --declared publicly in events.h.  */
extern int os_background(void);

/* debugging flag to show how OSFatal was called */
extern int os_key_fatal;
int os_key_fatal= 0;

/* Functions defined in this file --declared publicly in events.h.  */
extern void os_puts(const char *text, int nl);
extern void os_puterr(const char *text, int nl);

/* the following platform-dependent routines are required */
extern void *os_malloc(unsigned long);            /* osmem.c */
extern void *os_realloc(void *, unsigned long);   /* osmem.c */
extern void os_free(void *);                      /* osmem.c */
extern int os_ev_register(int (*fd)(void *context));  /* ospoll.c */

/* ------------------------------------------------------------------------ */

/* Need:  stdin, fgets, feof, ferror, clearerr,
          FILE, stdout, stderr, fputs, and fflush  */
#include <stdio.h>

/* The line buffer will be kept at length OS_MAXLINE unless more space
   is required, at which point up to a total of OS_BADLINE blocks each
   of length OS_MAXLINE will be added in search of a newline.
   On the next call to OSGetLine, the buffer will shrink to OS_MAXLINE
   initially if it has become longer.  */
#ifndef OS_MAXLINE
#define OS_MAXLINE 256
#endif
#ifndef OS_BADLINE
#define OS_BADLINE 64
#endif

static char normal_line[OS_MAXLINE];
static char *line= normal_line;

static char *os_fatal(int which);

char *os_get_line(void)
{
  long i, n;
  char *text;
  long len= OS_MAX_LINE;

  if (line!=normal_line) os_free_line();

  text= line;
  for (n=0 ; n<OS_BADLINE ; n++) {
    if (!fgets(text, OS_MAXLINE+n, stdin)) { /* see fgets comment below */
      /* EOF on stdin is very bad news */
      if (feof(stdin)) return os_fatal(1);
      /* error on stdin is bad news, but try to clear it anyway */
      else if (ferror(stdin)) clearerr(stdin);
      os_free_line();
      return 0;
    }
    if (!text[0]) return line;

    /* the return is impossible according to the semantics for the
       ANSI standard fgets -- take drastic action */
    for (i=1 ; text[i] ; i++) if (i==OS_MAXLINE+n-1) return os_fatal(3);

    if (text[i-1]=='\n') {
      text[i-1]= '\0';
    } else if (i==OS_MAXLINE+n-1) {
      /* if newline not found keep reading long line */
      if (line==normal_line) {
	line= os_malloc(len+OS_MAXLINE);
	if (!line) return os_fatal(4);
	for (i=0 ; i<len ; i++) line[i]= *text++;
      } else {
	line= os_realloc(line, len+OS_MAXLINE);
	if (!line) return os_fatal(4);
      }
      text= line+len-1; /* note increment by OS_MAXLINE-1
			   -- therefore fgets OS_MAXLINE+n above */
      len+= OS_MAXLINE;
      continue;
    }

    return line;
  }

  os_free_line();
  return 0;
}

void os_free_line(void)
{
  if (line!=normal_line) {
    char *l= line;
    line= normal_line;
    os_free(l);
  }
}

static char *os_fatal(int which)
{
  ev_remove(ev_find_key());
  os_key_fatal= which;
  return 0;
}

/* ------------------------------------------------------------------------ */

static void put_text(FILE *stream, const char *text, int nl);

void os_puts(const char *text, int nl)
{
  put_text(stdout, text, nl);
}

void os_puterr(const char *text, int nl)
{
  put_text(stderr, text, nl);
}

static void put_text(FILE *stream, const char *text, int nl)
{
  if (text) fputs(text, stream);
  if (nl) {
    if (text) {
      if (*text++) {
	while (*text) text++;
	if (text[-1]=='\n') nl= 0;
      }
    }
    if (nl) fputs("\n", stream);
  }
  fflush(stream);
}

/* ------------------------------------------------------------------------ */

void *os_key_context(void)
{
  return stdin;
}

static int key_type= -1;
static int os_fileno(void *context);

int os_key_type(void)
{
  if (key_type<0) key_type= os_ev_register(&os_fileno);
  return key_type;
}

/* If fileno is not a macro defined in stdio.h, it must be a function.
   Only other possibility is that some other macro has to be defined in
   order for stdio.h to pick up the proper fileno.  */
#ifndef fileno
extern int fileno(FILE *f);
#endif

static int os_fileno(void *context)
{
  return fileno((FILE *)context);
}

/* ------------------------------------------------------------------------ */

/* The select (and probably poll) function detects input on stdin even
   when your program is a background process.  When fgets attempts to
   read stdin from the background, however, SIGTTIN is send to the
   process, stopping it.  There is no easy way to stay informed about
   the transition from foreground to background mode -- you really want
   to treat this as a "focus event" which is somehow detected and
   reported to an appropriate event handler whenever it occurs.

   While it can't detect the occurence of a foreground-background
   transistion when it happens, the following routine can detect
   whether the program is currently in foreground or background.
 */

/* Use getpgrp to determine this process's process group, and
   tcgetpgrp to determine whether this file represents the controlling
   terminal of a different process group (see below for more).  */
extern int getpgrp(int);
extern int tcgetpgrp(int);
#undef NEED_TCGETPGRP
#ifdef NEED_TCGETPGRP
#include <sys/termios.h>
int tcgetpgrp(int fd)
{
  int tgid;
  if (ioctl(fd, TIOCGPGRP, &tgid)<0) return -1;
  else return tgid;
}
#endif

/* Before attempting to read stdin, check to be sure that if this file
   descriptor is the controlling terminal for this process, the process
   is in the foreground.  If this is the controlling terminal and the
   process is in the background, any attempt to read will generate a
   SIGTTIN signal.  Read the intro(2) and termio(4) man pages for a more
   complete discussion.  */
int os_background(void)
{
  /* never inhibit reading stdin if eof or error condition exists
     -- the handler must have a chance to see these */
  if (!feof(stdin) && !ferror(stdin)) {
    int fd= fileno(stdin);
    int tgid= tcgetpgrp(fd);     /* process group for this terminal--
				    could also use TIOCGPGRP ioctl */
    if (tgid>0) {
      /* This might be the tty of a background process, check */
      int pgid= getpgrp(0);      /* the System V version of getpgrp() has
				    no parameters, but this should be OK  */
      if (pgid!=tgid)            /* attempting to read would cause SIGTTIN */
	return 1;
    }
  }
  return 0;
}

/* ------------------------------------------------------------------------ */
