/*
   EVENTS.H
   Declarations for platform-independent event handling routines.

   $Id$
 */
/*    Copyright (c) 1994.  The Regents of the University of California.
                    All rights reserved.  */

/* ------------------------------------------------------------------------ */

/* The generic event handler works as follows:
   1. You register a callback routine (type EVCallback) for each
      I/O channel on which events may arrive.
   2. You call ev_handle_next (or ev_handle_forever or ev_handle_available)
      to get an event, which causes your callback routine to be called.
   3. You remove your callback routine when it is no longer relevant;
      a callback routine may remove itself, or it may be removed as a
      side effect of any other callback routine.

   Your callback routine takes a two arguments -- it's "context" --
   which you specify when you register the routine, and a "clear" flag.
   A context is typically some sort of I/O stream pointer which the
   callback routine needs in order to read the data associated with the
   event.  You are responsible for ensuring that the context pointer
   remains valid for the lifetime of your callback routine.  The clear
   flag will be non-zero only if ev_clear has been called.  It means
   that os_poll has detected input waiting on your channel, when the
   code is trying to clear out any pending input.  Your callback routine
   shoud read it's I/O stream, but discard the input and return
   immediately.  ev_clear will continue to call os_poll until no more
   input is available, so your callback must ensure that eventually no
   more input will be detected by os_poll, or it must remove itself, in
   order to assure that ev_clear doesn't loop forever.
 */
typedef void EVCallback(void *context, int clear);

/* The EVChannel data type is opaque; it is defined in events.c.  */
typedef struct EVChannel EVChannel;

/* ev_handle_next handles the next event, or calls the next registered
     idler routine, if no events are available and the timeout argument
     is non-zero.
     If positive, timeout is the maximum length of time which should
     be allowed for an event to arrive if none is available.  If this
     feature cannot be implemented on some platform, timeout>0 is
     equivalent to timeout==0.
     Negative timeout (-1) causes ev_handle_next to wait forever for an
     event to arrive, if necessary.  If an idler routine is available,
     but no events are available, the idler is called, and ev_handle_next
     returns immediately.

     ev_handle_next returns 1 if some callback routine was called, otherwise
     it returns 0.  The 0 return indicates either that a timeout has
     occurred (which implies that either timeout==0, or no idler routines
     are currently registered), or that no callbacks of any kind are
     currently registered.

   ev_handle_forever calls ev_handle_next(-1) until the result becomes 0.
   ev_handle_available calls ev_handle_next(0) until the result becomes 0.
 */
extern int ev_handle_next(long timeout);
extern void ev_handle_forever(void);
extern void ev_handle_available(void);

/* ------------------------------------------------------------------------ */

/* The high level routines for registering handlers are:
   ev_key_handler - registers the routine to be called when a line
                    of input from the keyboard is ready
     ev_key_handler fails (returning 0) if a keyboard handler has already
     been registered.  Call ev_remove(ev_find_key()) to remove it first.
     The KeyAction callback routine is not an EVCallback; it simply
     accepts a pointer to the 0-terminated line (not including any
     trailing newline) which has arrived from the keyboard.
     A line==0 argument to the KeyAction indicates an error.  If the
     error is unrecoverable or persistent, the keyboard handler may have
     been removed -- KeyAction can check this by calling ev_find_key().
   ev_idler      - registers routines to be run when no input is
                   immediately available, but handle_next_event has
		   been instructed to wait for input
     ev_idler fails (returning 0) is the same Action/context pair
     has already been registered.
   ev_timer      - registers a routine to be run after a certain period
                   of time (in milliseconds) has expired (but no other task
		   will be interrupted -- ev_handle_next must be called).
		   Just before its Action is called, the timer channel is
		   automatically removed.  It will be removed prior to that
		   time if ev_remove or ev_clear is called; in the case of
		   ev_clear, its Action is called with clear==1.
		   If the timeout argument to os_poll cannot be fully
		   implemented on this operating system, timer channels
		   not function properly.

   ev_find_key    - returns the channel pointer for the keyboard handler
   ev_find_idler  - returns the channel pointer corresponding to a given
                    Action and context (0 if none)

   Other high level handler registration and finder routines may be
   defined (e.g.- for X window system events), but these are platform
   dependent, and cannot be defined here.

   ev_remove     - removes a previously registered channel
 */
extern EVChannel *ev_key_handler(void (*KeyAction)(const char *line));
extern EVChannel *ev_find_key(void);
extern EVChannel *ev_idler(EVCallback *Action, void *context);
extern EVChannel *ev_find_idler(EVCallback *Action, void *context);
extern EVChannel *ev_timer(EVCallback *Action, void *context, long timeout);
extern EVChannel *ev_find_timer(EVCallback *Action, void *context);

extern void ev_remove(EVChannel *channel);

/* The line passed to the KeyAction routine is retrieved by the
   function ev_get_line.  The text is \0-terminated, and should not
   be modified.  The ev_get_line routine is also available for public
   use (e.g.- by the KeyAction routine itself).  ev_get_line will not
   block if no keyboard input is immediately available; instead
   non-keyboard events will be detected and handled normally.  A 0
   return value indicates an error; a severe or persistent error
   causes the keyboard handler to be removed (check with ev_find_key()).

   ev_put_back can be called to "put back" the line you just read using
   ev_get_line, so that the next call to ev_get_line will return it again,
   and ev_handle_next will return immediately without waiting for another
   line to be typed.  The KeyAction routine may also call ev_put_back to
   put its argument back.  */
char *ev_get_line(void);
void ev_put_back(void);

/* Any callback routine may set or read a busy flag:
   busy   - While this is non-zero, ev_handle_next will never call this
            routine, and the program will never block waiting for I/O
	    on this channel.
   For example, if an Action routine has partially serviced its input
   and wants to give other event handlers a chance before continuing,
   it should set its own busy bit, then call ev_handle_available(), then
   reset its busy bit.
   Note that the bit will be reset to 0 if ev_clear is called, as it
   should be if an error interrupts the normal program flow.

   The channel argument may be 0 in any of these routines; they then
   return 0 (if they have a return value), or are no-ops.
 */
extern int ev_busy(EVChannel *channel);
extern void ev_set_busy(EVChannel *channel, int busy);

/* ev_clear discards all available input (after some kind of error).
   This resets all busy flags to 0 as a side effect.  */
void ev_clear(void);

/* ------------------------------------------------------------------------ */

/* ev_handler registers a callback Action routine, which will be invoked
   with the given context argument.

   ev_handler is not intended to be called as a high level routine.
   The type argument must be recognized by the OSPoll routine, so a high
   level interface routine for  registering an Action callback for each
   possible event channel type should be provided in the ospoll.c file
   (which is platform-dependent).  It should be possible from the type
   argument to figure out the "meaning" of the context argument.

   ev_handler returns the channel pointer of the newly created event handler.
   If that context and type have already been registered, ev_handler fails,
   returning 0.  (It can also fail if the memory manager fails.)
   Save the returned EVChannel* if your handler needs ev_set_busy or the
   other public interface routines to the EVChannel data structure.

   ev_find finds the channel pointer given the Action, context, and type,
   returning 0 if there are no matching channels.
 */
extern EVChannel *ev_handler(EVCallback *Action, void *context, int type);
extern EVChannel *ev_find(void *context, int type);

/* ------------------------------------------------------------------------ */

/* The routines for terminal output are declared here, since the ev_get_line
   and ev_put_back routines for keyboard input are declared here.
   They have nothing to do with input events, however, and are defined
   in osterm.c, along with the platform dependent part of ev_get_line.

   os_puts is to be used for ordinary output; os_puterr for error
   output (as to stdout and stderr).  If the nl flag is non-zero, the
   routine appends a "\n" if text does not already end with one.
 */
extern void os_puts(const char *text, int nl);
extern void os_puterr(const char *text, int nl);

/* ------------------------------------------------------------------------ */
