#=======================================================================

iroots - integer roots of a polynomial over Q[x] 

Calling sequence:
iroots(p [, b ])


Parameters:
p -- a polynomial over Q[x]
b -- optional boolean

Summary:
The iroots function computes the integer roots of a univariate
polynomial over the rationals.                                 
iroots(p) returned a set of integer roots. 
iroots(p, TRUE) returned a set of pairs of the form
{ [r_1,m_1], ..., [r_n,m_n] } where r_i is a root of the polynomial p
with multiplicity m_i, that is, (x - r_i)^m_i divides p.

Examples:                                                      
iroots( 3/4*((x+1)^2*(x-2)^3*(x^2+2)) ) --> { -1, 2 }         
iroots( (5*x+2)*(2*x+4)*x )             --> { 0, -2 }        
iroots( (x^2+1) )                       --> { }             
iroots( (x+1)^2*(x-2)^3*(x^2+2), TRUE)  --> { [-1,2], [2,3] } 
iroots( (5*x+2)*(2*x+4), TRUE )         --> { [-2, 1] }      

Notes:
You have to load the sharelib library and to export the functions
defined in sharelib, for example loadlib("sharelib"): export(sharelib):

Copyright:
Algorithm developed by Michael B. Monagan.                 
Implemented in MuPAD by Oliver Kluge. (Aug 25 1994)  

======================================================================#


sharelib::iroots := proc(a, multiple) 

   local ldegree, check_lin_poly, iroots_k, d, l, r, x;

begin

   #---------------- Local procedures -----------------#

   # ldgree(a) - low degree of a polynomial in Q[x]    #
   #                                                   #
   #    a must be of type DOM_POLY.                    #
   #    The argument is not checked.                   #

   ldegree := proc(a)
      local n;
   begin
      n := nterms(a);
      if n = 0 then
         0
      else
         degree(nthmonomial(a,n))
      end_if
   end_proc:

   # check_lin_poly(a) - check if the linear polynom a #
   #                     in Z[x] has an integer root   #
   #                                                   #
   #    a must be of type DOM_POLY.                    #
   #    The argument is not checked.                   #

   check_lin_poly := proc(a)
      local c0, c1;
   begin
      c0 := coeff(a, x, 0);
      c1 := coeff(a, x, 1);
      if c0 mod c1 = 0 then
         r := r union { - (c0 div c1) };
      end_if
   end_proc:

   # iroots_k is the main routine to compute the   #
   # integer roots of a polynomial in Z[x].        #
   #                                               #
   # a must be a primitive polynomial of type      #
   # DOM_POLY in Z[x], ldegree(a)=0, degree(a)>0   #
   # if multiple = FALSE then a must be squarefree.#
   # The arguments are not checked.                #

   iroots_k := proc(a, x, multiple)
 
     local b, c, d, i, j, k, l, r, s, t;
 
   begin
 
     r := {};
     d := degree(a);
 
     if d = 1 then
        check_lin_poly(a);
        return(r);
     end_if;
 
     # From now on: degree(a) > 1 , ldegree(a) = 0 #
 
     l := abs(coeff(a,x,d));
     b := 1 + (norm(a) div l);  # Cauchy's bound on the roots #
 
     # Evaluate a between -b and b to find all integer roots  #
     # This is faster than to compute all linear factors of a #
     # (We have to check if b < 50 holds in MuPAD !)          #

     if b < 50 then
        t := coeff(a,x,0);
        for k from -b to b do
                if k <> 0 then if t mod k <> 0 then next end_if
                          else next end_if;
                if multiple then
                   c := 0;
                   while evalp(a, x=k) = 0 do
                         a := divide(a, poly(x-k), Quo);
                         c := c+1;
                   end_while;
                   if c > 0 then r := r union { [k, c] } end_if;
                else
                   if evalp(a, x=k) = 0 then r := r union {k}; end_if;
                end_if
        end_for;
        return(r);
     end_if;
 
     if multiple then
 
        l := sqrfree(a);

        # l = [u, f_1, e_1, ... , f_n, e_n] where              #
        #     a = u * f_1^e_1 * ... * f_n^e_n                  #
        #     f_i squarefree, degree(f_i) > 0                  #
        #     e.g. f_i is primitive and                        #
        #     gcd(f_i,f_j) = 1 for all i <> j                  #
 
        for i from 2 to nops(l)-1 step 2 do
          s := iroots_k( op(l, i) , x, FALSE);
          for j from 1 to nops(s) do
              r := r union { [ op(s,j), op(l,i+1) ] }
          end_for
        end_for;

     else
 
        # Compute all linear factors of a, e.g. find roots via #
        # mapping from Z[x] to Zp[x] for a small prime p       #
         
        l := faclib::linear(a,x);       # a must be squarefree #
 
        # l = [p, b, l_1, ... , l_n] where                     #
        #     a = b * l_1 * ... * l_n , degree(b) > 1          #
        #     l_i are all linear factors of a                  #
         
        for k from 3 to nops(l) do
            check_lin_poly(op(l, k));
        end_for;
         
     end_if;
 
     return(r);
 
   end_proc:

   #---------------------------------------------------#

   a := poly(a);   # convert to DOM_POLY #

   #---------------- Check argument -------------------#

   if testargs() then

      case args(0)

        of 1 do   multiple := FALSE; break;

        of 2 do   if not testtype(args(2), DOM_BOOL) then
                     error("second argument must be a boolean")
                  end_if;
                  break;

        otherwise error("wrong number of arguments.")

     end_case;

     if not testtype(a, Type::PolyOf(Type::Rational,1)) then
        error("argument must be a polynomial in Q[x].");
     end_if;

   end_if;

   #---------------------------------------------------#
   
   x := op(a, [2,1]);	# indeterminate of a #

   l := ldegree(a);

   #  Divide out zero roots #

   if l > 0 then 
      a := divide(a, poly(x^l), Quo); 
      if multiple then r := { [0,1] } else r := {0} end_if
   else 
      r := {};
   end_if;

   d := degree(a);
   if d = 0 then 
      return(r);
   end_if;

   a := divide(a, poly(icontent(a), [x]), Quo);	 # Make a primitive over Z[x] #

   if multiple<>TRUE then # better than if not multiple #
      a := divide(a, gcd(a, diff(a,x)), Quo);    # Discard multiple factors   #
   end_if;

   r union iroots_k(a, x, multiple);

end_proc:
