################################################################################
### LIBRARY : SGML.mu  -  SGML-conversion for euromath project               ###
### AUTHOR  : Andreas Sorgatz                                                ###
### VERSION : V-0.1                                                          ###
### CREATED : 25.10.93                                                       ###
### CHANGED : 17.12.93                                                       ###
###                                                                          ###
### This file contains the procedure 'SGML()' which converts MuPAD-Objects   ###
### into a SGML-notation, accepted by the DTD 'compalg.dtd' from EUROMATH.   ###
### Read file 'SGML.doc' for further information.                            ###
###                                                                          ###
### 'SGML()' MUST be called from the interactiv level and NOT from any pro-  ###
### cedure, in order to guarantee the correct evaluation context !!!         ###
###                                                                          ###
###         YOU NEED AT LEAST MUPAD VERSION-1.2a TO USE THIS PROGRAMM        ###
################################################################################

##
### Global values  #############################################################
##

### DOM_EXPR '_index' has changed, so check the kernel-version ! ###############

SGML_OLD_INDEX  := not bool( op(hold(_x_[_y_]),0) = hold(_index) ):

SGML_VERSION	:= "MuPAD->SGML - V0.1 - Nov.93":

SGML_STR_ERROR	:= "<STMT><ERROR>Evaluation failed</ERROR></STMT>":
SGML_STR_FATAL	:= "<STMT><ERROR>Translation failed</ERROR></STMT>":

SGML_STACK	:= table():          ### Used as stack for tags and strings  ###
SGML_STACH_CNT  := 0:

SGML_SET_NO_FCN	:= {                 ### MuPAD-objects not to tag with <FCN> ###
		     "_plus", "_mult", "_range", "_index"		
		}:


################################################################################
# NAME     :  _type                                                            #
# ARGUMENTS:  any MuPAD object                                                 #
# FUNCTION :  More flexible 'type'-function                                    #
#                                                                              #
# WARNING: THIS MAY BE CHANGED IN THE FUTURE !!!                               #
################################################################################
_type:= proc( obj )
local	typ;
begin
	obj:= args();                              ### unflatt sequences !!! ###
	
	if( (typ:=domtype(obj)) <> DOM_EXPR ) then return( typ );
	else                                  typ:= type( obj );
	end_if;
	
	### name is known by 'type' or given by the 0th operand ###
	if( domtype(typ) = DOM_STRING and typ <> "function" ) then 
		return( typ );
	else
		typ:= op(obj,0);
	end_if;	
	
	### determine name of DOM_IDENT ###
	if( domtype(typ) = DOM_IDENT ) then return( expr2text(typ) );
	end_if;
	
	### reduce a DOM_FUNC_ENV to a DOM_EXEC ###
	if( domtype(typ) = DOM_FUNC_ENV ) then typ:= op(typ,1);
	end_if;
	
	### determine name of an DOM_EXEC ###
	if( domtype(typ) = DOM_EXEC ) then 
		if( domtype((typ:=op(typ,3))) = DOM_STRING ) then 
			return( typ );
		else
			return( "function" );
		end_if;
	end_if;
	
	### determine name of a DOM_PROC ###
	if( domtype(typ) = DOM_PROC ) then 
		if( domtype((typ:=op(typ,6))) = DOM_IDENT ) then 
			return( expr2text(typ) );
		else
			return( "function" );
		end_if;
	end_if;

	### determine name of a DOM_DOMAIN ###
	if( domtype(typ) = DOM_DOMAIN ) then 
		if( domtype((typ:=domattr(typ,"name"))) = DOM_STRING ) then 
			return( typ );
		else
			return( "function" );
		end_if;
	end_if;
	
	"function";
end_proc:



################################################################################
# NAME     :  SGML_create_stack                                                #
# ARGUMENTS:  -                                                                #
# FUNCTION :  Create the global stack 'SGML_STACK'                             #
################################################################################
SGML_create_stack:= proc()
begin
	SGML_STACK_CNT:= 1;
	SGML_STACK    := table( 1="" );	
end_proc:



################################################################################
# NAME     :  SGML_push                                                        #
# ARGUMENTS:  string = SGML-string                                             #
# FUNCTION :  Push element 'string' on the global stack 'SGML_STACK'           #
#                                                                              #
# The function returns 'TRUE' which can be used as a return value for parser   #
# functions,  to indicate that an object has been recognized and succesfully   #
# converted.                                                                   #
################################################################################
SGML_push:= proc( string )
begin
	SGML_STACK_CNT:= SGML_STACK_CNT +1;
	SGML_STACK[SGML_STACK_CNT]:= string;
	TRUE;
end_proc:



################################################################################
# NAME     :  SGML_pop                                                         #
# ARGUMENTS:  string = SGML-string                                             #
# FUNCTION :  Pop the top-element from the global stack 'SGML_STACK'           #
#                                                                              #
# The function returns 'FALSE' which can be used as a return value for parser  #
# functions,  to indicate that an object has not been converted.               #
################################################################################
SGML_pop:= proc()
begin
	SGML_STACK[SGML_STACK_CNT]:= NIL;
	SGML_STACK_CNT:= SGML_STACK_CNT -1;
	FALSE;
end_proc:



################################################################################
# NAME     :  SGML                                                             #
# ARGUMENTS:  ... = Sequence of MuPAD objects                                  #
# FUNCTION :  Entry point for evaluations with SGML output                     #
#                                                                              #
# 'SGML()'  evaluates it's arguments and prints the result in a SGML-notation, #
# defined in "compalg.dtd" by euromath. If the evaluation results in an error, #
# then 'SGML_STR_ERROR' will be returned.  If the SGML conversion fails,  then #
# 'SGML_STR_FATAL' will be returnd.                                            #
#                                                                              #
# 'SGML()' works on the global stack variable 'SGML_STACK'.  Every function of #
# the SGML-parser returns 'TRUE', if it has recognized and converted an object #
# and it returns  'FALSE' otherwise  (in this case the stack is also correct). #
# If an expression can't be handled in a correct way,  then conversion is left #
# by an 'error()'-call.                                                        #
################################################################################
sharelib::SGML:= proc( SGML_obj )
local	PRETTY_PRINT, SGML_error, SGML_tmp, TEXTWIDTH;
option	hold;
begin
	SGML_obj:= args();              ### a given sequence was flatted !!! ###
	
	### Output must not be formatted with pretty print or linebreaks !!! ###
	
	PRETTY_PRINT:= FALSE; 
	TEXTWIDTH   := 100000;
	
	print( Unquoted, "<COMPALG>" );

	### No statement is given  #############################################

	if( args(0) = 0 ) then 
		print( Unquoted, "<STMT></STMT></COMPALG>" );
		return( null() );
	end_if;
		
	### One statement is given #############################################
	
	if( _type(SGML_obj) <> "_stmtseq" ) then
		
		### Full evaluation with manually error handling ###############
		
		print( Unquoted, "<OUTPUT>" );
		SGML_error:= traperror((SGML_tmp:=context(SGML_obj)));
		print( Unquoted, "</OUTPUT>" );
		
		### MuPAD->SGML conversion #####################################
		
		if( SGML_error <> 0 ) then print( Unquoted, SGML_STR_ERROR );
		else
			SGML_create_stack();
			SGML_error:= traperror( SGML_stmt(SGML_tmp) );
			if( SGML_error = 0 ) then 
				SGML_pop();SGML_push( "</STMT>" );
				print( Unquoted, tbl2text(SGML_STACK) );
			else
				print( Unquoted, SGML_STR_FATAL );
			end_if;		
		end_if;

	### A sequence of statements is given  #################################
	
	else for SGML_tmp in [op(SGML_obj)] do

		### Full evaluation with manually error handling ###############
		
		print( Unquoted, "<OUTPUT>" );
		SGML_error:= traperror((SGML_tmp:=context(SGML_tmp)));
		print( Unquoted, "</OUTPUT>" );
		
		### MuPAD->SGML conversion #####################################
		
		if( SGML_error <> 0 ) then print( Unquoted, SGML_STR_ERROR );
		else
			SGML_create_stack();
			SGML_error:= traperror( SGML_stmt(SGML_tmp) );
			if( SGML_error = 0 ) then 
				SGML_pop();SGML_push( "</STMT>" );
				print( Unquoted, tbl2text(SGML_STACK) );
			else
				print( Unquoted, SGML_STR_FATAL );
			end_if;		
		end_if;
	     end_for;
	end_if;

	print( Unquoted, "</COMPALG>" );
	
	return( null() );
end_proc:



## ## ## ## ## ## ## ## ## ## ## ## ## ## ## ## ## ## ## ## ## ## ## ## ## ## ##



################################################################################
# NAME:  SGML_algexpr                                                          #
# The empty expression and all unknown objects are handled as SGML-PCDATA. Be  #
# carefull, the order of the following 'if'-checkes is important.              #
################################################################################
SGML_algexpr:= proc( SGML_obj )
local	SGML_tmp;
begin	
	if( args(0) = 0 ) then return( TRUE ) end_if;   ### empty expression ###

	SGML_obj:= args();                      ### a sequence was flatted ! ###

	if( _type(SGML_obj) = "_stmtseq" ) then
		if( not SGML_stmt_plus(SGML_obj) ) then error(); 
		                                   else return( TRUE ); 
		end_if;
	end_if;
	
		### check lists first ! (may be flatted) #######################

	case TRUE
		of SGML_lists    ( SGML_obj ) do return( TRUE );
	
		of SGML_oper	 ( SGML_obj ) do return( TRUE );
		of SGML_assign	 ( SGML_obj ) do return( TRUE );
		of SGML_const	 ( SGML_obj ) do return( TRUE );
		of SGML_string	 ( SGML_obj ) do return( TRUE );
		of SGML_array	 ( SGML_obj ) do return( TRUE );

		### <CTRL> #####################################################
	
		of SGML_if       ( SGML_obj ) do return( TRUE );
		of SGML_for      ( SGML_obj ) do return( TRUE );
		of SGML_forin    ( SGML_obj ) do return( TRUE );
		of SGML_repeat   ( SGML_obj ) do return( TRUE );
		of SGML_case	 ( SGML_obj ) do return( TRUE );
		of SGML_while  	 ( SGML_obj ) do return( TRUE );

		### <CALCFUNC> #################################################

		of SGML_power	 ( SGML_obj ) do return( TRUE );
		of SGML_evalf	 ( SGML_obj ) do return( TRUE );
		of SGML_diff	 ( SGML_obj ) do return( TRUE );

		of SGML_lastout  ( SGML_obj ) do return( TRUE );
		of SGML_ldpack	 ( SGML_obj ) do return( TRUE );
		of SGML_graph	 ( SGML_obj ) do return( TRUE );
		of SGML_proc	 ( SGML_obj ) do return( TRUE );
		of SGML_fcn	 ( SGML_obj ) do return( TRUE );
		otherwise  
		   SGML_pcdata   ( SGML_obj );
	end_case;
end_proc:
	
	

################################################################################
# NAME:  SGML_algexpr_plus                                                     #
# At this time there's no differenz in algexpr and algexpr+.                   #
################################################################################
SGML_algexpr_plus:= SGML_algexpr:



##
### SGML_array #################################################################
##
SGML_array:= proc( SGML_obj )
local	SGML_dim, SGML_tmp;
begin
	if( domtype(SGML_obj) = DOM_ARRAY ) then		
		SGML_push( "<ARRAY>" );
		SGML_dim:= op(SGML_obj, 0);
		for SGML_tmp in [op(SGML_dim, 2..nops(SGML_dim))] do
			if( _type(SGML_tmp) = "_range" ) then
				SGML_push( "<RANGE><FROM>" );
				if( not SGML_algexpr_plus(op(SGML_tmp,1)) ) then
					error();
				end_if;
				SGML_push( "</FROM><TO>" );
				if( not SGML_algexpr_plus(op(SGML_tmp,2)) ) then
					error();
				end_if;
				SGML_push( "</TO></RANGE>" );
			else
				if( not SGML_lists(SGML_tmp) ) then
				print(SGML_tmp);
				error();
				end_if;
			end_if;
		end_for;

		if( not SGML_lists(op(SGML_obj,1..nops(SGML_obj))) ) then
			error();
		end_if;
		SGML_push( "</ARRAY>" );
	else			
		FALSE;
	end_if;
end_proc:



################################################################################
# NAME:  SGML_assign                                                           #
# Also if the left side isn't an indexed ident, it is expressed as an ident.   #
################################################################################
SGML_assign:= proc( SGML_obj )
local	SGML_tmp;
begin	
	if  ( _type(SGML_obj) = "_assign" ) then
		SGML_push( "<ASSIGN>" );
		if( not SGML_ident_index(op(SGML_obj,1)) ) then
			SGML_push( 
			    "<IDENT>".expr2text(op(SGML_obj,1))."</IDENT>" 
			);
		end_if;
		SGML_push( "<VAL>" );
		if( not SGML_algexpr_plus( op(SGML_obj,2) ) ) then error();
		end_if;
		SGML_push( "</VAL></ASSIGN>" );
	else			
		FALSE;
	end_if;
end_proc:



################################################################################
# NAME:  SGML_case                                                             #
#                                                                              #
################################################################################
SGML_case:= proc( SGML_obj )
local	SGML_tmp, SGML_max;
begin	
	if ( _type(SGML_obj) = "_case" ) then
		SGML_push( "<CTRL><CASE>" );
		if( not SGML_algexpr_plus(op(SGML_obj,1)) ) then error();
		end_if;
		if( nops(SGML_obj) mod 2 = 0 ) then SGML_max:= nops(SGML_obj)-2;
		                               else SGML_max:= nops(SGML_obj)-1;
		end_if;
		for SGML_tmp from 2 to SGML_max step 2 do
			SGML_push( "<CASEOF><CONDIT>" );
			if( not SGML_algexpr_plus(op(SGML_obj,SGML_tmp)) ) then
				error(); 
			end_if;
			SGML_push( "</CONDIT>" );
			if( not SGML_stmt_plus(op(SGML_obj, SGML_tmp+1)) ) then
				error(); 
			end_if;
			SGML_push( "</CASEOF>" );
		end_for;
		if( nops(SGML_obj) mod 2 = 0 ) then
			SGML_push( "<UNKNBL>"  );
			SGML_tmp:= op(SGML_obj, nops(SGML_obj));
			if( not SGML_stmt_plus(SGML_tmp) ) then error(); 
			end_if;
			SGML_push( "</UNKNBL>" );
		end_if;
		SGML_push( "</CASE></CTRL>" );
	else			
		FALSE;
	end_if;
end_proc:



################################################################################
# NAME:  SGML_const                                                            #
# A dirty trick is used to recognize the value "NIL".                          #
################################################################################
SGML_const:= proc( SGML_obj )
begin	
	if  ( SGML_obj = hold(PI)      ) then 
		SGML_push( "<CONST VAL=\"PI\">" );
	elif( SGML_obj = hold(E)       ) then 
		SGML_push( "<CONST VAL=\"E\">" );
	elif( SGML_obj = hold(I)       ) then 
		SGML_push( "<CONST VAL=\"I\">" );
	elif( SGML_obj = hold(TRUE)    ) then 
		SGML_push( "<CONST VAL=\"TRUE\">" );
	elif( SGML_obj = hold(FALSE)   ) then 
		SGML_push( "<CONST VAL=\"FALSE\">" );
	elif( SGML_obj = hold(FAIL)    ) then 
		SGML_push( "<CONST VAL=\"FAIL\">" );
	elif( SGML_obj = hold(SGML_tmp)) then 
		SGML_push( "<CONST VAL=\"NIL\">" );
	elif( SGML_obj = hold(SGML_obj)) then 
		SGML_push( "<CONST VAL=\"NIL\">" );
	else			
		return(	FALSE );
	end_if;
end_proc:



################################################################################
# NAME:  SGML_diff                                                             #
#                                                                              #
################################################################################
SGML_diff:= proc( SGML_obj )
local	SGML_tmp;
begin
	if( _type(SGML_obj) = "diff" ) then
		SGML_push( "<CALCFCN><DIFF><OPERAND>"  );
		if( not SGML_algexpr_plus(op(SGML_obj,1)) ) then error(); 
		end_if;
		SGML_push( "</OPERAND>" );
					
		for SGML_tmp in [op(SGML_obj, 2..nops(SGML_obj))] do
			if( _type(SGML_tmp) = "_seqgen" ) then
				SGML_push( 
					"<WRESPTO>".expr2text(op(SGML_tmp,1)).
					"</WRESPTO><ORDER>"
				);
				if( not SGML_algexpr_plus(op(SGML_tmp,2)) ) then
					error();
				end_if;
				SGML_push( "</ORDER>" );
			else
				SGML_push( 
				    "<WRESPTO>".expr2text(SGML_tmp)."</WRESPTO>" 
				);
			end_if;
		end_for;
		SGML_push( "</DIFF></CALCFCN>" );
	else			
		FALSE;
	end_if;
end_proc:



################################################################################
# NAME:  SGML_evalf                                                            #
#                                                                              #
################################################################################
SGML_evalf:= proc( SGML_obj )
begin	
	if  ( _type(SGML_obj) = "float" ) then
		SGML_push( "<CALCFCN><EVALF><ARG>"  );
		if( not SGML_algexpr_plus(op(SGML_obj,1)) ) then error();
		end_if;
		SGML_push( "</ARG></EVALF></CALCFCN>" );
	else			
		FALSE;
	end_if;
end_proc:



################################################################################
# NAME:  SGML_fcn                                                              #
#                                                                              #
################################################################################
SGML_fcn:= proc( SGML_obj )
local	SGML_tmp;
begin
	SGML_obj:= args();
	
	if( domtype(SGML_obj) <> DOM_EXPR ) then return( FALSE ); 
	end_if;
	SGML_tmp:= _type( SGML_obj );
	
	if( not contains(val(SGML_SET_NO_FCN), SGML_tmp) ) then
		if( SGML_tmp = "function" ) then
			SGML_tmp:= expr2text(op(SGML_obj,0));
		end_if;
		SGML_push("<FCN><NAME>".SGML_tmp."</NAME>");
		for SGML_tmp in [op(SGML_obj)] do
			SGML_push( "<ARG>"  );
			if( not SGML_algexpr_plus(SGML_tmp) ) then error(); 
			end_if;
			SGML_push( "</ARG>" );
		end_for;
		SGML_push( "</FCN>" );
	else			
		FALSE;
	end_if;
end_proc:



################################################################################
# NAME:  SGML_for                                                              #
#                                                                              #
################################################################################
SGML_for:= proc( SGML_obj )
local	SGML_tmp;
begin
	SGML_tmp:= _type( SGML_obj );
	
	if  ( SGML_tmp = "_for" or SGML_tmp = "_for_down" ) then
		SGML_push( 
		    "<CTRL><FOR><IDENT>".expr2text(op(SGML_obj,1)).
		    "</IDENT><FROM>"
		);
		if( not SGML_algexpr_plus(op(SGML_obj,2)) ) then error();
		end_if;
		SGML_push( "</FROM>" );
		if( SGML_tmp = "_for" ) then SGML_push( "<TO>"      );
		                            else SGML_push( "<DOWNTO>"  );
		end_if;
		if( not SGML_algexpr_plus(op(SGML_obj,3)) ) then error();
		end_if;
		if( SGML_tmp = "_for" ) then SGML_push( "</TO>"     );
		                            else SGML_push( "</DOWNTO>" );
		end_if;
		if( op(SGML_obj,4) <> hold(NIL) ) then
			SGML_push( "<STEP>" );
			if( not SGML_algexpr_plus(op(SGML_obj,4)) ) then
				error();
			end_if;
			SGML_push( "</STEP>" );
		end_if;
		if( not SGML_stmt_plus(op(SGML_obj,5)) ) then error(); 
		end_if;
		SGML_push( "</FOR></CTRL>" );
	else			
		FALSE;
	end_if;
end_proc:



################################################################################
# NAME:  SGML_forin                                                            #
#                                                                              #
################################################################################
SGML_forin:= proc( SGML_obj )
begin	
	if( _type(SGML_obj) = "_for_in" ) then
		SGML_push( "<CTRL><FORIN><IDENT>".expr2text(op(SGML_obj,1)).
			   "</IDENT><CONDIT>"  
		);
		if( not SGML_algexpr_plus(op(SGML_obj,2)) ) then error();
		end_if;
		SGML_push( "</CONDIT>" );
		if( not SGML_stmt_plus(op(SGML_obj,3)) ) then error(); 
		end_if;
		SGML_push( "</FORIN></CTRL>" );
	else			
		FALSE;
	end_if;
end_proc:



################################################################################
# NAME:  SGML_graph                                                            #
#                                                                              #
################################################################################
SGML_graph:= proc( SGML_obj )
local	SGML_tmp;
begin
	SGML_tmp:= _type( SGML_obj );
	
	if( SGML_tmp = "function" ) then SGML:= expr2text(SGML_obj,0);
	end_if;
	if( SGML_tmp ="plot" or SGML_tmp ="plot2d" or SGML_tmp ="plot3d" ) then
		SGML_push( "<GRAPH><NAME>".expr2text(SGML_tmp)."</NAME>" );
		if( nops(SGML_obj) = 0 ) then SGML_push( "<ARG></ARG>" );
		else
			for SGML_tmp in [op(SGML_obj)] do
				SGML_push( "<ARG>"  );
				if( not SGML_algexpr_plus(SGML_tmp) ) then
					error();
				end_if;
				SGML_push( "</ARG>" );
			end_for;
		end_if;
		SGML_push( "</GRAPH>" );
	else			
		FALSE;
	end_if;
end_proc:



################################################################################
# NAME:  SGML_ident_index                                                      #
#                                                                              #
################################################################################
SGML_ident_index:= proc( SGML_obj )
local	SGML_tmp;
begin
	if  ( _type(SGML_obj) = DOM_IDENT ) then
		SGML_push( "<IDENT>".expr2text(SGML_obj)."</IDENT>" );
	elif( _type(SGML_obj) = "_index" ) then
	
		if( SGML_OLD_INDEX ) then ### alt ###
		if( not SGML_ident_index(op(SGML_obj,0)) ) then return( FALSE );
		end_if;
		SGML_push( "<INDEX>".expr2text(op(SGML_obj))."</INDEX>" );
		
		else                      ### neu ###
		if( not SGML_ident_index(op(SGML_obj,1)) ) then return( FALSE );
		end_if;
		SGML_push( "<INDEX>".expr2text(op(SGML_obj,2..nops(SGML_obj))).
			   "</INDEX>" );
		end_if;
		
	else
		FALSE;
	end_if;
end_proc:



################################################################################
# NAME:  SGML_if                                                               #
# The MuPAD-parser converts all elif-structures into if-then-else structures.  #
# In MuPAD there are no empty else-blocks,  because this is expressed by the   #
# value "NIL".                                                                 #
################################################################################
SGML_if:= proc( SGML_obj )
begin	
	if  ( _type(SGML_obj) = "_if" ) then
		SGML_push( "<CTRL><IF>" );
		if( not SGML_algexpr_plus( op(SGML_obj,1) ) ) then error();
		end_if;
		SGML_push( "<THENBL>"  );
		if( not SGML_stmt_plus(    op(SGML_obj,2) ) ) then error();
		end_if;
		SGML_push( "</THENBL><ELSEBL>"  );
		if( not SGML_stmt_plus(    op(SGML_obj,3) ) ) then error();
		end_if;
		SGML_push( "</ELSEBL></IF></CTRL>" );
	else			
		FALSE;
	end_if;
end_proc:



################################################################################
# NAME:  SGML_lastout                                                          #
#                                                                              #
################################################################################
SGML_lastout:= proc( SGML_obj )
begin	
	if( _type(SGML_obj) = "last" ) then
		SGML_push( "<LASTOUT>" );
		if( not SGML_algexpr_plus(op(SGML_obj,1)) ) then error();
		end_if; 
		SGML_push( "</LASTOUT>" );
	else			
		FALSE;
	end_if;
end_proc:



################################################################################
# NAME:  SGML_ldpack                                                           #
#                                                                              #
################################################################################
SGML_ldpack:= proc( SGML_obj )
local	SGML_tmp;
begin
	SGML_tmp:= _type( SGML_obj );
	
	if( SGML_tmp = "load"    or SGML_tmp = "loadlib"  or
	    SGML_tmp = "loadmod" or SGML_tmp = "loadproc" or
	    SGML_tmp = "read" ) then
		SGML_push( "<LDPACK>".expr2text(SGML_obj)."</LDPACK>" );
	else			
		FALSE;
	end_if;
end_proc:



################################################################################
# NAME:  SGML_lists                                                            #
#                                                                              #
################################################################################
SGML_lists:= proc( SGML_obj )
local	SGML_tmp, SGML_typ;
begin	
	SGML_obj:= args();              ### a given sequence was flatted !!! ###
	SGML_typ:= _type( SGML_obj );
	
	if  ( SGML_typ = DOM_LIST       ) then SGML_typ:= "LIST";
	elif( SGML_typ = DOM_SET        ) then SGML_typ:= "SET";
	elif( SGML_typ = DOM_TABLE      ) then SGML_typ:= "TABLE";
	elif( SGML_typ = "_exprseq"     ) then SGML_typ:= "SEQUENCE";
	else  return( FALSE );
	end_if;

	SGML_push( "<".SGML_typ.">" );
	for SGML_tmp in [op(SGML_obj)] do
		SGML_push( "<ELEM>"  );
		if( not SGML_algexpr_plus(SGML_tmp) ) then error();
		end_if;
		SGML_push( "</ELEM>" );
	end_for;
	SGML_push( "</".SGML_typ.">" );
end_proc:



################################################################################
# NAME:  SGML_oper                                                             #
# The operator "NOT" has only one argument. All others have at least two and   #
# are given out in infix notation.                                             #
################################################################################
SGML_oper:= proc( SGML_obj )
local	SGML_tmp, SGML_str;
begin
	SGML_tmp:= _type( SGML_obj );

	if( SGML_tmp = "_not" ) then 
		SGML_push( "<OPER SYMBOL=\"NOT\">(" );
		if( not SGML_algexpr_plus(op(SGML_obj,1)) ) then error(); 
		end_if;
		return( SGML_push(")") );
	       
	elif( SGML_tmp = "_or"        ) then SGML_str:= "OR";
	elif( SGML_tmp = "_and"       ) then SGML_str:= "AND";
	
	elif( SGML_tmp = "_less"      ) then SGML_str:= "LT";
	elif( SGML_tmp = "_leequal"   ) then SGML_str:= "LE";
	elif( SGML_tmp = "_equal"     ) then SGML_str:= "EQ";
	elif( SGML_tmp = "_unequal"   ) then SGML_str:= "NE";
	
	elif( SGML_tmp = "_union"     ) then SGML_str:= "UNION";
	elif( SGML_tmp = "_intersect" ) then SGML_str:= "INTERSEC";
	elif( SGML_tmp = "_minus"     ) then SGML_str:= "SETMINUS";
	
	elif( SGML_tmp = "_mod"       ) then SGML_str:= "MOD";
	elif( SGML_tmp = "_div"       ) then SGML_str:= "INTDIV";
	else                                     
		return( FALSE )
	end_if;
	
	SGML_push( "(" );
	for SGML_tmp in [op(SGML_obj)] do
		if( not SGML_algexpr_plus(SGML_tmp) ) then error(); 
		end_if;
		SGML_push( "<OPER SYMBOL=\"".SGML_str."\">" );
	end_for;
	SGML_pop();
	SGML_push( ")" );
end_proc:



################################################################################
# NAME:  SGML_pcdata                                                           #
# If SGML_obj is a function-call, then it is in the set SGML_SET_NO_FCN.       #
################################################################################
SGML_pcdata:= proc( SGML_obj )
local	SGML_n, SGML_str, SGML_tmp;
begin	
	SGML_tmp:= _type( SGML_obj );
	
	if( SGML_tmp = DOM_NULL ) then return( TRUE ); 
	end_if;

	if( domtype(SGML_obj) <> DOM_EXPR ) then
		if( domtype(SGML_obj) = DOM_COMPLEX ) then
			if( not SGML_algexpr(op(SGML_tmp,1)) ) then error();
			end_if;
			SGML_push( "+" );
			if( not SGML_algexpr(op(SGML_tmp,2)) ) then error();
			end_if;
			SGML_push( "*" );
			return( SGML_const(hold(I)) );
		end_if;		
		if( testtype(SGML_obj,NUMERIC) ) then
			if( SGML_obj < 0 ) then 
				return(SGML_push("(".expr2text(SGML_obj).")"));
			end_if;
		end_if;
		return( SGML_push( expr2text(SGML_obj) ) );
	end_if;

	case SGML_tmp                                  ### special operators ###
	of "_index" do	if( SGML_OLD_INDEX ) then ### alt ###
			if( not SGML_algexpr(op(SGML_obj,0)) ) then error();
			end_if;
			SGML_push( "[" );
			if( not SGML_algexpr_plus(op(SGML_obj)) ) then error();
			end_if;
			return( SGML_push( "]" ) );
			
			else                      ### neu ###
			if( not SGML_algexpr(op(SGML_obj,1)) ) then error();
			end_if;
			SGML_push( "[" );
			SGML_tmp:= op( SGML_obj, 2..nops(SGML_obj) );
			if( not SGML_algexpr_plus(SGML_tmp) ) then error();
			end_if;
			return( SGML_push( "]" ) );
			end_if;
			
	of "_range" do	if( nops(SGML_obj) < 2 ) then error();
			end_if;
			SGML_str:= "..";
			
			break;
			
	of "_plus"  do  SGML_n:= nops( SGML_obj );
			if  ( SGML_n = 0 ) then return( SGML_push("_plus()") );
			elif( SGML_n = 1 ) then 
				SGML_push( "_plus(" );
				if( not SGML_algexpr(op(SGML_obj,1)) ) then
					error(); 
				end_if;
				return( SGML_push(")") );
			else
				SGML_str:= " + ";
			end_if;
			break;

	of "_mult"  do	SGML_n:= nops( SGML_obj );
			if  ( SGML_n = 0 ) then return( SGML_push("_mult()") );
			elif( SGML_n = 1 ) then 
				SGML_push( "_mult(" );
				if( not SGML_algexpr(op(SGML_obj),1) ) then
					error(); 
				end_if;
				return( SGML_push(")") );
			elif( SGML_n = 2 and op(SGML_obj,2) = -1 ) then 
				SGML_push( "(-" );
				if( not SGML_algexpr(op(SGML_obj,1)) ) then
					error(); 
				end_if;
				return( SGML_push(")") );
			elif( SGML_n = 2 and op(SGML_obj,1) = 1 ) then 
				if( not SGML_algexpr(op(SGML_obj,2)) ) then
					error(); 
				end_if;
				return( TRUE );
			else
				SGML_str:= " * ";
			end_if;
			break;
			
	otherwise:
			if( SGML_tmp = "function" ) then
				SGML_tmp:= expr2text(op(SGML_obj,0));
			end_if;
			SGML_push( SGML_tmp."(" );
			for SGML_tmp in [op(SGML_obj)] do
				if( not SGML_algexpr_plus(SGML_tmp) ) then
					error();
				end_if;
				SGML_push( ", " );
			end_for;
			SGML_pop();
			return( SGML_push(")") );
	end_case;
	
	SGML_push( "(" );
	for SGML_tmp in [op(SGML_obj)] do
		if( not SGML_algexpr_plus(SGML_tmp) ) then error(); 
		end_if;
		SGML_push( SGML_str );
	end_for;
	SGML_pop();
	SGML_push( ")" );
end_proc:



################################################################################
# NAME:  SGML_power                                                            #
#                                                                              #
################################################################################
SGML_power:= proc( SGML_obj )
begin	
	if  ( _type(SGML_obj) = "_power" ) then
		SGML_push( "<CALCFCN><POWER><OPERAND>"  );
		if( not SGML_algexpr_plus(op(SGML_obj,1)) ) then error();
		end_if;
		SGML_push( "</OPERAND><ARG>"  );
		if( not SGML_algexpr_plus(op(SGML_obj,2)) ) then error();
		end_if;
		SGML_push( "</ARG></POWER></CALCFCN>" );
	else			
		FALSE;
	end_if;
end_proc:



################################################################################
# NAME:  SGML_proc                                                             #
# The 'name' entry of a procedure get lost.                                    #
################################################################################
SGML_proc:= proc( SGML_obj )
local	SGML_tmp;
begin
	if  ( domtype(SGML_obj) = DOM_PROC ) then
		SGML_push( "<PROC>" );
		for SGML_tmp in [op(SGML_obj,1)] do
			SGML_push( "<IDENT>".expr2text(SGML_tmp)."</IDENT>" );
		end_for;
		if ( op(SGML_obj,2) <> NIL ) then
			SGML_push( "<LOCVAR>"  );
			for SGML_tmp in [op(SGML_obj,2)] do
				SGML_push( 
				    "<IDENT>".expr2text(SGML_tmp)."</IDENT>" 
				);
			end_for;
			SGML_push( "</LOCVAR>" );
		end_if;
		if ( op(SGML_obj,3) <> NIL ) then
			SGML_push( "<OPTION>"  );
			for SGML_tmp in [op(SGML_obj,3)] do
				SGML_push( 
				    "<IDENT>".expr2text(SGML_tmp)."</IDENT>" 
				);
			end_for;
			SGML_push( "</OPTION>" );
		end_if;
		if( not SGML_stmt_plus(op(SGML_obj,4)) ) then error(); 
		end_if;
		if ( op(SGML_obj,5) <> NIL ) then
			SGML_push( "<REMEMBER>"  );
			if( not SGML_algexpr(op(SGML_obj,5)) ) then error();
			end_if;
			SGML_push( "</REMEMBER>" );
		end_if;		
		if ( op(SGML_obj,6) <> NIL ) then
			SGML_push( "<NAME>".expr2text(op(SGML_obj,6))."</NAME>"
			);
		end_if;		
		SGML_push( "</PROC>" );
	else			
		FALSE;
	end_if;
end_proc:



################################################################################
# NAME:  SGML_repeat                                                           #
#                                                                              #
################################################################################
SGML_repeat:= proc( SGML_obj )
begin	
	if  ( _type(SGML_obj) = "_repeat" ) then
		SGML_push( "<CTRL><REPEAT>" );
		if( not SGML_stmt_plus(   op(SGML_obj,1)) ) then error(); 
		end_if;
		SGML_push( "<CONDIT>"  );
		if( not SGML_algexpr_plus(op(SGML_obj,2)) ) then error();
		end_if;
		SGML_push( "</CONDIT></REPEAT></CTRL>" );
	else			
		FALSE;
	end_if;
end_proc:



################################################################################
# NAME:  SGML_stmt                                                             #
#                                                                              #
################################################################################
SGML_stmt:= proc( # ... # )
begin
	SGML_push( "<STMT>"  );	
	if( not SGML_algexpr_plus(args()) ) then error(); 
	end_if;
	SGML_push( "</STMT>\n" );
end_proc:



################################################################################
# NAME:  SGML_stmt_plus                                                        #
# If an expression contains more than one statement, then they are packed in a #
# statement-sequence '_stmtseq'.                                               #
################################################################################
SGML_stmt_plus:= proc( SGML_obj )
local	SGML_tmp;
begin
	SGML_obj:= args();                      ### a sequence was flatted ! ###
	
	if( _type(SGML_obj) = "_stmtseq" ) then
		for SGML_tmp in [op(SGML_obj)] do
			if( not SGML_stmt(SGML_tmp) ) then error(); 
			end_if;
		end_for;
		TRUE;
	else	
		SGML_stmt( SGML_obj );
	end_if;
end_proc:



################################################################################
# NAME:  SGML_string                                                           #
# Conversion: '<'-->'&ls', '&'-->'&amp'                                        #
################################################################################
SGML_string:= proc( SGML_obj )
local	SGML_i, SGML_str, SGML_tmp;
begin
	if( domtype(SGML_obj) <> DOM_STRING ) then return( FALSE );
	end_if;

	SGML_str:= "";
	for SGML_i from 0 to strlen(SGML_obj)-1 do
		SGML_tmp:= substring(SGML_obj,SGML_i,1);
		if  ( SGML_tmp = "<" ) then SGML_tmp:= "&lt;";
		elif( SGML_tmp = ">" ) then SGML_tmp:= "&gt;";
		elif( SGML_tmp = "&" ) then SGML_tmp:= "&amp;";
		end_if;
		SGML_str:= SGML_str.SGML_tmp;
	end_for;
	
	SGML_push( "<STRING>".SGML_str."</STRING>" );
end_proc:



################################################################################
# NAME:  SGML_while                                                            #
#                                                                              #
################################################################################
SGML_while:= proc( SGML_obj )
begin	
	if  ( _type(SGML_obj) = "_while" ) then
		SGML_push( "<CTRL><WHILE><CONDIT>"  );
		if( not SGML_algexpr_plus(op(SGML_obj,1)) ) then error();
		end_if;
		SGML_push( "</CONDIT>" );
		if( not SGML_stmt_plus(   op(SGML_obj,2)) ) then error();
		end_if;
		SGML_push( "</WHILE></CTRL>" );
	else			
		FALSE;
	end_if;
end_proc:



##
### ENDE #######################################################################
##
SGML_VERSION:
