alias(BranchOrder(a)=extop(a,1)):
alias(val(a)=extop(a,2)): # valuation #
alias(err(a)=extop(a,3)):
alias(ListCoeffs(a)=extop(a,4)):
alias(variable(a)=extop(a,5)): # series variable #

Puiseux := domain():

Puiseux::match_x_power_k:=proc(e,x) # matches x^k with k a rational #
# returns k if match successful, FALSE otherwise #
begin
   if e=1 then 0
   elif e=x then 1
   elif type(e)="_power" and op(e,1)=x and contains({DOM_INT,DOM_RAT},type(op(e,2))) then op(e,2)
   else FALSE
   end_if
end_proc:

# these procedures are to be used from outside (for example gfun) to
  create Puiseux expansions efficiently without knowing their
  internal structure #
Puiseux::create := proc(bo,val,ord,l,x)
begin
   new(Puiseux,bo,val,ord,l,x)
end_proc:

Puiseux::ldegree:=proc(a) local l,n;
begin
   l:=ListCoeffs(a);
   n:=val(a);
   while l<>[] do
      if Puiseux::iszero(l[1]) then l[1]:=NIL; n:=n+1
      else return(n/BranchOrder(a))
      end_if
   end_while;
   FAIL
end_proc:

Puiseux::order := proc(a) begin err(a)/BranchOrder(a) end_proc:

Puiseux::remove_head:=proc(a) # a0+a1*x+a2*x^2+... --> a1*x+a2*x^2+... #
begin
   Puiseux::normal(extsubsop(a,2=val(a)+1,4=subsop(ListCoeffs(a),1=null())))
end_proc:

Puiseux::iszero:=iszero: # default simplification #

Puiseux::normal:=proc(a) # remove zero coefficients in head #
local l,v;
begin
   l:=ListCoeffs(a); # removes a0 #
   if nops(l)=0 then a
   elif not Puiseux::iszero(l[1]) then a
   else
      v:=val(a);
      while l<>[] do
         if Puiseux::iszero(l[1]) then l[1]:=NIL; v:=v+1 else break end_if
      end_while;
      extsubsop(a,2=v,4=l)
   end_if
end_proc:

# [branch order, val, ord, [coeffs]] #
Puiseux::new := proc(e,x) 
   local i,big_O,ord,k,d,bo,l;
   begin
      ord:=proc(t) begin if type(t)="_power" and op(t,1)=x and
	type(op(t,2))=DOM_RAT then denom(op(t,2)) else 1 end_if end_proc:
      l:=indets(e,PolyExpr);
      bo:=ilcm(1,1,ord(op(l,i))$i=1..nops(l)); # the 1 is needed because ilcm() -> FAIL #
      # find the O-term if any, and the branching order #
      if type(e)="_plus" then e:=[op(e)] else e:=[e] end_if;
      big_O:=0;
      for i from 1 to nops(e) do
         if type(op(e,i))=O then big_O:=op(e,i); e:=subsop(e,i=null()); break end_if
      end_for;
      if big_O<>0 then 
         if nops(big_O)<>1 or (ord:=Puiseux::match_x_power_k(op(big_O),x))=FALSE
         then error("invalid big-O") end_if;
         bo:=ilcm(bo,denom(ord));
         ord:=ord*bo;
      end_if;
      e:=_plus(op(e));
      e:=subs(e,x=x^bo); # should have integer exponents now #
      # now find the valuation and the degree #
      k:=ldegree(e,[x]);
      if type(ord)<>DOM_INT then 
         ord:=k+ORDER 	 # ORDER is set in series.mu #
      end_if;
      d:=min(degree(e,[x]),ord-1);
      new(Puiseux,bo,k,ord,[coeff(e,[x],i)$hold(i)=k..d],x);
   end_proc:

Puiseux::set_var := proc(a,X,x) # specify variable of expansion #
begin
   extsubsop(subs(a,X=x),5=x)
end_proc:

Puiseux::print :=
   proc(a,x) local l,i,bo,k;
   begin
      bo:=BranchOrder(a);
      k:=val(a);
      l:=ListCoeffs(a);
      if extnops(a)>=5 then x:=variable(a) end_if;
      l:=[l[i]*x^((k+i-1)/bo) $ i=1..nops(l)];
      l:=map(l,func((if x=0 then null() else x end_if),x));
      i:=O(x^(err(a)/bo));
      if l=[] then i else hold(_plus)(op(l),i) end_if
end_proc:

Puiseux::expr := proc(a)
local l,x,i,bo,k;
begin
   bo:=BranchOrder(a);
   k:=val(a);
   l:=ListCoeffs(a);
   x:=variable(a);
   _plus(l[i]*x^((k+i-1)/bo) $ i=1..nops(l))
end_proc:

Puiseux::multbo := proc(a,n) # multiply branching order of a by n (integer) #
local l,i,z;
begin
   l:=ListCoeffs(a);
   if l<>[] then 
      z:=0$(n-1);
      l:=[(l[i],z)$i=1..nops(l)-1,l[nops(l)]]
   end_if;
   extsubsop(a,1=BranchOrder(a)*n,2=val(a)*n,3=err(a)*n,4=l)
end_proc:

Puiseux::plus := proc(a,b) local ka,kb,qa,qb,la,lb,i,dif,da,db,lc,boa,bob,x;
begin
   if type(a)<>Puiseux then 
      if extnops(b)>=5 then a:=Puiseux::new(a,variable(b)) else a:=Puiseux::new(a,x) end_if;
      qb:=err(b); qa:=qb;
   elif type(b)<>Puiseux then
      # is b in a more general domain ? #
      b::convert(a); if %<>FAIL then return(%+b) end_if;
      if extnops(a)>=5 then b:=Puiseux::new(b,variable(a)) else b:=Puiseux::new(b,x) end_if;
      qa:=err(a); qb:=qa;
   else
      qa:=err(a); qb:=err(b);
   end_if;
   boa:=BranchOrder(a); bob:=BranchOrder(b);
   if boa<>bob then
      i:=ilcm(boa,bob);
      Puiseux::plus(Puiseux::multbo(a,i/boa),Puiseux::multbo(b,i/bob))
   else
      ka:=val(a); kb:=val(b);
      if kb>=qa then a
      elif ka>=qb then b
      elif ka<=kb then
         dif:=kb-ka; # dif>=0 #
         la:=ListCoeffs(a); lb:=ListCoeffs(b);
         da:=nops(la); db:=nops(lb)+dif;
         if da<=db then
            if dif<=da then 
               lc:=[la[i]$i=1..dif,la[i]+lb[i-dif]$i=dif+1..da,lb[i-dif]$i=da+1..min(db,qa-ka)]
            else # no overlapping #
               lc:=[la[i]$i=1..da,0$(dif-da),lb[i-dif]$i=dif+1..db]
            end_if
         else # da>db thus da>dif ==> always overlapping #
            lc:=[la[i]$i=1..dif,la[i]+lb[i-dif]$i=dif+1..db,la[i]$i=db+1..da];
         end_if;
         while nops(lc)<>0 do
            if lc[1]=0 then
               lc:=subsop(lc,1=null()); ka:=ka+1; if lc=[] then break end_if
            else break
            end_if
         end_while;
         extsubsop(a,2=ka,3=min(qa,qb),4=lc)
      else Puiseux::plus(b,a)
      end_if
   end_if
end_proc:

Puiseux::_plus:=misc::genassop(_plus,args(1),Puiseux::plus(args(1),args(2))):

Puiseux::negate := proc(a) local l,i;
begin
   l:=ListCoeffs(a);
   extsubsop(a,4=[-l[i]$i=1..nops(l)])
end_proc:

Puiseux::mult := proc(a,b) local ka,kb,qa,qb,da,db,la,lb,i,j,i1,i2,l1,l2,l3,imax,boa,bob;
begin
   if type(a)<>Puiseux then
      if a=1 then return(b) else a:=Puiseux::new(a,variable(b)) end_if
   elif type(b)<>Puiseux then 
      if contains({DOM_EXPR,DOM_IDENT,DOM_INT},domtype(b)) then
         b:=Puiseux::new(b,variable(a))
      else return(b::_mult(b::convert(a),b))
      end_if
   end_if;
   boa:=BranchOrder(a); bob:=BranchOrder(b);
   if boa<>bob then
      i:=ilcm(boa,bob);
      Puiseux::mult(Puiseux::multbo(a,i/boa),Puiseux::multbo(b,i/bob))
   else
      ka:=val(a); kb:=val(b);
      qa:=err(a); qb:=err(b);
      la:=ListCoeffs(a); lb:=ListCoeffs(b);
      da:=nops(la); db:=nops(lb);
      imax:=min(qa-ka,qb-kb); # maximal number of terms in the product #
      if da<=db then i1:=da; i2:=db
      else i1:=db; i2:=da
      end_if;
      l1:=_plus(la[j]*lb[i+1-j]$hold(j)=1..i)$hold(i)=1..i1;
      if da<=db then    
	 l2:=_plus(la[j]*lb[i+1-j]$hold(j)=1..da)$hold(i)=da+1..min(db,imax)
      else
	 l2:=_plus(la[i+1-j]*lb[j]$hold(j)=1..db)$hold(i)=db+1..min(da,imax)
      end_if;
      l3:=_plus(la[j]*lb[i+1-j]$hold(j)=i+1-db..da)$hold(i)=i2+1..min(da+db-1,imax);
      extsubsop(a,2=ka+kb,3=min(ka+qb,kb+qa),4=[l1,l2,l3])
   end_if
end_proc:

Puiseux::_mult:=misc::genassop(_mult,args(1),Puiseux::mult(args(1),args(2))):

Puiseux::square := proc(a) local ka,qa,da,la,i,j,l1,l3,imax;
begin
   ka:=val(a);
   qa:=err(a);
   la:=ListCoeffs(a);
   da:=nops(la); 
   imax:=qa-ka; # maximal number of terms in the product #
   l1:=_plus(la[j]*la[i+1-j]$hold(j)=1..i)$hold(i)=1..da;
   l3:=_plus(la[j]*la[i+1-j]$hold(j)=i+1-da..da)$hold(i)=da+1..min(2*da-1,imax);
   extsubsop(a,2=2*ka,3=ka+qa,4=[l1,l3])
end_proc:

Puiseux::_power := proc(a,n)
local r;
begin
   if type(n)=DOM_RAT then Puiseux::power(a,n)
   elif n<0 then 
      a:=normal(a);
      if n=-1 then Puiseux::invert(a)
      else Puiseux::invert(Puiseux::_power(a,-n))
      end_if
   elif n=0 then Puiseux::one(n,variable(a))
   elif n=1 then a
   elif n mod 2 = 0 then Puiseux::_power(Puiseux::square(a),n div 2)
   else Puiseux::mult(a,Puiseux::_power(Puiseux::square(a),n div 2))
   end_if
end_proc:

# from Zippel, page 165 #
Puiseux::power := proc(a,s)
local v,p,ps,k,q,l,d,p0,B,C,bo;
begin
   # works only for valence 0 #
   v:=val(a);
   if v<>0 then 
      # a = x^(v/bo) * (1 + ...) thus a^s = B * C
        where B = x^(s*v/bo) and C = (1 + ...)^s #
      # B has branching order denom(s*v/bo) and C has branching order bo #
      bo:=BranchOrder(a);
      p:=s*v/bo;
      C:=Puiseux::power(extsubsop(a,2=0,3=err(a)-v),s);
      if bo mod denom(p) <> 0 then
         k:=ilcm(bo,denom(p));
         C:=Puiseux::multbo(C,k/bo);
      else k:=bo
      end_if;
      extsubsop(C,2=p*k,3=err(C)+p*k)
   else
      q:=err(a); # O(x^q) #
      p:=ListCoeffs(a);
      d:=nops(p);
      if d=0 then return(extsubsop(a,3=s*q)) end_if;
      p0:=p[1];
      ps[1]:=p0^s;
      for k from 1 to q-1 do
         ps[k+1]:=_plus((l*s-k+l)*p[l+1]*ps[k+1-l]$l=1..min(k,d-1))/p0/k
      end_for;
      extsubsop(a,4=ps)
   end_if
end_proc:

# from Zippel, page 167 #
Puiseux::_fconcat:=proc(P,Q)
local q,R,i,l,d,k,vQ,lR,boP,boQ;
begin
      boP:=BranchOrder(P);
      # P(x<-Q) = P(x<-x^boP)(x<-Q^(1/boP)) #
      if boP<>1 then P:=extsubsop(P,1=1); Q:=Puiseux::power(Q,1/boP) end_if; 
      boQ:=BranchOrder(Q);
      vQ:=val(Q);
      if vQ<=0 then error("invalid composition") end_if;
      # P = a[k]*x^k + ... + a[p]*x^p + O(x^q) #
      k:=val(P);
      l:=ListCoeffs(P);
      d:=nops(l);
      q:=err(P);
      # initialize with a[p] + O(x^(vQ*(q-p))) #
      R:=extsubsop(Q,2=0,3=vQ*(q-k-d+1),4=[l[d]]);
      for i from d-1 downto 1 do
	 R:=Puiseux::_mult(R,Q); # now R = b*x^vQ + ... #
	 lR:=[l[i],0$(vQ-1),op(ListCoeffs(R))];
         R:=extsubsop(R,2=0,4=lR);
      end_for;
      # now R = a[k] + a[k+1]*Q + ... #
      if k=0 then R else Puiseux::_mult(R,Puiseux::_power(Q,k)) end_if
end_proc:

Puiseux::scalmult:=proc(p,a,k) # a*x^k*p #
local l,i;
begin
   l:=ListCoeffs(p);
   if args(0)<3 then extsubsop(p,4=[a*l[i]$i=1..nops(l)])
   else
      extsubsop(p,4=[a*l[i]$i=1..nops(l)],2=val(p)+k*BranchOrder(p),
	3=err(p)+k*BranchOrder(p))
   end_if
end_proc:

Puiseux::one:=proc(n,x) begin new(Puiseux,1,0,n,[1],x) end_proc:

Puiseux::frompoly := proc(p,n,x)
local i,v;
begin
   # to avoid coefficients like 0.0 #
   p:=mapcoeffs(p,fun((if args(1)=0.0 then 0 else args(1) end_if)));
   v:=ldegree(p);
   new(Puiseux,1,v,n,[coeff(p,i)$i=v..min(degree(p),n-1)],x)
end_proc:

# the following is about twice faster than Puiseux::power(a,-1) #
Puiseux::invert := proc(a) local k,q,d,l,i,a1,b,j;
begin
   l:=ListCoeffs(a);
   d:=nops(l);
   q:=err(a);
   if d=0 then # 1/O(x^q) -> division by zero #
      error("division by zero")
   else # 1/(a1*x^k+...+ad*x^(k+d-1)+O(x^q)) = x^(-k)/(a1+...+ad*x^(d-1)+O(x^(q-k))) #
      a1:=l[1];
      k:=val(a);
      b:=table(); b[1]:=1/a1;
      for i from 3 to d+1 do b[i-1]:=-_plus(l[j]*b[i-j]$j=2..i-1)/a1 end_for;
      for i from d+2 to q-k+1 do b[i-1]:=-_plus(l[j]*b[i-j]$j=2..d)/a1 end_for;
      extsubsop(a,2=-k,3=q-2*k,4=[b[i]$hold(i)=1..q-k])
   end_if;
end_proc:

Puiseux::equalx := proc(a) # is a x + O(...) #
begin
   if val(a)=1 and nops(ListCoeffs(a))>=1 and ListCoeffs(a)[1]=1 then TRUE else FALSE end_if
end_proc:

Puiseux::coeff := proc(a,n) # gives the coefficient of x^n #
local k,q,l,bo;
begin
   bo:=BranchOrder(a);
   k:=val(a);
   if n<k/bo then 0
   else
      q:=err(a);
      if n>=q/bo then FAIL
      else # x^k/bo corresponds to l[1], thus x^n corresponds to l[n*bo+1-k] #
         n:=n*bo+1-k;
         if type(n)<>DOM_INT then return(0) end_if;
         l:=ListCoeffs(a);
         if n<=nops(l) then l[n] else 0 end_if
      end_if
   end_if
end_proc:

Puiseux::nthterm := proc(a,n) local l;
begin
   l:=ListCoeffs(a);
   if n>nops(l) then FAIL else variable(a)^((val(a)+n-1)/BranchOrder(a)) end_if
end_proc:

Puiseux::lcoeff := proc(_a) local l;
begin
   l:=ListCoeffs(_a);
   if Puiseux::iszero=FAIL then Puiseux::iszero:=iszero end_if;
   while l<>[] do
      if Puiseux::iszero(l[1]) then l[1]:=NIL
      else return(l[1])
      end_if
   end_while;
   FAIL
end_proc:

Puiseux::lmonomial := proc(_a) local l,n;
begin
   l:=ListCoeffs(_a);
   if Puiseux::iszero=FAIL then Puiseux::iszero:=iszero end_if;
   n:=0; while l<>[] do
      if Puiseux::iszero(l[1]) then l[1]:=NIL
      else 
         return(l[1]*variable(_a)^((val(_a)+n)/BranchOrder(_a)))
      end_if;
      n:=n+1;
   end_while;
   FAIL
end_proc:

Puiseux::evaluate := proc(_a_)
begin
   ListCoeffs(_a_);
   extsubsop(_a_,4=eval(%))
end_proc:

Puiseux::name := "Puiseux":

# reverse(a[k]*x^(k/b) + a[k+1]*x^((k+1)/b) + ... ) = 
	reverse(a[k]*x^k + a[k+1]*x^(k+1) + ...)^b #
Puiseux::revert := proc(a) # using poly is faster #
local k,c,l,n,m,i,p,y,j,q,eq,sol;
begin
   if BranchOrder(a)<>1 then
      Puiseux::revert(extsubsop(a,1=1))^BranchOrder(a)
   else # branching order is 1 #
      k:=val(a); # y = x^k + ... ==> x = y^(1/k) + y^(2/k) + ... #
      if k<0 then # y = 1/x^j + ... ==> 1/y = x^j + x^(j+1) + ... #
         1/Puiseux::revert(1/a)
      elif k=0 then # y = c + x^j + ... ==> (y-c) = x^j + ... #
         c:=lcoeff(a);
         a:=Puiseux::remove_head(a);
         extsubsop(Puiseux::revert(a),5=variable(a)-c)
      else # k>0 #
         n:=err(a)-k; # number of knowns terms of y #
         m:=ListCoeffs(a);
         if nops(m)<n then m:=m . [0$(n-nops(m))] end_if;
         l[1]:=1/m[1]^(1/k); # coefficient of y^(1/k) #
         p:=l[1]*y; q:=poly(m[1]*y,[y]);
         for i from 2 to n do
            # compute the coefficient of y^(i/k) #
            # we need only the coefficients m[1]...m[i] #
            p:=p+c*y^i;
            q:=q+poly(m[i]*y^i,[y]);
            eq:=coeff(q(p),y,i); # this is the expensive operation #
            l[i]:=-coeff(eq,c,0)/coeff(eq,c,1);
            p:=subs(p,c=l[i]);
         end_for;
         extsubsop(a,1=k,2=1,3=1+n,4=[l[i]$hold(i)=1..n])
      end_if
   end_if
end_proc:

# suggestion from Frank Postel #
Puiseux::map := fun(	
     extsubsop( args(1),4=map(ListCoeffs(args(1)),args(i) $ hold(i)=2..args(0)) )
   ):

Puiseux::contfrac := proc(a)
local res,l,bo,n,j,v,k,a1,b1,z,q;
begin
   if extnops(a)>=5 then z:=variable(a) end_if;
   a:=Puiseux::normal(a);
   bo:=BranchOrder(a);
   v:=val(a);
   q:=err(a);
   l:=ListCoeffs(a);
   n:=nops(l);
   if n=0 then return(O(z^(q/bo)))
   elif n=1 then return(l[1]*z^(v/bo)+O(z^(q/bo)))
   end_if;
   # a is normalized ==> l[1]<>0 #
   # find second non zero term #
   for j from 2 to n do if l[j]<>0 then break end_if end_for;
   if j=n then return(l[1]*z^(v/bo)+l[j]*z^((v+j-1)/bo)+O(z^(q/bo))) end_if;
   # is there a third non zero term ? #
   for k from j+1 to n do if l[k]<>0 then break end_if end_for;
   if l[k]=0 then # there are only two non zero terms #
      l[1]*z^(v/bo)+l[j]*z^((v+j-1)/bo)+O(z^(q/bo))
   else
      a1:=l[1]*z^(v/bo);
      b1:=z^((v+j-1)/bo);
      a1+b1/Puiseux::contfrac(b1/(a-a1))
   end_if
end_proc:

Puiseux::series:=proc() local i;
begin
  Series::series(Puiseux::print(new(Puiseux,args(i)$i=1..args(0)-2)),args(args(0)-1),args(args(0)))
end_proc:

Puiseux::diff := proc(a,x)
local l,i,k,bo,ord;
begin
   bo:=BranchOrder(a);
   k:=val(a);
   ord:=err(a);
   l:=ListCoeffs(a);
   if x=variable(a) then # diff(c*x^((k+i-1)/bo),x)=x*(k+i-1)/bo*x^((k+i-1)/bo-1) #
      extsubsop(a,2=k-bo,3=ord-bo,4=[l[i]*(k+i-1)/bo$i=1..nops(l)])
   else # only differentiate coefficients #
      map(a,diff,x)
   end_if
end_proc:

unalias(BranchOrder, val, err, ListCoeffs, variable):

