# #
# $Date: 1995/06/06 16:15:53 $ $Author: frankp $ $Revision: 1.11 $ #
# #
#++
	gseries  --  a domain for generalized series expansions 

	Paul Zimmermann
	Frank Postel

	This domain is used by 'series' and 'asympt'.
++#

gseries:=domain():
gseries::name:="gseries":
gseries::info:="gseries -- generalized series expansions [try ?asympt for options]":

#--
 frankp, 15.03.1995

        gseries::new  --  computing asymptotic expansions

        gseries::new(ex, x [=v] [,ord])

        ex : expression in x
        x  : ident
	v  : expression
        ord: (optional) positive integer

        Compute the asymptotic series of ex of order ord with respect
	to x at the point v. If v is missing then x is assumed to goes to
	zero.  If ord is missing then the value of ORDER is used 
	(6 by default).
        If an expansion could not be computed, then the expression
        ex will be returned (without any bigO term).
	Otherwise an element of the domain 'gseries' is returned,
	which has 3 operands:

	1. A list containing lists of the factors for each product in ex;
	2. the bigO term
	3. the series variable.

	Example: 3*exp(1/x) + y*ln(x) + O(x^4) in x 
	         -> new(gseries, [[3,exp(1/x)],[y,ln(x)]], x^4, x)

	The algorithm based on the dissertation of Dominik Gruntz:
	"On Computing Limits in a Symbolic Manipulation System", 
	submitted to the SWISS FEDERAL INSTITUTE OF TECHNOLOGY ZUERICH
	in 1995.
--#

# force loading #
limit:
if type(ORDER)<>DOM_INT then ORDER:=6 end_if: # default order #
loadlib("Series"):

gseries::new := proc(ex,dirx)
    local n, asy, x, vx, X, _X, __LIMITx;
begin
    if args(0) < 2 or args(0) > 3 then error("wrong no of args")
    elif type(dirx) = "_equal" then
        x := op(dirx, 1); 
	if type(x) <> DOM_IDENT then
	    error("invalid series variable")
	end_if;
        vx := op(dirx, 2);
        if has(vx, x) then error("invalid arguments")
        elif vx = infinity then X := __LIMITx; _X := x
        elif vx = -infinity then X := -__LIMITx; _X := -x
	else X := 1/__LIMITx+vx; _X := 0
	end_if
    elif type(dirx) <> DOM_IDENT then error("invalid series variable") 
    else X := 1/__LIMITx; _X := 1/x
    end_if;
    if args(0) = 3 then
	n := args(3);
	if not testtype(n,Type::PosInt) then
	    error("3nd argument must be a positive integer")
	end_if
    else n := ORDER 
    end_if;

    asy := eval( subs(ex,x=X) );
    asy := gseries::asympt( 
	(stdlib::limit)::simp2explog(asy,__LIMITx), __LIMITx , n
    );
    if asy = FAIL then return( ex )
    else
        if _X <> 0 then asy := map(asy,eval@subs,__LIMITx=_X);
	elif _X = 0 then 
	    asy := map(asy,eval@subs,__LIMITx=1/__LIMITx,__LIMITx=x-vx)
	end_if;
        new(gseries,map(op(asy,1),simplify),simplify(op(asy,2)),_X)
    end_if
end_proc:

#--
  frankp, 15.03.1995

        gseries::asympt(ex,x,n)

        ex : expression in x
        x  : ident
	n  : order of series computation

        Compute the asymptotic series of ex, assuming that x goes to
        infinity. This function returns FAIL if an expansion can not
        be computed, and the list [aex,ord] otherwise where aex is
        an expansion of ex and ord the corresponding big-O term.
--#

gseries::asympt := proc(ex,__LIMITx,n)
    local S, s, w, i, ____OMEGA, nupmov, t, lc, Ex, j;
begin
    Ex := ex;  # save expression #

    userinfo(1,"(1)  Find mrv set of ",ex);

    S := (stdlib::limit)::getMRV( ex,__LIMITx );
    if S = FAIL then
        userinfo(1,"** getMRV fails");
        return( FAIL )
    elif nops(S) = 0 then
        return( [gseries::mkgseries(ex,__LIMITx),__LIMITx^ORDER] )
    end_if;

    # Upward Movement #
    nupmov := 0;
    if S intersect {__LIMITx} = {__LIMITx} then
        S := S minus {__LIMITx};
        repeat
            userinfo(2,"Upward Movement");
            ex := eval(
                subs(ex, [__LIMITx=exp(__LIMITx),ln(__LIMITx)=__LIMITx])
	    );
            S := map( S,eval@subs,
		    [__LIMITx=exp(__LIMITx),ln(__LIMITx)=__LIMITx] 
	    );
            nupmov := nupmov + 1
        until has(ex,exp(__LIMITx)) end_repeat;
        S := S union {exp(__LIMITx)}
    end_if;

    userinfo(1,"(2)  Rewrite it");

    s := (stdlib::limit)::rewrite( S,__LIMITx );
    w := op(s,1); S := op(s,2);
    ex := eval(subs(ex,op(S,i)=op(s,[3,i]) $ hold(i)=1..nops(S),w=____OMEGA));

    userinfo(1,"(3)  Compute the Puiseux series in ____OMEGA of ",ex);

    s := Series::series(ex,____OMEGA,n);
    if s = FAIL then
        userinfo(1,"** series computation fails");
        return( FAIL )
    end_if;

    # Downward Movement #
    (w := eval(
        subs(w,[__LIMITx=ln(__LIMITx),exp(__LIMITx)=__LIMITx]) )
    ) $ i=1..nupmov;
    (s := s::map( s,eval@subs,[__LIMITx=ln(__LIMITx),exp(__LIMITx)=__LIMITx]))
	$ i=1..nupmov;

    userinfo(2,"Result is ",s::expr(s));

    s := eval(subs(s,____OMEGA=w));

    if (t:=Puiseux::lmonomial(s)) = Ex then
        userinfo(2,"series approx. was an exact one");
	userinfo(1,"(4)  Asymptotic approx. is ",
	    [gseries::mkgseries(t,__LIMITx),0]
	);
        return( [gseries::mkgseries(t,__LIMITx),0] )
    end_if;

    i := extop(s,2)/extop(s,1) - 1; # ldegree(s) - 1 #
    lc := [];
    repeat
        i := i + 1;
        t := Puiseux::coeff(s,i);
        S := bool( has(t,__LIMITx) or t = FAIL );
        if not S then 
	    if not Puiseux::iszero(t) then lc := lc.[[t,w^i]] end_if
	end_if
    until S end_repeat;

    if t = FAIL then
        userinfo(2,"all coefficients are constant");
        userinfo(1,"(4)  Asymptotic approx. is ",[lc,w^Puiseux::order(s)]);
        return( [lc,w^Puiseux::order(s)] )
    end_if;

    repeat
        userinfo(1,"--> map 'gseries::asympt' to ",t);
        S := gseries::asympt(t,__LIMITx,n);
        if S = FAIL then return( FAIL )
	elif S[2] = 0 then
	    lc := lc.gseries::insFactor(S[1],w^i);
            i := i + 1;
            t := Puiseux::coeff(s,i)
        end_if
    until t = FAIL or S[2] <> 0 end_repeat;
    if t = FAIL then
	userinfo(1,"(4)  Asymptotic approx. is ",[lc,w^i]);
	[lc,w^i]
    else
        userinfo(1,"(4)  Asymptotic approx. is ", 
	    [lc.gseries::insFactor(S[1],w^i), S[2]*w^i]
        );
        [lc.gseries::insFactor(S[1],w^i), S[2]*w^i]
    end_if
end_proc:

#--
    gseries::insFactor(S,f)
--#
gseries::insFactor := fun(
    subsop(args(1),j=[(args(1))[j][1],(args(1))[j][2]*args(2)] $ hold(j)=1..nops(args(1)))

):

#--
    gseries::mkgseries(ex,x) 

    Returns a list containing respectively that coefficients of an
    addend of ex that are constant with respect to x and the product
    of all subterms of ex dependend on x.
    If ex is a constant (in respect to x), an error message occurs.
--#
gseries::mkgseries := proc(ex,__LIMITx)
    local lc, f, g, xx, cc;
begin
    userinfo(2," of ",ex);
    ex := expand(ex);
    lc := [];
    if type(ex) = "_plus" then
	for f in ex do
	    if type(f) = "_mult" then
		cc := 1; xx := 1;
		for g in f do
		    if has(g,__LIMITx) then xx := xx*g 
		    else cc := cc*g 
		    end_if
		end_for;
		lc := lc.[[cc,xx]]
	    elif has(f,__LIMITx) then lc := lc.[[1,f]]
	    else lc := lc.[[f,1]]
	    end_if
	end_for
    elif has(ex,__LIMITx) then lc := [[1,ex]] 
    else error("illegal argument")
    end_if
end_proc:

#--
    map  --  maps onto the coefficients
--#
gseries::map := proc(ex)
    local l;
begin
    l := extop(ex,1);
    ( l[j][1] := map(l[j][1],args(i) $ hold(i)=2..args(0)) )
	$ hold(j)=1..nops(l);
    extsubsop(ex,1=l)
end_proc: 

#--
    print  --  print an asymptotic series
--#
gseries::print := fun(
    (extop(args(1),1);
    if extop(args(1),2) = 0 then
	_plus(_mult(op(%[i]))$hold(i)=1..nops(%))
    else
	hold(_plus)(_mult(op(%[i]))$hold(i)=1..nops(%),hold(O)(extop(args(1),2)))
    end_if)
):

#--
    convert  --  convert a Puiseux series to an asymptotic series
--#
gseries::convert := proc(a)
local bo,k,l,i,x,ll;
begin
   if type(a)=Puiseux then
      bo:=extop(a,1);
      k:=extop(a,2);
      l:=extop(a,4);
      x:=extop(a,5);
      ll:=[];
      for i from 1 to nops(l) do
         if l[i]<>0 then ll:=append(ll,[l[i],x^((k+i-1)/bo)]) end_if
      end_for;
      new(gseries,ll,x^(extop(a,3)/bo),x)
   else error("unknown type")
   end_if
end_proc:

gseries::scalmult := proc(a,c,f) # multiplies coefficients by c and terms by f #
local l,i;
begin
   if args(0)<3 then f:=1 end_if;
   l:=extop(a,1);
   extsubsop(a,1=[[l[i][1]*c,l[i][2]*f]$i=1..nops(l)],2=extop(a,2)*f)
end_proc:

gseries::set_var := proc(a,X,x)
begin
   extsubsop(a,1=subs(extop(a,1),X=x),2=subs(extop(a,2),X=x),3=x)
end_proc:

gseries::lmonomial := proc(a)
local l;
begin
   l:=extop(a,1);
   if nops(l)=0 then FAIL else l[1][1]*l[1][2] end_if
end_proc:

# lterm(3*exp(1/x)+y*ln(x)+O(x^4)) -> exp(1/x) #
gseries::lterm := proc(a)
local l;
begin
   l:=extop(a,1);
   if nops(l)=0 then FAIL else l[1][2] end_if
end_proc:

# lcoeff(3*exp(1/x)+y*ln(x)+O(x^4)) -> 3 #
gseries::lcoeff := proc(a)
local l;
begin
   l:=extop(a,1);
   if nops(l)=0 then FAIL else l[1][1] end_if
end_proc:

gseries::coeff := proc(a,n)
local l;
begin
   if not testtype(n,Type::PosInt) then error("only positive integer allowed") end_if;
   l:=extop(a,1);
   if n>nops(l) then FAIL else l[n][1] end_if
end_proc:

gseries::nthterm := proc(a,n)
local l;
begin
   if not testtype(n,Type::PosInt) then error("only positive integer allowed") end_if;
   l:=extop(a,1);
   if n>nops(l) then FAIL else l[n][2] end_if
end_proc:

gseries::nthmonomial := proc(a,n)
local l;
begin
   if not testtype(n,Type::PosInt) then error("only positive integer allowed") end_if;
   l:=extop(a,1);
   if n>nops(l) then FAIL else l[n][1]*l[n][2] end_if
end_proc:

# error term : order(3*exp(1/x)+y*ln(x)+O(x^4)) -> x^4 #
gseries::order := proc(a) begin extop(a,2) end_proc:

gseries::expr := proc(a)
local l,i;
begin
   l:=extop(a,1);
   _plus(_mult(op(l[i]))$i=1..nops(l))
end_proc:

gseries::_plus := proc(a,b)
begin
   if args(0)=1 then return(a) end_if;
   hold(_plus)(args())
end_proc:

# returns 1 if a >> b when x->0, 0 if a=b, and -1 if a << b #
gseries::compare := proc(a,b,x)
begin
   if a=b then 0 else error("cannot get dominant function") end_if
end_proc:

gseries::_mult := proc(a,b)
local err1,err2,la,lb,ea,eb;
begin
   if args(0)=1 then return(a) end_if;
   la:=extop(a,1); lb:=extop(b,1);
   ea:=extop(a,2); eb:=extop(b,2);
   if la=[] then err1:=ea else err1:=la[1][2] end_if;
   if lb=[] then err2:=eb else err2:=lb[1][2] end_if;
   err1:=err1*eb; err2:=err2*ea;
   if gseries::compare(err1,err2,extop(a,3))=-1 then err1:=err2 end_if;
   # now err1 is the dominant function #
   if nops(la)=1 and nops(lb)=1 then extsubsop(a,1=[zip(la[1],lb[1],_mult)],2=err1)
   else hold(_mult)(a,b)
   end_if
end_proc:

