# #
# $Author: frankp $ $Date: 1995/06/11 15:45:50 $ $Revision: 1.21 $ #
# #
# frankp, 23/09/94 #

#++
  sylv.mu

       linalg::sylvester -- computes the sylvester matrix of two polynomials

       sylvester(f,g,[x])

       sylvester(f,g) computes the sylvester matrix of f and g.
       The polynomials f and g must be univariate polynomials
       both of category 'PolynomialCat(R)' or both of type DOM_POLY.

       sylvester(f,g,x) computes the sylvester matrix of f and g
       in respect to the indeterminate x.
       f and g must be both of category 'PolynomialCat' or both of 
       type DOM_EXPR or DOM_POLY.

       If the polynomials are univariate then the coefficient ring 
       of the sylvester matrix is R where R is the coefficient ring 
       of the polynomials. Are the polynomials of type DOM_POLY 
       and their coefficient ring is the ring Expr, the coefficient 
       ring of the sylvester matrix is 'ExpressionField()'. 
       Are the polynomials multivariate then the coefficient domain 
       of the sylvester matrix is the ring 'DistributedPolynomial(ind,R)',
       where ind are all the unknowns of f and g without the unknown x.
       If in this case R has not the axiom 'normalRep' then the
       ring 'Polynomial(R)' will be used.	
++#

linalg::sylvester := proc(f,g)
    name linalg::sylvester;
    local cf, cg, m, n, ffcn, gfcn, R, X, i, j;
begin
    if testargs() then
	if args(0) < 2 or args(0) > 3 then error("wrong no of args") end_if;
	if not contains( {DOM_IDENT,DOM_EXPR,DOM_POLY},domtype(f) )
	or not contains( {DOM_IDENT,DOM_EXPR,DOM_POLY},domtype(g) )
	then
	    if f::hasProp( PolynomialCat ) <> TRUE
	    or g::hasProp( PolynomialCat ) <> TRUE
	    then
		error("types of polynomials are different")
	    end_if
	end_if;
	if args(0) = 2 then
	    if contains( {DOM_IDENT,DOM_EXPR},domtype(f) )
	    or contains( {DOM_IDENT,DOM_EXPR},domtype(g) )
	    then
		error("missing indeterminate")
	    end_if;
	    if domtype(f) = DOM_POLY then
		X := op(f,2); R := op(g,2)
	    else
		X := f::indets(f); R := g::indets(g)
	    end_if;	
	    if X <> R or nops(X) <> 1 then
		error("missing indeterminate") 
	    end_if;
	    X := op( X )
	end_if;
	if args(0) = 3 then
	    X := args(3);
	    if domtype(X) <> DOM_IDENT then
		error("illegal indeterminate")
	    end_if
	end_if;
	if contains( {DOM_IDENT,DOM_EXPR},domtype(f) ) then
	    f := poly(f,[X])
	end_if;
	if contains( {DOM_IDENT,DOM_EXPR},domtype(g) ) then
	    g := poly(g,[X])
	end_if;
	if f = FAIL or g = FAIL then
	    error("unable to convert to polynomials of ".expr2text(X))
	end_if;
	if domtype(f) = DOM_POLY then
	    if op(f,3) <> op(g,3) then
		error("expecting polynomials over the same coefficient ring")
	    end_if;
	    if degree(f,X) = 0 or degree(g,X) = 0 then
		error("expecting polynomials with positive degree in ".expr2text(X))
	    end_if
	else
	    if f::coeffRing <> g::coeffRing then
		error("expecting polynomials over the same coefficient ring")
	    end_if;
	    if f::degree(f,X) = 0 or g::degree(g,X) = 0 then
		error("expecting polynomials with positive degree in ".expr2text(X))
	    end_if
	end_if
    end_if;	

    if args(0) = 2 then
	if domtype(f) = DOM_POLY then
	    X := op( op(f,2) ) 
	else
	    X := op( f::indets(f) )
	end_if
    else
	X := args(3)
    end_if;

    if f::hasProp( PolynomialCat ) = TRUE then
	m := f::degree(f,X); n := g::degree(g,X);
	cf := f::indets(f); cg := g::indets(g);
	R := f::coeffRing;
	if nops(cf) > 1 or nops(cg) > 1 then
	    if not R::hasProp( normalRep ) then R := Polynomial( R )
	    else
	        if (j := contains(cf,X)) > 0 then cf[j] := NIL end_if;
	        for j in cg do
		    if contains(cf,j) = 0 then cf := cf.[j] end_if
	        end_for;
	        R := DistributedPolynomial( cf,R )
	    end_if
	end_if;
	cf := f::coeff(f,X,m-i) $ i=0..m;
	cg := g::coeff(g,X,n-i) $ i=0..n
    else
	if domtype(f) <> DOM_POLY then f := poly(f,[X]) end_if;
        if domtype(g) <> DOM_POLY then g := poly(g,[X]) end_if;
	m := degree(f,X); n := degree(g,X);
	R := op(f,3);
	if R = hold(Expr) then
	    R := ExpressionField()
	elif type(R) = "function" then
	    if op(R,0) = hold(IntMod) then
		R := IntegerMod( op(R) )
	    else
		error("unknown coefficient ring")
	    end_if
	elif R::hasProp( CommutativeRing ) <> TRUE then
	    error("illegal coefficient ring")
	end_if;
	if not R::hasProp( normalRep ) then
	    R := Polynomial( R )
	else
	    cf := op(f,2); cg := op(g,2);
	    if nops(cf) > 1 or nops(cg) > 1 then 
                if (j := contains(cf,X)) > 0 then cf[j] := NIL end_if;
                for j in cg do
                    if contains(cf,j) = 0 and j <> X then cf := cf.[j] end_if 
                end_for;
                R := DistributedPolynomial( cf,R )
	    end_if
	end_if;
	cf := coeff(f,X,m-i) $ i=0..m;
	cg := coeff(g,X,n-i) $ i=0..n
    end_if;

    ffcn := fun( [ 0 $ args(1)-1, cf, 0 $ n-args(1) ] );
    gfcn := fun( [ 0 $ args(1)-1, cg, 0 $ m-args(1) ] );

    domattr(Matrix(R),"convert")(
      array( 1..n+m,1..n+m,[ ffcn(i) $ i=1..n, gfcn(i) $ i=1..m ] )
    )
end_proc:

# end of file #
