#-
 kamo, 15.2.95
 Network 
 Library package for working with networks (directed graphs)
-#


proc()
  local path;
begin
  loadlib("plotlib"):
  Network := domain():
  Network::name := "Network":
  Network::info := "Library 'Network': the package for handling directed graphs":
  Network::interface := {
 hold(AddEdge), hold(AddVertex), hold(AllShortPath), 
 hold(blockflow), hold(Cycle), hold(ChangeEdge), 
 hold(ChangeVertex), hold(Complete), hold(ConvertSSQ), 
 hold(ECapacity), hold(EWeight),
 hold(DelEdge), hold(DelVertex), hold(Edge), hold(Epost), hold(Epre), 
 hold(InDegree), hold(IsEdge), hold(IsVertex), hold(LongPath), 
 hold(MaxFlow), hold(MinCost), hold(MinCut), hold(OutDegree), hold(PrintGraph), 
 hold(Random), hold(ShortPath), hold(ShortPathTo), hold(ShowGraph), 
 hold(TopSort), hold(Vertex), hold(VWeight)
	}:
  
  
#-
 new - generates a new network
 new(V,Ed)
 new(V,Ed, <EWeight=[c1,..,cm]>,<Capacity=[t1,..,tm]>,<Vweight=[v1,..,vn]>)
 Parameters:
   V - list of nodes 
   Ed - list of edges
 If Ed = [e1,..,em] then the cost (weight) of arc ei is ci and the capacity
 is ti. The node weights are specified by the Vweight list. If the lists 
 Eweight, Capacity and Vweight contains fewer elements than Ed and V respectively,
 then the cost and capacity lists are filled up with 1 and 0 resp., and the 
 vertex weigth list is filled up with 0.
-#
Network::new := proc(V,Edges)
local Epr, Epo, e, i, j, lhs, rhs, Vw, Ecap, Ew, arg;
begin
#--
 Creates a new network.
 Parameters:
   1 - V, list of nodes
   2 - Edges, list of edges
   3,... - equations 
	Vweight = [...] - weights for the nodes
	Capacity = [...] - capacities for the edges
	Eweight   = [...] - costs for the edges

 Data structure:
	new(network, V, Edges, Vw, Ecap, Ew, Epo, Epr);
  (1)  V - list of nodes
  (2)  Edges - list of edges
  (3)  Vw - weights of nodes
  (4)  Ecap - edge capacities
  (5)  Ew - edge costs
  (6)  Epo - adjacency lists (outgoing)
  (7)  Epr - adjacency lists (incoming)
--#

   if testargs() then
      if args(0) < 2 then
	   error("Wrong number of arguments!");
      end_if;
      if not testtype(V,DOM_LIST) then
	   error("Wrong argument type: list expected!");
      end_if;
      if not testtype(Edges, Type::ListOf(DOM_LIST)) then
	   error("Wrong argument type: list of lists expected!");
      end_if;
   end_if;

   for i from 3 to args(0) do
	arg := args(i);
	if not testtype(arg, "_equal") then
		error("Wrong argument type: equation expected!");
	end_if;
	lhs := op(arg,1); rhs := op(arg,2);
	case lhs 
	   of hold(Vweight) do
		case type(rhs)
		   of DOM_TABLE do 
		       Vw := {op(rhs,[1..nops(rhs),1])}; 
		       if {op(V)} <> Vw then
			      error("Illegal nodes specified!");
		       end_if;
		       Vw := rhs;
		       break;
		   of DOM_LIST do 
		       if nops(rhs) < nops(V) then
			      rhs := rhs.[0 $ j=1..(nops(V)-nops(rhs))];
		       end_if;
		       Vw := table(V[j] = rhs[j] $ j=1..nops(V));
		       break;
		   otherwise
			   error("Wrong argument type: list or table expected!");
		end_case;

		break;
	   of hold(Capacity) do
		case type(rhs)
		   of DOM_LIST do
		       if nops(rhs) < nops(Edges) then
			      rhs := rhs.[0 $ (nops(Edges)-nops(rhs))];
               end_if;
		       Ecap := table(Edges[j] = rhs[j] $ j=1..nops(Edges));
		       break;
		   of DOM_TABLE do
		       Ecap := {op(rhs,[1..nops(rhs),1])}; 
		       if {op(Edges)} <> Ecap then
			      error("Illegal edges specified!");
		       end_if;
		       Ecap := rhs;
		       break;
		   otherwise
			   error("Wrong argument type: list or table expected!");
		end_case;
		break;
	   of hold(Eweight) do
		case type(rhs)
		   of DOM_LIST do
		       if nops(rhs) < nops(Edges) then
			      rhs := rhs.[0 $ (nops(Edges)-nops(rhs))];
               end_if;
		       Ew := table(Edges[j] = rhs[j] $ j=1..nops(Edges));
		       break;
		   of DOM_TABLE do
		       Ew := {op(rhs,[1..nops(rhs),1])}; 
		       if {op(Edges)} <> Ew then
			      error("Illegal edges specified!");
		       end_if;
		       Ew := rhs;
		       break;
		   otherwise
			   error("Wrong argument type: list or table expected!");
		end_case;
		break;
	   otherwise
		error("Unknown type!");
	end_case;
   end_for;

   i := NIL;
   #-
    If there are no node weights specified, assign the 
    weight 0 to every node.
   -#
   if Vw = hold(Vw) then
	Vw := table(V[i] = 0 $ i=1..nops(V));
   end_if;

   #-
    If there are no edge capacities specified, assign the 
    capacity 1 to every edge.
   -#
   if Ecap = hold(Ecap) then
	Ecap := table(Edges[i] = 1 $ i=1..nops(Edges));
   end_if;

   #-
    If there are no edge costs specified, assign the 
    cost 1 to every node.
   -#
   if Ew = hold(Ew) then
	Ew := table(Edges[i] = 1 $ i=1..nops(Edges));
   end_if;

   #-
    Generate the adjacency lists
   -#
   Epr := table((V[i]=[] $ i=1..nops(V))):
   Epo := Epr;
   for e in Edges do
        Epr[e[2]] := append(Epr[e[2]], e[1]);
        Epo[e[1]] := append(Epo[e[1]], e[2]);
   end_for;

   new(Network, V, Edges, Vw, Ecap, Ew, Epo, Epr);
end_proc:

#-
 Output of networks
-#
Network::print := proc(G)
begin 
   hold(Network())
end_proc:

#- 
 testtype  - test if expression is a Network 
-#
Network::testtype:= proc(e,T) begin
    if T = Network then
        if domtype(e) = Network then
            return(TRUE)
        end_if
    end_if;
    FAIL
end_proc:  
  
 
  path := pathname("NETWORK"):


  Network::AddEdge := loadproc(Network::AddEdge, path, "addedge"):
  Network::AddVertex := loadproc(Network::AddVertex, path, "addvert"):
  Network::AllShortPath := loadproc(Network::AllShortPath, path, "allspath"):
  Network::blockflow := loadproc(Network::blockflow, path, "blockflow"):
  Network::ChangeVertex := loadproc(Network::ChangeVertex, path, "chvertex"):
  Network::ChangeEdge := loadproc(Network::ChangeEdge, path, "chedge"):
  Network::Complete := loadproc(Network::Complete, path, "complete"):
  Network::ConvertSSQ := loadproc(Network::ConvertSSQ, path, "convssq"):
  Network::Cycle := loadproc(Network::Cycle,  path, "cycle"):
  Network::DelEdge := loadproc(Network::DelEdge, path, "deledge"):
  Network::DelVertex := loadproc(Network::DelVertex, path, "delvert"):
  Network::ECapacity := loadproc(Network::ECapacity, path, "ecap"):
  Network::Edge := loadproc(Network::Edge, path, "edge"):   
  Network::Epost := loadproc(Network::Epost, path, "epost"):
  Network::Epre := loadproc(Network::Epre, path, "epre"):
  Network::EWeight := loadproc(Network::EWeight, path, "eweight"):
  Network::InDegree := loadproc(Network::InDegree, path, "indegre"):
  Network::IsEdge := loadproc(Network::IsEdge, path, "isedge"):
  Network::IsVertex := loadproc(Network::IsVertex, path, "isvertex"):
  Network::LongPath := loadproc(Network::LongPath, path, "lpath"):
  Network::MaxFlow := loadproc(Network::MaxFlow, path, "maxflow"):
  Network::MinCost := loadproc(Network::MinCost, path, "mincost"):
  Network::MinCut := loadproc(Network::MinCut, path, "mincut"):
  Network::OutDegree := loadproc(Network::OutDegree, path, "outdegre"):
  Network::PrintGraph := loadproc(Network::PrintGraph, path, "printnet"):
  Network::Random := loadproc(Network::Random, path, "random"):
  Network::ShortPath := loadproc(Network::ShortPath, path, "spath"):
  Network::ShortPathTo := loadproc(Network::ShortPathTo, path, "sspath"):
  Network::ShowGraph := loadproc(Network::ShowGraph, path, "showgr"):
  Network::TopSort := loadproc(Network::TopSort, path, "topsort"):
  Network::Vertex := loadproc(Network::Vertex, path, "vertex"):
  Network::VWeight := loadproc(Network::VWeight, path, "vweight"):
   
# functions without interface: #
  Network::graph_check := loadproc(Network::graph_check, path, "gc"):
  
# for AllShortPath: #
  Network::floyd := loadproc(Network::floyd, path, "floyd"):

# for LongPath: #
  Network::lbellman := loadproc(Network::lbellman, path, "lbellman"):

# for ShortPath: #
  Network::bellman := loadproc(Network::bellman, path, "bellman"):
  Network::dijkstra := loadproc(Network::dijkstra, path, "dijkstra"):
  Network::rbellman := loadproc(Network::rbellman, path, "rbellman"):
  Network::rdijkstra := loadproc(Network::rdijkstra, path, "rdijkstra"):
  Network::spath_arg_check := loadproc(Network::spath_arg_check, path, "spargc"):
  Network::spath_eval := loadproc(Network::spath_eval, path, "speval"):
  
# for MinCost: #
  Network::relaxation := loadproc(Network::relaxation, path, "relax"):
  
# for MinCost: #
  Network::pp_fifo := loadproc(Network::pp_fifo, path, "pp_fifo"):
  
  null():
end_proc():


