
# Control routine for numeric integration.				#
#									#
# References:								#
# [1] Numerical Integration in a Symbolic Context, K.O. Geddes,  	#
#     pp. 185 - 191, Proceedings of SYMSAC '86, ACM, 1986.		#
# [2] Numerical Integration using Symbolic Analysis, K.O. Geddes,	#
#     pp. 8 - 17, Maple Technical Newsletter 6, 1991			#
#									#
# Input:								#
# f -- an expression (integrand)					#
# y -- an equation of the form x=a..b . 				#
#									#
# Output:								#
# The value of the integral or FAIL					#


intlib::numint :=  proc(f,y)
   local DIGITS, a, b, eps, x;
begin   
   if type(args(2)) <> "_equal" then
      return(FAIL)
   end_if; 
   eps := 10^(2-DIGITS);
   x := op(y, 1); a := op(y,[2,1]); b := op(y,[2,2]);
   intlib::control(f, x, a, b, eps);
end_proc:

intlib::control :=  proc(f, x, a, b, eps)
   local c, DIGITS, err, ffunc, h, l, left, 
         nofun, pleft, pright, r, right, tol;
begin

if testtype(f, DOM_PROC) then
   # We have to modify this control routine to	#
   # handle user defined procedures/functions.	#
   return(FAIL);
end_if;

if has({a,b}, [infinity, -infinity]) then
        return( intlib::improper(f,x,a,b,eps) )
end_if;

if not testtype(a, NUMERIC) or not testtype(b, NUMERIC) then
   return(FAIL)
end_if;

userinfo(1,"integrating on ", a, "..", b, "the integrand ", f);

left := float(a);  right := float(b);  h := float(b-a);
if abs(h) < 10^(-DIGITS) then 
   return( 0.0 )
elif not testtype(left,NUMERIC) or not testtype(right,NUMERIC) then 
   return( FAIL )
elif left > right then 
   return( intlib::control(-f,x,b,a,eps) )
end_if;

pleft  := intlib::taylor(f, x=a); 
pright := intlib::taylor(f, x=b);

# If not "pure Taylor series" then there is an end-point singularity. 	#

if pleft = FAIL and pright = FAIL then
        # Singularity in both endpoints.                 #  
        # Split interval in two, a..(a+b)/2 , (a+b)/2..b #
        userinfo(1, "splitting interval into ", a, "..", (a+b)/2, "and ", (a+b)/2, "..", b);
	if h < sqrt(eps) then
	    error("integrand is apparently singular everywhere")
	end_if;
        DIGITS:=DIGITS+2;
	return( intlib::control(f,x,a,(a+b)/2,eps/2)
		+  intlib::control(f,x,(a+b)/2,b,eps/2) )

elif pright = FAIL then
        # Singularity in the right endpoint.                 #
	# Transform singularity to be at the left end-point. #
        userinfo(1, "singularity at right end-point. transforming to the interval 0..", b-a);
	return( intlib::control(subs(f, x=b-x),x,0,b-a,eps) )

elif pleft = FAIL then
	# Deal with singularity at the left end-point. 		#
        userinfo(1, "singularity at left end-point.");
	if left <> 0.0 then
	    # Transform left end-point to zero. 		#
            userinfo(1, "transforming to the interval 0..", b-a);
	    return( intlib::control(subs(f,x=x+a),x,0,b-a,eps) )
	else
            # Not implemented yet.              		#
	    # return( intlib::singleft(f,x,right,eps) ) 	#
	    return(FAIL)
	end_if
else
	# Attempt numerical integration. 			#

        # Before calling the numerical integrator it is useful	#
      	# to analyze the integrand for removable singularities	#
	# in the interval and for accurate evaluation (for	#
	# example to avoid cancellation in floating-point	#
	# arithmetic. For example: ln(1-cos(2*x))-2*ln(x) is 	#
	# analytic in [0,1], but it has a singularity at x=0.	#
	# See [1], makeproc. This isn't implemented yet. 	#

	# Convert expression f to a functional expression.	#

	ffunc := subs(hold(func(f)), hold(f)=f); ffunc := level(ffunc,2);

	userinfo(1, "calling Clenshaw-Curtis Quadrature.");
        if traperror( (
              l := intlib::ccquad(ffunc,left,right,eps,487)
           ) ) <> 0 then
           userinfo(1, "singularity in or near interval of integration");
           return(FAIL)
        end_if;
        r := l[1]; err := l[2]; nofun := l[3]; c := l[4];
	if r = FAIL then
            
	    if traperror( (
                 osc := intlib::oscillate(ffunc, left, right, 500, 10)
               ) ) <> 0 then
               userinfo(1, "singularity in or near interval of integration");
	       # Not implemented yet.                           #
               # return( intlib::singnear(f,x,left,right,eps) ) #
               return(FAIL) 
	    elif osc = FAIL then
               userinfo(1, "function apparently oscillates infinitely");
	       return(FAIL)
            else
               userinfo(1, "oscillating integral detected.");
	       userinfo(1, "splitting interval into ", a, "..", (a+b)/2, "and ", (a+b)/2, "..", b);
	       DIGITS:=DIGITS+2;
	       return( intlib::control(f,x,a,(a+b)/2,eps/2)
                +  intlib::control(f,x,(a+b)/2,b,eps/2) )
	    end_if
	end_if;

	# Calculate error tolerance, with eps as a relative error	#
	#   criterion;  if too small then use absolute error.		#
	tol := max(eps*abs(r), 0.001*eps);
        userinfo(1, "error = ", err, ", error tolerance = ", tol);
	if err <= tol then 
	   return( r ) 
        else
	   error("numerical integration failed to converge")
	end_if
end_if
end_proc:

intlib::taylor := proc() 
   local s;
begin
   s := series(args()); 
   if s = FAIL then
      return(FAIL)
   elif testtype(s, Type::Series(Taylor)) then
      return(s)
   else
      return(FAIL)
   end_if
end_proc:
