
# Handle improper integrals, i.e. integrals where the lower and/or 	#
# upper limit of integration is infinite.				#
#									#
# References:                                                           #
# [1] Numerical Integration in a Symbolic Context, K.O. Geddes,         #
#     pp. 185 - 191, Proceedings of SYMSAC '86, ACM, 1986.              #
# [2] Numerical Integration using Symbolic Analysis, K.O. Geddes,       #
#     pp. 8 - 17, Maple Technical Newsletter 6, 1991                    #

intlib::improper :=  proc(f, x, a, b, eps)
   	local DIGITS, lower, newf;
begin
	userinfo(1, "integrating on interval ", a, "..", b);
	if has(a,[infinity, -infinity]) and 
           not has(b,[infinity, -infinity]) then
	    intlib::improper(-f,x,b,a,eps)
	elif has(a,[infinity, -infinity]) and 
             has(b,[infinity, -infinity]) then
	    if a=b then
		error("integration limits are both plus (or minus) infinity")
	    end_if;
	    DIGITS:=DIGITS+2;
	    intlib::improper(-f,x,0,a,eps/2) + intlib::improper(f,x,0,b,eps/2)
	elif has(b,[infinity, -infinity]) then
	    if b = -infinity then
		return(intlib::improper(-subs(f, x=-x), x, -a, -b, eps))
	    end_if;
	    lower := float(a);
	    if not testtype(lower,NUMERIC) then
		error("non-numeric integration limit encountered")
	    end_if;
	    if lower < 0.0 then
		DIGITS:=DIGITS+2;
		intlib::control(f,x,lower,0,eps/2) 
                + intlib::improper(f,x,0,b,eps/2)
	    else
		userinfo(1, "applying transformation x = ", 1/x);
		newf := normal(subs(f, x=1/x) * (-1/x^2));
		if abs(lower) < 10^(-DIGITS) then # lower = 0 ? #
		    userinfo(1, "interval is 0..1 for the two functions ", f, " and ", -newf);
		    DIGITS:=DIGITS+2;
		    intlib::control(f,x,0,1,eps/2) 
                    + intlib::control(-newf,x,0,1,eps/2)
		else
		    userinfo(1, "interval is 0..", 1/lower, " for the function ",  -newf);
		    intlib::control(-newf,x,0,1/lower,eps)
		end_if
	    end_if
	else
	    error("expecting an infinite integration limit")
	end_if
end_proc:
