
# Reference:								#
# [1] Algorithm 424: Clenshaw-Curtis Quadrature [D-1], W.M. Gentleman,	#
#     pp. 353-355, Communications of the ACM, Vol. 15, Nr. 5, 1972.	#
# [2] Remark on Algorithm 424: Clenshaw-Curtis Quadrature, K.O. Geddes, #
#     ACM Transactions on Mathematical Software, Vol. 5, Nr. 2, 1979.	#

# DESCRIPTION-								#
# USING CLENSHAW-CURTIS QUADRATURE, THIS FUNCTION SUBPROGRAM ATTEMPTS	#
# TO INTEGRATE THE FUNCTION f FROM a TO b TO AT LEAST THE REQUESTED	#
# RELATIVE ACCURACY tolerr, WHILE USING NO MORE THAN limit FUNCTION	#
# EVALUATIONS. IF THIS CAN BE DONE, intlib::ccquad RETURNS THE VALUE 	#
# NEWINT OF THE INTEGRAL, ESTERR RETURNS AN ESTIMATE OF THE ABSOLUTE	#
# ERROR ACTUALLY COMMITTED, USED RETURNS THE NUMBER OF FUNCTION VALUES 	#
# ACTUALLY USED, AND csxfrm[1], ..., csxfrm[used] CONTAINS N = USED-1	#
# TIMES THE DISCRETE COSINE TRANSFORM, AS USUALLY DEFINED, OF THE 	#
# INTEGRAND IN THE INTERVAL.  IF THE REQUESTED ACCURACY CANNOT BE	#
# ATTAINED WITH THE NUMBER OF FUNCTION EVALUATIONS PERMITTED, THE LAST	#
# (AND PRESUMABLY BEST) ANSWER OBTAINED IS RETURNED.			#
# REGARDING THE CHOICE OF THE VALUE "limit", IT SHOULD BE NOTED THAT	#
# THE NUMBER OF FUNCTION EVALUATIONS ACTUALLY USED WILL ALWAYS BE A 	#
# NUMBER OF THE FORM  2*3^M + 1  AND THEREFORE A GOOD VALUE FOR limit 	#
# MIGHT BE 163 OR 487. IF f IS NOT ANALYTIC IN A SUFFICIENTLY LARGE	#
# REGION SURROUNDING THE INTERVAL [a,b] THEN THE VALUE FAIL IS RETURNED.#

# This algorithm is an adaption of the algorithm implemented by 	#
# K.O. Geddes (see [2]), which is based on the Fortran implemention	#
# described in [1].							#

# Input:								#
# f 	 - a function or procedure, the integrand.			#
# a, b   - finite closed interval on which integral is taken.		#
# tolerr - requested relative accuracy.					#
# limit  - maximal number of function/procedure evaluations.		#
#									#
# * If f is an expression, for example sin(x)+cos(x)+Pi, you have to 	#
#   use func(f) instead of f.						#
# * Choose 10^(2-DIGITS) <= tolerr <= 10^(-5). If tolerr > 10^(-5)  	#
#   then tolerr := 10^(-5), if tolerr < 10^(2-DIGITS), then 		#
#   tolerr := 10^(2-DIGITS). Increase DIGITS if your requested relative	#
#   accuracy is smaller than 10^(2-DIGITS).				#
#									#
# Output:								#
# [ NEWINT, ESTERR, USED, csxfrm ] or [ FAIL, ESTERR, USED, csxfrm ]	#
#									#
# Note: f must be continuous in [a, b]. No check is done.		#
# Use traperror to trap an error condition during evaluation of f: 	#
#									#
# if traperror((l := ccquad(f, a, b, eps, 487))) <> 0 then		#
#    userinfo(1, "singularity in interval of integration");		#
#    return(FAIL)							#
# end_if;								#
# r := l[1]; err := l[2]; nofun := l[3]; c := l[4];			#
# if r = FAIL then							#
#    userinfo(1, "singularity in or near interval of integration");	#
#    return(FAIL)							#
# end_if;								#
#									#
# How to use the estimated error:					#
# Calculate error tolerance, with eps as a relative error criterion;	#
# if too small then use absolute error.           			#
# tol := max(eps*abs(r), 0.001*eps);					#
# userinfo(1, "error = ", err, ", error tolerance = ", tol);		#
# if err <= tol then							#
#    return( r )							#
# else									#
#    userinfo(1, "numerical integration failed to converge");		#
#    return(FAIL)							#
# end_if								#
#									#
# if ccquad return FAIL, or the estimated error is greater than requested 	#
# relative accuracy, if f is an oscilating function, for example sin(x)	#
# in the interval from 0 to 100. Split interval to achieve the requested#
# relative accuracy, for example split into [0,25], [25,50], [50,75] 	#
# [75,100] and integrate f on each interval.				#

intlib::ccquad := proc(f,a,b,tolerr,limit)
   local TOLERR,ESTERR,USED,log2TOL,maxcoef,K,tol,RT3,CENTRE,WIDTH,SHIFT,FUND,
         ANGLE,C,S,OLDINT,NEWINT,T1,T2,T3,T4,T5,T6,T7,T8,T9,T10,T11,T12,N,N2,N3,
         N_LESS_1,N_LESS_3,MAX,M_MAX,J,STEP,L,J1,J2,J3,J4,J5,J6,J7,J8,J_REV,
         csxfrm;
begin

    #		INITIALIZATION		#

    a := float(a); b := float(b);
    TOLERR := min(tolerr,0.1*10^(-4));
    TOLERR := max(TOLERR,10.0^(2-DIGITS)); 
    log2TOL := float(ln(TOLERR)/ln(2));
    RT3 := float(sqrt(3));
    M_MAX := 8;
    CENTRE := 0.5*a+0.5*b;
    WIDTH := 0.5*b-0.5*a;
    MAX := min(limit,2*3^(M_MAX+1));
    L := array(1 .. M_MAX);
    for J from 1 to M_MAX do  L[J] := 1 end_for;

    #		COSINE TRANSFORM			#
    # COMPUTE DOUBLE THE COSINE TRANSFORM WITH N = 6	#

    N := 6;
 
    # 		SAMPLE FUNCTION		#

    csxfrm[1] := f(b);
    csxfrm[7] := f(a);
    SHIFT := -0.5*WIDTH*RT3;
    csxfrm[2] := f(CENTRE-SHIFT);
    csxfrm[6] := f(CENTRE+SHIFT);
    SHIFT := -0.5*WIDTH;
    csxfrm[3] := f(CENTRE-SHIFT);
    csxfrm[5] := f(CENTRE+SHIFT);
    csxfrm[4] := f(CENTRE);

    # EVALUATE THE FACTORED N = 6 COSINE TRANSFORM	#

    T1 := csxfrm[1]+csxfrm[7];
    T2 := csxfrm[1]-csxfrm[7];
    T3 := 2.0*csxfrm[4];
    T4 := csxfrm[2]+csxfrm[6];
    T5 := (csxfrm[2]-csxfrm[6])*RT3;
    T6 := csxfrm[3]+csxfrm[5];
    T7 := csxfrm[3]-csxfrm[5];
    T8 := T1+2.0*T6;
    T9 := 2.0*T4+T3;
    T10 := T2+T7;
    T11 := T1-T6;
    T12 := T4-T3;
    csxfrm[1] := T8+T9;
    csxfrm[2] := T10+T5;
    csxfrm[3] := T11+T12;
    csxfrm[4] := T2-2.0*T7;
    csxfrm[5] := T11-T12;
    csxfrm[6] := T10-T5;
    csxfrm[7] := T8-T9;
    USED := 7;

    # 	EVALUATE ESTIMATE OF INTEGRAL	#

    OLDINT := -0.01428571429*csxfrm[7]-1/15*csxfrm[5]
              -1/3*csxfrm[3]+0.5*csxfrm[1];
    OLDINT := 1/3*WIDTH*OLDINT;

    # COMPUTE REFINED APPROXIMATION				#
    # SAMPLE FUNCTION AT INTERMEDIATE POINTS IN DIGIT REVERSED	#
    # ORDER. AS THE SEQUENCE IS GENERATED, COMPUTE THE FIRST	#
    # (RADIX FOUR TRANSFORM) PASS OF THE FAST FOURIER TRANSFORM.#

    repeat
        for J from 2 to M_MAX do  L[J-1] := L[J] end_for;
        L[M_MAX] := 3*L[M_MAX-1];
        J := USED;
        FUND := 1/3*PI/N;
        for J1 from 1 to L[1] do
            for J2 from J1 to L[2] step L[1] do
                for J3 from J2 to L[3] step L[2] do
                    for J4 from J3 to L[4] step L[3] do
                        for J5 from J4 to L[5] step L[4] do
                            for J6 from J5 to L[6] step L[5] do
                                for J7 from J6 to L[7] step L[6] do
                                    for J8 from J7 to L[8] step L[7] do
                                    ANGLE := FUND*(3*J8-2);
                                    SHIFT :=
                                        -WIDTH*float(sin(1/2*PI-ANGLE));
                                    T1 := f(CENTRE-SHIFT);
                                    T3 := f(CENTRE+SHIFT);
                                    SHIFT := -WIDTH*float(sin(ANGLE));
                                    T2 := f(CENTRE+SHIFT);
                                    T4 := f(CENTRE-SHIFT);
                                    T5 := T1+T3;
                                    T6 := T2+T4;
                                    csxfrm[J+1] := T5+T6;
                                    csxfrm[J+2] := T1-T3;
                                    csxfrm[J+3] := T5-T6;
                                    csxfrm[J+4] := T2-T4;
                                    J := J+4
                                    end_for
                                end_for
                            end_for
                        end_for
                    end_for
                end_for
            end_for
        end_for;

        # DO RADIX 3 PASSES OF FAST FOURIER TRANSFORM	#

        N2 := 2*N;
        STEP := 4;
        while STEP < N do
            J1 := USED+STEP;
            J2 := USED+2*STEP;
            csxfrm := intlib::R3PASS(N2,STEP,csxfrm,USED,J1,J2);
            STEP := 3*STEP
        end_while;

	# 	COMBINE RESULTS		#
	# FIRST DO J = 0 AND J = N	#

        T1 := csxfrm[1];
        T2 := csxfrm[USED+1];
        csxfrm[1] := T1+2.0*T2;
        csxfrm[USED+1] := T1-T2;
        T1 := csxfrm[N+1];
        T2 := csxfrm[N2+2];
        csxfrm[N+1] := T1+T2;
        csxfrm[N2+2] := T1-2.0*T2;

	# NOW DO REMAINING VALUES OF J	#

        N3 := 3*N;
        N_LESS_1 := N-1;
        for J from 1 to N_LESS_1 do
            J1 := N+J;
            J2 := N3-J;
            ANGLE := FUND*J;
            C := float(sin(1/2*PI-ANGLE));
            S := float(sin(ANGLE));
            T1 := C*csxfrm[J1+2]-S*csxfrm[J2+2];
            T2 := (S*csxfrm[J1+2]+C*csxfrm[J2+2])*RT3;
            csxfrm[J1+2] := csxfrm[J+1]-T1-T2;
            csxfrm[J2+2] := csxfrm[J+1]-T1+T2;
            csxfrm[J+1] := csxfrm[J+1]+2.0*T1
        end_for;

	# 	NOW UNSCRAMBLE		#

        T1 := csxfrm[N2+1];
        T2 := csxfrm[N2+2];
        for J from 1 to N_LESS_1 do
            J1 := USED+J;
            J2 := N2+J;
            csxfrm[J2] := csxfrm[J1];
            csxfrm[J1] := csxfrm[J2+2]
        end_for;
        csxfrm[N3] := T1;
        csxfrm[N3+1] := T2;
        N := N3;
        USED := N+1;

	#	INTEGRAL EVALUATION		#
	#					#
	# EVALUATE NEW ESTIMATE OF INTEGRAL	#

        N_LESS_3 := N-3;
        NEWINT := 0.5*csxfrm[USED]/(1-N^2);
        for J from 1 to N_LESS_3 step 2 do
            J_REV := N-J; NEWINT := NEWINT+csxfrm[J_REV]/J_REV/(2-J_REV)
        end_for;
        NEWINT := NEWINT+0.5*csxfrm[1];
        NEWINT := 2*WIDTH*NEWINT/N;

	# 	TEST IF DONE			#
	# Determine where the coefficients start#
	# to be negligible in magnitude.	#

        maxcoef := max(abs(csxfrm[1]),abs(csxfrm[2]),abs(csxfrm[3]),
            abs(csxfrm[4]),abs(csxfrm[5]));
	K := USED;
        while K >= 8 and abs(csxfrm[K]) < TOLERR*maxcoef do 
              K := K-1 
        end_while;

	# For an integrand analytic in a sufficient region surrounding 	#
	# the interval, the coefficients should decrease no slower than #
	# C * 2^(-i). Thus we expect K <= -log2(TOLERR). Allow it to be	#
	# 1.5 times larger.						#

        if -1.5*log2TOL < K then
            return( [ FAIL, NEWINT, USED, csxfrm ] );
        end_if;
	
	# The error estimate is 8 * abs(A[N]) / N^3 (see Gentleman's 	#
	# paper). We assume  abs(A[i]) <= C * 2^(-i)  and we estimate	#
	# N*C from csxfrm[i], i=K-4..K. Recall that csxfrm[i] = N*A[i-1]#

        ESTERR := max(2^(K-5)*abs(csxfrm[K-4]),2^(K-4)*abs(csxfrm[K-3]),
            2^(K-3)*abs(csxfrm[K-2]),2^(K-2)*abs(csxfrm[K-1]),
            2^(K-1)*abs(csxfrm[K]))/2^(N-3)/N^4;
        ESTERR := min(ESTERR,abs(NEWINT-OLDINT));
        ESTERR := max(ESTERR,10.0^(2-DIGITS)*abs(NEWINT));
        tol := max(TOLERR*abs(NEWINT),0.001*TOLERR);
	OLDINT := NEWINT;

	# IF REFINEMENT NOT PERMITTED, OR IF ESTIMATED ERROR 	#
	# SATISFACTORY, RETURN.					#

    until (ESTERR <= tol) or (MAX < 3*N+1) end_repeat;

    [ NEWINT, ESTERR, USED, csxfrm ]

end_proc:


# RADIX 3 PASS FOR FAST FOURIER TRANSFORM OF REAL SEQUENCE OF LENGTH N2	#
# M IS THE LENGTH OF THE TRANSFORM ALREADY ACCOMPLISHED, I.E.THE NUMBER	#
# OF DISTINCT VALUES OF THE FREQUENCY INDEX C HAT OF THESE TRANSFORMS,	#
# AND THE SPACING OF THE SEQUENCES TO BE TRANSFORMED.  EXPLICIT USE IS	#
# MADE OF THE FACT THAT M IS EVEN AND NOT LESS THAN FOUR.		#

intlib::R3PASS := proc(N2,M,csxfrm,SHIFT0,SHIFT1,SHIFT2)
   local M_OVER_2,M3,K,K0,K1,J,J0,J1,HAFRT3,RSUM,RDIFF,RSUM2,ISUM,
         IDIFF,IDIFF2,FUND,ANGLE,C1,S1,C2,S2,R0,R1,R2,I0,I1,I2;
begin
    HAFRT3 := 1/2*float(sqrt(3));
    M_OVER_2 := (M-1) div 2;
    M3 := 3*M;
    FUND := 2*PI/M3;

    # DO ALL TRANSFORMS FOR C HAT = 0, I.E. TWIDDLE FACTOR UNITY        #

    for K from 1 to N2 step M3 do
        RSUM := csxfrm[SHIFT1+K]+csxfrm[SHIFT2+K];
        RDIFF := (csxfrm[SHIFT1+K]-csxfrm[SHIFT2+K])*HAFRT3;
        csxfrm[SHIFT1+K] := csxfrm[SHIFT0+K]-0.5*RSUM;
        csxfrm[SHIFT2+K] := RDIFF;
        csxfrm[SHIFT0+K] := csxfrm[SHIFT0+K]+RSUM
    end_for;

    # DO ALL TRANSFORMS FOR C HAT = CAP C/2, I.E. TWIDDLE FACTOR E(B/6)	#

    J := 1/2*M+1;
    for K from J to N2 step M3 do
        RSUM := (csxfrm[SHIFT1+K]+csxfrm[SHIFT2+K])*HAFRT3;
        RDIFF := csxfrm[SHIFT1+K]-csxfrm[SHIFT2+K];
        csxfrm[SHIFT1+K] := csxfrm[SHIFT0+K]-RDIFF;
        csxfrm[SHIFT2+K] := RSUM;
        csxfrm[SHIFT0+K] := csxfrm[SHIFT0+K]+0.5*RDIFF
    end_for;

    # DO ALL TRANSFORMS FOR REMAINING VALUES OF C HAT.  OBSERVE THAT C	#
    # HAT AND CAP C-C HAT MUST BE PAIRED. CHOOSE A FREQUENCY INDEX.	#

    for J from 1 to M_OVER_2 do
        J0 := J+1;
        J1 := M-J+1;

	# COMPUTE THE TWIDDLE FACTOR	#

        ANGLE := FUND*J;
        C1 := float(sin(1/2*PI-ANGLE));
        S1 := float(sin(ANGLE));
        C2 := C1^2-S1^2;
        S2 := 2.0*S1*C1;

	# CHOOSE THE REPLICATION	#

        for K0 from J0 to N2 step M3 do
            K1 := K0-J0+J1;
	    
	    # OBTAIN TWIDDLED VALUES	#

            R0 := csxfrm[SHIFT0+K0];
            I0 := csxfrm[SHIFT0+K1];
            R1 := C1*csxfrm[SHIFT1+K0]-S1*csxfrm[SHIFT1+K1];
            I1 := S1*csxfrm[SHIFT1+K0]+C1*csxfrm[SHIFT1+K1];
            R2 := C2*csxfrm[SHIFT2+K0]-S2*csxfrm[SHIFT2+K1];
            I2 := S2*csxfrm[SHIFT2+K0]+C2*csxfrm[SHIFT2+K1];
	
	    # COMPUTE TRANSFORMS AND RETURN IN PLACE	#

            RSUM := R1+R2;
            RDIFF := (R1-R2)*HAFRT3;
            RSUM2 := R0-0.5*RSUM;
            ISUM := I1+I2;
            IDIFF := (I1-I2)*HAFRT3;
            IDIFF2 := I0-0.5*ISUM;
            csxfrm[SHIFT0+K0] := R0+RSUM;
            csxfrm[SHIFT0+K1] := RSUM2+IDIFF;
            csxfrm[SHIFT1+K0] := RSUM2-IDIFF;
            csxfrm[SHIFT1+K1] := RDIFF+IDIFF2;
            csxfrm[SHIFT2+K0] := RDIFF-IDIFF2;
            csxfrm[SHIFT2+K1] := I0+ISUM
        end_for
    end_for;
    csxfrm
end_proc:
