# $Date: 1995/05/26 07:58:56 $ $Author: kg $ $Revision: 1.16 $ #

#++
TruncatedPowerSeries -- the constructor of truncated power series domains

TruncatedPowerSeries([x [, R]])

x - unknown (indeterminate)
R - coefficient field

Representation:-
1- list of coefficients
2- valence (integer)
3- truncated series? (boolean)
4- branching order (integer)

The order of the series is (nops(coeffs) + valence)).
The coefficient list never is empty. A pure order term O(t^n) is
represented as new(this, [ CF::zero ], n-1, TRUE, 1).

For a description of the representation and the algorithms see:
R. Zippel, 'Effective Polynomial Computation', Kluwer 1993, pp161.
++#

alias(
    theCoeffs(x)=extop(x,1),
    theValence(x)=extop(x,2),
    isTruncated(x)=extop(x,3),
    theOrder(x)=nops(extop(x,1))+extop(x,2),
    theBranchOrder(x)=extop(x,4)):

if ORDER = hold(ORDER) then ORDER:= 6 end_if:

TruncatedPowerSeries:= DomainConstructor(
    TruncatedPowerSeries,
    [ X, CF ],
    [ ],
    ( case args(0)
      of 0 do X:= hold(t); CF:= Rational; break;
      of 1 do CF:= Rational; break;
      of 2 do break;
      otherwise error("wrong no of args");
      end_case;
      if domtype(X) <> DOM_IDENT then error("no unknown") end_if;
      if CF::hasProp(Field) <> TRUE then
          error("no field")
      end_if ),
    BaseDomain,
    [ Field, Algebra(CF) ],
    [ ],

    "characteristic" = CF::characteristic,
    
    "coeffs" = fun(theCoeffs(args(1))),
    
    "valence" = fun(theValence(args(1))),
    
    "truncated" = fun(isTruncated(args(1))),

    "order" = fun((
	if not isTruncated(args(1)) then infinity
	else theOrder(args(1)) / theBranchOrder(args(1))
	end_if
    )),
    
    "branchOrder" = fun(theBranchOrder(args(1))),
    
    "expr" = proc(s) local bo, c, o, i; begin
    	bo:= theBranchOrder(s);
    	c:= theCoeffs(s);
    	o:= theValence(s)-1;
    	_plus((CF::expr(c[i])*hold(X)^((i+o)/bo)) $ i=1..nops(c),
    	      (if isTruncated(s) then O(hold(X)^(theOrder(s)/bo))
    	       else null() end_if))
    end_proc,
    
    "convert" = proc(e) local c; begin
    	case type(e)
    	of this do
    	    return(e);
    	of O do
    	    c:= extop(e,1);
    	    if c = hold(X) then
    	    	return(new(this, [CF::zero], 0, TRUE, 1))
    	    end_if;
    	    if testtype(c, NUMERIC) then
    	    	return(new(this, [CF::zero], -1, TRUE, 1))
    	    end_if;
    	    if type(c) <> "_power" then break end_if;
    	    if op(c,1) <> hold(X) then break end_if;
    	    c:= op(c,2);
    	    case domtype(c)
    	    of DOM_INT do return(new(this, [CF::zero], c-1, TRUE, 1));
    	    of DOM_RAT do return(new(this, [CF::zero], op(c,1)-1, TRUE, op(c,2)));
    	    end_case;
    	of DOM_IDENT do
	    if e <> hold(X) then break end_if;
	    return(new(this, [CF::one], 1, FALSE, 1));
    	of "_plus" do
    	    c:= map([op(e)], this::convert);
    	    if contains(c, FAIL) <> 0 then break end_if;
    	    return(this::_plus(op(c)));
    	of "_mult" do
    	    c:= map([op(e)], this::convert);
    	    if contains(c, FAIL) <> 0 then break end_if;
    	    return(this::_mult(op(c)));
    	of "_power" do
    	    if not testtype(op(e,2), Type::Rational) then break end_if;
    	    c:= this::convert(op(e,1));
    	    if c = FAIL then break end_if;
    	    return(this::_power(c, op(e,2)));
    	of "function" do
    	    if op(e,0) <> hold(O) then break end_if;
    	    c:= op(e,1);
    	    if c = hold(X) then
    	    	return(new(this, [CF::zero], 0, TRUE, 1))
    	    end_if;
    	    if testtype(c, NUMERIC) then
    	    	return(new(this, [CF::zero], -1, TRUE, 1))
    	    end_if;
    	    if type(c) <> "_power" then break end_if;
    	    if op(c,1) <> hold(X) then break end_if;
    	    c:= op(c,2);
    	    case domtype(c)
    	    of DOM_INT do return(new(this, [CF::zero], c-1, TRUE, 1));
    	    of DOM_RAT do return(new(this, [CF::zero], op(c,1)-1, TRUE, op(c,2)));
    	    end_case;
    	end_case;

        # is e a coeff? #
    	c:= CF::convert(e);
    	if c = FAIL then FAIL else new(this, [c], 0, FALSE, 1) end_if
    end_proc,
    
    "zero" = new(this, [ CF::zero ], 0, FALSE, 1),
    
    "one" = new(this, [ CF::one ], 0, FALSE, 1),
    
    "iszero" = fun(bool(nops(select(theCoeffs(args(1)), not CF::iszero)) = 0)),
    
    "_plus" = proc(s,t) local c, v, o, i; begin
    	case args(0)
    	of 1 do return(args(1));
    	of 2 do break;
    	otherwise
    	    s:= _plus(args(i) $ i=1..(args(0) div 2));
    	    t:= _plus(args(i) $ i=((args(0) div 2)+1)..args(0));
    	    return(s+t);
    	end_case;
    	
	if map({args()}, domtype) <> {this} then return(FAIL) end_if;
	
	if theBranchOrder(s) <> theBranchOrder(t) then
	    return(this::_plus(this::adaptBranchOrders(s, t)))
	end_if;
	
	# compute valence and order #
	v:= min(theValence(s), theValence(t));
	if isTruncated(s) then
	    if isTruncated(t) then
		o:= min(theOrder(s), theOrder(t))
	    else
		o:= theOrder(s)
	    end_if
	elif isTruncated(t) then
	    o:= theOrder(t)
	else
	    o:= max(theOrder(s), theOrder(t))
	end_if;

	if theOrder(s) <> o then
	    s:= this::truncOrder(s,o)
	elif theOrder(t) <> o then
	    t:= this::truncOrder(t,o)
	end_if;

	# add coeffs #
	c:= zip([ CF::zero $ (theValence(s)-v) ] . theCoeffs(s),
		[ CF::zero $ (theValence(t)-v) ] . theCoeffs(t),
		CF::_plus, CF::zero);
		
	# remove leading zeros #
	while nops(c) > 1 do
	    if not CF::iszero(c[1]) then break end_if;
	    c[1]:= NIL;
	    v:= v+1
	end_while;
	new(this, c, v, isTruncated(s) or isTruncated(t),
	    theBranchOrder(s))
    end_proc,

    "negate" = fun(
    	extsubsop(args(1), 1=map(theCoeffs(args(1)), CF::negate))
    ),

    "_mult" = proc(s,t) local c, cs, ct, ncs, nct, v, n, i, k; begin
    	case args(0)
    	of 2 do break;
    	of 1 do return(s);
    	otherwise
    	    k:= args(0) div 2;
    	    cs:= _mult(args(i) $ i=1..k);
    	    ct:= _mult(args(i) $ i=(k+1)..args(0));
    	    return(cs * ct);
    	end_case;
    	
    	if domtype(t) <> this then
    	    if domtype(t) = DOM_INT then
    	    	return(this::intmult(s, t))
    	    end_if;
    	    c:= CF::convert(t);
    	    if c = FAIL then 
    	        return((domtype(t))::_mult(s,t))
    	    end_if; 
    	    t:= this::convert(c);
    	elif domtype(s) <> this then
    	    if domtype(s) = DOM_INT then
    	    	return(this::intmult(t, s))
    	    end_if;
    	    c:= CF::convert(s);
    	    if c = FAIL then 
    	        return(FAIL) 
    	    end_if; 
    	    s:= this::convert(c);
    	end_if;

	if theBranchOrder(s) <> theBranchOrder(t) then
	    return(this::_mult(this::adaptBranchOrders(s, t)))
	end_if;

	# compute valence and order #
	v:= theValence(s) + theValence(t);
	cs:= theCoeffs(s); ncs:= nops(cs);
	ct:= theCoeffs(t); nct:= nops(ct);
	if isTruncated(s) then
	    if isTruncated(t) then
	    	n:= min(ncs, nct)
	    else
	    	n:= ncs
	    end_if
	elif isTruncated(t) then
	    n:= nct
	else
	    n:= ncs + nct - 1
	end_if;
	
	# multiply coeffs #
	c:= [ CF::_plus(CF::_mult(cs[i], ct[k-i+1])
	    		$ i=max(1,k+1-nct)..min(k,ncs))
	      $ k=1..n ];

	# remove leading zeros #
	while nops(c) > 1 do
	    if not CF::iszero(c[1]) then break end_if;
	    c[1]:= NIL;
	    v:= v+1
	end_while;

	new(this, c, v, isTruncated(s) or isTruncated(t),
	    theBranchOrder(s))
    end_proc,
    
    "_power" = proc(s,n) local c, o, v, bo, p, k, i; begin
    	if not testtype(n, Type::Rational) then
    	    error("not a rational power")
    	end_if;
    	case n
    	of 0 do return(this::one);
    	of 1 do return(s);
    	end_case;

	if theValence(s) <> 0 then
	    v:= theValence(s);
	    bo:= theBranchOrder(s);
	    s:= this::_power(extsubsop(s, 2=0), n);
	    v:= v * n / bo;
	    if domtype(v) = DOM_INT then bo:= 1
	    else bo:= op(v,2); v:= op(v,1) end_if;
	    return(this::_mult(s, new(this, [CF::one], v, FALSE, bo)));
	end_if;
	
	c:= theCoeffs(s);
	if nops(c) = 1 then
	    bo:= theBranchOrder(s);
	    v:= theValence(s) * n / bo;
	    if domtype(v) = DOM_INT then bo:= 1
	    else bo:= op(v,2); v:= op(v,1) end_if;
	    return(new(this, [ CF::_power(c[1],n) ], v,
	    	       isTruncated(s), bo));
	end_if;
	
	if isTruncated(s) or n < 0 or domtype(n) = DOM_RAT then
	    if isTruncated(s) then o:= theOrder(s)
	    else o:= ORDER * theBranchOrder(s) end_if;
	    bo:= nops(c)-1;
	    v:= c[1];
	    p:= [ CF::_power(v,n) ];
	    for k from 1 to o-1 do
	    	p:= append(p, CF::divex(
	    	    CF::_plus(CF::_mult(c[i+1], p[k-i+1], i*n-k+i)
	    	    	      $ i=1..min(k,bo)),
	    	    CF::intmult(v, k)))
	    end_for;
	    extsubsop(s, 1=p, 3=TRUE)
	else # repeated squaring #
	    p:= this::one;
	    while TRUE do
	    	if n mod 2 = 1 then p:= this::_mult(p, s) end_if;
	    	if n < 2 then break end_if;
	    	n:= n div 2;
	    	s:= this::_mult(s, s)
	    end_while;
	    p
	end_if
    end_proc,
    
    "invert" = fun(this::_power(args(1), -1)),
    
    "random" = (if CF::hasProp(systemRep) then
	proc() local p; begin
	    p:= randpoly([X], hold(Expr), hold(Coeffs)=CF::random);
	    this::convert(expr(p) + O(X^degree(p)))
	end_proc
    else
	proc() local p; begin
	    p:= randpoly([X], CF);
	    this::convert(expr(p) + O(X^degree(p)))
	end_proc
    end_if),
    
    # truncOrder(s,o) - truncates s to order o;
      o must be less than the actual order of s if s is truncated #
    "truncOrder" = fun((
	if args(2) <= theValence(args(1)) then
    	    extsubsop(args(1), 1=[ CF::zero ], 2=(args(2)-1))
    	elif theOrder(args(1)) > args(2) then
     	    extsubsop(args(1),
    	    	1=[ op(theCoeffs(args(1)),
			1..(args(2) - theValence(args(1)))) ])
   	else
   	    args(1)
   	end_if
    )),
    

    # adaptBranchOrders(s,t) -- makes the branching orders of s and t equal #
    "adaptBranchOrders" = proc(s,t) local bo, l, c, z, v, adapt; begin
	if theBranchOrder(s) <> theBranchOrder(t) then

	    adapt:= fun((
	    	c:= theCoeffs(args(1));
	    	v:= args(2) * theValence(args(1));
	    	z:= CF::zero $ (args(2)-1);
	    	c:= [ (c[i], z) $ hold(i)=1..nops(c) ];
	    	while nops(c) > 1 do
	    	    if not CF::iszero(c[1]) then break end_if;
	    	    c[1]:= NIL;
	    	    v:= v+1
	    	end_while;
	    	new(this, c, v, isTruncated(args(1)), args(3))
	    ));
	    
	    bo:= ilcm(theBranchOrder(s), theBranchOrder(t));
	    l:= bo / theBranchOrder(s);
	    if l <> 1 then
	    	s:= adapt(s, l, bo)
	    end_if;
	    l:= bo / theBranchOrder(t);
	    if l <> 1 then
	    	t:= adapt(t, l, bo)
	    end_if;
	end_if;
	(s,t)
    end_proc
    
):

unalias(theCoeffs, theValence, isTruncated, theOrder, theBranchOrder):

# end of file #
