/*
 * server.c
 *
 * (c)24/12/1993 Stuart N. John
 *
 *
 * History:
 *
 *   24-12-1993 - Created module
 *   29-12-1993 - Added TCP/IP sockets
 *
 */

#include <sys/types.h>
#include <linux/fs.h>
#include <netinet/in.h>
#include <sys/file.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <signal.h>
#include <stdio.h>

#include "powerglove.h"
#include "mode.h"
#include "server.h"
#include "message.h"


#define QLEN 5  /* max connection queue length */


/* global variables */

u_short port  = PORT;     /* server connection port                   */
int     msock;            /* master server socket                     */
int     csock;            /* socket for client connection             */
Mode_t  *mode = NULL;     /* control for client connected to server   */
fd_set  afds;             /* active file descriptor set               */
int     nfds  = -1;       /* process file descriptor table size       */
int     msglen;           /* length of received message               */
char    bytes [G_BYTES];  /* raw glove data values & sample tick diff */
int     sendtick   = 0;   /* tick time last sample was sent           */
int     tickoffset = 0;   /* tick time since last sample sent         */


/* prototypes */

static int new_client(void);
static int delete_client(void);
static int reply_message(int id);
static int create_daemon();
static int serversock(u_short portnumber, int qlen);
void quit();


/*
 * Server process to dispatch Powerglove data to a client
 *
 */
int main(int argc, char *argv[])
{
  struct sockaddr_in fsin;      /* the from address of a client     */
  int                alen;      /* from address length              */
  fd_set             rfds;      /* read file descriptor set         */
  struct timeval     timeout;   /* timeout settings for select call */
  int                id;        /* id of client messages            */
  int                tmpsock;   /* socket for unwanted client connection */
  struct sockaddr_in tmpfsin;   /* the unwanted from address of a client */
  char               recvbuf[MAX(G_BYTES, M_BYTES) + 1];  /* buffer to receive incoming messages   */

  switch (argc)  /* check for alternative port number option */
  {
    case 1:  break;
    case 2:  port = (u_short) atoi(argv[1]);
             break;
    default: fprintf(stderr, "usage: %s [port]\n", argv[0]);
             exit(0);
  }

  /* make this server a Unix daemon process, and set up signal handling */

  create_daemon();

  /* sync with the glove output and start receiving data */

  if (glove_init() < 0)
  {
    fprintf(stderr, "server: failed to initialise glove\n");
    exit(1);
  }

  /* clear the active file descriptor set, create a passive master server
     socket to accept client connections, and add it to the active file
     descriptor set */

  FD_ZERO(&afds);
  msock = serversock(port, QLEN);
  FD_SET(msock, &afds);

  nfds = getdtablesize();  /* get max number of file descriptors for process */

  /* main server loop */

  for (;;)
  {
    /* sample the latest glove data */

    if (glove_update(bytes, mode->filter) < 0)
    {
      fprintf(stderr, "server: no more glove data\n");
      (void) quit();
    }

    /* add new sample tick time difference to total tick time
       offset since last sample was sent to client */

    tickoffset += (int) bytes[G_T];

    /* construct the read file descriptor set, and set timeouts for the
       select call  */

    bcopy((char *) &afds, (char *) &rfds, sizeof(rfds));
    timeout.tv_sec  = 0;
    timeout.tv_usec = 0;

    /* select sockets to check for incoming client messages */

    if (select(nfds, &rfds, (fd_set *)0, (fd_set *)0, &timeout) < 0)
    {
      fprintf(stderr, "server: select error\n");
      perror("server");
      (void) quit();
    }

    /* we now have a bit mask which specifies which of
       the descriptors in the set is ready (if any) */

    /* check for incoming connection request from a client */

    if (FD_ISSET(msock, &rfds))
    {
      if (mode == NULL)
      {
        /* no current client connection, so accept and create a new one */

        alen = sizeof(fsin);
        csock = accept(msock, (struct sockaddr *) &fsin, &alen);

        if (csock < 0)
        {
          fprintf(stderr, "server: accept error\n");
          if (errno == EWOULDBLOCK)
            fprintf(stderr, "server: no connection to accept on non-blocking socket\n");
          perror("server");
          (void) quit();
        }

        new_client();

        /* add new client socket into the active file descriptor set */

        FD_SET(csock, &afds);

#ifdef DEBUGSERVER
        fprintf(stderr, "server: accepted new client connection %d\n", csock);
#endif
      }
      else
      {
        /* already have a client connection, so reject the attempted new one */

        alen = sizeof(tmpfsin);
        tmpsock = accept(msock, (struct sockaddr *) &tmpfsin, &alen);

        fprintf(stderr, "server: sorry, this server already has a client\n");

        /* kill unwanted client socket */
        close(tmpsock);
      }
    } /* end connection request */

    /* decode and respond to client message (if any) */

    if (FD_ISSET(csock, &rfds))
    {
      id = get_message(csock, mode, recvbuf);
      if (id < 0)
      {
        fprintf(stderr, "server: receive message error, terminating connection\n");
        delete_client();
        continue;
      }
      else
        reply_message(id);
    }

    /* send out glove data if connection in one-way mode */

    if (mode->comms == PGCOMMS_ONEWAY)
    {
      /* *** handle different combinations of data - NOT IMPLEMENTED *** */

      /* set tick time difference since last data sent to client */

      bytes[G_T] = (char) tickoffset;

      send_message(PG_REPLY_GLOVEDATA, csock, NULL, bytes);

      /* store tick time of data sent to client, reset offset tick time */

      sendtick += tickoffset;
      tickoffset = 0;
    }

  } /* end main server loop */
} /* end main() */


/*
 * Creates a new client connection mode structure and initialises it
 *
 */
static int new_client(void)
{
  mode = (Mode_t *) malloc(sizeof(Mode_t));
  if (mode == NULL)
  {
    fprintf(stderr, "server: not enough memory for client\n");
    (void) quit();
  }

  mode->datalist = PGDATA_NONE;     /* don't send data to client      */
  mode->comms    = PGCOMMS_TWOWAY;  /* wait for client to poll server */
  mode->filter   = PGFILTER_OFF;    /* no smoothing of glove data     */
}


static int delete_client(void)
{
  FD_CLR(csock, &afds);  /* remove client from active file descriptor set */
  free(mode);            /* delete the client mode data structure         */
  mode = NULL;

#ifdef DEBUGSERVER
  fprintf(stderr, "server: closing client connection %d...\n", csock);
#endif

  /* shutdown full-duplex client socket */

  close(csock);

  return 0;
}


/*
 * Reply to client messages
 *
 */
static int reply_message(int id)
{
  switch (id)
  {
    case PG_GET_GLOVEDATA : /* send latest glove data */

                   /* set tick time difference since last data sent to client */

                   bytes[G_T] = (char) tickoffset;

                   send_message(PG_REPLY_GLOVEDATA, csock, NULL, bytes);

                   /* store tick time of data sent to client, reset offset tick time */

                   sendtick += tickoffset;
                   tickoffset = 0;

                   break;

    case PG_GET_CONFIG : /* send config control data */

                   send_message(PG_REPLY_CONFIG, csock, mode, NULL);
                   break;

    case PG_GET_DATALIST : /* send data list combination */

                   send_message(PG_REPLY_DATALIST, csock, mode, NULL);
                   break;

    case PG_GET_COMMS : /* send communication mode */

                   send_message(PG_REPLY_COMMS, csock, mode, NULL);
                   break;

    case PG_GET_FILTER : /* send filter mode */

                   send_message(PG_REPLY_FILTER, csock, mode, NULL);
                   break;

    case PG_QUIT : /* close connection */

                   delete_client();
                   break;

    default:
                   break;
  }
}


/*
 * Create a background server process
 * This server runs in the background as a Unix daemon
 *
 */
static int create_daemon()
{
  int  parent;    /* store return value of fork function */
  int  fd;        /* temp file descriptor                */
  char pbuf[10];  /* array to hold an ASCII pid          */

  /* block termination signals while we set up a daemon */

  (void) signal(SIGHUP, SIG_IGN);   /* ignore hang-up signal   */
  (void) signal(SIGTERM, SIG_IGN);  /* ignore hang-up signal   */
  (void) signal(SIGINT, SIG_IGN);   /* ignore interrupt signal */

  /* fork server process into background and become a daemon */

  parent = fork();
  if (parent < 0)
  {
    fprintf(stderr, "server: error when forking\n");
    perror("server");
    exit(1);
  }

  if (parent) exit(0); /* foreground server process exit */

  /* child process becomes orphaned and then inherited by the
     initial system process (init), ensuring clean exit */


  /* background server process (daemon) continues here... */


  /* close all inherited file descriptors */

/*
  for (fd = getdtablesize() - 1; fd >= 0; --fd) (void) close(fd);
*/

  /* detach server from its controlling tty */

  fd = open("/dev/tty", O_RDWR);
  (void) ioctl(fd, TIOCNOTTY, 0);
  (void) close(fd);

  /* move server to a known directory */

  (void) chdir("/tmp");

  /* set server's umask for file creation privaleges */
  /* rwx for the user, rx for the user's group       */

  (void) umask(027);

  /* place server in its own private process group */

  (void) setpgrp(0, getpid());

  /* open file descriptors for stdin, stdout & stderr */
  /* Unix convention starts from file descriptor 0    */

/*  fd = open("/dev/null", O_RDWR); */ /* stdin  */
/*  (void) dup(fd);                 */ /* stdout */

/*  fd = open(ERRORCHANNEL, O_RDWR);*/ /* stderr  */

  /* acquire an exclusive lock or fail because another server must be running */
  /* a process only holds a lock established by flock while it executes,
     and the lock is released if the server crashes or the system reboots */

  fd = open(LOCKFILE, O_RDWR | O_CREAT, 0640);
  if (fd < 0)
  {
    fprintf(stderr, "server: could not open lock file\n");
    perror("server");
    exit(1);
  }

  if (flock(fd, LOCK_EX | LOCK_NB))
  {
    read(fd, pbuf, 10);
    fprintf(stderr, "server: lock file in use\n");
    fprintf(stderr, "server: another server is running as process %d\n", atoi(pbuf));
    close(fd);
    exit(0);
  }

  /* we now have a lock file, so we can write the server process id
     to it which enables quick lookup of the id when we want to kill
     the process */

  (void) sprintf(pbuf, "%6d\n", getpid());
  (void) write(fd, pbuf, strlen(pbuf));

  fprintf(stderr, "\nserver: running as process %d\n", getpid());

  /* daemon should now be running happily in the background... */

  /* set up signal handling for clean server termination */

  (void) signal(SIGINT, quit);   /* set error handler for interrupt signal */
  (void) signal(SIGTERM, quit);  /* set error handler for terminate signal */
}


/*
 * Creates a passive socket for the master server
 *
 */
static int serversock(u_short portnumber, int qlen)
{
  struct servent  *pse;    /* pointer to service information entry  */
  struct protoent *ppe;    /* pointer to protocol information entry */
  struct sockaddr_in sin;  /* an Internet endpoint address          */
  int s;                   /* socket descriptor                     */
  int setflag = 1;

  bzero((char *) &sin, sizeof(sin));
  sin.sin_family      = AF_INET;
  sin.sin_addr.s_addr = htonl(INADDR_ANY);
  sin.sin_port        = htons(portnumber);

  /* map protocol name to protocol number */

  if ((ppe = getprotobyname("tcp")) == 0)
  {
    fprintf(stderr, "server: can't get tcp protocol entry\n");
    perror("server");
    (void) quit();
  }

  /* allocate a socket */

  s = socket(AF_INET, SOCK_STREAM, ppe->p_proto);
  if (s < 0)
  {
    fprintf(stderr, "server: can't open master socket\n");
    perror("server");
    (void) quit();
  }

  /* bind the socket */

  if (bind(s, (struct sockaddr *) &sin, sizeof(sin)) < 0)
  {
    fprintf(stderr, "server: can't bind master socket\n");
    perror("server");
    close(s);
    (void) quit();
  }

  if (listen(s, qlen) < 0)
  {
    fprintf(stderr, "server: can't listen on master socket\n");
    perror("server");
    (void) quit();
  }

  if (fcntl(s, F_SETFL, O_NONBLOCK) < 0)
  {
    fprintf(stderr,"server: can't set no-blocking on master socket\n");
    perror("server");
  }

  return s;
}


/*
 * Shutdown routine, also called if we get SIGINT, SIGTERM signals
 *
 */
void quit()
{
  fprintf(stderr, "\n");

  /* close the serial connection */
  glove_quit();

  if (nfds >= 0)  /* we have sockets to close */
  {
    fprintf(stderr, "server: closing client socket\n");

    if (FD_ISSET(csock, &afds))
      delete_client();

    fprintf(stderr, "server: closing master socket\n");

    if (FD_ISSET(msock, &afds))
    {
      FD_CLR(msock, &afds);
      close(msock);
    }
  }

  fprintf(stderr, "server: exit\n");

  exit(0);
}
