/*
 * posture.c
 *
 * (c)28/11/1993 Stuart N. John
 *
 *
 * History:
 *
 *   28-11-1993 - Created module
 *
 */

#include <sys/types.h>
#include <stdio.h>

#include "glove.h"
#include "posture.h"


/*
 * Posture Data List
 *
 * This list is used to store information about all postures which can be
 * classified by the system. The data for each posture is parsed from the
 * posture definition file.
 *
 */

postureData_t *posture_list = NULL;


/*
 * Posture Look-Up Table
 *
 * The posture look-up table is used to unambiguously classify a posture
 * from the sampled values of the finger flex sensors and the rotation of
 * the wrist. The posture table is generated from values in the posture
 * data list by the function fill_posture_table().
 *
 * The table is a 5-dimensional array where the dimensions are:
 *
 * thumb          - 4 flex values (Open(0), Semi_Open(1), Semi_Closed(2), Closed(3))
 * index finger   - 4 flex values                    "
 * middle finger  - 4 flex values                    "
 * ring finger    - 4 flex values                    "
 * wrist rotation - 12 increments (0-11) of 30 degrees
 *
 * N.B. Wrist rotation is in a clockwise direction.
 *
 */

static postureData_t *posture_table [4][4][4][4][12];


/* function prototypes */

static int read_posture_file();
static int fill_posture_table();

#ifdef DEBUGPOS
static int dump_posture_list();
#endif


/* functions */

postureData_t *classify_posture(Flex t, Flex i, Flex m, Flex r, int w)
{
  return posture_table [t][i][m][r][w];
}


int create_posture_table()
{
  read_posture_file();

#ifdef DEBUGPOS
  dump_posture_list();
#endif

  fill_posture_table();
}


int delete_posture_table()
{
  postureData_t *next_p;

  while (posture_list)
  {
    next_p = posture_list->next;
    free(posture_list->name);
    free(posture_list);
    posture_list = next_p;
  }
}


/* static functions */

/*
 * Fill Posture Table
 *
 * For each posture in the posture list, this routine generates all the
 * possible valid combiations of finger flex and rotation between the
 * min/max values specified for a posture, and places pointers to that
 * posture data structure at the appropriate locations in the table.
 *
 * N.B. Wrist rotation wraps around from 11 to 0, so when generating
 *      values in the range rotation.min to rotation.max we make sure
 *      to mod by 12 before using each value.
 *
 */

static int fill_posture_table()
{
  postureData_t *p = posture_list;  /* pointer to traverse posture list */
  Flex t, i, m, r;                  /* finger loop values               */
  int w;                            /* wrist loop values                */
  postureData_t *element;           /* temp pointer to test conflicts   */

  while (p)
  {
    for (t = p->thumb.min; t <= p->thumb.max; t++)
      for (i = p->index.min; i <= p->index.max; i++)
        for (m = p->middle.min; m <= p->middle.max; m++)
          for (r = p->ring.min; r <= p->ring.max; r++)
            for (w = (p->rotation.min) % 12; ; ++w, w %= 12)
            {
              /* we have identified a position in the posture table to
                 record the posture data pointer p, but we must first check
                 to see if the table already contains a pointer here,
                 indicating an ambiguous posture definition */

              element = posture_table [t][i][m][r][w];

              if (element != NULL && element != p && element != p->parent)
                fprintf(stderr, "posture: posture conflict %s <- %s\n", element->name, p->name);

              posture_table [t][i][m][r][w] = p;

              if (w == p->rotation.max)
                break;
            }

    p = p->next;
  }
}


static int read_posture_file()
{
  FILE          *posture_file;
  postureData_t *posture, *parent;
  postureData_t *last_posture = posture_list;
  char          buffer[20];
  char          buf[256];
  int           nextchar;
  int           id = 0;

  if ((posture_file = fopen(POSTURE_FILE, "r")) == NULL)
  {
    fprintf(stderr, "posture: can't open posture defintion file\n");
    perror("posture");
    return -1;
  }

  /* get posture name */
  while ((fgets(buf, 256, posture_file)) != NULL)
  {
    sscanf(buf, "\"%[^\"]s\"\n", buffer);

    if ((posture = (postureData_t *) malloc(sizeof(postureData_t))) == NULL)
    {
      fprintf(stderr, "out of memory");
      return -1;
    }
  
    if ((posture->name = (char *) malloc((strlen(buffer) + 1) * sizeof(char))) == NULL)
    {
      fprintf(stderr, "out of memory");
      return -1;
    }

    strcpy(posture->name, buffer);  /* store posture name       */
    posture->id = id++;             /* store posture identifier */

    fscanf(posture_file, "{\n");
  
    posture->sub_posture = FALSE;
    posture->parent = NULL;
    parent = posture;
    fscanf(posture_file, "  T %d %d\n", &posture->thumb.min, &posture->thumb.max);
    fscanf(posture_file, "  I %d %d\n", &posture->index.min, &posture->index.max);
    fscanf(posture_file, "  M %d %d\n", &posture->middle.min, &posture->middle.max);
    fscanf(posture_file, "  R %d %d\n", &posture->ring.min, &posture->ring.max);
    fscanf(posture_file, "  W %d %d\n", &posture->rotation.min, &posture->rotation.max);
  
    posture->next = NULL;
  
    if (last_posture == NULL)
      posture_list = posture;
    else
      last_posture->next = posture;

    last_posture = posture;
 
    fgets(buf, 256, posture_file);

    while (*buf == '\"')
    {
      /* get posture name */
      sscanf(buf, "\"%[^\"]s\"\n", buffer);

      if ((posture = (postureData_t *) malloc(sizeof(postureData_t))) == NULL)
      {
        fprintf(stderr, "out of memory");
        return -1;
      }
      
      if ((posture->name = (char *) malloc((strlen(buffer) + 1) * sizeof(char))) == NULL)
      {
        fprintf(stderr, "out of memory");
        return -1;
      }
      
      strcpy(posture->name, buffer);  /* store posture name       */
      posture->id = id++;             /* store posture identifier */
      
      fscanf(posture_file, "  {\n");

      posture->sub_posture = TRUE;
      posture->parent = parent;
      posture->thumb.min  = last_posture->thumb.min;
      posture->thumb.max  = last_posture->thumb.max;
      posture->index.min  = last_posture->index.min;
      posture->index.max  = last_posture->index.max;
      posture->middle.min = last_posture->middle.min;
      posture->middle.max = last_posture->middle.max;
      posture->ring.min   = last_posture->ring.min;
      posture->ring.max   = last_posture->ring.max;

      fscanf(posture_file, "    W %d %d\n", &posture->rotation.min, &posture->rotation.max);
      
      posture->next = NULL;
  
      last_posture->next = posture;
      last_posture = posture;

      fscanf(posture_file, "  }\n");

      fgets(buf, 256, posture_file);
    }

    fscanf(posture_file, "}\n");

    if (fgetc(posture_file) == EOF) break;
  }

  return 0;
}


/*
 * Dump Posture List
 *
 * Simple debug function to display the contents of the Posture Data List.
 *
 */

#ifdef DEBUGPOS
static int dump_posture_list()
{
  postureData_t *p = posture_list;  /* pointer to traverse posture list */

  while (p)
  {
    printf("\"%s\" %s\n", p->name, (p->sub_posture == TRUE)?"(sub)":"(class)");
    printf("{\n");
    printf("  T %d %d\n", p->thumb.min, p->thumb.max);
    printf("  I %d %d\n", p->index.min, p->index.max);
    printf("  M %d %d\n", p->middle.min, p->middle.max);
    printf("  R %d %d\n", p->ring.min, p->ring.max);
    printf("  W %d %d\n", p->rotation.min, p->rotation.max);
    printf("}\n\n");

    p = p->next;
  }
}
#endif


#ifdef DEBUGMAIN
int main()
{
  create_posture_table();
}
#endif
