
/********************************************************************************
 *  Projektname:	AERO
 *  Filename:		rayausg.c
 *  Filetyp:		Modul
 ********************************************************************************
 *  WICHTIG:            Dieser Quelltext benoetigt einen Compiler, der Struktur-
 *                      Zuweisungen beherrscht (z.B. ANSI-C, gcc). Sind also z.B.
 *                      a und b Strukturen gleichen Typs, muss bei b=a; der ge-
 *                      samte Inhalt von a nach b kopiert werden.
 ********************************************************************************
 *  Modulname:		Anzeige
 *  Revision:		9
 *  letzte Aenderung:	19.4.94
 *  Autor:  		Hartmut 
 *  Status:		Erzeugung von Quelltext fuer POV-Ray Raytracer
 *		       							
 *  imp. Bezeichner:    ###### ???? alles aus autils.c
 *  ----------------
 *
 *  exp. Bezeichner:
 *  ----------------
 *  void ErzeugeRaytracerBild(char *grundname, TKamera kamera, TZustand *zustand, 
 *			      int modus, int rayvers, int framenumber)
 *          Ausgabe des angegebenen Zustands als Raytracer-Quelltext auf ein File.
 *          Dabei wird durch modus die Art der Anzeige gesteuert (modus ist
 *          ein Bitvektor, gleiche Funktion wie bei AnzeigeDarstellung()):
 *  (ex. nicht)  Bit 0: Fussboden zeichnen (1) oder nicht (0)
 *  (in Planung) Bit 1: Koordinatenachsen der Fenster zeichnen (1) oder nicht (0)
 *  (in Planung) Bit 2: Koordinatenachsen der einzelnen Koerper zeichnen (1) oder
 *                      nicht (0)
 *  (in Planung) Bit 3: Kraefte zeichnen (1) oder nicht (0)
 *  realisiert:  Bit 4: Ausgabe fuer 3D-Bildschirm (1) oder normalen Schirm (0) 
 *          rayvers gibt an, fuer welchen Raytracer der Output erfolgen soll:
 *               1:     POV-Ray V1.0
 *               2:     POV-Ray V2.x
 *
 *  Beschreibung:
 *  -------------
 *  In diesem Modul sind die Routinen zusammengefasst, die einen Zustand als
 *  Quelltext fuer den Raytracer POV-Ray ausgeben. Es werden die Versionen
 *  V1.0 und V2.x von POV-Ray unterstuetzt. Weitere Raytracer (z.B.
 *  Rayshade) sind in Planung.
 *
 *  Fehler:
 *  -------
 *  Es kann sein, dass durch Rechenungenauigkeit manche Objekte im Verlauf
 *  einer Animation ploetzlich und ohne "Anlass" um ca. 160 Grad huepfen.
 *  Der Fehler liegt daran, dass fuer den Raytracer die Quaternions in
 *  Eulerwinkel (also Drehungen nacheinander um die 3 Achsen) umgerechnet
 *  werden muessen. Diese Berechnung ist anscheinend sehr fehleranfaellig
 *  bei ungenauen Quaternionwerten.
 *
 *  Revisiongeschichte:					       	
 *  -------------------						       	
 *   1:   Grundlegende Version 
 *   2:   Materialien eingebaut (noch unvollstaendig)
 *   3:   Zusammengesetzte Objekte eingebaut
 *   4:   3D-Ausgabe eingebaut: Kameraversatz und leichte Parallaxe; 
 *        Farbe, Durchsichtigkeit und Rauhigkeit beziehen sich auf eine
 *        2. Textur, die aehnlich einem Lack oder einer durchsichtigen 
 *        Folie die Oberflaeche veraendert.
 *   5:   Ausgabe ins File optimiert, Rauhigkeit mit Bump-Pattern. (-> V1.3)
 *   6:   Wenn Files nicht zu oeffnen gehen, dann werden jetzt auch keine
 *        Daten mehr ausgegeben. (-> V1.4)
 *   7:   In EulerWinkelFromQuaternion() die Rechengenauigkeit erhoeht
 *   8:   Anpassung an POV-Ray V2.x, neue Parameter rayvers und framenumber
 *        in EulerWinkelFromQuaternion()  (-> V1.5)
 *   9:   Scale <1 1 0> bei AusgabeVerbParameter() entfernt um Warning des
 *        Raytracers bei Links der Laenge 0 zu vermeiden (haupts. Gelenk).
 *        Zugriff auf falsches Matrixelement in EulerWinkelFromQuaternion()
 *        korrigiert. Berechnung der Stereokameras in ErzeugeRaytracerBild()
 *        korrigiert.
 *   
 *  To do:
 *  ------
 *        - Kommentare ueberpruefen, korrigieren, ergaenzen (z.B. imp. Bez.)
 *        - Zusammengesetzte Objekte ueberpruefen! Speziell MERGE-Object in
 *          POV-Ray V2.x pruefen.
 *        - Farbe fuer Glas aendern (in POV-Ray V2.x)
 *        - evtl. Verbindungsnummern als Kommentare in die Files (Problem: 
 *          Verbindungen sind nicht intern durchnumeriert)
 *
 ********************************************************************************/

#define RAYAUSG_C

#include <stdio.h>
#include <stdlib.h>

#include "rayausg.h"
#include "autils.h"
#include "Kamera.h"

/*======================================================================*/

/***********************************************************************
 *  Routinen und Variablen fuer das Animationsfenster
 ***********************************************************************/


static FILE *handle;		  /* Filehandle des gerade geoeffneten Files */
static int raytracer;		  /* Raytracer-Version: 1: POV-Ray V1.0 */
				  /*                    2: POV-Ray V2.x */

static TMaterialTabelle *materialtabelle; /* gerade aktive Materialtabelle */



/*****************************************************************************
 *  Funktion:       void AusgabeKamera(TVektor kampos, TQuaternion kq, TReal zoom)
 *  Parameter:      kampos: Position der Kamera
 *                  kq:     Drehlage der Kamera
 *                  zoom:   Zoomfaktor der Kamera
 *
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   handle: File, in das ausgegeben wird
 *
 *  Beschreibung:
 *  -------------
 *  Die Kamera wird an ihre Position in den Raum gesetzt. Dazu werden etwas
 *  ueber der Kamera links 2 Lichtquellen gesetzt. Weitere Lichtquellen
 *  rechts ueber der Kamera und sehr weit oben (als Diffusbeleuchtung)
 *  koennen sehr leicht eingefuegt werden.
 *****************************************************************************/

#define KameraGrundZoom 1.9
#define LichtSeitlich   3
#define LichtOben       5
#define DiffuslichtOben 30
#define HauptlichtIntensitaet 2  /* Hauptlicht 2-fach ausgeben, damit Bild heller */

/*****************************************************************************
 *  Funktion:       void POV1AusgabeKamera(...)
 *
 *  Beschreibung:
 *  -------------
 *  Ausgabe der Daten fuer POV-Ray V1.0
 *****************************************************************************/

static void POV1AusgabeKamera(TVektor kampos, TVektor vorne, TVektor oben,
			      TVektor rechts, TReal kzoom)
{
     int i;	     /* Zaehler fuer Lichtintensitaet */

     fprintf(handle, "camera\n{\n");
     fprintf(handle, "    location  < %f %f %f >\n", kampos[0], kampos[1], kampos[2]);
     fprintf(handle, "    direction < %f %f %f >\n", vorne[0]*kzoom, vorne[1]*kzoom, vorne[2]*kzoom); /* Zoom-Faktor und Grundvergroesserung einbringen */
     fprintf(handle, "    up        < %f %f %f >\n", oben[0], oben[1], oben[2]);
     fprintf(handle, "    right     < %f %f %f >\n", rechts[0]*1.333333, rechts[1]*1.333333, rechts[2]*1.333333);
     fprintf(handle, "}\n\n");

     /* Lichtquellen ausgeben, ggf. mehrfach zur Helligkeitssteigerung */
     for (i=1; i<=HauptlichtIntensitaet; i++)
     {
	  fprintf(handle, "object\n{\n");
	  fprintf(handle, "    light_source\n    {\n");
	  fprintf(handle, "        < %f %f %f >\n", 
		  kampos[0]-LichtSeitlich*rechts[0]+LichtOben*oben[0],
		  kampos[1]-LichtSeitlich*rechts[1]+LichtOben*oben[1], 
		  kampos[2]-LichtSeitlich*rechts[2]+LichtOben*oben[2]);
	  fprintf(handle, "        color red 1 green 1 blue 1\n    }\n}\n\n");
     } /* for */

#if 0    /* weitere Lichtquellen und Diffusbeleuchtung auf Wunsch */
     fprintf(handle, "object\n{\n");
     fprintf(handle, "    light_source\n    {\n");
     fprintf(handle, "        < %f %f %f >\n", 
	     kampos[0]+LichtSeitlich*rechts[0]+LichtOben*oben[0],
	     kampos[1]+LichtSeitlich*rechts[1]+LichtOben*oben[1],
	     kampos[2]+LichtSeitlich*rechts[2]+LichtOben*oben[2]);
     fprintf(handle, "        color red 1 green 1 blue 1\n    }\n}\n\n");

     fprintf(handle, "object\n{\n");
     fprintf(handle, "    light_source\n    {\n");
     fprintf(handle, "        < %f %f %f >\n", 
	     kampos[0]+DiffuslichtOben*oben[0],
	     kampos[1]+DiffuslichtOben*oben[1],
	     kampos[2]+DiffuslichtOben*oben[2]);
     fprintf(handle, "        color red .1 green .1 blue .1\n    }\n}\n\n");
#endif
}


/*****************************************************************************
 *  Funktion:       void POV2AusgabeKamera(...)
 *
 *  Beschreibung:
 *  -------------
 *  Ausgabe der Daten fuer POV-Ray V2.x. Hierbei wird von den neuen
 *  Moeglichkeiten von V2.x Gebrauch gemacht. Darum werden hier nur noch
 *  alle noetigen Vektoren deklariert und die eigentliche Definition steht
 *  dann im Include-File POVRAY2INCLUDE.
 *****************************************************************************/

static void POV2AusgabeKamera(TVektor kampos, TVektor vorne, TVektor oben,
			      TVektor rechts, TReal kzoom)
{

     fprintf(handle, "\n/*----------------------------------------------------------------------*/\n\n");
     fprintf(handle, "/* Definitions for camera and light sources */\n\n");

     fprintf(handle, "#declare %s = < %f, %f, %f >\n", POVRAY2CAMPOS,
	     kampos[0], kampos[1], kampos[2]);
     fprintf(handle, "#declare %s = %f\n", POVRAY2ZOOM, kzoom);
     fprintf(handle, "#declare %s = < %f, %f, %f >\n", POVRAY2CAMX, rechts[0], rechts[1], rechts[2]);
     fprintf(handle, "#declare %s = < %f, %f, %f >\n", POVRAY2CAMY, oben[0], oben[1], oben[2]);
     fprintf(handle, "#declare %s = < %f, %f, %f >\n", POVRAY2CAMZ, vorne[0], vorne[1], vorne[2]);


     /* Die Kamera ist im File POVRAY2INCLUDE angegeben. Alle noetigen */
     /* Parameter sind oben deklariert. */

#if 0
     fprintf(handle, "\n/*----------------------------------------------------------------------*/\n\n");
     fprintf(handle, "/* Definition of camera */\n\n");
     fprintf(handle, "camera\n{\n");
     fprintf(handle, "    location  < %f, %f, %f >\n", kampos[0], kampos[1], kampos[2]);
     fprintf(handle, "    direction %s*%f\n", POVRAY2CAMZ, kzoom);
     fprintf(handle, "    up        %s\n", POVRAY2CAMY);
     fprintf(handle, "    right     %s*4/3\n", POVRAY2CAMX);
     fprintf(handle, "}\n\n");
#endif

     /* Lichtquellen sind im File POVRAY2INCLUDE angegeben. Mittels */
     /* der oben definierten Vektoren POVRAY2CAMX, POVRAY2CAMY und */
     /* POVRAY2CAMZ koennen sie ja auch dort relativ zur Kamera angegeben */
     /* werden. Darum fehlen sie hier voellig! */

#if 0
     fprintf(handle, "object\n{\n");
     fprintf(handle, "    %s1\n", POVRAY2LIGHT);
     fprintf(handle, "    translate <%f, %f, %f>\n",
	     kampos[0], kampos[1], kampos[2]);
     fprintf(handle, "              + (< %f, %f, %f > * %s1x)\n",
	     rechts[0], rechts[1], rechts[2], POVRAY2LIGHTPOS);
     fprintf(handle, "              + (< %f, %f, %f > * %s1y)\n",
	     oben[0], oben[1], oben[2], POVRAY2LIGHTPOS);
     fprintf(handle, "              + (< %f, %f, %f > * %s1z)\n}\n\n",
	     vorne[0], vorne[1], vorne[2], POVRAY2LIGHTPOS);

     fprintf(handle, "object\n{\n");
     fprintf(handle, "    %s2\n", POVRAY2LIGHT);
     fprintf(handle, "    translate <%f, %f, %f>\n",
	     kampos[0], kampos[1], kampos[2]);
     fprintf(handle, "              + (< %f, %f, %f > * %s2x)\n",
	     rechts[0], rechts[1], rechts[2], POVRAY2LIGHTPOS);
     fprintf(handle, "              + (< %f, %f, %f > * %s2y)\n",
	     oben[0], oben[1], oben[2], POVRAY2LIGHTPOS);
     fprintf(handle, "              + (< %f, %f, %f > * %s2z)\n}\n\n",
	     vorne[0], vorne[1], vorne[2], POVRAY2LIGHTPOS);
#endif

}


/*****************************************************************************
 *  Funktion AusgabeKamera() selbst.
 *****************************************************************************/

static void AusgabeKamera(TVektor kampos, TQuaternion kq, TReal zoom)
{
     AMatrix M_kamera;
     TVektor vorne, oben, rechts;

     /* Kamera ausgeben */

     MatrixFromQuaternion(kq, M_kamera);
     ATransformation(vorne, M_kamera, 0.0, 0.0, -1.0);

     ATransformation(oben, M_kamera, 0.0, 1.0, 0.0);
     ATransformation(rechts, M_kamera, 1.0, 0.0, 0.0);

     switch (raytracer)
     {
       case POVRAY1:			  /* Code fuer POV-Ray V1.0 */
	  POV1AusgabeKamera(kampos, vorne, oben, rechts, KameraGrundZoom*zoom);
	  break;
	  
       default:     /* defaultmaessig Code fuer POV-Ray V2.x ausgeben */
	  POV2AusgabeKamera(kampos, vorne, oben, rechts, KameraGrundZoom*zoom);
	  break;
     } /* switch (raytracer) */
}




/*****************************************************************************
 *  Funktion:       void EulerWinkelFromQuaternion(TQuaternion q, TVektor winkel)
 *  Parameter:      q:      Drehlage des Koerpers
 *                  winkel: Winkel, um die um die 3 Achsen gedreht werden
 *                          muss (in Grad)
 *
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   -
 *
 *  Beschreibung:
 *  -------------
 *  Aus dem Quaternion werden die 3 Euler-Winkel berechnet, d.h. die Winkel
 *  um die jeweils um die Koordinatenachsen gedreht werden muss. Dabei kommt
 *  es auf die Reihenfolge an! Hier ist diese: 
 *      1. Drehung um z-Achse
 *      2. Drehung um x-Achse
 *      3. Drehung um y-Achse
 *****************************************************************************/

#define asinInDeg(x) 180/PI*asin(x)
#define M1TO1(x) (x>1.0) ? 1.0 : ((x<-1.0) ? -1.0 : x)

static void EulerWinkelFromQuaternion(TQuaternion q, TVektor winkel)
{
     AMatrix M_lage;
     TReal sinalpha, cosalpha, sinbeta, cosbeta, singamma, cosgamma;

     /* Da hohe Genauigkeit gefragt ist, hier Quaternion normieren */
     NormiereQuaternion(q);

     /* ##### Hier kann optimiert werden: die unnoetigen Matrix-Elemente */
     /* muessen ja gar nicht berechnet werden. Also MatrixFromQuaternion */
     /* weg und dafuer die Element-Berechnung direkt hin. Allerdings ist es */
     /* so uebersichtlicher! ##################*/

     MatrixFromQuaternion(q, M_lage);

     sinalpha = -M_lage[1][2];
     sinalpha = M1TO1(sinalpha);
     cosalpha = sqrt(1-sinalpha*sinalpha);
     winkel[0] = asinInDeg(sinalpha);

#if 0     /* unnoetig, da nach obiger Berechnungsweise immer cosalpha>=0 */
     if (cosalpha<0) winkel[0]=180-winkel[0];
#endif

     if (NAHE_0(fabs(sinalpha)-1))
     {					  /* waehle gamma=0 */
	  sinbeta  = -M_lage[2][0];
	  sinbeta  = M1TO1(sinbeta);
	  cosbeta  = M_lage[0][0];
	  cosbeta  = M1TO1(cosbeta);
	  winkel[1] = asinInDeg(sinbeta);
	  if (cosbeta<0) winkel[1]=180-winkel[1];

	  singamma = 0;
	  cosgamma = 1;
	  winkel[2] = 0;
     }
     else
     {
	  sinbeta = M_lage[0][2]/cosalpha;
	  sinbeta = M1TO1(sinbeta);
	  cosbeta = M_lage[2][2]/cosalpha;
	  cosbeta = M1TO1(cosbeta);
	  winkel[1] = asinInDeg(sinbeta);
	  if (cosbeta<0) winkel[1]=180-winkel[1];
	  singamma= M_lage[1][0]/cosalpha;
	  singamma= M1TO1(singamma);
	  cosgamma= M_lage[1][1]/cosalpha;
	  cosgamma= M1TO1(cosgamma);
	  winkel[2] = asinInDeg(singamma);
	  if (cosgamma<0) winkel[2]=180-winkel[2];
     }

}



/*****************************************************************************
 *  Funktion:       void AusgabeLage(TQuaternion q, TVektor pos)
 *  Parameter:      q:      Drehlage des Koerpers
 *                  pos:    Position des Koerpers
 *
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   handle: File, in das ausgegeben wird
 *
 *  Beschreibung:
 *  -------------
 *  Die Lage des Koerpers wird in das File ausgegeben. Dazu wird zuerst das
 *  Quaternion in Euler-Winkel umgerechnet und als 3 einzelne rotate-Befehle
 *  ausgegeben. Anschliessend wird noch ein translate-Befehl ausgegeben, der
 *  den Koerper an die Endgueltige Position verschiebt.
 *****************************************************************************/

static void AusgabeLage(TQuaternion q, TVektor pos)
{
     TVektor winkel;

     EulerWinkelFromQuaternion(q, winkel);

     switch(raytracer)
     {
       case POVRAY1:			  /* Ausgabe fuer POV-Ray V1.0 */
	  if (!NAHE_0(winkel[2]))
	  {
	       fprintf(handle, "    rotate < 0 0 %f >\n",winkel[2]);
	  }
	  if (!NAHE_0(winkel[0]))
	  {
	       fprintf(handle, "    rotate < %f 0 0 >\n",winkel[0]); 
	  }
	  if (!NAHE_0(winkel[1]))
	  {
	       fprintf(handle, "    rotate < 0 %f 0 >\n",winkel[1]);
	  }
	  if (!NAHE_0(pos[0]) || !NAHE_0(pos[1]) || !NAHE_0(pos[2]))
	  {
	       fprintf(handle, "    translate < %f %f %f >\n", pos[0], pos[1], pos[2]);
	  }
	  break;

       default:				  /* Ausgabe fuer POV-Ray V2.x */
  	  if (!NAHE_0(winkel[2]))
	  {
	       fprintf(handle, "    rotate < 0, 0, %f >\n",winkel[2]);
	  }
	  if (!NAHE_0(winkel[0]))
	  {
	       fprintf(handle, "    rotate < %f, 0, 0 >\n",winkel[0]); 
	  }
	  if (!NAHE_0(winkel[1]))
	  {
	       fprintf(handle, "    rotate < 0, %f, 0 >\n",winkel[1]);
	  }
	  if (!NAHE_0(pos[0]) || !NAHE_0(pos[1]) || !NAHE_0(pos[2]))
	  {
	       fprintf(handle, "    translate < %f, %f, %f >\n", pos[0], pos[1], pos[2]);
	  }
	  break;
     } /* switch (raytracer) */
}



/*****************************************************************************
 *  Funktion:       void AusgabeTexture(TKoerper *koerper)
 *  Parameter:      koerper:  Aktueller Koerper
 *
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   handle: File, in das ausgegeben wird
 *
 *  Beschreibung:
 *  -------------
 *  Ausgabe der Textur. Diese ist abhaengig von Material, Farbe,
 *  Durchsichtigkeit, Rauhigkeit und Reflexion des Koerpers.
 *****************************************************************************/

static void AusgabeTexture(TKoerper *koerper)
{
     char **matname;

     matname = MaterialNamen(materialtabelle);

     switch (raytracer)
     {
       case POVRAY1:			  /* Code fuer POV-Ray V1.0 */
	  fprintf(handle, "    texture\n    {\n");
	  fprintf(handle, "        aero%s\n",matname[koerper->Material]);
	  fprintf(handle, "    }\n");
	  fprintf(handle, "    texture\n    {\n");
	  fprintf(handle, "        color red %f green %f blue %f alpha %f\n",
		  ((TReal)koerper->R/(TReal)USHRT_MAX), 
		  ((TReal)koerper->G/(TReal)USHRT_MAX), 
		  ((TReal)koerper->B/(TReal)USHRT_MAX),
		  koerper->Durchsichtigkeit);
	  fprintf(handle, "        reflection %f\n",koerper->Reflexion);
	  fprintf(handle, "        roughness %f\n",koerper->Rauhigkeit*0.9995+0.0005);
	  if (!NAHE_0(koerper->Rauhigkeit))
	  {
	       fprintf(handle, "        wrinkles %f\n",koerper->Rauhigkeit);
	  }
	  fprintf(handle, "    }\n");
	  break;

       default:				  /* Code fuer POV-Ray V2.x */
	  /* Rauhigkeit, Reflexion und Transparenz innerhalb von AERO werden */
	  /* nicht mehr unterstuetzt! */
	  fprintf(handle, "    texture\n    {\n");
	  fprintf(handle, "        pigment {color red %f green %f blue %f filter aero%sfilter}\n",
		  ((TReal)koerper->R/(TReal)USHRT_MAX), 
		  ((TReal)koerper->G/(TReal)USHRT_MAX), 
		  ((TReal)koerper->B/(TReal)USHRT_MAX),
		  matname[koerper->Material]);
	  fprintf(handle, "    }\n");
	  fprintf(handle, "    texture {aero%s}\n",matname[koerper->Material]);

	  break;

     } /* switch (raytracer) */
}



/*****************************************************************************
 *  Funktion:       void AusgabeQuader(TKoerper *koerper)
 *  Parameter:      koerper:  Koerper-Struktur des Quaders
 *
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   handle:  File, in das ausgegeben wird
 *
 *  Beschreibung:
 *  -------------
 *  Der Quader wird in das File ausgegeben.
 *****************************************************************************/

static void AusgabeQuader(TKoerper *koerper)
{
     switch (raytracer)
     {
       case POVRAY1:			  /* Code fuer POV-Ray V1.0 */
	  fprintf(handle, "object\n{\n");
	  fprintf(handle, "    box { %s }\n", PovrayQuader);
	  fprintf(handle, "    scale < %f %f %f >\n",
		  koerper->Form.Quader.KantenLaenge[0],
		  koerper->Form.Quader.KantenLaenge[1],
		  koerper->Form.Quader.KantenLaenge[2]);
	  AusgabeTexture(koerper);
	  AusgabeLage(koerper->q, koerper->Position);
	  fprintf(handle, "}\n\n");
	  break;

       default:				  /* Code fuer POV-Ray V2.x */
	  fprintf(handle, "object\n{\n");
	  fprintf(handle, "    %s\n", PovrayQuader);
	  fprintf(handle, "    scale < %f, %f, %f >\n",
		  koerper->Form.Quader.KantenLaenge[0],
		  koerper->Form.Quader.KantenLaenge[1],
		  koerper->Form.Quader.KantenLaenge[2]);
	  AusgabeTexture(koerper);
	  AusgabeLage(koerper->q, koerper->Position);
	  fprintf(handle, "}\n\n");
	  break;
     } /* switch (raytracer) */ 
}		    




/*****************************************************************************
 *  Funktion:       void AusgabeZylinder(TKoerper *koerper)
 *  Parameter:      koerper:  Struktur des Zylinders
 *
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   handle:  File, in das ausgegeben wird
 *
 *  Beschreibung:
 *  -------------
 *  Den Zylinder wird in das File ausgegeben. Es handelt sich dabei um einen
 *  unendlich langen Zylinder (Quadric), der auf beiden Seiten durch eine
 *  Ebene begrenzt wird.
 *****************************************************************************/

static void AusgabeZylinder(TKoerper *koerper)
{
     switch (raytracer)
     {
       case POVRAY1:			  /* Code fuer POV-Ray V1.0 */
	  fprintf(handle, "object\n{\n");
	  fprintf(handle, "    intersection { %s }\n", PovrayZylinder);
	  fprintf(handle, "    scale < %f %f %f >\n",
		  koerper->Form.Zylinder.Radius,
		  koerper->Form.Zylinder.Radius,
		  koerper->Form.Zylinder.Hoehe);
	  AusgabeTexture(koerper);
	  AusgabeLage(koerper->q, koerper->Position);
	  fprintf(handle, "}\n\n");
	  break;

       default:				  /* Code fuer POV-Ray V2.x */
	  fprintf(handle, "object\n{\n");
	  fprintf(handle, "    %s\n", PovrayZylinder);
	  fprintf(handle, "    scale < %f, %f, %f >\n",
		  koerper->Form.Zylinder.Radius,
		  koerper->Form.Zylinder.Radius,
		  koerper->Form.Zylinder.Hoehe);
	  AusgabeTexture(koerper);
	  AusgabeLage(koerper->q, koerper->Position);
	  fprintf(handle, "}\n\n");
	  break;
     } /* switch (raytracer) */
}		    




/*****************************************************************************
 *  Funktion:       void AusgabeKugel(TKoerper *koerper)
 *  Parameter:      koerper:  Koerper-Struktur der Kugel
 *
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   handle:  File, in das ausgegeben wird
 *
 *  Beschreibung:
 *  -------------
 *  Die Kugel wird in das File ausgegeben.
 *****************************************************************************/

static void AusgabeKugel(TKoerper *koerper)
{
     switch (raytracer)
     {
       case POVRAY1:			  /* Code fuer POV-Ray V1.0 */
	  fprintf(handle, "object\n{\n");
	  fprintf(handle, "    sphere { %s }\n",PovrayKugel);
	  fprintf(handle, "    scale < %f %f %f >\n",
		  koerper->Form.Kugel.Radius,
		  koerper->Form.Kugel.Radius,
		  koerper->Form.Kugel.Radius);
	  AusgabeTexture(koerper);
	  AusgabeLage(koerper->q, koerper->Position);
	  fprintf(handle, "}\n\n");
	  break;

       default:				  /* Code fuer POV-Ray V2.x */
	  fprintf(handle, "object\n{\n");
	  fprintf(handle, "    %s\n",PovrayKugel);
	  fprintf(handle, "    scale %f\n", koerper->Form.Kugel.Radius);
	  AusgabeTexture(koerper);
	  AusgabeLage(koerper->q, koerper->Position);
	  fprintf(handle, "}\n\n");
	  break;
     } /* switch (raytracer) */
}		    




/*****************************************************************************
 *  Funktion:       void AusgabeNagel(TKoerper *koerper)
 *  Parameter:      koerper:  Koerper-Struktur des Punktes/Nagels
 *
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   handle:  File, in das ausgegeben wird
 *
 *  Beschreibung:
 *  -------------
 *  Momentan wird bei einem Nagel gar nichts ausgegeben.
 *****************************************************************************/

static void AusgabeNagel(TKoerper *koerper)
{
     /* Bei einem Nagel wird gar nichts ausgegeben. Vielleicht kann man auch */
     /* eine kleine Kugel ausgeben. Mal sehen. */
}



/*****************************************************************************
 *  Funktion:       void AusgabeEbene(TKoerper *koerper)
 *  Parameter:      koerper:  Koerper-Struktur der Ebene
 *
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   handle:  File, in das ausgegeben wird
 *
 *  Beschreibung:
 *  -------------
 *  Die Ebene wird in das File ausgegeben.
 *****************************************************************************/

static void AusgabeEbene(TKoerper *koerper)
{
     switch (raytracer)
     {
       case POVRAY1:			  /* Code fuer POV-Ray V1.0 */
	  fprintf(handle, "object\n{\n");
	  fprintf(handle, "    plane { %s }\n", PovrayEbene);
	  AusgabeTexture(koerper);
	  AusgabeLage(koerper->q, koerper->Position);
	  fprintf(handle, "}\n\n");
	  break;

       default:				  /* Code fuer POV-Ray V2.x */
	  fprintf(handle, "object\n{\n");
	  fprintf(handle, "    %s\n", PovrayEbene);
	  AusgabeTexture(koerper);
	  AusgabeLage(koerper->q, koerper->Position);
	  fprintf(handle, "}\n\n");
	  break;
     } /* switch (raytracer) */
}



/*****************************************************************************
 *  Funktion:       void AusgabeKoerper(TKoerper *koerper)
 *
 *  Parameter:      koerper: Zeiger auf die 3D-Koerper-Daten im aktuellen Zustand
 *
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   handle:  File, in das die Koerper ausgegeben werden.
 *
 *  Beschreibung:
 *  -------------
 *  Die Liste mit verzeigerten Koerpern wird als Raytracer-Quelltext in das
 *  File ausgegeben.
 *****************************************************************************/

static void AusgabeKoerper(TKoerper *koerper)
{
     do
     {

	  if (!(koerper->AStatus & HiddenON))
	  {
#if 0
	       /* Falls Koerper-Koordinatenachsen gezeichnet werden sollen, 
		  diese jetzt hier berechnen */
	       if (modus & ObjectCoordON) 
	       {
		    switch (koerper->Art)
		    {
		      case KUGEL:	  /* nur bei KUGEL, QUADER, ZYLINDER,  */
		      case QUADER:	  /* EBENE und ZUSGESOBJ sind Koordina- */
		      case EBENE:	  /* tenachsen sinnvoll */
		      case ZYLINDER:
		      case ZUSGESOBJ:
			 {
			      TVektor m, xe, ye, ze;	  
			      
			      /* Koerperschwerpunkt */
			      TransformationPunkt(m, 0, 0, 0);

			      /* Nacheinander die 3 Achsenrichungen */
			      TransformationPunkt(xe, 0.2, 0, 0); 
			      TransformationPunkt(ye, 0, 0.2, 0);
			      TransformationPunkt(ze, 0, 0, -0.2);

			      /* Koerper-Achsen clippen und hinzeichnen */
			      ClipLinie3d(m, xe);
			      ClipLinie3d(m, ye);
			      ClipLinie3d(m, ze);
			 }
			 break;
		      default:
			 break;
		    }  /* switch (koerper->Art) */
	       }  /* if (modus & ObjectCoordON) */
#endif

	       /* Koerpernummer als Kommentar ausgeben */
	       switch (raytracer)
	       {
		 case POVRAY1:
		    fprintf(handle, "/* O#%li */\n",koerper->KoerperId);
		    break;
		 default:
		    fprintf(handle, "/* O#%li */\n",koerper->KoerperId);
		    break;
	       }

	       switch (koerper->Art)
	       {
		 case KUGEL:
		    AusgabeKugel(koerper);
		    break;
		    
		 case QUADER:
		    AusgabeQuader(koerper);
		    break;
		    
		 case ZYLINDER:
		    AusgabeZylinder(koerper);
		    break;
		
		 case EBENE:
		    AusgabeEbene(koerper);
		    break;
    
		 case MPUNKT:		  /* wie Nagel */
		 case PUNKT:		  /* wie Nagel */
		 case NAGEL:
		    AusgabeNagel(koerper);
		    break;

		 case ZUSGESOBJ:
		    switch (raytracer)
		    {
		      case POVRAY1:
			 AusgabeKoerper(koerper->Form.ZusGesObj.KoerperListe);
			 break;
		      default:
			 fprintf(handle, "merge{\n");
			 AusgabeKoerper(koerper->Form.ZusGesObj.KoerperListe);
			 fprintf(handle, "} /*merge*/\n");
			 break;
		    }
		    
		 default: 
		    break;
	       } /* switch (koerper->Art) */

	  } /* if (!(koerper->AStatus & HiddenON)) */

	  koerper = koerper->Naechster;

     } while (koerper!=NULL);

}




/*****************************************************************************
 *  Funktion:       QuaternionVonVektor(TVektor v, TReal laenge, TQuaternion q)
 *
 *  Parameter:      v:       Zielvektor
 *                  laenge:  Laenge von v
 *                     
 *  Rueckgabewert:  q:       Quaternion, das (0,0,laenge) auf v dreht
 *
 *  Import. Bez.:   handle:  Handle des Files, in das ausgegeben wird
 *
 *  Beschreibung:
 *  -------------
 *  Aus dem angegebenen Vektor v und dessen Laenge wird ein Quaternion
 *  berechnet, das den Vektor (0,0,laenge) auf v dreht. Dies dient zur
 *  Berechnung der Lage einer Verbindung.
 *****************************************************************************/

void QuaternionVonVektor(TVektor v, TReal laenge, TQuaternion q)
{
     if (NAHE_0(laenge))
     {					  /* keine Drehung */
	  q[0] = 1;
	  q[1] = 0;
	  q[2] = 0;
	  q[3] = 0;
     }
     else if (NAHE_0(laenge+v[2]))
     {					  /* Drehung um 180 deg um x */
	  q[0] = 0;			  
	  q[1] = 1;
	  q[2] = 0;
	  q[3] = 0;
     }
     else				  /* Quaternion wie fuer Matrix in */
     {					  /* MatrixVon2Punkten() */
	  q[0] = sqrt((v[2]+laenge)/(2*laenge));
	  q[1] = v[1]/sqrt(2*laenge*(v[2]+laenge));
	  q[2] = -v[0]/sqrt(2*laenge*(v[2]+laenge));
	  q[3] = 0;
     }
}     



/*****************************************************************************
 *  Funktion:       void AusgabeVerbParameter(TVektor p1, TVektor p2)
 *
 *  Parameter:      p1, p2:  Endpunkte der Verbindung
 *                     
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   handle:  Handle des Files, in das ausgegeben wird
 *
 *  Beschreibung:
 *  -------------
 *  Ausgabe aller Parameter der Verbindung: Skalierung auf die richtige
 *  Laenge, Ausgabe der Textur, Drehung in die korrekte Lage und
 *  Verschiebung an die richtige Position.
 *****************************************************************************/

static void AusgabeVerbParameter(TVektor p1, TVektor p2, char *texturename)
{
     TVektor v, winkel;
     TQuaternion q;
     TReal laenge;

     v[0] = p2[0]-p1[0];		  /* Vektor von p1 nach p2 */
     v[1] = p2[1]-p1[1];
     v[2] = p2[2]-p1[2];
     
     laenge = sqrt(v[0]*v[0]+v[1]*v[1]+v[2]*v[2]);
     
     QuaternionVonVektor(v, laenge, q);
     EulerWinkelFromQuaternion(q, winkel);

     if (NAHE_0(laenge))
	  laenge = EPSILON;		  /* sonst Warning im Raytracer */

     switch (raytracer)
     {
       case POVRAY1:			  /* Code fuer POV-Ray V1.0 */
	  fprintf(handle, "    texture { %s }\n", texturename);
	  fprintf(handle, "    scale < 1.0 1.0 %f >\n", laenge);

	  if (!(NAHE_0(winkel[2])))
	       fprintf(handle, "    rotate < 0 0 %f >\n",winkel[2]);
	  if (!(NAHE_0(winkel[0])))
	       fprintf(handle, "    rotate < %f 0 0 >\n",winkel[0]);
	  if (!(NAHE_0(winkel[1])))
	       fprintf(handle, "    rotate < 0 %f 0 >\n",winkel[1]);

	  fprintf(handle, "    translate < %f %f %f >\n", p1[0], p1[1], p1[2]);
	  break;

       default:				  /* Code fuer POV-Ray V2.x */
	  fprintf(handle, "    texture { %s }\n", texturename);
	  fprintf(handle, "    scale < 1.0, 1.0, %f >\n", laenge);
	  
	  if (!(NAHE_0(winkel[2])))
	       fprintf(handle, "    rotate < 0, 0, %f >\n",winkel[2]);
	  if (!(NAHE_0(winkel[0])))
	       fprintf(handle, "    rotate < %f, 0, 0 >\n",winkel[0]);
	  if (!(NAHE_0(winkel[1])))
	       fprintf(handle, "    rotate < 0, %f, 0 >\n",winkel[1]);

	  fprintf(handle, "    translate < %f, %f, %f >\n", p1[0], p1[1], p1[2]);
	  break;
     } /* switch (raytracer) */
}




/*****************************************************************************
 *  Funktion:       void AusgabeStange(TVektor p1, TVektor p2)
 *
 *  Parameter:      p1, p2:  Endpunkte der Stange
 *                     
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   handle:  Handle des Files, in das ausgegeben wird
 *
 *  Beschreibung:
 *  -------------
 *  Es wird eine Stange passender Laenge ins File geschrieben.
 *****************************************************************************/

static void AusgabeStange(TVektor p1, TVektor p2)
{
     switch (raytracer)
     {
       case POVRAY1:			  /* Code fuer POV-Ray 1.0 */
	  fprintf(handle, "object\n{\n");
	  fprintf(handle, "    intersection { %s }\n",PovrayStange);
	  AusgabeVerbParameter(p1, p2, PovrayStangenTextur);
	  fprintf(handle, "}\n\n");
	  break;

       default:				  /* Code fuer POV-Ray 2.x */
	  fprintf(handle, "object\n{\n");
	  fprintf(handle, "    %s\n",PovrayStange);
	  AusgabeVerbParameter(p1, p2, PovrayStangenTextur);
	  fprintf(handle, "}\n\n");
	  break;
     } /* switch (raytracer) */
}



/*****************************************************************************
 *  Funktion:       void ProjektionFeder(x1,y1,x2,y2, TReal ruhelaenge, 
 *                                                    TReal federkonst)
 *
 *  Parameter:      x1, y1,
 *                  x2, y2:  2D-Koordinaten der beiden Stangenendpunkte
 *                  ruhelaenge: Ruhelaenge der Feder
 *                  federkonst: Federkonstante der Feder
 *
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   -
 *
 *  Beschreibung:
 *  -------------
 *  Es wird eine Spiralfeder dreidimensional hingezeichnet. Die Anzahl der
 *  Windungen ist abhaengig von der Ruhelaenge der Feder.
 *  Die Feder wird in einer Defaultposition definiert und dann durch eine
 *  Matrix an die entsprechende Verbindungsposition im Raum abgebildet.
 *****************************************************************************/

#define WindungenProM 30
#define FR 0.05
#define PunkteProWindung 12

/* Punkte fuer eine Windung bei 12 Punkten pro Windung: */
#define C00  1
#define C30  0.8660254
#define C60  0.5
#define C90  0
#define S00  C90
#define S30  C60
#define S60  C30
#define S90  C00

static TReal windungx[PunkteProWindung] = 
                {C00, C30, C60, C90, -C60, -C30, -C00, -C30, -C60, C90, C60, C30};
static TReal windungy[PunkteProWindung] = 
                {S00, S30, S60, S90, S60, S30, S00, -S30, -S60, -S90, -S60, -S30};

static void POV1AusgabeFeder(TVektor p1, TVektor p2, TReal ruhelaenge, TReal federkonst)
{
     TVektor v, winkel;
     TQuaternion q;
     TReal laenge, ganghoehe, teillaenge, schraeg;
     int   windungen, i, j;

     v[0] = p2[0]-p1[0];		  /* Vektor von p1 nach p2 */
     v[1] = p2[1]-p1[1];
     v[2] = p2[2]-p1[2];	
     
     laenge = sqrt(v[0]*v[0]+v[1]*v[1]+v[2]*v[2]);
     if (NAHE_0(laenge)) return;
     
     windungen = (int) (ruhelaenge*WindungenProM);
     ganghoehe = laenge*.8/(windungen+0.5);
     
     teillaenge = FR*0.535898384862;	  /* FR*2*sqrt((2-sqrt(3))/(2+sqrt(3))) */
			           /* ist genauer als FR*2*PI/PunkteProWindung, */
                                   /* da 12-Eck und nicht Kreis */
     teillaenge = sqrt(teillaenge*teillaenge+ganghoehe*ganghoehe/
		       (PunkteProWindung*PunkteProWindung));
     schraeg = asinInDeg(ganghoehe/(PunkteProWindung*teillaenge));

     fprintf(handle, "object\n{\n");
     fprintf(handle, "    union\n    {\n");

     /* Anfang der Feder */
     fprintf(handle, "        intersection { %s scale <1.0 1.0 %f> }\n",
	     PovrayStange, laenge*0.1);
     fprintf(handle, "        intersection { %s scale <1.0 1.0 %f> rotate <0 90 0> rotate <0 0 -15> translate <0.0 0.0 %f> }\n", PovrayStange, FR, 0.1*laenge);

     for (i=0; i<windungen; i++)
     {
	  for (j=0; j<PunkteProWindung; j++)
	  {
	       fprintf(handle, "        intersection { %s scale <1.0 %f 1.0> rotate <%f 0 0> rotate <0 0 %d> translate <%f %f %f> }\n",PovrayFederTeil, teillaenge, schraeg, j*30, FR*windungx[j], FR*windungy[j], 0.1*laenge+i*ganghoehe+j*ganghoehe/PunkteProWindung);
	  }
     }

     /* noch eine halbe Windung am Ende */
     for (j=0; j<PunkteProWindung/2; j++)
     {
	       fprintf(handle, "        intersection { %s scale <1.0 %f 1.0> rotate <%f 0 0> rotate <0 0 %d> translate <%f %f %f> }\n",PovrayFederTeil, teillaenge, schraeg, j*30, FR*windungx[j], FR*windungy[j], 0.1*laenge+windungen*ganghoehe+j*ganghoehe/PunkteProWindung);
     }

     /* Ende der Feder */
     fprintf(handle, "        intersection { %s scale <1.0 1.0 %f> rotate <0 -90 0> rotate <0 0 -15> translate <0.0 0.0 %f> }\n", PovrayStange, FR, 0.9*laenge);

     fprintf(handle, "        intersection { %s scale <1.0 1.0 %f> translate <0.0 0.0 %f> }\n", PovrayStange, laenge*0.1, laenge*0.9);
     
     fprintf(handle, "    }\n");

     QuaternionVonVektor(v, laenge, q);
     EulerWinkelFromQuaternion(q, winkel);

     fprintf(handle, "    texture { %s }\n", PovrayFederTextur);
     fprintf(handle, "    rotate < 0 0 %f >\n",winkel[2]);
     fprintf(handle, "    rotate < %f 0 0 >\n",winkel[0]);
     fprintf(handle, "    rotate < 0 %f 0 >\n",winkel[1]);
     fprintf(handle, "    translate < %f %f %f >\n", p1[0], p1[1], p1[2]);


     /* als Begrenzungsobjekt einen Zylinder angeben, der die Feder umschliesst */
     fprintf(handle, "    bounded_by\n    {\n");
     fprintf(handle, "        intersection\n        {\n           %s\n", 
	     PovrayFederBound);
     fprintf(handle, "            scale <%f %f %f>\n", FR*1.5, FR*1.5, laenge);
     fprintf(handle, "            rotate < 0 0 %f >\n",winkel[2]);
     fprintf(handle, "            rotate < %f 0 0 >\n",winkel[0]);
     fprintf(handle, "            rotate < 0 %f 0 >\n",winkel[1]);
     fprintf(handle, "            translate < %f %f %f >\n", p1[0], p1[1], p1[2]);
     fprintf(handle, "        }\n    }\n");
     

     fprintf(handle, "}\n\n");		  /* Ende des Objekts */

}

static void POV2AusgabeFeder(TVektor p1, TVektor p2, TReal ruhelaenge, TReal federkonst)
{
     TVektor v, winkel;
     TQuaternion q;
     TReal laenge, ganghoehe, teillaenge, schraeg;
     int   windungen, i, j;

     v[0] = p2[0]-p1[0];		  /* Vektor von p1 nach p2 */
     v[1] = p2[1]-p1[1];
     v[2] = p2[2]-p1[2];	
     
     laenge = sqrt(v[0]*v[0]+v[1]*v[1]+v[2]*v[2]);
     if (NAHE_0(laenge)) return;
     
     windungen = (int) (ruhelaenge*WindungenProM);
     ganghoehe = laenge*.8/(windungen+0.5);
     
     teillaenge = FR*0.535898384862;	  /* FR*2*sqrt((2-sqrt(3))/(2+sqrt(3))) */
			           /* ist genauer als FR*2*PI/PunkteProWindung, */
                                   /* da 12-Eck und nicht Kreis */
     teillaenge = sqrt(teillaenge*teillaenge+ganghoehe*ganghoehe/
		       (PunkteProWindung*PunkteProWindung));
     schraeg = asinInDeg(ganghoehe/(PunkteProWindung*teillaenge));
     
     fprintf(handle, "object\n{\n");
     fprintf(handle, "    union\n    {\n");

     /* Anfang der Feder */
     fprintf(handle, "        object { %s scale <1.0, 1.0, %f> }\n",
	     PovrayStange, laenge*0.1);
     fprintf(handle, "        object { %s scale <1.0, 1.0, %f> rotate <0, 90, 0> rotate <0, 0, -15> translate <0.0, 0.0, %f> }\n", PovrayStange, FR, 0.1*laenge);

     for (i=0; i<windungen; i++)
     {
	  for (j=0; j<PunkteProWindung; j++)
	  {
	       fprintf(handle, "        object { %s scale <1.0, %f, 1.0> rotate <%f, 0, 0> rotate <0, 0, %d> translate <%f, %f, %f> }\n",PovrayFederTeil, teillaenge, schraeg, j*30, FR*windungx[j], FR*windungy[j], 0.1*laenge+i*ganghoehe+j*ganghoehe/PunkteProWindung);
	  } /* for (j=0; j<PunkteProWindung; j++) */
     } /* for (i=0; i<windungen; i++) */

     /* noch eine halbe Windung am Ende */
     for (j=0; j<PunkteProWindung/2; j++)
     {
	  fprintf(handle, "        object { %s scale <1.0, %f, 1.0> rotate <%f, 0, 0> rotate <0, 0, %d> translate <%f, %f, %f> }\n",PovrayFederTeil, teillaenge, schraeg, j*30, FR*windungx[j], FR*windungy[j], 0.1*laenge+windungen*ganghoehe+j*ganghoehe/PunkteProWindung);
     } /* for (j=0; j<PunkteProWindung/2; j++) */

     /* Ende der Feder */
     fprintf(handle, "        object { %s scale <1.0, 1.0, %f> rotate <0, -90, 0> rotate <0, 0, -15> translate <0.0, 0.0, %f> }\n", PovrayStange, FR, 0.9*laenge);
     fprintf(handle, "        object { %s scale <1.0, 1.0, %f> translate <0.0, 0.0, %f> }\n", PovrayStange, laenge*0.1, laenge*0.9);
     fprintf(handle, "    }\n");

     QuaternionVonVektor(v, laenge, q);
     EulerWinkelFromQuaternion(q, winkel);

     fprintf(handle, "    texture { %s }\n", PovrayFederTextur);
     fprintf(handle, "    rotate < 0, 0, %f >\n",winkel[2]);
     fprintf(handle, "    rotate < %f, 0, 0 >\n",winkel[0]);
     fprintf(handle, "    rotate < 0, %f, 0 >\n",winkel[1]);
     fprintf(handle, "    translate < %f, %f, %f >\n", p1[0], p1[1], p1[2]);


     /* als Begrenzungsobjekt einen Zylinder angeben, der die Feder umschliesst */
     fprintf(handle, "    bounded_by\n    {\n");
     fprintf(handle, "        object\n        {\n           %s\n", 
	     PovrayFederBound);
     fprintf(handle, "            scale <%f, %f, %f>\n", FR*1.5, FR*1.5, laenge);
     fprintf(handle, "            rotate < 0, 0, %f >\n",winkel[2]);
     fprintf(handle, "            rotate < %f, 0, 0 >\n",winkel[0]);
     fprintf(handle, "            rotate < 0, %f, 0 >\n",winkel[1]);
     fprintf(handle, "            translate < %f, %f, %f >\n", p1[0], p1[1], p1[2]);
     fprintf(handle, "        }\n    }\n");
     
     fprintf(handle, "}\n\n");		  /* Ende des Objekts */

}

static void AusgabeFeder(TVektor p1, TVektor p2, TReal ruhelaenge, TReal federkonst)
{
     switch (raytracer)
     {
       case POVRAY1:			  /* Code fuer POV-Ray 1.0 */
	  POV1AusgabeFeder(p1, p2, ruhelaenge, federkonst);
	  break;

       default:				  /* Code fuer POV-Ray 2.x */
	  POV2AusgabeFeder(p1, p2, ruhelaenge, federkonst);
	  break;
     } /* switch (raytracer) */
}




/*****************************************************************************
 *  Funktion:       void AusgabeGelenk(TVektor p1, TVektor p2)
 *
 *  Parameter:      p1, p2:  Endpunkte des Gelenks
 *                     
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   handle:  Handle des Files, in das ausgegeben wird
 *
 *  Beschreibung:
 *  -------------
 *  Es wird ein Gelenk ins File geschrieben.
 *****************************************************************************/

static void AusgabeGelenk(TVektor p1, TVektor p2)
{
     switch (raytracer)
     {
       case POVRAY1:			  /* Code fuer POV-Ray 1.0 */
	  fprintf(handle, "object\n{\n");
	  fprintf(handle, "    union { %s }\n",PovrayGelenk);
	  AusgabeVerbParameter(p1, p2, PovrayGelenkTextur);
	  fprintf(handle, "}\n\n");
	  break;

       default:				  /* Code fuer POV-Ray 2.x */
	  fprintf(handle, "object\n{\n");
	  fprintf(handle, "    %s\n",PovrayGelenk);
	  AusgabeVerbParameter(p1, p2, PovrayGelenkTextur);
	  fprintf(handle, "}\n\n");
	  break;
     } /* switch (raytracer) */
}



/*****************************************************************************
 *  Funktion:       void AusgabeDaempfer(TVektor p1, TVektor p2)
 *
 *  Parameter:      p1, p2:  Endpunkte des Daempfers
 *                     
 *  Rueckgabewert:  -
 *
 *  Import. Bez.:   handle:  Handle des Files, in das ausgegeben wird
 *
 *  Beschreibung:
 *  -------------
 *  Es wird ein Daempfer passender Laenge ins File geschrieben.
 *****************************************************************************/

static void AusgabeDaempfer(TVektor p1, TVektor p2)
{
     switch (raytracer)
     {
       case POVRAY1:			  /* Code fuer POV-Ray 1.0 */
	  fprintf(handle, "object\n{\n");
	  fprintf(handle, "    union { %s }\n",PovrayDaempfer);
	  AusgabeVerbParameter(p1, p2, PovrayDaempferTextur);
	  fprintf(handle, "}\n\n");
	  break;

       default:				  /* Code fuer POV-Ray 2.x */
	  fprintf(handle, "object\n{\n");
	  fprintf(handle, "    %s\n",PovrayDaempfer);
	  AusgabeVerbParameter(p1, p2, PovrayDaempferTextur);
	  fprintf(handle, "}\n\n");
	  break;
     } /* switch (raytracer) */
}



/*****************************************************************************
 *  Funktion:       void AusgabeVerbindungen(TVerbindung *verbindung)
 *
 *  Parameter:      verbindung: Zeiger auf die 3D-Verbindungs-Daten im aktuellen
 *                              Zustand
 *
 *  Import. Bez.:   handle:     File, in das die Verbindungen ausgegeben
 *                              werden sollen
 *                            
 *  Rueckgabewert:  -
 *
 *  Beschreibung:
 *  -------------
 *  Die Liste mit verzeigerten Verbindungen wird in das Raytracer-File
 *  ausgegeben. Dabei werden die Verbindungen durch Grundkoerper
 *  dargestellt, die auf korrekte Laenge und in die korrekte Lage gebracht
 *  werden. Nach wie vor stellt die Feder das groesste Problem dar.
 *****************************************************************************/

static void AusgabeVerbindungen(TVerbindung *verbindung)
{
     AMatrix M_verb;			  /* Matrix fuer Verbindungsdrehung */
     TVektor p1, p2;			  /* 3D-Koordinaten der Verbindungspunkte */

     do
     {
	  MatrixFromQuaternion(verbindung->Koerper1->q, M_verb);
	  
	  M_verb[0][3] = verbindung->Koerper1->Position[0];
	  M_verb[1][3] = verbindung->Koerper1->Position[1];
	  M_verb[2][3] = verbindung->Koerper1->Position[2];
	  
	  ATransformation(p1, M_verb, verbindung->VPunkt1[0], 
			  verbindung->VPunkt1[1],
			  verbindung->VPunkt1[2]);
	  
	  MatrixFromQuaternion(verbindung->Koerper2->q, M_verb);
	  
	  M_verb[0][3] = verbindung->Koerper2->Position[0];
	  M_verb[1][3] = verbindung->Koerper2->Position[1];
	  M_verb[2][3] = verbindung->Koerper2->Position[2];
	  
	  ATransformation(p2, M_verb, verbindung->VPunkt2[0], 
			  verbindung->VPunkt2[1],
			  verbindung->VPunkt2[2]);


	  if (!(verbindung->AStatus & HiddenON))
	  {
	       switch (verbindung->Art)
	       {
		 case FEDER:
		    AusgabeFeder(p1, p2,
				 verbindung->VerParameter.Feder.Ruhelaenge,
				 verbindung->VerParameter.Feder.Federkonstante);
		    break;
		    
		 case STANGE:
		    AusgabeStange(p1, p2);
		    break;
		    
		 case DAEMPFER:
		    AusgabeDaempfer(p1, p2);
		    break;
		    
		 case GELENK:
		    AusgabeGelenk(p1, p2);
		    break;
	       } /* switch (verbindung->Art) */
	  } /* if (!(verbindung->AStatus & HiddenON)) */

	  verbindung = verbindung->Naechste;
    } while (verbindung!=NULL);

}




/*****************************************************************************
 *  Funktion:       void AusgabeBild(TVektor kampos, TQuaternion kq, TReal zoom, 
 *                                   TZustand *zustand)
 *
 *  Parameter:      kampos:  Absolute Position der Kamera
 *                  kq:      Ausrichtung der Kamera
 *                  zoom:    Zoomfaktor der Kamera
 *                  zustand: Abzubildender Zustand
 *
 *  Rueckgabewert:  -
 *
 *  Beschreibung:
 *  -------------
 *  Der aktuelle Zustand wird als Raytracer-Quelltext in das (schon
 *  geoeffnete) File ausgegeben.
 *****************************************************************************/

static void AusgabeBild(TVektor kampos, TQuaternion kq, TReal zoom, 
			TZustand *zustand, int framenumber)
{
     TKoerper *koerper;
     

     switch (raytracer)
     {
       case POVRAY1:			  /* Code fuer POV-Ray V1.0 */
	  fprintf(handle, "/* POV-Ray V1.0 raytracer scene file created by AERO */\n\n");
	  fprintf(handle, "#include \"%s\"\n",POVRAY1BASICFILENAME);
	  fprintf(handle, "#include \"%s\"\n\n",POVRAY1MATERIALFILENAME);

	  fprintf(handle, "\n/*----------------------------------------------------------------------*/\n\n");
	  fprintf(handle, "/* Definition of camera and light_sources */\n\n");
	  AusgabeKamera(kampos, kq, zoom);	  /* Kamera mit Lichtquellen ausgeben */
	  
	  if (zustand == NULL) return;	  /* Wenn Zustand leer, ist nichts zu tun */
     
	  koerper = zustand->Koerper;
	  if (koerper != NULL)		  /* Wenn keine Koerper existieren, dann */
					  /* existieren auch keine Verbindungen */
	  {
	       /* alle Koerper ins File schreiben */
	       fprintf(handle, "\n/*----------------------------------------------------------------------*/\n\n");
	       fprintf(handle, "/* Definition of Objects */\n\n");
	       AusgabeKoerper(koerper);
	       if (zustand->Verbindungen != NULL)
	       {
		    /* alle Verbindungen ins File schreiben */
		    fprintf(handle, "\n/*----------------------------------------------------------------------*/\n\n");
		    fprintf(handle, "/* Definition of Links */\n\n");
		    AusgabeVerbindungen(zustand->Verbindungen);
	       }
	  }
     
	  /* #### Fussboden ausgeben #### */
	  break;

       default:				  /* Code fuer POV-Ray V2.x */
	  fprintf(handle, "/* POV-Ray V2.x raytracer scene file created by AERO */\n\n");
	  fprintf(handle, "#declare %s = %f\n", POVRAY2TIME, zustand->Zeit);
	  fprintf(handle, "#declare %s = %i\n\n", POVRAY2FRAME, framenumber);

	  AusgabeKamera(kampos, kq, zoom);	  /* Kamera mit Lichtquellen ausgeben */
	  
	  
	  fprintf(handle, "\n/*----------------------------------------------------------------------*/\n\n");
	  fprintf(handle, "/* Include basic shapes, materials and extensions */\n\n");
	  fprintf(handle, "#include \"%s\"\n", POVRAY2INCLUDE);

	  if (zustand == NULL) return;	  /* Wenn Zustand leer, ist nichts zu tun */
     
	  koerper = zustand->Koerper;
	  if (koerper != NULL)		  /* Wenn keine Koerper existieren, dann */
					  /* existieren auch keine Verbindungen */
	  {
	       /* alle Koerper ins File schreiben */
	       fprintf(handle, "\n/*----------------------------------------------------------------------*/\n\n");
	       fprintf(handle, "/* Definition of Objects */\n\n");
	       AusgabeKoerper(koerper);
	       if (zustand->Verbindungen != NULL)
	       {
		    /* alle Verbindungen ins File schreiben */
		    fprintf(handle, "\n/*----------------------------------------------------------------------*/\n\n");
		    fprintf(handle, "/* Definition of Links */\n\n");
		    AusgabeVerbindungen(zustand->Verbindungen);
	       }
	  }
     
	  /* #### Fussboden ausgeben #### */
	  break;
     } /* switch (raytracer) */
}




/*****************************************************************************
 *  Funktion:       void ErzeugeRaytracerBild(char *grundname, TKamera kamera, 
 *                                   TZustand *zustand, int modus, int rayvers,
 *                                                             int framenumber)
 *
 *  Parameter:      grundname:   Pfad und Grundname des Files (einschliesslich 
 *                               Framenummer, ohne Extension)
 *                  kamera:      Position, Lage und zoom der Kamera
 *                  zustand:     Zeiger auf den darzustellenden Zustand
 *                  modus:       Art der Ausgabe (Bitvektor)
 *                               Bit 0: Fussboden zeichnen (1) oder nicht (0)
 *                               Bit 1: Koordinatenachsen der Fenster zeichnen (1)
 *                                      oder nicht (0)
 *                               Bit 2: Koordinatenachsen der einzelnen Koerper
 *                                      zeichnen (1) oder nicht (0)
 *                  rayvers:     Zu verwendende Raytracer-Version:
 *                                  1: POV-Ray V1.0
 *                                  2: POV-Ray V2.x
 *                  framenumber: Nummer des aktuellen AERO-Frames
 *
 *  Rueckgabewert:  -
 *
 *  Importierte Bezeichner: -
 *
 *  Beschreibung:
 *  -------------
 *  Mit dieser Routine wird der angegebene Zustand als Quelltext fuer den
 *  Raytracer ausgegeben. Dabei wird davon ausgegangen, dass es ein
 *  File mit den Texturen fuer die einzelnen Materialien gibt.
 *  ######## Hier fehlt noch eine ausfuerlichere Beschreibung ########
 *****************************************************************************/

void ErzeugeRaytracerBild(char *grundname, TKamera kamera, TZustand *zustand, 
			  int animmodus, int rayvers, int framenumber)
{ 
     char filename[500];

     /* Werte global setzen, um Parameter zu sparen */
     materialtabelle = zustand->MaterialDaten;
     raytracer = rayvers;

     strcpy(filename, grundname);
     if (animmodus & Show3dON)
     {
	  TQuaternion augenrichtung;	  /* Blickrichtung links/rechts */
	  TVektor augenposition;	  /* Augenposition links/rechts */
	  AMatrix M_auge;		  /* Matrix fuer Augenabstand */
	  TVektor rechts;		  /* Basis-Vektor fuer "rechts von Kamera" */
	  /* Quaternions fuer die Augenparallaxe linkes und rechtes Auge */
	  TQuaternion parallaxe_l = { 0.99999846876, 0, 0.00174999196, 0};
	  TQuaternion parallaxe_r = { 0.99999846876, 0, -0.00174999196, 0};

	  MatrixFromQuaternion(kamera.Richtung, M_auge);
	  ATransformation(rechts, M_auge, AUGENABSTAND/2, 0.0, 0.0);

	  /* Bild fuer das linke Auge berechnen */
	  augenposition[0] = kamera.Position[0]-rechts[0];
	  augenposition[1] = kamera.Position[1]-rechts[1];
	  augenposition[2] = kamera.Position[2]-rechts[2];
	  AQuaternionMult(kamera.Richtung, parallaxe_l, augenrichtung);

	  /* Filenamens-Extension je nach Raytracer anhaengen */
	  switch (raytracer)
	  {
	    default:	  
	       strcat(filename, ".l.pov");
	       break;
	  }

	  handle = fopen(filename, "w");
	  if (handle==NULL)
	       fprintf(stderr, "Could not open file %s\n",filename);
	  else
	  {
	       AusgabeBild(augenposition, augenrichtung, kamera.Zoom, zustand, framenumber);
	       fclose(handle);
	  }



	  /* Bild fuer das rechte Auge berechnen */
	  augenposition[0] = kamera.Position[0]+rechts[0];
	  augenposition[1] = kamera.Position[1]+rechts[1];
	  augenposition[2] = kamera.Position[2]+rechts[2];
	  AQuaternionMult(kamera.Richtung, parallaxe_r, augenrichtung);

	  strcpy(filename, grundname);

	  /* Filenamens-Extension je nach Raytracer anhaengen */
	  switch (raytracer)
	  {
	    default:	  
	       strcat(filename, ".r.pov");
	       break;
	  }
	  handle = fopen(filename, "w");
	  if (handle==NULL)
	       fprintf(stderr, "Could not open file %s\n",filename);
	  else
	  {
	       AusgabeBild(augenposition, augenrichtung, kamera.Zoom, zustand, framenumber);
	       fclose(handle);
	  }
     }
     else
     {
	  /* Filenamens-Extension je nach Raytracer anhaengen */
	  switch (raytracer)
	  {
	    default:	  
	       strcat(filename, ".pov");
	       break;
	  }
	  handle = fopen(filename, "w");
	  if (handle==NULL)
	       fprintf(stderr, "Could not open file %s\n",filename);
	  else
	  {
	       AusgabeBild(kamera.Position, kamera.Richtung, kamera.Zoom, zustand, framenumber);
	       fclose(handle);
	  }
     }


}

