;;
;; universal .emacs for Linux and Suns
;;
;; Sets Fonts for the Font-Lock mode
;; 
;; key bindings ctrl-Arrows (move by words)
;;              shift-Arrows (move and set mark at the same time)
;;
;; Loads some extension that should reside in the site-lisp 
;; directory or another directory in the load-path
;; The extensions are loaded inside a condition-case error-handler
;; so execution won't stop in case of missing files
;;
;; 
;; Created by Stefan Strobel 
;; strobel@sun1.rz.fh-heilbronn.de
;; Please send bug reports and comments to the above adress
;; 
;; A detailed description of this file and the extensions
;; together with an introduction in emacs-lisp programming
;; will be available in the next edition of our book :
;;
;; Linux - from PC to Workstation 
;; Stefan Strobel and Thomas Uhl
;; Springer Verlag, ISBN 3-540-57383-6
;;
;;


;;;
;;; Check for special command-line args
;;; (actually we just check for -ne which inhibits loading
;;;  of some extensions and speeds up the startup)
;;;  

(defvar load-exts 
  (if (member "-ne" command-line-args)
      (progn
	(setq command-line-args (delete "-ne" command-line-args))
	nil)
    t)
  "should we load our extensions")
    

;;;
;;; Settings
;;;

;; if files are in a different directory and don't get loaded 
;; extend the load-path here !
;; example : 

(setq load-path (cons "/usr/local/lib/emacs/site-lisp/auctex" load-path))
(setq load-path (cons "/usr/local/lib/emacs/site-lisp/swi" load-path))
(setq load-path (cons "/usr/local/lib/emacs/site-lisp/ilisp" load-path))

;;
;; In some Unix environments, emacs is located in /usr/lib/emacs, so
;; you might have to change the above path settings or create a symlink:
;;   ln -s /usr/lib/emacs /usr/local/lib/emacs

;; On a Linux System you will need the following 
;; (and you can delete the above path settings)

(setq load-path (cons "/usr/lib/emacs/site-lisp/auctex" load-path))
(setq load-path (cons "/usr/lib/emacs/site-lisp/swi" load-path))
(setq load-path (cons "/usr/lib/emacs/site-lisp/ilisp" load-path))

(setq default-major-mode 'text-mode)          ; Default mode for unknown files
(setq default-fill-column 78)                 ; Text width
(setq transient-mark-mode t)                  ; Make the region visible
(put 'eval-expression 'disabled nil)          ; allow eval with Meta-esc

;;(setq make-backup-files nil)                  ; no backup files

;;(setq vc-suppress-confirm t)                  ; for sccs/rcs commands
;; this confirmation should not be suppressed. 
;; Otherwise the revert commands wont work

(if load-exts
    (standard-display-european t))            ; enable display of accents.


;;;
;;; Set the default faces for the font-lock-mode
;;;

(setq font-lock-function-name-face 'bold)
(setq font-lock-comment-face       'italic)
(setq font-lock-string-face        'default)
(setq font-lock-doc-string-face    'default)

;;;
;;; Set special face attributes 
;;; (inside an error handler in case something is not available)
;;;

(if window-system
    (condition-case nil 
	(progn
	  (set-face-foreground  'region "black")
	  (set-face-background  'region "grey90")
	  (if (not (x-display-color-p))
	      (set-face-underline-p 'region t))
	  
	  (set-face-foreground  'bold   "black")
	  (set-face-background  'bold   "white")
	  (set-face-underline-p 'bold    nil)
	  
	  (if (x-display-color-p)
	      (progn
		(copy-face 'default   'comment)
		(set-face-foreground  'comment "grey60")
		(set-face-background  'comment "white")
		(setq font-lock-comment-face 'comment)
		
		(copy-face 'default   'string)
		(set-face-foreground  'string "dark slate grey")
		(set-face-background  'string "white")
		(setq font-lock-string-face 'string)
		
		(copy-face 'default   'docstr)
		(set-face-foreground  'docstr "blue")
		(set-face-background  'docstr "white")
		(setq font-lock-doc-string-face 'docstr))))
      
      (error nil)))


;;;
;;; Set the fonts in Emacs
;;; (also in an error handler in case some fonts dont exist)
;;;

(if window-system
    (condition-case nil 
	(progn
	  (set-face-font 'bold 
			 "-adobe-courier-bold-r-normal--*-*-75-75-m-*-*-1")
	  (set-face-font 'italic 
			 "-adobe-courier-medium-o-normal--*-*-75-75-m-*-*-1")
	  (set-face-font 'comment 
			 "-adobe-courier-medium-o-normal--*-*-75-75-m-*-*-1")
	  (set-face-font 'bold-italic 
			 "-adobe-courier-bold-o-normal--*-*-75-75-m-*-*-1"))
      (error nil)))



;;
;; Set up the new C / C++ Mode
;; (this should no longer be necessary when emacs 19.23 comes out)
;;

(fmakunbound 'c-mode)
(makunbound 'c-mode-map)
(fmakunbound 'c++-mode)
(makunbound 'c++-mode-map)
(autoload 'c++-mode "cc-mode" "C++ Editing Mode" t)
(autoload 'c-mode   "cc-mode" "C Editing Mode" t)

(makunbound 'c-style-alist)

(setq auto-mode-alist
      (append '(("\\.C$"  . c++-mode)
		("\\.cc$" . c++-mode)
		("\\.hh$" . c++-mode)
		("\\.c$"  . c-mode)
		("\\.h$"  . c-mode)
		) auto-mode-alist))



;;;
;;; Load some additions / extensions
;;;

      
(condition-case nil               ; saveplace - save the current position 
    (progn                        ; for files (has to activated with C-x p
      (load "saveplace"))
  (error nil))


(condition-case nil
    (quietly-read-abbrev-file)    ; read our abreviation file
  (error nil))



(if load-exts
    (progn

      (condition-case nil         ; jka-compress - load / save compressed files
	  (progn                 
	    (load "jka-compr")
	    (setq auto-mode-alist 
		  (cons '("\\.c\\(\\.gz\\|\\.Z\\)?\\'" . c-mode)
			auto-mode-alist))
	    (setq auto-mode-alist 
		  (cons '("\\.cc\\(\\.gz\\|\\.Z\\)?\\'" . c++-mode)
			auto-mode-alist)))
	(error nil))
      

      (condition-case nil         ; autoinsert - insert a file in new files
	  (progn                  ; depending on its extension (for templates)
	    (load "autoinsert")
	    ;; default path for insert files is ~/insert
	    (setq auto-insert-alist
		  (nconc '(("\\.el$" . "elisp-insert.el"))
			 auto-insert-alist)))
	(error nil))


      (condition-case nil         ; functions for shift-arrows (allow to 
	  (load "add-functions")  ; set the mark and move at the same time)
	(error nil))
      

      (condition-case nil         ; functions and configuration for ilisp 
	  (load "add-ilispconf")  ; with the cmu-lisp system
	(error nil))


      (condition-case nil         ; Mode line dragging function
	  (progn 
	    (load "mldrag")
	    (global-set-key [mode-line down-mouse-1] 
			    'mldrag-drag-mode-line)
	    (global-set-key [vertical-line down-mouse-1] 
			    'mldrag-drag-vertical-line)
	    (global-set-key [vertical-scroll-bar S-down-mouse-1]
			    'mldrag-drag-vertical-line))
	(error nil))
      
      ))


;;;
;;; Set up hooks for other modes
;;;

(add-hook 'text-mode-hook 
	  '(lambda ()
	     (turn-on-auto-fill)))           ; Auto fill in Text mode
	 

(add-hook 'emacs-lisp-mode-hook 
	  '(lambda () 
	     (if window-system 
		 (condition-case nil     
		     (progn 
		       (load "func-menu")    ; load function menu
		       (define-key global-map [S-down-mouse-3] 'function-menu))
		   (error nil)))
	     ;;
	     ;; set up a new menu in the elisp mode
	     ;;
	     (define-key emacs-lisp-mode-map [menu-bar elisp]
	       (cons "Elisp" (make-sparse-keymap "elisp")))
	     (define-key emacs-lisp-mode-map [menu-bar elisp debonenoff]
	       '("Cancel Debug on Entry" . cancel-debug-on-entry))
	     (define-key emacs-lisp-mode-map [menu-bar elisp debonen]
	       '("Debug on Entry" . debug-on-entry))
	     (define-key emacs-lisp-mode-map [menu-bar elisp debdefun]
	       '("Debug Defun" . edebug-defun))
	     (define-key emacs-lisp-mode-map [menu-bar elisp evalbuff]
	       '("Eval Buffer" . eval-buffer))
	     (define-key emacs-lisp-mode-map [menu-bar elisp evalreg]
	       '("Eval Region" . eval-region))
	     (define-key emacs-lisp-mode-map [menu-bar elisp evaldef]
	       '("Eval Defun" . eval-defun))
	     (font-lock-mode 1)    
	     (line-number-mode 1)))          ; line-numbers

(add-hook 'c++-mode-hook
	  '(lambda () 
	     (if window-system 
		 (condition-case nil     
		     (progn 
		       (load "func-menu")    ; load function menu
		       (define-key global-map [S-down-mouse-3] 'function-menu))
		   (error nil)))
	     ;;(font-lock-mode 1)
	     (line-number-mode 1)))

(add-hook 'c-mode-hook
	  '(lambda () 
	     (if window-system 
		 (condition-case nil     
		     (progn 
		       (load "func-menu")    ; load function menu
		       (define-key global-map [S-down-mouse-3] 'function-menu))
		   (error nil)))
	     (define-key c-mode-map  [delete]  'delete-char)
	     (font-lock-mode t)
	     (line-number-mode 1)))

(add-hook 'lisp-mode-hook
	  '(lambda () 
	     (line-number-mode 1)))
 
(add-hook 'gud-mode-hook 
	  '(lambda ()
	     (load-inspect-mode)))

(add-hook 'tcl-mode-hook
	  '(lambda ()
	     (define-key tcl-mode-map  [delete]  'delete-char)
	     (if window-system
		 (font-lock-mode t))
	     (line-number-mode 1)))


(add-hook 'protalk-mode-hook 
	  '(lambda ()
	     (line-number-mode 1)
	     (condition-case nil
		 (load "pkhelp")
	       (error nil))))

(add-hook 'outline-mode-hook
	  '(lambda ()
	     (line-number-mode 1)
	     (setq outline-regexp "[^#\n\^M]")
	     (setq outline-level 'outline-indent-level)
	     (define-key outline-mode-map [return] 'outline-newline)
	     (define-key outline-mode-map [up] 'outline-up)
	     (define-key outline-mode-map [down] 'outline-down)))

(or (assoc "\\.outl$" auto-mode-alist)
    (setq auto-mode-alist (cons '("\\.outl$" . outline-mode) 
				auto-mode-alist)))



;;
;; Modes to autoload
;; -----------------
;;


;;
;; Inspector -- Inspect variables in another window while debugging with gdb
;;

(autoload 'load-inspect-mode "inspect" 
  "Inspector addition" t)


;;
;; ediff  -- show diffs between two files in two windows, patch selectively
;; epatch -- apply patches selectively to a file
;;

(autoload 'ediff-buffers "ediff" "Visual interface to diff(1)" t)
(autoload 'ediff  "ediff"  "Visual interface to diff(1)" t)
(autoload 'ediff-files "ediff" "Visual interface to diff(1)" t)
(autoload 'ediff-files-remote "ediff" "Visual interface to diff(1)") 
(autoload 'epatch  "ediff"  "Visual interface to patch(1)" t)
(autoload 'ediff-patch-file "ediff" "Visual interface to patch(1)" t)
(autoload 'ediff-patch-buffer "ediff" "Visual interface to patch(1)" t)
(autoload 'vc-ediff "ediff"
  "Interface to diff & version control via vc.el" t) 
(autoload 'rcs-ediff "ediff"
  "Interface to diff & version control via rcs.el" t)


;;
;; Folding Mode - turn emacs into a folding editor
;;

(autoload 'folding-mode "folding" 
  "Minor mode that simulates a folding editor" t)
;;
;; If you want files folded automatically , 
;; then uncomment the following two lines.
;;(folding-mode t)
;;(folding-mode-add-find-file-hook)


;;
;; HTML mode
;;

(autoload 'html-mode "html-mode" "HTML major mode." t)
(or (assoc "\\.html$" auto-mode-alist)
    (setq auto-mode-alist (cons '("\\.html$" . html-mode) 
				auto-mode-alist)))



;;
;; TCL mode
;; --------
;;

(autoload 'tcl-mode "tcl-mode" "TCL major mode." t)
(or (assoc "\\.tcl$" auto-mode-alist)
    (setq auto-mode-alist (cons '("\\.tcl$" . tcl-mode) 
				auto-mode-alist)))



;;
;; Postscript Mode
;; --------------
;;

(autoload 'postscript-mode "postscript.el" "" t)
(setq auto-mode-alist
      (cons '("\\.c?ps$".postscript-mode) auto-mode-alist))


;;
;; Smalltalk-mode
;; --------------
;;
(setq auto-mode-alist
      (append  '(("\\.st$" . smalltalk-mode))
	       auto-mode-alist))

(autoload 'smalltalk-mode "st.el" "" t)


;;
;; lispdir -- Search for/retrieve additional packages in Emacs-Lisp-Archive
;; -------
;;

(autoload 'format-lisp-code-directory "lispdir" nil t)
(autoload 'lisp-dir-apropos "lispdir" nil t)
(autoload 'lisp-dir-retrieve "lispdir" nil t)
(autoload 'lisp-dir-verify "lispdir" nil t)


;;
;; auc-tex
;; -------
;;

(autoload 'tex-mode "auc-tex" "Automatic select TeX or LaTeX mode" t)
(autoload 'plain-tex-mode "auc-tex" "Mode for Plain TeX" t)
(autoload 'latex-mode "auc-tex" "Mode for LaTeX" t)
(autoload 'LaTeX-math-mode    "tex-math"      "Math mode for TeX." t)
(autoload 'outline-minor-mode "outline-m" "Minor Outline Mode." t)



;;
;; swi-prolog 
;; ----------
;;

(autoload 'run-prolog "swi-prolog" "Run an inferior prolog process" t)
(autoload 'prolog-mode "swi-prolog" "SWI-Prolog mode" t)



;;
;; Font-lock for dired 
;; ===================
;;


(setq dired-font-lock-keywords
      '(("\\S +\\([~%#]\\)$" . font-lock-doc-string-face)
	("\\S +\\.\\([oszZ]\\|elc\\|gz\\)$" . font-lock-string-face)
	("^  \\(/.+\\)$" 1 font-lock-type-face)
	("[^ ]+ -> [^ ]+$" . font-lock-function-name-face)
	("^..\\(.....w....\\|........w.\\)" 1 font-lock-comment-face)
	("^[^ ].*$" 0 font-lock-comment-face t)
	("^..d.* \\([^ ]+\\)$" 1 font-lock-keyword-face)))

(add-hook 'dired-mode-hook
	  (function (lambda ()
		      (if window-system
			  (font-lock-mode t))
		      (setq font-lock-keywords dired-font-lock-keywords))))


;;;
;;; Kappa
;;; -----
;;;

(setq auto-mode-alist
      (append '(("\\.pt$"  . protalk-mode)
		("\\.ptk$" . protalk-mode)
		("\\.pkc$" . c-mode))
              auto-mode-alist))

(autoload 'protalk-mode "protalk"
  "ProTalk editing mode" t)



;;;
;;; Fvwm config Files
;;; -----------------
;;;

(setq auto-mode-alist
      (append '(("\\.fvwm$" . fvwm-mode)
  		(".fvwmrc$" . fvwm-mode)) 
	      auto-mode-alist))

(autoload 'fvwm-mode "fvwm-mode"
  "Mode for editing fvwm config files")

(add-hook 'fvwm-mode-hook
	  (function (lambda ()
		      (line-number-mode t)
		      (if window-system
			  (progn
			    (font-lock-mode t)
			    (setq font-lock-keywords fvwm-font-lock-keywords)
			    (font-lock-fontify-buffer))))))



;;;
;;; Keys
;;;

(define-key ctl-x-map "p"  'toggle-save-place)

(global-set-key [f31]      'blink-matching-open) 
(global-set-key [f27]      'beginning-of-line)              ; Home
(global-set-key [f33]      'end-of-line)                    ; End
(global-set-key [f29]      'scroll-down)                    ; PgUp
(global-set-key [f35]      'scroll-up)                      ; PgDn
(global-set-key [C-f29]    'beginning-of-buffer)            ; C-PgUp
(global-set-key [C-f35]    'end-of-buffer)                  ; C-PgDn

(global-set-key [home]     'beginning-of-line)              ; Home
(global-set-key [end]      'end-of-line)                    ; End
(global-set-key [prior]    'scroll-down)                    ; PgUp
(global-set-key [next]     'scroll-up)                      ; PgDn

(global-set-key [C-right]  'forward-word)
(global-set-key [C-left]   'backward-word)
(global-set-key [C-prior]  'beginning-of-buffer)
(global-set-key [C-next]   'end-of-buffer)

(global-set-key [cancel]   'keyboard-quit)
(global-set-key [find]     'search-forward)

(global-set-key [f5]       'goto-line)
(global-set-key [S-f5]     'what-line)                           
(global-set-key [f6]       'tags-search)
(global-set-key [S-f6]     'visit-tags-table)
                           
(global-set-key [M-f8]     'next-error)
(global-set-key [f9]       'compile)                           
(global-set-key [f10]      'next-error)                           
(global-set-key [f12]      'advertised-undo)

                  

;;;
;;; remap Backspace and Delete
;;;
         
(global-unset-key [backspace])
(global-set-key   [backspace] 'delete-backward-char) 

(define-key emacs-lisp-mode-map       [delete] 'delete-char)
(define-key lisp-mode-map             [delete] 'delete-char)
(define-key lisp-interaction-mode-map [delete] 'delete-char)
(define-key text-mode-map             [delete] 'delete-char)
(define-key minibuffer-local-map      [delete] 'delete-char)
(define-key minibuffer-local-completion-map [delete]  'delete-char)
;(define-key tex-mode-map             [delete] 'delete-char)


;;;
;;; for rect-mark
;;;
(define-key ctl-x-map "r\C-@"     'rm-set-mark)
(define-key ctl-x-map [?r ?\C-\ ] 'rm-set-mark)
(define-key ctl-x-map "r\C-x"     'rm-exchange-point-and-mark)

(autoload 'rm-set-mark "rect-mark"
  "Set mark for rectangle." t)
(autoload 'rm-exchange-point-and-mark "rect-mark"
  "Exchange point and mark for rectangle." t)


;;
;; console key settings for linux
;; ------------------------------
;;

(defvar cursor-map-2 (make-keymap) 
  "Keymap for ESC-[")
(fset 'Cursor-Map-2 cursor-map-2)
(define-key esc-map "[" 'Cursor-Map-2)
 
(define-key esc-map "[A"  'previous-line)
(define-key esc-map "[B"  'next-line)
(define-key esc-map "[C"  'forward-char)
(define-key esc-map "[D"  'backward-char)
(define-key esc-map "[H"  'beginning-of-line)
(define-key esc-map "[Y"  'end-of-line)
(define-key esc-map "[1~" 'beginning-of-line)
(define-key esc-map "[4~" 'end-of-line)
(define-key esc-map "[5^" 'scroll-down)
(define-key esc-map "[6^" 'scroll-up)
(define-key esc-map "[[A" 'help-for-help)
(define-key esc-map "[[B" 'byte-compile-file)
(define-key esc-map "[[C" 'isearch-forward)
(define-key esc-map "[[D" 'query-replace-regexp)
(define-key esc-map "[[E" 'eval-defun)
(define-key esc-map "[[F" 'eval-current-buffer)
(define-key esc-map "[[G" 'buffer-menu)
(define-key esc-map "[[H" 'global-set-key)
(define-key esc-map "[[I" 'save-buffer)
(define-key esc-map "[[J" 'save-buffers-kill-emacs)
(define-key esc-map "[2^" 'set-mark-command)
(define-key esc-map "[3^" 'delete-char)

 
(defun Ctl-C-prefix ()
  Ctl-C-keymap)
(setq Ctl-C-keymap (make-keymap))
(global-set-key "\C-c" (Ctl-C-prefix))
 
(define-key Ctl-C-keymap "m" 'LaTeX-math-mode)
(define-key Ctl-C-keymap "o" 'outline-minor-mode)
 
(make-variable-buffer-local 'outline-prefix-char)
(make-variable-buffer-local 'outline-regexp)
(make-variable-buffer-local 'outline-level-function)

(setq-default outline-prefix-char "\C-l")
(setq-default outline-regexp      "[*\l]+")
(setq-default outline-level-function 'outline-level-default)

