;;;
;;; A slightly modified and heavily documented version of the logic
;;; programming query system presented in section 4.5 of Abelson &
;;; Sussman.  Sections 4.4 and 4.5 should be read before attempting to
;;; understand this code.
;;;

;;;; Utilities.

;;; Make-Table returns a Dispatch function that in turn returns either
;;; the Lookup function or the Insert! function.  The two-key table is kept
;;; in the shared local variable Local-Table, which is an association list
;;; that maps one key to an association that maps a second key to some value.
;;; (From A&S section 3.3.3.)

;;; We've added a Remove operation so that we can clear things out of the
;;; database.  Remove takes two keys.  If either one is Nil, everything along
;;; that "axis" will be removed.

(define (make-table)
  (let ((local-table (list '*table*)))
    (define (lookup key-1 key-2)
      (let ((subtable (assq key-1 (cdr local-table))))
        (if (null? subtable)
            nil
            (let ((record (assq key-2 (cdr subtable))))
              (if (null? record)
                  nil
                  (cdr record))))))

    (define (insert! key-1 key-2 value)
      (let ((subtable (assq key-1 (cdr local-table))))
        (if (null? subtable)
            (set-cdr! local-table
                      (cons (list key-1
                                  (cons key-2 value))
                            (cdr local-table)))
            (let ((record (assq key-2 (cdr subtable))))
              (if (null? record)
                  (set-cdr! subtable
                            (cons (cons key-2 value)
                                  (cdr subtable)))
                  (set-cdr! record value)))))
       'ok)
    (define (remove! key-1 key-2)
      (mapcar (lambda (subtable)
		(if (or (null? key-1)
			(eq? key-1 (car subtable)))
		    (mapcar (lambda (entry)
			      (if (or (null? key-2)
				      (eq? key-2 (car entry)))
				  (set-cdr! entry nil)))
			    (cdr subtable))))
	      (cdr local-table))
      'ok)
    (define (dispatch m)
      (cond ((eq? m 'lookup-proc) lookup)
            ((eq? m 'insert-proc!) insert!)
	    ((eq? m 'remove-proc!) remove!)
            (else (error "Unknown operation -- TABLE" m))))

    dispatch))


;;; The global Operation-Table stores the table we use for all our
;;; Get, Put, and Remove operations.

(define operation-table (make-table))
(define get (operation-table 'lookup-proc))
(define put (operation-table 'insert-proc!))
(define remove (operation-table 'remove-proc!))


;;; Singleton is a shorthand we use to make a one-element stream.

(define (singleton x) (cons-stream x the-empty-stream))

;;;; The Driver Loop and Instantiation (4.5.1).

;;; Query-Driver-Loop is a sort of Read-Eval-Print loop that prompts for
;;; a query system command and "evaluates" it.  If the command is an
;;; assertion to be added to the system, we call Add-Rule-Or-Assertion
;;; with it. Otherwise, it is taken to be a query.  In this case, we
;;; Qeval the query to get a stream of frames, each element of which we
;;; instantiate and print out.

(define (query-driver-loop)
  (newline)
  (princ "Query==> ")
  (let ((q (query-syntax-process (read))))
    (cond ((assertion-to-be-added? q)
	   (add-rule-or-assertion!
	    (add-assertion-body q))
	   (princ "Assertion added to data base."))
	  (else
	   (sequence
	    (print-stream-elements-on-separate-lines
	     (map-stream (lambda (frame)
			   (instantiate q
					frame
					(lambda (v f) 
					  (contract-question-mark v))))
			 (qeval q (singleton '()))))))))
  (query-driver-loop))

;;; To instantiate a frame, we copy it, replacing any variables in the
;;; expression by their values in the given Frame.  The values are
;;; themselves instantiated, since they could contain variables.  The
;;; Unbound-Var-Handler, passed in, gets called with any variables that
;;; do not have values specified in the Frame.

(define (instantiate exp frame unbound-var-handler)
  (define (copy exp)
    (cond ((constant? exp) exp)
          ((var? exp)
           (let ((vcell (binding-in-frame exp frame)))
             (if (null? vcell)             
                 (unbound-var-handler exp frame)
                 (copy (binding-value vcell)))))
          (else (cons (copy (car exp))
                      (copy (cdr exp))))))
  (copy exp))

;;;; The Evaluator (4.5.2).

;;; Qeval takes as inputs a Query and a stream of frames, Frame-Stream,
;;; and returns a stream of extended frames.  "Special forms" are
;;; identified in a data-directed way using Get.  Any query not
;;; identified as a special form is a simple query, and is processed by
;;; Asserted?

(define (qeval query frame-stream)
  (show-stream frame-stream) (pp query)
  (let ((qproc (get (type query) 'qeval)))
    (if (not (null? qproc))
        (qproc (contents query) frame-stream)
        (asserted? query frame-stream))))


;;; Asserted? handles simple queries.  For each frame in the
;;; Frame-Stream, Find-Assertions matches the Query-Pattern against all
;;; assertions in the database, producing a stream of extended frames,
;;; and Apply-Rules applies all possible rules, producing another stream
;;; of extended frames.  These two streams are combined by
;;; Append-Delayed to make a stream of all the ways the Query-Pattern
;;; can be satisfied, consistent with the original frame.

(define (asserted? query-pattern frame-stream)
  (flatmap
   (lambda (frame)
     (append-delayed
      (find-assertions query-pattern frame)
      (delay (apply-rules query-pattern frame))))
   frame-stream))

;;; Append-Delayed appends two streams, delaying the second if possible.

(define (append-delayed s1 delayed-s2)
  (if (empty-stream? s1)
      (force delayed-s2)
      (cons-stream (head s1)
                   (append-delayed (tail s1) delayed-s2))))

;;;; Compound Queries.

;;; Conjoin implements the And special form.  Conjoin first processes
;;; the Frame-Stream by finding the stream of all possible extensions to
;;; the first query in the conjunction.  Then it recursively applies
;;; Conjoin to the rest of the queries.  Thus, each recursive call acts
;;; as one stage in a sequence of conjuncts, one stage for each conjunt.

;;; New procedures follow

(define (conjoin conjuncts frame-stream)
  (if (empty-conjunction? conjuncts)
      frame-stream
      (merge-frame-streams 
       (qeval (first-conjunct conjuncts) frame-stream)
       (conjoin (rest-conjuncts conjuncts) frame-stream))))

(define (merge-frame-streams stream1 stream2)
  (remove-failed-frames
   (flatmap
    (lambda (frame1)
      (map-stream
       (lambda (frame2)
	 (merge-frames frame1 frame2))
       stream2))
    stream1)))

(define (remove-failed-frames stream)
  (cond ((empty-stream? stream)
	 the-empty-stream)
	((equal? (head stream) 'failed)
	 (remove-failed-frames (tail stream)))
	(else (cons-stream (head stream)
			   (remove-failed-frames (tail stream))))))

(define (merge-frames frame1 frame2)
  (if (empty-frame? frame1)
      frame2
      (let ((var (binding-variable (first-binding frame1)))
	    (val (binding-value (first-binding frame1)))
	    (rest (rest-bindings frame1)))
	(let ((result (extend-if-possible var val frame2)))
	  (if (equal? result 'failed)
	      'failed
	      (merge-frames rest result))))))


(define (first-binding frame)
  (car frame))

(define (rest-bindings frame)
  (cdr frame))

(define (empty-frame? frame)
  (null? frame))

;;; End new procedures


;;; Install And as a Qeval special form, implemented with Conjoin:

(put 'and 'qeval conjoin)


;;; Disjoin implements the Or special form.  Each of the disjuncts is
;;; processed separately, and the resulting streams are interleaved.

(define (disjoin disjuncts frame-stream)
  (if (empty-disjunction? disjuncts)
      the-empty-stream
      (interleave-delayed
       (qeval (first-disjunct disjuncts) frame-stream)
       (delay (disjoin (rest-disjuncts disjuncts)
                       frame-stream)))))

;;; Install Or as a Qeval special form, implemented with Disjoin:

(put 'or 'qeval disjoin)


;;; Negate implements the Not special form.  If we cannot extend a frame
;;; in the Frame-Stream to match the query A, we include that frame in
;;; the output stream.

;;; Section 4.5.2 -- Filters

(define (negate a frame-stream)
  (flatmap
   (lambda (frame)
     (if (empty-stream? (qeval (negated-query a)
                               (singleton frame)))
         (singleton frame)
         the-empty-stream))
   frame-stream))

;;; Install Not as a Qeval special form, implemented with Negate:

(put 'not 'qeval negate)


;;; Lisp-Value implements the Lisp-Value special form.  Each frame in
;;; the Frame-Stream is instantiated and the indicated predicate
;;; applied, and the frames for which the predicate returns false are
;;; filtered out of the input stream.  We signal an error if there are
;;; any unbound pattern variables in the instantiated frame.

(define (lisp-value call frame-stream)
  (flatmap
   (lambda (frame)
     (if (execute
          (instantiate call
                       frame
                       (lambda (v f)
			   (error "Unknown pat var--LISP-VALUE"
                                v))))
         (singleton frame)
         the-empty-stream))
   frame-stream))


;;; Execute is a little helper function that applies the predicate to
;;; the instantiated frame.

(define (execute exp)
  (apply (eval (predicate exp) user-initial-environment)
         (args exp)))

;;; Install Lisp-Value as a Qeval special form:

(put 'lisp-value 'qeval lisp-value)


;;; Always-True passes on the Frame-Stream ignoring its arguments.  This
;;; is used by Rule-Body to provide bodies for rules that were defined
;;; without bodies (i.e., rules whose conclusions are always satisfied.

(define (always-true ignore frame-stream)
  frame-stream)                                        

;;; Install Always-True as a Qeval special form:

(put 'always-true 'qeval always-true)

;;;; Finding Assertions by Pattern Matching (4.5.3).

;;; Find-Assertions returns a stream of frames, each one an extension of
;;; the given Frame formed by matching the given pattern against
;;; assertions in the database.

(define (find-assertions pattern frame)
  (flatmap (lambda (datum)
             (pattern-match pattern datum frame))
           (fetch-assertions pattern frame)))

;;; Pattern-Match tries to extend the Frame according to the given
;;; pattern.  Returns The-Empty-Stream if the match fails, or a
;;; singleton stream of the extended frame if it works.

(define (pattern-match pat dat frame)
  (let ((result (internal-match pat dat frame)))
    (if (eq? result 'failed)
        the-empty-stream
        (singleton result))))


;;; Internal-Match returns Failed if the frame cannot be extended
;;; according to the pattern, or an extended frame if it can be.

(define (internal-match pat dat frame)
  (cond

   ;; If a recursive call failed, we fail:
   ((eq? frame 'failed) 'failed)

   ;; If pattern is a variable, extend the frame by binding the variable
   ;; to the data:
   ((var? pat) (extend-if-consistent pat dat frame))

   ;; If pattern is a constant, it has to match the data.
   ((constant? pat)
    (if (constant? dat)
	(if (same-constant? pat dat) frame 'failed)
	'failed))

   ;; If the data is a constant, it can't match.
   ((constant? dat) 'failed)

   ;; Otherwise, they're both lists, so try matching the car and the
   ;; cdr... 
   (else (internal-match (cdr pat)
			 (cdr dat)
			 (internal-match (car pat)
					 (car dat)
					 frame)))))


;;; Extend-If-Consistent extends the Frame if the Var is currently
;;; unbound in that frame, or if the new and old values are the same (as
;;; determined by Internal-Match).

(define (extend-if-consistent var dat frame)
  (let ((value (binding-in-frame var frame)))
    (if (null? value)
        (extend var dat frame)
        (internal-match (binding-value value) dat frame))))

;;;; Rules and Unification (4.5.4).

;;; Apply-rules forms a stream of extension frames by applying rules
;;; from the database ro the Pattern and Frame.

(define (apply-rules pattern frame)
  (flatmap (lambda (rule)
             (apply-a-rule rule pattern frame))
           (fetch-rules pattern frame)))

;;; Apply-A-Rule first augments the Query-Frame by unifying the Rule
;;; conclusion with the Query-Pattern.  If this succeeds, it evaluates
;;; the rule body in this new frame.

(define (apply-a-rule rule query-pattern query-frame)
  (let ((clean-rule (rename-variables-in rule)))
    (let ((unify-result (unify-match query-pattern
                                     (conclusion clean-rule)
                                     query-frame)))
      (if (empty-stream? unify-result)
          the-empty-stream
          (qeval (rule-body clean-rule) unify-result)))))

;;; Rename-Variables-In is used to generate unique variables for
;;; applying the rule.

(define (rename-variables-in rule)
  (let ((rule-application-id (new-rule-application-id)))
    (define (tree-walk exp)
      (cond ((constant? exp) exp)
            ((var? exp)
             (make-new-variable exp rule-application-id))
            (else (cons (tree-walk (car exp))
                        (tree-walk (cdr exp))))))
    (tree-walk rule)))


;;; Unify-Match tries to extend the given Frame by unifying the two
;;; given patterns.  If this works, a singleton stream of the extended
;;; frame is returned, otherwise, The-Empty-Stream is returned.

(define (unify-match p1 p2 frame)
  (let ((result (internal-unify p1 p2 frame)))
    (if (eq? result 'failed)
        the-empty-stream
        (singleton result))))

;;; Internal-Unify is much like the pattern matcher, except that it's
;;; symmetrical -- variables are allowed on both sides of the match.

(define (internal-unify p1 p2 frame)
  (cond

   ;; If a recursive call failed, we fail:
   ((eq? frame 'failed) 'failed)

   ;; If either pattern is a variable, try extending by binding it to
   ;; the other thing:
   ((var? p1) (extend-if-possible p1 p2 frame))
   ((var? p2) (extend-if-possible p2 p1 frame))

   ;; If one pattern is a constant, it has to match the other.
   ((constant? p1)
    (if (constant? p2)
	(if (same-constant? p1 p2) frame 'failed)
	'failed))

   ;; If the other is a constant, it can't match.
   ((constant? p2) 'failed)

   ;; Otherwise, they're both lists, so try matching the car and the cdr
   ;; of each...
   (else (internal-unify (cdr p1)
			 (cdr p2)
			 (internal-unify (car p1)
					 (car p2)
					 frame)))))

;;; Extend-If-Possible extends the Frame if the Var is currently
;;; unbound in that frame, or if the new and old values are the same (as
;;; determined by Internal-Unify).  Just like Extend-if-Consistent,
;;; except for some esoteric cases described on pp. 371-372.

(define (extend-if-possible var val frame)
  (if (equal? var val)                         ;***
      frame
      (let ((value-cell (binding-in-frame var frame)))
        (if (null? value-cell)
            (if (freefor? var val frame)       ;***
                (extend var val frame)
                'failed)
            (internal-unify (binding-value value-cell)
                            val
                            frame)))))

;;; Freefor? tests whether an expression that might become the value of
;;; a variable contains that variable.  A simple tree walk, substituting
;;; the values of variables wherever necessary.

(define (freefor? var exp frame)
  (define (freewalk e)
    (cond ((constant? e) t)
          ((var? e)
           (if (equal? var e)
               nil
               (freewalk (lookup-in-frame e frame))))
          ((freewalk (car e)) (freewalk (cdr e)))
          (else nil)))
  (freewalk exp))

;;;; Maintaining the Database (4.5.5).

;;; To speed up the examination of patterns, we store all assertions
;;; whose cars are constant symbols in separate streams.  Such
;;; assertions are called "indexed assertions".


;;; THE-ASSERTIONS is a stream of all the assertions.

(define THE-ASSERTIONS the-empty-stream)


;;; Fetch-Assertions checks to see if we can use the indexed assertions,
;;; and, if so, gets just those assertions that are relevant to the
;;; Pattern.  Otherwise, gets all the assertions.

(define (fetch-assertions pattern frame)
  (if (use-index? pattern)
      (get-indexed-assertions pattern)
      (get-all-assertions)))

(define (get-all-assertions) THE-ASSERTIONS)

(define (get-indexed-assertions pattern)
  (get-stream (index-key-of pattern) 'assertion-stream))

;;; Get-Stream looks up a stream in the table and returns an empty
;;; stream if there's no stream stored there.

(define (get-stream key1 key2)
  (let ((s (get key1 key2)))
    (if (null? s) the-empty-stream s)))


;;; THE-RULES is a stream of all the rules.

(define THE-RULES the-empty-stream)


;;; Fetch-Rules checks to see if we can use the indexed rules, and, if
;;; so, gets just those rules that a relavant to the Pattern.
;;; Otherwise, gets all the rules.

(define (fetch-rules pattern frame)
  (if (use-index? pattern)
      (get-indexed-rules pattern)
      (get-all-rules)))

(define (get-all-rules) THE-RULES)

;;; Rules that begin with variables are stored with the key of ? in the
;;; table.

(define (get-indexed-rules pattern)
  (append-streams
   (get-stream (index-key-of pattern) 'rule-stream)
   (get-stream '? 'rule-stream)))


;;; Add-Rule-Or-Assertion! adds the thing to either the rules or the
;;; assertions, depending on what kind of thing it is.

(define (add-rule-or-assertion! assertion)
  (if (rule? assertion)
      (add-rule! assertion)
      (add-assertion! assertion)))

(define (add-assertion! assertion)
  (store-assertion-in-index assertion)
  (let ((old-assertions THE-ASSERTIONS))
    (set! THE-ASSERTIONS
          (cons-stream assertion old-assertions))
    'ok))

(define (add-rule! rule)
  (store-rule-in-index rule)
  (let ((old-rules THE-RULES))
    (set! THE-RULES (cons-stream rule old-rules))
    'ok))

(define (store-assertion-in-index assertion)
  (if (indexable? assertion)
      (let ((key (index-key-of assertion)))
        (let ((current-assertion-stream
               (get-stream key 'assertion-stream)))
          (put key
               'assertion-stream
               (cons-stream assertion
                            current-assertion-stream))))))

(define (store-rule-in-index rule)
  (let ((pattern (conclusion rule)))
    (if (indexable? pattern)
        (let ((key (index-key-of pattern)))
          (let ((current-rule-stream
                 (get-stream key 'rule-stream)))
            (put key
                 'rule-stream
                 (cons-stream rule
                              current-rule-stream)))))))

;;; A thing is Indexable? if its car is a constant or variable.

(define (indexable? pat)
  (or (constant-symbol? (car pat))
      (var? (car pat))))

;;; The index key for a thing is its car, or, if it's a variable, ?.

(define (index-key-of pat)
  (let ((key (car pat)))
    (if (var? key) '? key)))

(define (use-index? pat)
  (constant-symbol? (car pat)))

;;;; More Utilities (4.5.6).

;;; Abstract syntax of queries.  Pretty straightfoward list
;;; manipulation.


;;; Type and Contents are used by Qeval to take apart special forms.

(define (type exp)
  (if (atom? exp) 
      (error "Unknown expression TYPE" exp)
      (if (symbol? (car exp)) (car exp) nil)))

(define (contents exp)
  (if (atom? exp) 
      (error "Unknown expression CONTENTS" exp)
      (cdr exp)))


;;; Every assertion is typed at top level as (assert! <body>).

(define (assertion-to-be-added? exp)
  (eq? (type exp) 'assert!))

(define (add-assertion-body exp) 
  (car (contents exp)))


;;; Taking apart arguments to special forms:

(define empty-conjunction? null?)
(define first-conjunct car)
(define rest-conjuncts cdr)

(define empty-disjunction? null?)
(define first-disjunct car)
(define rest-disjuncts cdr)

(define negated-query car)

(define predicate car)
(define args cdr)


;;; Rules look like (rule <conclusion> . <body>).

(define (rule? statement)
  (if (atom? statement)
      nil
      (eq? (car statement) 'rule)))

(define conclusion cadr)

(define (rule-body rule)
  (if (null? (cddr rule))
      '(always-true)
      (caddr rule)))


;;; Variables are turned from ?x to (? x) for easier manipulation
;;; internally.

(define (query-syntax-process exp)
  (map-over-atoms expand-question-mark exp))

(define (map-over-atoms proc exp)
  (if (atom? exp)
      (proc exp)
      (cons (map-over-atoms proc (car exp))
            (map-over-atoms proc (cdr exp)))))

(define (expand-question-mark atom)
  (if (symbol? atom)
      (let ((characters (explode atom)))
        (if (eq? (car characters) '?)
            (list '? (implode (cdr characters)))
            atom))
      atom))

(define (var? exp)
  (if (atom? exp)
      nil
      (eq? (car exp) '?)))

(define constant? atom?)
(define constant-symbol? symbol?)
(define same-constant? equal?)


;;; Unique variables are constructed during rule application with the
;;; following functions.  The unique identifier for a rule application
;;; is a number, which is incremented every time a rule is applied.

(define rule-counter 0)

(define (new-rule-application-id)
  (set! rule-counter (1+ rule-counter))
  rule-counter)

(define (make-new-variable var rule-application-id)
  (cons '? (cons rule-application-id (cdr var))))

(define (contract-question-mark variable)
  (if (number? (cadr variable))  ;rule application id
      (implode (append '(?)
                       (explode (caddr variable))
                       '(-)
                       (explode (cadr variable))))
      (implode (append '(?) (explode (cadr variable))))))


;;; Frames are represented as association lists of variables and their
;;; values.  Since variables are represented internally as lists of ?
;;; and their name, we use Assoc instead of Assq to do the lookup.

(define (make-binding variable value)
  (cons variable value))

(define (binding-variable binding)
  (car binding))

(define (binding-value binding)
  (cdr binding))

(define (binding-in-frame variable frame)
  (assoc variable frame))

(define (extend variable value frame)
  (cons (make-binding variable value) frame))

(define (lookup-in-frame variable frame)
  (let ((binding (binding-in-frame variable frame)))
    (if binding
	(binding-value binding)
	())))


;;; Print-Stream-Elements-On-Separate-Lines does just that.

(define (print-stream-elements-on-separate-lines s)
  (if (empty-stream? s)
      (sequence (newline)
		(princ "Done."))
      (sequence (print (head s))
                (print-stream-elements-on-separate-lines
                 (tail s)))))

;;; Load-Assertions lets us load a whole database easily.

(define (load-assertions filename)
  (define (load-loop)
    (let ((thing (read)))
      (if (not (eof-object? thing))
	  (sequence
	   (add-rule-or-assertion! (query-syntax-process thing))
	   (newline)
	   (princ "Added ")
	   (princ thing)
	   (princ ".")
	   (load-loop)))))
  (with-input-from-file filename load-loop)
  (newline)
  (princ "Done."))


;;; Clear-Assertions clears all the known rules and assertions out of
;;; the database.  In addition to clearing the streams of all rules
;;; and assertions, we have to clear the indexed rules and assertions.
;;; We do that by using the Remove operation.

(define (clear-assertions)
  (set! THE-RULES the-empty-stream)
  (set! THE-ASSERTIONS the-empty-stream)
  (remove nil 'assertion-stream)
  (remove nil 'rule-stream)
  'ok)

(define (show-stream stream)
  (if (not (empty-stream? stream))
      (sequence (pp (head stream)) (show-stream (tail stream)))))
