# Bindings for textual documents
# Uses the regions module...regions are paragraphs here,
# Paragraphs are divided by two newlines, and
# so the only filler is the 2nd newline.

load_library_module regions.tcl

# The region functions delimit by blank lines. Looking for a newline as the
# first char in successive lines is faster than [string first [$t get ...]]

proc region_prev {t index} {
	scan [$t index $index] "%d.%d" i dummy
	for {incr i -1} {$i > 0} {incr i -1} {
		if {([$t get "$i.0"] == "\n") && (
			[$t get "$i.0 +1 char"] != "\n")} {
			incr i
			return "$i.0"
	}}
	if {[$t get 1.0] != "\n"} {return 1.0} else {return ""}
}

proc region_end {t index} {
	scan [$t index $index] "%d.%d" i dummy
	scan [$t index end] "%d.%d" e dummy
	for {incr i} {$i < $e} {incr i} {
		if {([$t get "$i.0"] == "\n")} {return "$i.0 -1c"
	}}
	if {[$t get "$e.0"] == "\n"} {return "$e.0 -1c"} else {return ""}
}

proc region_next {t index} {
	scan [$t index $index] "%d.%d" i dummy
	scan [$t index end] "%d.%d" e dummy
	for {incr i} {$i < $e} {incr i} {
		if {([$t get "$i.0"] == "\n") &&
			([$t get "$i.0 +1c"] != "\n")} {return "$i.0 +1c"
	}}
	return ""
}

# We don't want anything particularly special to be noted about paragraphs,
# however fillers should not be expanded because they are empty lines.
# Besides, any text inserted to represent the lines would confuse the region
# code.
proc filler_collapse_indicate {t start end} {return ""}


if $edit_flag {

set adjust_line_width 0

# Splits line if it is longer than length. Returns number of extra lines
# produced (0 is if line was not broken). Index is on line to break, length
# is desired length, string is contents of line.
proc split_line {t {index insert} {length ""} {string ""}} {
	if {($string == "")} {
		set string [$t get "$index linestart" "$index lineend"]
	}
	if {($length == "")} {
		global adjust_line_width 
		if $adjust_line_width {
			set length $adjust_line_width
		} else {set length [lindex [$t configure -width] 4]}}
	if {([string length $string] < $length)} {return 0}

	set offset [string last " " [string range $string 0 $length]]
	if {($offset < 0)} {return 0}
	set break [$t index "$index linestart +$offset chars"]
	$t delete $break
	$t insert $break \n
	set breaks [split_line $t "$break +1 chars" $length]
	global modified ; set modified 1
	return [incr breaks]
}

# Adjusts selected region to fit in length columns, so that no lines wrap
# If unspecified, length defaults to window width.
proc adjust_region {t {length ""}} {
	set chars [$t get sel.first sel.last]
	set m1 [gensym] ; set m2 [gensym]
	$t mark set $m1 sel.first ; $t mark set $m2 sel.last
	register_undoable_cmd $t [list undo_filter $t $m1 $m2 $chars] "Adjust $chars" "$m1 $m2"
	do_adjust_region $t $length
}

proc do_adjust_region {t length} {
	set trace sel.first
	while {([set offset [string first \n [$t get $trace sel.last]]] >= 0)} {
		set trace [$t index "$trace +$offset chars"]
		global modified ; set modified 1
		$t delete $trace ;	$t insert $trace " "
	}
	split_line $t sel.first $length
}

# A re-definition of indent_width, to ensure paragraph wrap-around.
proc indent_with {t l e} {
	set prefix [$e get]
	destroy_f_entry $t $l $e

	if {([catch {$t index sel.first}])} {beep ; return}
	if {($prefix == "")} {beep ; return}
	set mark1 [$t index sel.first]
	set mark2 [$t index sel.last]

	set chars [$t get sel.first sel.last]
	set m1 [gensym] ; set m2 [gensym]
	register_undoable_cmd $t [list undo_filter $t $m1 $m2 $chars] "Indent $chars" "$m1 $m2"

	set length [lindex [$t configure -width] 4]
	set length [expr "$length - [string length $prefix]"]
	do_adjust_region $t $length

	global modified
	set modified 1
	set sellast [$t index "sel.last linestart"]
	for {set mark [$t index "sel.first linestart"]} \
			{[$t compare $mark < $sellast]} \
			{set mark [$t index "$mark +1 lines linestart"]} {
		$t insert $mark $prefix
		}
	$t insert $mark $prefix
	$t mark set $m1 "sel.first linestart"; $t mark set $m2 sel.last
	$t tag remove sel 1.0 end
	$t tag add sel $m1 $m2
}

proc do_paragraph {t fn args} {
	if {([catch {$t get sel.first}])} {
		select_group $t insert
		eval $fn $args
		$t tag remove sel 1.0 end
	} else {eval $fn $args
}}


# Textual bindings.
proc textbind {f m} {
	global Keys
	bind all <Control-i> "do_paragraph %W indent_region %W $f"
	parse_bindings Text \
M-j			{do_paragraph %W adjust_region %W} \
$Keys(C_m)		{+split_line %W {insert -1 chars}}
	bind Text <Shift-Return> "[bind Text <Return>] ; [bind Text <Return>]"

	if {[winfo exists $m]} {
		parse_menuentries $m.edit.m {
					{"Split Paragraph" 0 S-Return}
					{"Reformat Paragraph" 0 M-j}}
}}

}

region_bind $frame $menu "Paragraph" 0
if $edit_flag {textbind $frame $menu}
$text configure -wrap word
set regions_defined 1
