#
# Module for editing Text widgets
#

# Undos are handled thus: Each cmd that can be undone stores an entry into
# TH(Undo,Data,$w). This entry contains the code necessary to undo itself, a
# human-readable title to prompt the user with what it did, and a list of marks
# that were created to accomodate the undo code.

# Clears the undo log.
proc th_Text_kill_undos {w} {
  global TH
  if {[catch "set TH(Undo,Data,$w)"]} {set TH(Undo,Data,$w) ""}

  foreach data $TH(Undo,Data,$w) {
    foreach mark [lindex $data 2] {  $w mark unset $mark}}
}

# Undoes last command.
proc th_Text_undo {w} {
  global TH
  if {[catch "set TH(Undo,Data,$w)"]} {set TH(Undo,Data,$w) ""}
  if {([llength $TH(Undo,Data,$w)] == 0)} {th_beep ; return}

  set data [lindex $TH(Undo,Data,$w) 0]
  set TH(Undo,Data,$w) [lrange $TH(Undo,Data,$w) 1 end]
  set TH(Modified,$w) 1
  eval [lindex $data 0]
  foreach mark [lindex $data 2] {  $w mark unset $mark}
}

proc th_Text_register_undoable_cmd {w code {name ""} {marks ""}} {
  global TH ;  set TH(Modified,$w) 1
  if {[catch "set TH(Undo,Data,$w)"]} {set TH(Undo,Data,$w) ""}
  if {[catch "set TH(Undo,Max,[winfo class $w])"]} {set TH(Undo,Max,[winfo class $w]) 0}

  set new_entry [list $code $name $marks]
  set TH(Undo,Data,$w) [concat [list $new_entry] $TH(Undo,Data,$w)]
  if {($TH(Undo,Max,[winfo class $w]) >= 0) && ($TH(Undo,Max,[winfo class $w]) < [llength $TH(Undo,Data,$w)])} {
    set purge_data [lindex $TH(Undo,Data,$w) $TH(Undo,Max,[winfo class $w])]
    foreach mark [lindex $purge_data 2] {  $w mark unset $mark}
    set TH(Undo,Data,$w) [lrange $TH(Undo,Data,$w) 0 [expr "$TH(Undo,Max,[winfo class $w]) - 1"]]
}}


# Simple insertion routines

proc th_Text_insert {w chars} {
  $w insert insert $chars
  $w yview -pickplace insert
  global TH ; set TH(Modified,$w) 1
}

# Figures out how to undo inserting output into text widget.
# (A common undoable option)
proc th_Text_figure_out_undo_insert {w output} {
  # Figure out how to undo pasting command.
  set ustart [th_gensym];  set uend [th_gensym]
  $w mark set $uend insert
  $w mark set $ustart "insert -[string length $output] chars"
  th_Text_register_undoable_cmd $w [list $w delete $ustart $uend] "Insert $output" "$ustart $uend"
}

proc th_Text_undoable_insert {w string} {
  th_Text_insert $w $string
  th_Text_figure_out_undo_insert $w $string
}

proc th_Text_paste_selection {w} {
  if {[catch {set chars [$w get sel.first sel.last]}]} {
    if {[catch {set chars [selection get]}]} {
      set chars "" ; th_beep}}
  th_Text_undoable_insert $w $chars
}

proc th_Text_self_insert {w {c ""}} {
  if {(![regexp . $c])} {return}
  th_Text_insert $w $c
  global TH
  if {[catch "set TH(Overwrite,$w)"]} {set TH(Overwrite,$w) 0}
  if $TH(Overwrite,$w) {if {[$w get insert] != "\n"} {$w delete insert}}
}


# Text killing

proc th_Text_kill_line {w} {
  global TH ;  set TH(Modified,$w) 1
  if {([$w get insert] == "\n")} {
    set end {insert +1 chars}
  } else {set end {insert lineend}}
  set killed [$w get insert $end]
  $w delete insert $end
  set offset [string length $killed]
  if {[catch "set TH(Kill,Mark,$w)"]} {set TH(Kill,Mark,$w) ""}

  if {($TH(Kill,Mark,$w) == "") || [$w compare insert != $TH(Kill,Mark,$w)] \
    || [catch {.th_kill index sel.first}]} {
    catch ".th_kill delete killed.first killed.last"
    set TH(Kill,Mark,$w) [$w index insert]
  }

  catch {.th_kill tag remove sel sel.first sel.last}
  if {[catch ".th_kill index killed.last" index]} {set index 1.0}
  .th_kill insert $index $killed
  .th_kill tag add sel 1.0 "$index +$offset chars"
  .th_kill tag add killed 1.0 "$index +$offset chars"
}

proc th_Text_kill_range {w start end} {
  if {([$w compare $start >= $end])} {th_beep ; return}
  set s [$w index $start]
  set e [$w index $end]

  global TH ;  set TH(Modified,$w) 1
  catch {.th_kill delete killed.first killed.last}
  catch {$w tag remove sel sel.first sel.last}
  .th_kill insert 1.0 [$w get $s $e]
  set offset [string length [$w get $s $e]]
  .th_kill tag add sel 1.0 "1.0 +$offset chars"
  .th_kill tag add killed 1.0 "1.0 +$offset chars"
  $w delete $s $e
}

proc th_Text_kill_region {w} {
  if {![catch "$w index sel.last"]} {th_Text_kill_range $w sel.first sel.last
  } elseif {![catch {set m [$w index mark]}]} {
    if {[$w compare $m <= insert]} {
      th_Text_kill_range $w $m insert
    } else {th_Text_kill_range $w insert $m}
  } else {th_beep}
}


# Text deletion

proc th_Text_delete_range {w start end {undo 1}} {
  if {([$w compare $start >= $end])} {th_beep ; return}
  set s [$w index $start]
  set e [$w index $end]

  if $undo {
    # Figure out how to undo delete
    set dead [$w get $s $e]
    th_Text_register_undoable_cmd $w [list $w insert [$w index $s] $dead] "Delete $dead"
  } else {global TH ; set TH(Modified,$w) 1}
  $w delete $s $e
  $w yview -pickplace insert
}

proc th_Text_delete_selection {w} {
  if {[catch "$w index sel.last"]} {th_beep ; return}
  th_Text_delete_range $w sel.first sel.last
}


# Filtering commands

proc th_Text_filter {w filter} {
  if {([catch {$w get sel.first}])} {
    set start insert ; set end "insert wordend"
    set selected 0
  } else {set start sel.first ; set end sel.last
    set selected 1
  }
  set word [$w get $start $end]
  set new_w [$filter $word]

  if {($word == $new_w)} {$w mark set insert "insert wordend" ; return}

  set new_wl [string length $new_w]
  $w delete $start $end
  $w insert insert $new_w
  if $selected {$w tag add sel "insert -$new_wl chars" insert}
  $w yview -pickplace insert

  # Figure out how to undo filter
  set ustart [th_gensym] ; set uend [th_gensym]
  $w mark set $uend insert
  $w mark set $ustart "insert -$new_wl chars"
  th_Text_register_undoable_cmd $w [list th_Text_undo_filter $w $ustart $uend $word] "Change $word" "$ustart $uend"
}

proc th_Text_undo_filter {w ustart uend word} {$w delete $ustart $uend ; $w insert $ustart $word}


# Transposition commands

proc th_Text_transpose_chars {w} {
  if {([$w compare insert <= 1.0])} {th_beep ; return}
  set transedchars [th_Text_transchars $w]
  # Figure out how to undo transpose
  set uinsert [th_gensym]
  $w mark set $uinsert "insert -1 chars"
  th_Text_register_undoable_cmd $w [list th_Text_undo_transpose $w $uinsert th_Text_transchars] "Transpose $transedchars" "$uinsert"
}

proc th_Text_transchars {w} {
  set c1 [$w get {insert -1 chars}]
  set c2 [$w get insert]
  $w delete {insert -1 chars} {insert +1 chars}
  $w insert insert $c2
  $w insert insert $c1
  $w yview -pickplace insert
  return "$c1 $c2"
}

proc th_Text_transpose_words {w} {
  $w mark set insert {insert wordstart}
  if {([$w compare {insert -2 chars wordstart} <= 1.0])} {th_beep ; return}
  set transedwords [th_Text_transwords $w]
  # Figure out how to undo transpose
  set uinsert [th_gensym]
  $w mark set $uinsert "insert -1 chars wordstart"
  th_Text_register_undoable_cmd $w [list th_Text_undo_transpose $w $uinsert th_Text_transwords] "Transpose $transedwords" "$uinsert"
}

proc th_Text_transwords {w} {
  set c1 [$w get insert {insert wordend}]
  $w delete insert {insert wordend}
  set c2 [$w get {insert -1 chars wordstart} insert]
  $w delete {insert -1 chars wordstart} insert
  set c3 [$w get {insert -1 chars wordstart} insert]
  $w delete {insert -1 chars wordstart} insert
  $w insert insert $c1
  $w insert insert $c2
  $w insert insert $c3
  $w yview -pickplace insert
  return "$c1$c2$c3"
}

proc th_Text_undo_transpose {w uinsert fn} {
  $w mark set insert $uinsert
  $fn $w
}


# Indenting text

proc th_Text_indent_add {w} {
  set prefix ["[th_frame $w].indent.e" get]
  th_hide_entry $w indent

  if {([catch {$w index sel.first}])} {th_beep ; return}
  if {($prefix == "")} {th_beep ; return}
  set mark1 [$w index sel.first]
  set mark2 [$w index sel.last]

  set chars [$w get sel.first sel.last]
  set m1 [th_gensym] ; set m2 [th_gensym]
  th_Text_register_undoable_cmd $w [list th_Text_undo_filter $w $m1 $m2 $chars] "Indent $chars" "$m1 $m2"
  th_Text_add_prefix $w sel.first sel.last $prefix
  $w mark set $m1 "sel.first linestart"; $w mark set $m2 sel.last
  $w tag remove sel 1.0 end
  $w tag add sel $m1 $m2
}

proc th_Text_add_prefix {w start end prefix} {
  scan [$w index $start] "%d.%d" s dummy
  scan [$w index $end] "%d.%d" e dummy
  if {$dummy == 0} {incr e -1}
  for {set t $s} {$t <= $e} {incr t} {
    $w insert "$t.0" $prefix
}}

proc th_Text_indent_delete {w} {
  set prefix ["[th_frame $w].indent.e" get]
  th_hide_entry $w indent

  if {([catch {$w index sel.first}])} {th_beep ; return}
  if {($prefix == "")} {th_beep ; return}
  set mark1 [$w index sel.first]
  set mark2 [$w index sel.last]

  set chars [$w get sel.first sel.last]
  set m1 [th_gensym] ; set m2 [th_gensym]
  th_Text_register_undoable_cmd $w [list th_Text_undo_filter $w $m1 $m2 $chars] "Indent $chars" "$m1 $m2"
  th_Text_delete_prefix $w sel.first sel.last $prefix
  $w mark set $m1 "sel.first linestart"; $w mark set $m2 sel.last
  $w tag remove sel 1.0 end
  $w tag add sel $m1 $m2
}

proc th_Text_delete_prefix {w start end prefix} {
  scan [$w index $start] "%d.%d" s dummy
  scan [$w index $end] "%d.%d" e dummy
  if {$dummy == 0} {incr e -1}
  set l [string length $prefix]
  for {set t $s} {$t <= $e} {incr t} {
    if {[string first $prefix [$w get "$t.0" "$t.0 lineend"]] == 0} {
      $w delete "$t.0" "$t.0 +$l c"
}}}


