
/* TclQddb_View.c - TCL interface routines for Qddb views.
 *
 * Copyright (C) 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "tcl.h"
#include "Qddb.h"
#include "tclQddb.h"

static Tcl_HashTable 	TclQddb_ViewHashTable;
static int		TclQddb_ViewHashTableInit = 0;
static Tcl_HashTable 	TclQddb_VarOwnerHashTable;
static int		TclQddb_VarOwnerHashTableInit = 0;
static unsigned int	TclQddb_ViewNextNumber = 0;

/* For {Get,Set}VarOwner().   var_name is the TCL
 * variable, view_ptr is the current owner.   view_ptr
 * is NULL to indicate no owner.
 *
 * This whole mess is to allow one view to supercede another
 * preexisting view.   For example, a TCL variable 'a' might
 * be used in one view, then reused in another.   Since a
 * TCL variable can be bound to only one view, we allow
 * its redefinition instead of causing an error when it is
 * used again.    This allows things like:
 * 	qddb_view define $t1 { {name var(name)} {addr var(addr)} }
 *	qddb_view define $t2 { {name var(name)} {addr var(addr)} }
 * to be valid.   The view defined on $t1 lost its TCL variables
 * when the second call is made.   The view still exists, but
 * it no longer has any TCL variables associated with it.
 */

static int TclQddb_InitView _ANSI_ARGS_((Tcl_Interp *, Schema *, char *, char *, TclQddb_View **));
static int TclQddb_DefineView _ANSI_ARGS_((Tcl_Interp *, Schema *, TclQddb_Tuple *, 
					   TclQddb_View *, char *));
static int TclQddb_SetVariable _ANSI_ARGS_((Tcl_Interp *, Schema *, TclQddb_View *, char *, char *));
static int TclQddb_RelinkDataTree _ANSI_ARGS_((Tcl_Interp *, DataTree *, char *));
static void TclQddb_FreeView _ANSI_ARGS_((TclQddb_View *));
static int TclQddb_NewView _ANSI_ARGS_((Tcl_Interp *, TclQddb_View *));
static int TclQddb_InitInstance _ANSI_ARGS_((Tcl_Interp *, Schema *, DataTree **, TclQddb_View *, Boolean));
static int TclQddb_GetViewRow _ANSI_ARGS_((Tcl_Interp *, char *));
static int TclQddb_SetViewRow _ANSI_ARGS_((Tcl_Interp *, char *, char *));
static int TclQddb_ViewLock _ANSI_ARGS_((Tcl_Interp *, TclQddb_View *, Boolean));
static int TclQddb_ViewUnlock _ANSI_ARGS_((Tcl_Interp *, TclQddb_View *));
static TclQddb_View *GetVarOwner _ANSI_ARGS_((char *));
static void SetVarOwner _ANSI_ARGS_((char *, TclQddb_View *));

/* TclQddb_View -- Manipulate qddb views.
 *
 * <view desc> <- qddb_view define <tuple desc> <attribute name/TCL var pairs>
 * qddb_view set <view desc> <row desc>
 * <row desc>  <- qddb_view get <view desc>
 * qddb_view delete <view desc>|all
 */

int TclQddb_ViewProc(clientData, interp, argc, argv)
    ClientData			clientData;
    Tcl_Interp			*interp;
    int				argc;
    char			*argv[];
{	
    Schema			*schema;
    TclQddb_View		*view = NULL;
    TclQddb_Tuple		*tuple;
    char			*tuple_name = NULL;

    if (argc < 3 || argc > 5) {
	Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
	return TCL_ERROR;
    }
    if (TclQddb_ViewHashTableInit == 0) {
	TclQddb_ViewHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_ViewHashTable, TCL_STRING_KEYS);
    }
    switch (argv[1][0]) {
    case 'd': /* define & delete */
	if (strcmp(argv[1], "define") == 0) {
	    if (argc != 4) {
		Tcl_AppendResult(interp, argv[0], " define: wrong # args", NULL);
		return TCL_ERROR;
	    }
	    if (strncmp(argv[2], "qddb_row", 8) == 0) {
		TclQddb_RowHeader *row = TclQddb_GetRows(interp, argv[2]);
		if (row == NULL)
		    return TCL_ERROR;
		tuple_name = row->tuple_name;
		tuple = TclQddb_GetTuple(interp, tuple_name);
	    } else {
		tuple_name = argv[2];
		tuple = TclQddb_GetTuple(interp, argv[2]);
	    }
	    if (tuple == NULL) {
		Tcl_AppendResult(interp, argv[0], ": bad <tuple desc> \"", argv[2], "\"", NULL);
		return TCL_ERROR;
	    }
	    if ((schema = TclQddb_GetSchema(tuple->schema_name)) == NULL) {
		Tcl_AppendResult(interp, argv[0], ": bad schema \"", tuple->schema_name, "\"", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_InitView(interp, schema, tuple->schema_name, tuple_name, &view) != TCL_OK)
		return TCL_ERROR;
	    if (TclQddb_DefineView(interp, schema, tuple, view, argv[3]) != TCL_OK) {
		TclQddb_FreeView(view);
		return TCL_ERROR;
	    }
	    if (TclQddb_NewView(interp, view) != TCL_OK)
		return TCL_ERROR;
	    if (strncmp(argv[2], "qddb_row", 8) == 0) {
		if (TclQddb_SetViewRow(interp, interp->result, argv[2]) != TCL_OK)
		    return TCL_ERROR;
	    }
	} else if (strcmp(argv[1], "delete") == 0) {
	    if (argc != 3) {
		Tcl_AppendResult(interp, argv[0], " delete: wrong # args", NULL);
		return TCL_ERROR;		
	    }
	    if (TclQddb_DeleteView(interp, argv[2]) != TCL_OK) {
		return TCL_ERROR;
	    }
	} else {
	    Tcl_AppendResult(interp, argv[0], ": bad command \"", argv[1], "\"", NULL);
	    return TCL_ERROR;
	}	    
	break;
    case 's':
	if (strcmp(argv[1], "set") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": bad command \"", argv[1], "\"", NULL);
	    return TCL_ERROR;
	}
	if (argc != 4) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_SetViewRow(interp, argv[2], argv[3]) != TCL_OK)
	    return TCL_ERROR;
	break;
    case 'g':
	if (strcmp(argv[1], "get") != 0) {
	    Tcl_AppendResult(interp, argv[0], ": bad command \"", argv[1], "\"", NULL);
	    return TCL_ERROR;
	}
	if (argc != 3) {
	    Tcl_AppendResult(interp, argv[0], ": wrong # args", NULL);
	    return TCL_ERROR;
	}
	if (TclQddb_GetViewRow(interp, argv[2]) != TCL_OK)
	    return TCL_ERROR;
	break;
    default:
	Tcl_AppendResult(interp, argv[0], ": bad command \"", argv[1], "\"", NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/* GetVarOwner -- 
 *	Gets the current owner of a TCL variable; returns
 * NULL if either the variables doesn't exist or is currently
 * unowned.
 */
static TclQddb_View *GetVarOwner(var)
    char		*var;
{
    Tcl_HashEntry	*hash_ptr;

    if (TclQddb_VarOwnerHashTableInit == 0)
	return NULL;
    hash_ptr = Tcl_FindHashEntry(&TclQddb_VarOwnerHashTable, var);
    if (hash_ptr == NULL)
	return NULL;
    return (TclQddb_View *)Tcl_GetHashValue(hash_ptr);
}

/* SetVarOwner --
 *	Sets the var owner of 'var' to 'view'.  SetVarOwner
 * should be called when a view is deleted to set its
 * variable ownership to NULL.
 */
static void SetVarOwner(var, view)
    char		*var;
    TclQddb_View	*view;
{
    Tcl_HashEntry	*hash_ptr;
    int			newPtr;

    if (TclQddb_VarOwnerHashTableInit == 0) {
	TclQddb_VarOwnerHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_VarOwnerHashTable, TCL_STRING_KEYS);
    }
    hash_ptr = Tcl_FindHashEntry(&TclQddb_VarOwnerHashTable, var);
    if (hash_ptr == NULL) {
	hash_ptr = Tcl_CreateHashEntry(&TclQddb_VarOwnerHashTable, var, &newPtr);
    }
    Tcl_SetHashValue(hash_ptr, (ClientData)view);
}

static int TclQddb_SetViewRow(interp, view_token, row_token)
    Tcl_Interp			*interp;
    char			*view_token, *row_token;
{
    Schema			*schema;
    TclQddb_View		*view_ptr;
    TclQddb_Tuple		*tuple_ptr;
    TclQddb_RowHeader		*row_header;
    TclQddb_Rows		*rows;
    char			**attr_names;
    size_t			instance[MAXIMUM_NUMBER_OF_SCHEMA_LEVELS];
    int				num, i;

    if ((view_ptr = TclQddb_GetView(interp, view_token)) == NULL)
	return TCL_ERROR;
    if ((schema = TclQddb_GetSchema(view_ptr->schema_name)) == NULL) {
	Tcl_AppendResult(interp, "cannot find schema: \"", view_ptr->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    if ((row_header = TclQddb_GetRows(interp, row_token)) == NULL)
	return TCL_ERROR;
    if (strcmp(view_ptr->schema_name, row_header->schema_name) != 0 ||
	strcmp(view_ptr->tuple_name, row_header->tuple_name) != 0) {
	char			buf[BUFSIZ];

	sprintf(buf, "view(%s,%s) does not match row(%s,%s)", view_ptr->schema_name, view_ptr->tuple_name,
		row_header->schema_name, row_header->tuple_name);
	Tcl_AppendResult(interp, buf, NULL);
	return TCL_ERROR;
    }
    if ((tuple_ptr = TclQddb_GetTuple(interp, view_ptr->tuple_name)) == NULL)
	return TCL_ERROR;
    rows = row_header->row;
    while (rows != NULL) {
	attr_names = TclQddb_ParseAttr(rows->attr_name);
	if ((num = StringToInstance(rows->instance, instance)) == 0) {
	    for (i = 0; attr_names[i] != NULL; i++)
		Free(attr_names[i++]);
	    Free(attr_names);
	    Tcl_AppendResult(interp, "failed to convert string \"", rows->instance,
			     "\" to an instance identifier", NULL);
	    return TCL_ERROR;
	}
	for (i = 0; i < num; i++) {
#if defined(DIAGNOSTIC)
	    if (attr_names[i] == NULL) {
		Tcl_AppendResult(interp, "bad attribute \"", rows->attr_name, "\"", NULL);
		return TCL_ERROR;
	    }
#endif
	    TclQddb_SetInstance(interp, schema, tuple_ptr, view_ptr, attr_names[i], instance[i]);
	    Free(attr_names[i]);
	}
	Free(attr_names);
	rows = rows->next;
    }
    return TCL_OK;
}

static int TclQddb_GetViewRow(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    TclQddb_View		*view_ptr;
    TclQddb_RowHeader		*row_header = NULL;
    TclQddb_Rows		*rows = NULL;
    Schema			*schema;
    SchemaNode			*schemanode;
    int				attr_num, i;
    char			*attr_name, *inst_str;
    size_t			instance[MAXIMUM_NUMBER_OF_SCHEMA_LEVELS];

    if ((view_ptr = TclQddb_GetView(interp, token)) == NULL)
	return TCL_ERROR;
    if ((schema = TclQddb_GetSchema(view_ptr->schema_name)) == NULL) {
	Tcl_AppendResult(interp, "cannot find schema: \"", view_ptr->schema_name, "\"", NULL);
	return TCL_ERROR;
    }
    for (i = 1; i <= view_ptr->num_attrs; i++) {
#if defined(DIAGNOSTIC)
	if (view_ptr->view[i] == NULL) {
	    char		buf[BUFSIZ];

#if defined(notdef)
	    sprintf(buf, "view_ptr->view[%d] == %x", i, (unsigned int)view_ptr->view[i]);
#endif
	    Tcl_AppendResult(interp, "QddbTcl error: \"", buf, "\"", NULL);
	    return TCL_ERROR;
	}
#endif
	schemanode = view_ptr->view[i]->datatree_schema->schemanode;
	instance[schemanode->Level-1] = view_ptr->curinst[i];
	if (schemanode->IsLeaf != True)
	    continue;
	attr_num = schemanode - schema->Entries;
	attr_name = Qddb_ConvertAttributeNumberToName(schema, attr_num);
	if (attr_name == NULL) {
	    /* FIXME: free up memory */
	    Tcl_AppendResult(interp, "TclQddb_GetViewRow: cannot find attribute number (Qddb Error)", NULL);
	    return TCL_ERROR;
	}
	inst_str  = InstanceToString(instance, (size_t)schemanode->Level);
	if (row_header == NULL) {
	    row_header = (TclQddb_RowHeader *)Malloc(sizeof(TclQddb_RowHeader));
	    row_header->schema_name = Malloc(strlen(view_ptr->schema_name)+1);
	    strcpy(row_header->schema_name, view_ptr->schema_name);
	    row_header->tuple_name = Malloc(strlen(view_ptr->tuple_name)+1);
	    strcpy(row_header->tuple_name, view_ptr->tuple_name);
	    rows = row_header->row = (TclQddb_Rows *)Malloc(sizeof(TclQddb_Rows));
	    rows->next = NULL;
	    rows->attr_name = attr_name;
	    rows->instance = inst_str;
	} else {
	    rows->next = (TclQddb_Rows *)Malloc(sizeof(TclQddb_Rows));
	    rows = rows->next;
	    rows->next = NULL;
	    rows->attr_name = attr_name;
	    rows->instance = inst_str;
	}
    }
    if (TclQddb_SetRow(interp, row_header) != TCL_OK) {
	/* FIXME: free up memory */
	return TCL_ERROR;
    }
    return TCL_OK;
}

/* TclQddb_NewView -- Set up the 'qddb_view%d' hash value for 'view'
 */
static int TclQddb_NewView(interp, view)
    Tcl_Interp			*interp;
    TclQddb_View		*view;
{
    Tcl_HashEntry		*hash_ptr;
    int				newPtr;
    char			token[BUFSIZ];
    
    sprintf(token, "qddb_view%d", TclQddb_ViewNextNumber++);
    hash_ptr = Tcl_CreateHashEntry(&TclQddb_ViewHashTable, token, &newPtr);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot create hash entry \"", token, "\" (TCL error)", NULL);
	TclQddb_FreeView(view);
	return TCL_ERROR;	
    }
    Tcl_SetHashValue(hash_ptr, (ClientData)view);
    Tcl_SetResult(interp, token, TCL_VOLATILE);
    return TCL_OK;
}

/* TclQddb_View -- Get the view associated with 'token'
 */
TclQddb_View *TclQddb_GetView(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    TclQddb_View		*view;
    Tcl_HashEntry		*hash_ptr;

    if (TclQddb_ViewHashTableInit == 0) {
	TclQddb_ViewHashTableInit = 1;
	Tcl_InitHashTable(&TclQddb_ViewHashTable, TCL_STRING_KEYS);
	return NULL;
    }
    hash_ptr = Tcl_FindHashEntry(&TclQddb_ViewHashTable, token);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot find keylist \"", token, "\" (TCL error)", NULL);
	return NULL;
    }
    view = (TclQddb_View *)Tcl_GetHashValue(hash_ptr);
    return view;
}

static int TclQddb_DeleteViewOne(interp, token)
    Tcl_Interp			*interp;
    char			*token;
{
    TclQddb_View		*view;
    Tcl_HashEntry		*hash_ptr;

    hash_ptr = Tcl_FindHashEntry(&TclQddb_ViewHashTable, token);
    if (hash_ptr == NULL) {
	Tcl_AppendResult(interp, "cannot find view \"", token, "\" (TCL error)", NULL);
	return TCL_ERROR;	
    }
    view = (TclQddb_View *)Tcl_GetHashValue(hash_ptr);
    Tcl_DeleteHashEntry(hash_ptr);
    if (view != NULL) {
	TclQddb_ViewLock(interp, view, True);
	TclQddb_FreeView(view);
    } else 
	return TCL_OK;
    return TCL_OK;
}

/* TclQddb_DeleteView -- Delete view(s) from the hash table.
 */
int TclQddb_DeleteView(interp, Token)
    Tcl_Interp			*interp;
    char			*Token;
{
    Tcl_HashEntry		*hash_ptr;
    Tcl_HashSearch		hash_search;
    char			*hash_key;
    int				deletebytuple = 0;

    if (TclQddb_ViewHashTableInit == 0)
	return TCL_OK;
    if (strncmp("qddb_tuple", Token, 10) == 0)
	deletebytuple = 1;
    if (strcmp(Token, "all") == 0 || deletebytuple == 1) {
	hash_ptr = Tcl_FirstHashEntry(&TclQddb_ViewHashTable, &hash_search);
	while (hash_ptr != NULL) {
	    if (deletebytuple == 1) {
		TclQddb_View	*view;

		view = (TclQddb_View *)Tcl_GetHashValue(hash_ptr);
		if (strcmp(view->tuple_name, Token) != 0) {
		    hash_ptr = Tcl_NextHashEntry(&hash_search);
		    continue;
		}
	    }
	    hash_key = Tcl_GetHashKey(&TclQddb_ViewHashTable, hash_ptr);
	    if (hash_key == NULL) {
		Tcl_AppendResult(interp, "TclQddb_DeleteView: ", 
				 "Tcl_GetHashKey failed (TCL ERROR)", NULL);
		return TCL_ERROR;
	    }
	    if (TclQddb_DeleteViewOne(interp, hash_key) != TCL_OK)
		return TCL_ERROR;
	    hash_ptr = Tcl_NextHashEntry(&hash_search);
	}
	if (deletebytuple == 0) {
	    if (TclQddb_ViewHashTableInit == 1) {
		TclQddb_ViewHashTableInit = 0;
		Tcl_DeleteHashTable(&TclQddb_ViewHashTable);
	    }
	    if (TclQddb_VarOwnerHashTableInit == 1) {
		TclQddb_VarOwnerHashTableInit = 0;
		Tcl_DeleteHashTable(&TclQddb_VarOwnerHashTable);
	    }
	}
    } else if (TclQddb_DeleteViewOne(interp, Token) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}

void TclQddb_DeleteViewProc(clientData)
    ClientData			clientData;
{
#if defined(DEBUG_MALLOC)
    fprintf(stderr, "TclQddb_DeleteViewProc\n");
#endif
    (void)TclQddb_DeleteView((Tcl_Interp *)clientData, "all");
}

/* TclQddb_InitView -- Allocate memory for a view.
 */
static int TclQddb_InitView(interp, schema, schema_name, tuple_name, view)
    Tcl_Interp			*interp;
    Schema			*schema;
    char			*schema_name, *tuple_name;
    TclQddb_View		**view;
{
    int				num_entries;

    (*view) = (TclQddb_View *)Malloc(sizeof(TclQddb_View));
    (*view)->schema_name = Malloc(strlen(schema_name)+1);
    strcpy((*view)->schema_name, schema_name);
    (*view)->tuple_name = Malloc(strlen(tuple_name)+1);
    strcpy((*view)->tuple_name, tuple_name);
    num_entries = schema->NumberOfAttributes + 2;
    (*view)->num_attrs = num_entries-2;
    (*view)->view = (DataTree **)Calloc(sizeof(DataTree *)*num_entries);
    (*view)->tclvars = (char **)Calloc(sizeof(char *)*(num_entries));
    (*view)->curinst = (int *)Calloc(sizeof(int)*num_entries);
    return TCL_OK;
}

/* TclQddb_DefineView -- Set up the TCL variables associated with a view.
 */
static int TclQddb_DefineView(interp, schema, tuple, view, vars)
    Tcl_Interp			*interp;
    Schema			*schema;
    TclQddb_Tuple		*tuple;
    TclQddb_View		*view;
    char			*vars;
{
    int				i, argc, nargc;
    char			**argv, **nargv;

    if (Tcl_SplitList(interp, vars, &argc, &argv) != TCL_OK)
	return TCL_ERROR;
    for (i = 0; i < argc; i++) {
	if (Tcl_SplitList(interp, argv[i], &nargc, &nargv) != TCL_OK) {
	    Free(argv);
	    return TCL_ERROR;
	}
	if (nargc != 2) {
	    Tcl_AppendResult(interp, "list format must be {attribute variable} ...", NULL);
	    Free(argv);
	    Free(nargv);
	    return TCL_ERROR;
	}
	if (TclQddb_SetVariable(interp, schema, view, nargv[0], nargv[1]) != TCL_OK) {
	    Free(argv);
	    Free(nargv);	    
	    return TCL_ERROR;
	}
	Free(nargv);
    }
    Free(argv);
    if (TclQddb_InitInstance(interp, schema, tuple->datatree, view, False) != TCL_OK)
	return TCL_ERROR;
    return TCL_OK;
}

static int TclQddb_SetVariable(interp, schema, view, name, var)
    Tcl_Interp			*interp;
    Schema			*schema;
    TclQddb_View		*view;
    char			*name, *var;
{
    int				attr;

    if ((attr=Qddb_ConvertAttributeNameToNumber(schema, name)) == -1) {
	Tcl_AppendResult(interp, "cannot find attribute \"", name, "\" in relation ", 
			 schema->RelationName, ".", NULL);
	return TCL_ERROR;
    }
#if defined(DIAGNOSTIC)
    if (attr > schema->NumberOfAttributes || attr < 1) {
	char			buf[BUFSIZ];

	sprintf(buf, "%d", attr);
	Tcl_AppendResult(interp, "TclQddb_SetVariable: bad attribute ", buf, " (Qddb Error)", NULL);
	return TCL_ERROR;
    }
#endif
    view->tclvars[attr] = Malloc(strlen(var)+1);
    strcpy(view->tclvars[attr], var);
    return TCL_OK;
}

/* TclQddb_InitInstance -- Initialize all instances to 1 or the current instances.
 */
static int TclQddb_InitInstance(interp, schema, tree, view, keep_instances)
    Tcl_Interp			*interp;
    Schema			*schema;
    DataTree			**tree;
    TclQddb_View		*view;
    Boolean			keep_instances;
{
    int				i, j, schema_index;
    TclQddb_View		*old_view_ptr;

#if defined(DIAGNOSTIC)
    if (tree == NULL || view == NULL || schema == NULL) {
	Tcl_AppendResult(interp, "TclQddb_InitInstance: tree/view/schema NULL", NULL);
	return TCL_ERROR;
    }
#endif
    for (i = 0; tree[i] != NULL; i++) {
#if defined(DIAGNOSTIC)
	if (tree[i]->datatree_schema == NULL) {
	    Tcl_AppendResult(interp, "TclQddb_InitInstance: datatree_schema is NULL: (Qddb Error)", NULL);
	    return TCL_ERROR;
	}
	if (tree[i]->datatree_schema->schemanode == NULL) {
	    Tcl_AppendResult(interp, "TclQddb_InitInstance: schemanode is NULL: (Qddb Error)", NULL);
	    return TCL_ERROR;
	}
#endif
	schema_index =  tree[i][0].datatree_schema->schemanode - schema->Entries;
#if defined(DIAGNOSTIC)
	if (schema_index < 1 || schema_index > schema->NumberOfAttributes) {
	    char		buf[BUFSIZ];

	    sprintf(buf, "%d", schema_index);
	    Tcl_AppendResult(interp, "TclQddb_InitInstance: bad schema index ", buf, ": (Qddb Error)", NULL);
	    abort();
	}
#endif
#if defined(DIAGNOSTIC)
	if (view->num_attrs < schema_index)
	    abort();
#endif
	if (keep_instances == True) {
	    for (j = 0; tree[i][j].datatree_type != DATATREE_END; j++);
	    if (view->curinst[schema_index] > j)
		view->curinst[schema_index] = j; /* Reduce the instance number to a reasonable value */
	    view->view[schema_index] = tree[i] + view->curinst[schema_index] - 1;
	} else {
	    view->view[schema_index] = tree[i];
	    view->curinst[schema_index] = 1;
	}
	if (keep_instances == False && view->tclvars[schema_index] != NULL) {
	    old_view_ptr = GetVarOwner(view->tclvars[schema_index]);
	    SetVarOwner(view->tclvars[schema_index], view);
	    if (TclQddb_RelinkDataTree(interp, view->view[schema_index], view->tclvars[schema_index]) 
		!= TCL_OK) {
		return TCL_ERROR;
	    }
	}
	j = view->curinst[schema_index]-1;
	if (tree[i][j].datatree_type == DATATREE_CHILDREN) {
	    if (TclQddb_InitInstance(interp, schema, tree[i][j].datatree_children, view, keep_instances)
		!= TCL_OK)
		return TCL_ERROR;
	}
    }
    return TCL_OK;
}

int TclQddb_SetInstance(interp, schema, tuple, view, name, instance)
    Tcl_Interp			*interp;
    Schema			*schema;
    TclQddb_Tuple		*tuple;
    TclQddb_View		*view;
    char			*name;
    size_t			instance;
{
    char			**name_list, *newname;
    DataTree			**dt, *old_view;
    int				attr, which, i;

    name_list = TclQddb_SplitAttributeName(name);
    dt = tuple->datatree;
    if (dt == NULL) {
	Tcl_AppendResult(interp, "trying to use null dt (Qddb_Error)", NULL);
	TclQddb_FreeArgs(-1, name_list);
	return TCL_ERROR;
    }
    for (i = 0; name_list[i] != NULL && dt != NULL; i++) {
	newname = TclQddb_ConcatAttributeName(name_list, i+1);
	if ((attr = Qddb_ConvertAttributeNameToNumber(schema, newname)) == -1) {
	    Tcl_AppendResult(interp, "cannot find attribute \"", newname, "\" in relation ", 
			     schema->RelationName, ".", NULL);
	    Free(newname);
	    TclQddb_FreeArgs(-1, name_list);
	    return TCL_ERROR;
	}
	Free(newname);
	if (view->curinst[attr] == 0) {
	    /* Need to find the right datatree node 
	     * and set the curinst to 1.
	     */
	    view->curinst[attr] = 1;
	    which = TclQddb_FindAttribute(dt, name_list[i]);
	    if (which == -1) {
		Tcl_AppendResult(interp, "failed to find attribute name component \"", 
				 name_list[i], "\"", NULL);
		TclQddb_FreeArgs(-1, name_list);
		return TCL_ERROR;
	    }
	} else {
	    which = TclQddb_FindAttribute(dt, name_list[i]);
	    if (which == -1) {
		Tcl_AppendResult(interp, "failed to find attribute name component \"", 
				 name_list[i], "\"", NULL);
		TclQddb_FreeArgs(-1, name_list);
		return TCL_ERROR;
	    }
	}
	if (name_list[i+1] == NULL) {
	    if (view->curinst[attr] != instance) {
		if (dt[which]->datatree_type == DATATREE_CHILDREN) {
		    int 			j;

		    for (j = 0; dt[which][j].datatree_type != DATATREE_END; j++);
		    if (instance > j) {
			Tcl_AppendResult(interp, "bad instance number", NULL);
			TclQddb_FreeArgs(-1, name_list);
			return TCL_ERROR;
		    }
		    view->curinst[attr] = instance;
		    if (TclQddb_InitInstance(interp, schema, 
					     dt[which][instance-1].datatree_children, 
					     view, False) != TCL_OK) {
			TclQddb_FreeArgs(-1, name_list);
			return TCL_ERROR;
		    }
		} else
		    view->curinst[attr] = instance;
	    }
	}
	/* If it has a TCL variable associated with it,
	 * Unlink/Link the TCL variable to the node.
	 */
	if (view->tclvars[attr] != NULL) {
	    if (GetVarOwner(view->tclvars[attr]) != view) {
		view->view[attr] = dt[which] + view->curinst[attr] - 1;
	    } else {
		old_view = view->view[attr];
		view->view[attr] = dt[which] + view->curinst[attr] - 1;
		if (TclQddb_RelinkDataTree(interp, view->view[attr], view->tclvars[attr]) != TCL_OK) {
		    TclQddb_FreeArgs(-1, name_list);
		    return TCL_ERROR;
		}
	    }
	} else
	    view->view[attr] = dt[which] + view->curinst[attr] - 1;
	if ((dt[which] + view->curinst[attr] - 1)->datatree_type == DATATREE_CHILDREN)
	    dt = (dt[which] + view->curinst[attr] - 1)->datatree_children;
	else
	    dt = NULL;
    }
    TclQddb_FreeArgs(-1, name_list);
    return TCL_OK;
}


static int TclQddb_RelinkDataTree(interp, node, var)
    Tcl_Interp			*interp;
    DataTree			*node;
    char			*var;
{
    if (node == NULL) {
	Tcl_AppendResult(interp, "TclQddb_RelinkDataTree: pass in node == NULL", NULL);
	return TCL_ERROR;
    }
    TclQddb_UnlinkVar(interp, var);
    switch (node->datatree_type) {
    case DATATREE_STRING:
	if (TclQddb_LinkVar(interp, var, (char *)&(node->datatree_string), TCL_LINK_STRING,
			    node->datatree_schema->schemanode->Format) != TCL_OK)
	    return TCL_ERROR;
	break;
    case DATATREE_DATE:
	/* NOTE: dates are pre-formatted */
	if (TclQddb_LinkVar(interp, var, (char *)&(node->datatree_date), TCL_LINK_STRING, NULL) != TCL_OK)
	    return TCL_ERROR;
	break;
    case DATATREE_INT:
	if (TclQddb_LinkVar(interp, var, (char *)&(node->datatree_int), TCL_LINK_INT,
			    node->datatree_schema->schemanode->Format) != TCL_OK)
	    return TCL_ERROR;
	break;
    case DATATREE_REAL:
	if (TclQddb_LinkVar(interp, var, (char *)&(node->datatree_real), TCL_LINK_DOUBLE,
			    node->datatree_schema->schemanode->Format) != TCL_OK)
	    return TCL_ERROR;
	break;
    case DATATREE_NOVAL:
	if (TclQddb_LinkVar(interp, var, (char *)node->datatree_string, TCL_LINK_STRING, 
			    node->datatree_schema->schemanode->Format) != TCL_OK)
	    return TCL_ERROR;
	break;
    default: {
	char			buf[BUFSIZ];

	sprintf(buf, "%d", node->datatree_type);
	Tcl_AppendResult(interp, "cannot link to datatree_type ", buf, NULL);
	return TCL_ERROR;
    }
    }
    return TCL_OK;
}

static void TclQddb_FreeView(view)
    TclQddb_View		*view;
{
    char			**arr;
    int				i;

    Free(view->tuple_name);
    Free(view->schema_name);
    Free(view->view);
    arr = view->tclvars;
    for (i = 1; i <= view->num_attrs; i++)
	if (arr[i] != NULL)
	    Free(arr[i]);
    Free(view->tclvars);
    Free(view->curinst);
    Free(view);
}

/* TclQddb_ViewLock -- 
 *	Disassociate the view's pointers with any variables it owns.
 */
static int TclQddb_ViewLock(interp, view, free_var)
    Tcl_Interp			*interp;
    TclQddb_View		*view;
    Boolean			free_var;
{
    TclQddb_View		*old_view;
    char			*ptr;
    int				i;

    if (view != NULL) {
	for (i = 1; i <= view->num_attrs; i++) {
	    ptr = view->tclvars[i];
	    if (view->view[i] != NULL && ptr != NULL) {
		old_view = GetVarOwner(ptr);
		if (old_view == view) {
		    if (free_var == True) /* free the TCL variable for this view? */
			SetVarOwner(ptr, NULL);
		    TclQddb_UnlinkVar(interp, ptr);
		}
	    }
	}
    }
    return TCL_OK;
}

/* TclQddb_ViewUnlock -- 
 *	Relink the view's pointers with any variables it owns.
 */
static int TclQddb_ViewUnlock(interp, view)
    Tcl_Interp			*interp;
    TclQddb_View		*view;
{
    Schema			*schema;
    TclQddb_Tuple		*tuple;
    TclQddb_View		*old_ptr;
    char			*ptr;
    int				i;

    tuple = TclQddb_GetTuple(interp, view->tuple_name);
    if (tuple == NULL) {
	Tcl_AppendResult(interp, "cannot find tuple \"", view->tuple_name, "\"", NULL);
	return TCL_OK;
    }
    schema = TclQddb_GetSchema(view->schema_name);
    if (schema == NULL) {
	Tcl_AppendResult(interp, "cannot find schema \"", view->schema_name, "\"", NULL);
	return TCL_OK;
    }
    /* Re-initialize the view with the new values.  We can 
     * optimize this quite a bit, but this works well for now.
     */
    TclQddb_InitInstance(interp, schema, tuple->datatree, view, True);
    if (view != NULL) {
	for (i = 1; i <= view->num_attrs; i++) {
	    ptr = view->tclvars[i];
	    if (view->view[i] != NULL && ptr != NULL) {
		old_ptr = GetVarOwner(ptr);
		if (old_ptr == view || old_ptr == NULL) {
		    if (TclQddb_RelinkDataTree(interp, view->view[i], ptr) != TCL_OK)
			return TCL_ERROR;
		    SetVarOwner(ptr, view);
		}
	    }
	}
    }
    return TCL_OK;
}

/* TclQddb_ViewLockTuple --
 *	Lock down all views associated with a particular tuple.
 */
int TclQddb_ViewLockTuple(interp, tuple_name)
    Tcl_Interp			*interp;
    char			*tuple_name;
{
    Tcl_HashEntry		*hash_ptr;
    Tcl_HashSearch		hash_search;
    TclQddb_View		*view_ptr;

    if (TclQddb_ViewHashTableInit == 0)
	return TCL_OK; 
    hash_ptr = Tcl_FirstHashEntry(&TclQddb_ViewHashTable, &hash_search);
    while (hash_ptr != NULL) {
	view_ptr = (TclQddb_View *)Tcl_GetHashValue(hash_ptr);
	if (strcmp(view_ptr->tuple_name, tuple_name) != 0) {
	    hash_ptr = Tcl_NextHashEntry(&hash_search);
	    continue;
	}
	if (TclQddb_ViewLock(interp, view_ptr, False) != TCL_OK) {
	    /* FIXME: cleanup? */
	    return TCL_ERROR;
	}
	hash_ptr = Tcl_NextHashEntry(&hash_search);
    }
    return TCL_OK;
}

/* TclQddb_ViewLockTuple --
 *	Unlock all views associated with a particular tuple.
 */
int TclQddb_ViewUnlockTuple(interp, tuple_name)
    Tcl_Interp			*interp;
    char 			*tuple_name;
{
    Tcl_HashEntry		*hash_ptr;
    Tcl_HashSearch		hash_search;
    TclQddb_View		*view_ptr;
    
    if (TclQddb_ViewHashTableInit == 0)
	return TCL_OK; /* shouldn't happen */
    hash_ptr = Tcl_FirstHashEntry(&TclQddb_ViewHashTable, &hash_search);
    while (hash_ptr != NULL) {
	view_ptr = (TclQddb_View *)Tcl_GetHashValue(hash_ptr);
	if (strcmp(view_ptr->tuple_name, tuple_name) != 0) {
	    hash_ptr = Tcl_NextHashEntry(&hash_search);
	    continue;
	}
	if (TclQddb_ViewUnlock(interp, view_ptr) != TCL_OK) {
	    /* FIXME: cleanup? */
	    return TCL_ERROR;
	}
	hash_ptr = Tcl_NextHashEntry(&hash_search);
    }
    return TCL_OK;
}
