

/* qddb/Lib/LibQddb/Utils.c
 *
 * Copyright (C) 1993, 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2 
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"

/* EXPORTED:
 *	Open(char *, int, int)
 *	OpenDatabase(char *, int)
 *	....
 */

int Open(Name, Flags, Mode)
    char		*Name;
    int			Flags;
    int			Mode;
{
    int			FileDescriptor;

    FileDescriptor = open(Name, Flags, 0666);
    return FileDescriptor;
}

int OpenDatabase(RelationFN, Mode)
    char		*RelationFN;
    int			Mode;
{
    char		TmpRelationFN[MAXPATHLEN];
    int			retval;

    strcpy(TmpRelationFN, RelationFN);
    strcat(TmpRelationFN, "/Database");
    retval = Open(TmpRelationFN, O_RDWR, Mode);
    if (retval == -1)
	return Open(TmpRelationFN, O_RDONLY, Mode);
    else
	return retval;
}

int OpenChangeFile(RelationFN, Number, Mode)
    char		*RelationFN;
    size_t		Number;
    int			Mode;
{
    char		TmpRelationFN[MAXPATHLEN];
    int			retval;

    sprintf(TmpRelationFN, "%s/Changes/%d", RelationFN, (int)Number);
    retval = Open(TmpRelationFN, O_RDWR, Mode);
    if (retval == -1)
	return Open(TmpRelationFN, O_RDONLY, Mode);
    else
	return retval;
}

int OpenAdditionFile(RelationFN, Number, Mode)
    char		*RelationFN;
    size_t		Number;
    int			Mode;
{
    char		TmpRelationFN[MAXPATHLEN];
    int			retval;

    sprintf(TmpRelationFN, "%s/Additions/%d", RelationFN, (int)Number);
    retval = Open(TmpRelationFN, O_RDWR, Mode);
    if (retval == -1)
	return Open(TmpRelationFN, O_RDONLY, Mode);
    else
	return retval;
}

int OpenHashTable(RelationFN, Mode)
    char		*RelationFN;
    int			Mode;
{
    char		TmpRelationFN[MAXPATHLEN];

    strcpy(TmpRelationFN, RelationFN);
    strcat(TmpRelationFN, "/HashTable");
    return Open(TmpRelationFN, O_RDONLY, Mode);
}

void Close(FileDesc)
    int		FileDesc;
{
    if (close(FileDesc) == -1)
    	perror("Close");
}

#ifdef QDDBDIRS
/* Qddb_FindRelation --
 *	Return value must copied if routine reentered before the
 * string is used.
 */
char *Qddb_FindRelation(RelationName)
    char		*RelationName;
{
    static char		NewName[MAXFILENAMELEN];
    static char		*QddbDirs = NULL; 
    char		*getenv();
    char		*Dir, *TmpDirs;

    if (RelationName[0] == '.') {
	GETPWD(NewName);
	strcat(NewName, "/");
	strcat(NewName, RelationName);
	return NewName;
    }
    if (RelationName[0] == '/')
    	return RelationName;
    if (access(RelationName, 00) != -1) {
	char		ch[MAXFILENAMELEN];

	/* Check for normal directories with the same name as the
	 * relation.  All relations must have a HashTable.
	 */

	strcpy(ch, RelationName); 
	strcat(ch, "/HashTable");
	if (access(ch, 00) != -1) {
	    GETPWD(NewName);
	    strcat(NewName, "/");
	    strcat(NewName, RelationName);
	    return NewName;
	}
    }
    if (QddbDirs == NULL) {
    	QddbDirs = getenv("QDDBDIRS");
    	if (QddbDirs == NULL) {
	    return NULL;
	}
    }
    TmpDirs = QddbDirs;
    Dir = strtok(TmpDirs, ":");
    if (Dir == NULL)
    	return RelationName;
    TmpDirs = NULL;
    do {
    	strcpy(NewName, Dir);
    	strcat(NewName, "/");
    	strcat(NewName, RelationName);
    	if (access(NewName, 00) != -1)
    		return NewName;
    } while ((Dir=strtok(TmpDirs, ":")) != NULL);
    return NULL;
}
#endif
    
    	
void Read(File, Buffer, SizeOfBuffer)
    int			File;
    char		*Buffer;
    size_t		SizeOfBuffer;
{
    int			NumberRead;

    if ((NumberRead=read(File, Buffer, SizeOfBuffer)) != SizeOfBuffer) {
    	Buffer[NumberRead] = 0;
    	fprintf(stderr, "Attempted %d, actually read %d of filedesc %d\n", 
    		(int)SizeOfBuffer, NumberRead, File);
	fflush(stderr);
    	perror("Read():");
    	exit(FILE_ERROR);
    }
}

void Write(File, Buffer, SizeOfBuffer)
    int			File;
    char		*Buffer;
    size_t		SizeOfBuffer;
{
    if (write(File, Buffer, SizeOfBuffer) != SizeOfBuffer) {
    	perror("Write():");
    	exit(FILE_ERROR);
    }
}

size_t SizeOfFile(File)
    int			File;
{
    struct stat		FileStatus;

    if (fstat(File, &FileStatus) == -1) {
    	perror("SizeOfFile()");
    	exit(FILE_ERROR);
    }
    return (size_t)FileStatus.st_size;
}

void PANIC(string)
    char		*string;
{
    fprintf(stderr, "%s, errno = %d\n", string, errno);
    fprintf(stderr, "Dumping core for analysis....\n");
    fflush(stderr);
    abort();
    exit(2);
}


void Qddb_SplitBufferIntoLines(Lines, Buffer, BufLen)
    Entry		*Lines;
    char		*Buffer;
    size_t		BufLen;
{
    int			i, j;
    size_t		LineTop;
    char		*ch;
    
    /* Split the buffer into lines.  A line is
     * a set of characters beginning with '%' and
     * ending in a newline.  Newlines embedded in
     * the line must be backslashed ('\').
     */

    /* Free up any traces of an old entry. 
     */
    if (*Lines != NULL)
	Qddb_Free(QDDB_TYPE_ENTRY, *Lines);
    /* Mark the newlines that aren't backslashed and
     * count the number of lines.
     */
    for (i = 0, LineTop = 0; i < BufLen; i++) {
	if (Buffer[i] == '\\' && (Buffer[i+1] == '\\' || Buffer[i+1] == '\n')) {
	    i++;
	} else if (Buffer[i] == '\n') {
	    Buffer[i] = '\0';
	    LineTop++;
	}
    }
    LineTop++;
    *Lines = (Entry)Malloc(sizeof(char *)*(size_t)LineTop);
    for (i = 0, j = 0, ch = Buffer; i < BufLen; i++)
	if (Buffer[i] == '\0') { /* End of a line */
	    size_t	len;

	    if (i == BufLen - 1 && Buffer[i-1] == '\0')
		break;
	    len = strlen(ch) + 1;
	    (*Lines)[j] = Malloc(sizeof(char)*len);
/* FIXME: need to check for reduced attribute identifiers here */
	    strcpy((*Lines)[j], ch);
	    ch = Buffer+i+1;
	    j++;
	}
#if defined(DIAGNOSTIC)
    if (j >= LineTop)
	PANIC("SplitBufferIntoLines(Utils.c): j >= LineTop\n");
#endif
    (*Lines)[j] = NULL;
}

void Qddb_ReadEntry(File, OldEntry, Start, Length, DoLocking)
    int			File;
    Entry		*OldEntry;
    off_t 		Start;
    size_t		Length;
    Boolean		DoLocking;
{
    char		*Buffer;

#if defined(DIAGNOSTIC)
    if (OldEntry == NULL)
	PANIC("Qddb_ReadEntry: OldEntry == NULL\n");
#endif
#ifdef RECORDLOCKING
    if (DoLocking == True)
	LockSection(File, F_RDLCK, Start, (off_t)Length, True);
#endif
    if (Length == 0)
	PANIC("Qddb_ReadEntry: trying to read a zero length file");
    Buffer = Malloc(sizeof(char)*(size_t)(Length+1));
    lseek(File, Start, 0);
    Read(File, Buffer, Length);
    Buffer[Length] = '\0';
#ifdef RECORDLOCKING
    if (DoLocking == True)
	UnlockSection(File, Start, (off_t)Length);
#endif
    Qddb_SplitBufferIntoLines(OldEntry, Buffer, Length);
    Free(Buffer);
}


int Qddb_ReadEntryByKeyList(FileDesc, RelationFN, ThisEntry, TheKeyList, DoLocking)
    int			FileDesc;
    char		*RelationFN;
    Entry		*ThisEntry;
    KeyList		*TheKeyList;
    Boolean		DoLocking;
{
    int			fd;

    if (TheKeyList == NULL)
	return -1;
    switch(QDDB_KEYLIST_TYPE(TheKeyList)) {
    case ORIGINAL:
    	if (FileDesc == -1) {
	    fd = OpenDatabase(RelationFN, 0);
	    if (fd == -1)
		return -1;
	    Qddb_ReadEntry(fd, ThisEntry, TheKeyList->Start, 
			   TheKeyList->Length, DoLocking);
	    Close(fd);
    	} else {
	    Qddb_ReadEntry(FileDesc, ThisEntry, TheKeyList->Start, 
			   TheKeyList->Length, DoLocking);
	}
	if (EntryInvalid(*ThisEntry) == True) {
	    return -1;
	}
    	break;
    case CHANGE:
    	fd = OpenChangeFile(RelationFN, TheKeyList->Number, 0);
	if (fd == -1)
	    return -1;
    	Qddb_ReadEntry(fd, ThisEntry, (off_t)0, SizeOfFile(fd), DoLocking);
    	Close(fd);
    	break;
    case ADDITION:
    	fd = OpenAdditionFile(RelationFN, TheKeyList->Number, 0);
	if (fd == -1)
	    return -1;
    	Qddb_ReadEntry(fd, ThisEntry, (off_t)0, SizeOfFile(fd), DoLocking);
    	Close(fd);
    	break;
    default:
    	PANIC("Qddb_ReadEntryByKeyList: BAD TYPE IN KeyList");
    }	
    return 0;
}

int Qddb_ReadEntryByKeyList2(FileDesc, RelationFN, ThisEntry, TheKeyList, DoLocking)
    int			FileDesc;
    char		*RelationFN;
    Entry		*ThisEntry;
    KeyList		*TheKeyList;
    Boolean		DoLocking;
{
    int			fd;

    switch(QDDB_KEYLIST_TYPE(TheKeyList)) {
    case ORIGINAL:
    	if (FileDesc == -1) {
   		fd = OpenDatabase(RelationFN, 0);
		if (fd == -1)
		    return -1;
    		Qddb_ReadEntry(fd, ThisEntry, TheKeyList->Start, 
    			  TheKeyList->Length, DoLocking);
    		Close(fd);
    	} else 
    		Qddb_ReadEntry(FileDesc, ThisEntry, TheKeyList->Start, 
    			  TheKeyList->Length, DoLocking);
	if (EntryInvalid(*ThisEntry) == True) {
	    return -1;
	}
    	break;
    case CHANGE:
    	Qddb_ReadEntry(FileDesc, ThisEntry, (off_t)0, SizeOfFile(FileDesc), DoLocking);
    	break;
    case ADDITION:
    	Qddb_ReadEntry(FileDesc, ThisEntry, (off_t)0, SizeOfFile(FileDesc), DoLocking);
    	break;
    default:
    	PANIC("Qddb_ReadEntryByKeyList2: BAD TYPE IN KeyList");
    }	
    return 0;
}

/* WriteEntry doesn't use record locking because it writes from the current
 * position until it is finished.  Usually used by programs that have
 * complete control over the relation or already have it locked down.
 */
void WriteEntry(schema, File, WhichEntry)
    Schema		*schema;
    int			File;
    Entry		WhichEntry;
{
    Entry		AttributePtr = WhichEntry;

    Write(File, *AttributePtr, strlen(*AttributePtr));
    Write(File, "\n", 1);
    AttributePtr++;
    while (*AttributePtr != NULL) {
	Write(File, *AttributePtr, strlen(*AttributePtr));
    	Write(File, "\n", 1);
    	AttributePtr++;
    }
}

size_t GetEntryNumber(WhichEntry)
    Entry			WhichEntry;
{
    int				EntryNumber;

    sscanf(WhichEntry[0], "%%0 %*c %d\n", &EntryNumber);
    return (size_t)EntryNumber;
}

Boolean EntryInvalid(WhichEntry)
    Entry			WhichEntry;
{
    char			*Valid = WhichEntry[0];

    while (*Valid != 'I' && *Valid != 'V' && *Valid != '\0')
    	Valid++;
    if (*Valid == '\0') {
        fprintf(stderr, "%s\n", WhichEntry[0]);
        PANIC("EntryInvalid(): %0 entry doesn't contain 'V' or 'I'");
    }
    return *Valid == 'I'? True : False;
}

int EntryChanged(ChangeFN, Length)
    char		*ChangeFN; 
    size_t		*Length;
{
    int			ChangeFile;

    if ((ChangeFile=Open(ChangeFN, O_RDWR, 0)) == -1)
    	return -1;
    *Length = SizeOfFile(ChangeFile);
    return ChangeFile;
}

int yyerror(string)
    char		*string;
{
    extern int		LineNumber;
    char		buf[BUFSIZ], *ch;

    sprintf(buf, "syntax error at line %d\n", LineNumber);
    if (qddb_errmsg != NULL) {
	ch = qddb_errmsg;
    } else {
	ch = Calloc(1);
    }
    qddb_errmsg = Malloc(strlen(buf)+strlen(ch)+2);
    strcpy(qddb_errmsg, ch);
    strcat(qddb_errmsg, "\n");
    strcat(qddb_errmsg, buf);
    Free(ch);
    LineNumber = 1;
    return 1;
}

int StrcmpNoupper(string1, string2)
    char		*string1, *string2;
{
    int			i;

    if (strcmp(string1, string2) == 0)
        return 0;
    if (strlen(string1) != strlen(string2))
        return 1;
    for (i = 0; i < (int)strlen(string1); i++)
        if (tolower(string1[i]) != tolower(string2[i]))
            return 1;
    return 0;
}

#ifdef RECORDLOCKING
int LockSection(File, LockType, Start, Length, Wait)
    int			File;
    int			LockType;
    off_t		Start, Length;
    Boolean		Wait;
{
    struct flock	Lock;

    Lock.l_type = LockType;
    Lock.l_whence = 0;
    Lock.l_start = Start;
    Lock.l_len = Length;
    if (Wait == True)
    	return fcntl(File, F_SETLKW, &Lock);
    else
    	return fcntl(File, F_SETLK, &Lock);
}

int UnlockSection(File, Start, Length)
    int			File;
    off_t		Start, Length;
{
    struct flock	Lock;

    Lock.l_type = F_UNLCK;
    Lock.l_whence = 0;
    Lock.l_start = Start;
    Lock.l_len = Length;
    return fcntl(File, F_SETLKW, &Lock);
}
#else
int LockSection(File, LockType, Start, Length, Wait)
    int			File;
    int			LockType;
    off_t		Start, Length;
    Boolean		Wait;
{
    return 0;
}

int UnlockSection(File, Start, Length)
    int			File;
    off_t		Start, Length;
{
    return 0;
}
#endif

Boolean IsNumericString(string)
    char		*string;
{
    int			i, len;
    char		*tmpstr;
    Boolean		SawDot;

    len = strlen(string);
    tmpstr = string;
    SawDot = False;
    /* Optional negation */
    if (*tmpstr == '-') {
	i = 1;
	tmpstr++;
    } else {
	i = 0;
    }
    /* Check for integer */
    while (*tmpstr != '\0') {
	if (!isdigit(*tmpstr)) {
	    if (isspace(*tmpstr))
		return True;
	    if (*tmpstr != '.') {
		if (*tmpstr == 'e' || *tmpstr == 'E')
		    break;
		return False;
	    } else {
		if (SawDot == True)
		    return False;
		SawDot = True;
	    }
	}
	tmpstr++;
    }
    /* Matched? */
    if (*tmpstr == '\0')
        return True;
    /* Otherwise, we saw an 'e' or 'E' */
    tmpstr++;
    /* Optional sign or space */
    while (*tmpstr != '\0' && isspace(*tmpstr))
	tmpstr++;
    if (*tmpstr == '-')
	tmpstr++;
    while (*tmpstr != '\0' && isspace(*tmpstr))
	tmpstr++;
    /* Integer portion of exponent */
    while (*tmpstr != '\0') {
	if (isspace(*tmpstr))
	    return True;
	if (!isdigit(*tmpstr))
	    return False;
	tmpstr++;
    }
    return True;
}


void SkipSpaces(string)
    char		**string;
{
    while (*string != NULL && isspace((**string)))
	(*string)++;
}

void Qddb_InvalidateEntry(schema, Start, Length)
    Schema		*schema;
    off_t		Start;
    size_t		Length;
{
    char		*RelationFN = schema->RelationName, DatabaseFN[MAXFILENAMELEN];
    int			DatabaseFile = schema->database_fd;
    char		*Buffer;
    int			i;

#ifdef RECORDLOCKING
    LockSection(DatabaseFile, F_WRLCK, Start, (off_t)Length, True);
#endif
    Buffer = Malloc(Length);
    lseek(DatabaseFile, Start, 0);
    Read(DatabaseFile, Buffer, Length);
    for (i = 0; i < Length; i++)
	if (Buffer[i] == 'V') {
	    Buffer[i] = 'I';
	    break;
	} 
    lseek(DatabaseFile, Start, 0);
    Write(DatabaseFile, Buffer, Length);
    strcpy(DatabaseFN, RelationFN);
    strcat(DatabaseFN, "/Database.key");
#if defined(HAVE_UTIME)
#if defined(HAVE_UTIME_NULL)
    utime(DatabaseFN, NULL);
#endif
#endif
#ifdef RECORDLOCKING
    UnlockSection(DatabaseFile, Start, (off_t)Length);
#endif
    Free(Buffer);
}



void Qddb_InvalidateEntryByNumber(schema, Number)
    Schema		*schema;
    size_t		Number;
{
    char		*RelationFN = schema->RelationName;
    char		DatabaseFN[MAXFILENAMELEN], *entry;
    FILE		*DatabaseKeyFile;
    int			i;
    off_t		Start;
    size_t		Length;
    char		buf[BUFSIZ], *ptr = NULL;

    strcpy(DatabaseFN, RelationFN);
    strcat(DatabaseFN, "/Database.key");
    DatabaseKeyFile = fopen(DatabaseFN, "r");
    i = 1;
    while ((entry=fgets(buf, BUFSIZ, DatabaseKeyFile)) != NULL && i != Number)
	i++;
    if (entry == NULL)
	return;	/* must be an addition */
    Start = (off_t)strtoul(buf, &ptr, 10);
    Length = (size_t)strtoul(ptr, NULL, 10);
    Qddb_InvalidateEntry(schema, Start, Length);
    fclose(DatabaseKeyFile);
}

void DeleteEntryByNumber(schema, Number)
    Schema		*schema;
    size_t		Number;
{
    char		DatabaseFN[MAXFILENAMELEN];
    char		*RelationFN = schema->RelationName;

    sprintf(DatabaseFN, "%s/Changes/%d", RelationFN, (int)Number);
    unlink(DatabaseFN);
    sprintf(DatabaseFN, "%s/Additions/%d", RelationFN, (int)Number);
    unlink(DatabaseFN);
    Qddb_InvalidateEntryByNumber(schema, Number);
}

void DeleteEntryByKeyList(schema, node)
    Schema		*schema;
    KeyList		*node;
{
    switch (QDDB_KEYLIST_TYPE(node)) {
    case ORIGINAL:
	Qddb_InvalidateEntry(schema, node->Start, node->Length);
	break;
    case CHANGE:
    case ADDITION:
	DeleteEntryByNumber(schema, node->Number);
	break;
    default:
	;
    }
}

void StripBackslashes(buf)
    char		*buf;
{
    char		*tmpbuf, *tmp, *tmpstart;

    tmp = tmpstart = (char *)Malloc(strlen(buf)+1);
    tmpbuf = buf;
    while ((*tmp = *tmpbuf) != '\0') {
	if (*tmpbuf == '\\' && *(tmpbuf+1) == '\n') {
		tmpbuf++;
	} else {
	    tmp++;
	    tmpbuf++;
	}
    }
    strcpy(buf, tmpstart);
    Free(tmpstart);
}	    

char **ParseKeyFile(fd)
    int			fd;
{
    struct stat		stat_buf;
    char		**retval, *ch, *buf;
    int			i, retval_top;

    if (fstat(fd, &stat_buf) == -1)
	return NULL;
    retval = (char **)Malloc(sizeof(char *)*(size_t)(stat_buf.st_size/4));
    buf = (char *)Malloc(sizeof(char)*(size_t)stat_buf.st_size);
    if (read(fd, buf, (size_t)stat_buf.st_size) != stat_buf.st_size)
	PANIC("ERROR: Cannot read key file\n");
    ch = buf;
    retval_top = 0;
    for (i = 0; i < stat_buf.st_size; i++) {
	if (*buf == '\n') {
	    *buf++ = '\0';
	    retval[retval_top++] = ch;
	    ch = buf;
	} else
	    buf++;
    }
    retval[retval_top] = NULL;
    return retval;
}

KeyList *CatRelation(RelationFN)
    char		*RelationFN;
{
    char		KeyFN[BUFSIZ], **key_contents, buf[BUFSIZ];
    KeyList		*retval = NULL, *tmp, *last;
    int			fd;
    struct stat		stat_buf;
    off_t		start;
    size_t		len, number;
    struct dirent	*dir_ptr;
    DIR			*DIR_PTR;
    char		*ptr = NULL;

    strcpy(KeyFN, RelationFN);
    strcat(KeyFN,"/Database.key");
    if ((fd = Open(KeyFN, O_RDONLY, 0)) == -1)
	return NULL;
    key_contents = ParseKeyFile(fd);
    number = 1;
    last = NULL;
    while (*key_contents != NULL) {
	start = (off_t)strtoul(*key_contents, &ptr, 10);
	len = (size_t)strtoul(ptr, NULL, 10);
	tmp = (KeyList *)Malloc(sizeof(KeyList));
	tmp->Start = start;
	tmp->Length = len;
	tmp->Number = number++;
	QDDB_KEYLIST_SET_TYPE(tmp, ORIGINAL);
	sprintf(buf, "%s/Changes/%d", RelationFN, (int)tmp->Number);
	if (stat(buf, &stat_buf) != -1) {
	    QDDB_KEYLIST_SET_TYPE(tmp, CHANGE);
	}
	tmp->next = NULL;
	if (last == NULL) {
	    last = tmp;
	    retval = last;
	} else {
	    last->next = tmp;
	    last = tmp;
	}
	key_contents++;
    }
    sprintf(KeyFN, "%s/Additions", RelationFN);
    DIR_PTR = opendir(KeyFN);
    while ((dir_ptr = readdir(DIR_PTR)) != NULL) {
	int			dirlen;

	/* nothing beginning with '.' is supposed to be in this directory except '.' & '..' */
	dirlen = NLENGTH(dir_ptr);
	if (dirlen != 0 && dir_ptr->d_name[0] != '.' && dir_ptr->d_name[0] != 'N') {
	    tmp = (KeyList *)Malloc(sizeof(KeyList));
	    tmp->Start = (off_t)0;
	    tmp->Length = (size_t)0;
	    tmp->Number = atoi(dir_ptr->d_name);
	    QDDB_KEYLIST_SET_TYPE(tmp, ADDITION);
	    tmp->next = NULL;
	    if (last == NULL) {
		last = tmp;
		retval = last;
	    } else {
		last->next = tmp;
		last = tmp;
	    }
	}	    
    }
    closedir(DIR_PTR);
    Free(*key_contents);
    Free(key_contents);
    close(fd);
    return retval;
}

char *InstanceToString(Inst, Len)
    size_t		Inst[];
    size_t		Len;
{
    char		retval[BUFSIZ];
    char		*ptr;
    int			i, j;

    retval[0] = '\0';
    j = 0;
    for (i = 0; i < Len; i++) {
	if (i == Len-1)
	    sprintf(retval+j, "%d", (int)Inst[i]);
	else
	    sprintf(retval+j, "%d.", (int)Inst[i]);
	j = strlen(retval);
    }
    ptr = Malloc((size_t)j+1);
    strcpy(ptr, retval);
    return ptr;
}

int StringToInstance(str, Inst)
    char		*str;
    size_t		Inst[];
{
    char		*ch;
    char		*buf = alloca(strlen(str)+1);
    int			i;

    strcpy(buf, str);
    ch = strtok(buf, ".");
    for (i = 0; ch != NULL; i++) {
	Inst[i] = atoi(ch);
	ch = strtok(NULL, ".");
    }
    return i;
}

/* Rough date checking; don't actually check for a valid date */
int Qddb_CheckDate(s)
    char			*s;
{
    char			buf[BUFSIZ];
    time_t			mytime, clockVal;
    long			zone;

    mytime = time(NULL);
    zone = Qddb_GetTimeZone(mytime);
    if (strcmp(DEFAULT_DATE_FORMAT, "%d/%m/%y") == 0) {
	int		mm, dd, yy;

	if (sscanf(s, "%d/%d/%d", &dd, &mm, &yy) == 3) {
	    sprintf(buf, "%d/%d/%d", mm, dd, yy);
	    if (Qddb_GetDate(buf, mytime, zone, &clockVal) < 0) {
		return 1;
	    }
	} else {
	    if (Qddb_GetDate(s, mytime, zone, &clockVal) < 0) {
		return 1;
	    }
	}
    } else {
	if (Qddb_GetDate(s, mytime, zone, &clockVal) < 0) {
	    return 1;
	}
    }
    return 0;
}

/* Accept dates in any form */
char *Qddb_DateStringToTime(s)
    char			*s;
{
    char			buf[BUFSIZ], *retval;
    time_t			mytime, clockVal;
    long			zone;

    if (strcmp(s, " ") == 0 || s[0] == '\0') {
	retval = Malloc(strlen(s)+1);
	strcpy(retval, s);
	return retval;
    }	
    mytime = time(NULL);
    zone = Qddb_GetTimeZone(mytime);
    if (strcmp(DEFAULT_DATE_FORMAT, "%d/%m/%y") == 0) {
	int		mm, dd, yy;

	if (sscanf(s, "%d/%d/%d", &dd, &mm, &yy) == 3) {
	    sprintf(buf, "%d/%d/%d", mm, dd, yy);
	    if (Qddb_GetDate(buf, mytime, zone, &clockVal) < 0) {
		retval = Malloc(2);
		strcpy(retval, "");
		return retval;
	    }
	} else {
	    if (Qddb_GetDate(s, mytime, zone, &clockVal) < 0) {
		retval = Malloc(2);
		strcpy(retval, "");
		return retval;
	    }
	}
    } else {
	if (Qddb_GetDate(s, mytime, zone, &clockVal) < 0) {
	    retval = Malloc(2);
	    strcpy(retval, "");
	    return retval;
	}
    }
    if (sizeof(time_t) == sizeof(int))
	sprintf(buf, "%u", (int)clockVal);
    else if (sizeof(time_t) == sizeof(long))
	sprintf(buf, "%lu", (long)clockVal);
    else
	PANIC("sizeof(time_t) != sizeof(int) or sizeof(long)");
    retval = Malloc(sizeof(char)*(strlen(buf)+1));
    strcpy(retval, buf);
    return retval;
}

char *Qddb_DateTimeToString(s, format)
    char			*s;
    char 			*format;
{
    char			buf[BUFSIZ], *retval;
    time_t			mytime;
    struct tm			*tm_buf;

    if (strcmp(s, " ") == 0 || s[0] == '\0') {
	retval = Malloc(2);
	strcpy(retval, " ");
	return retval;
    }
    mytime = (time_t)strtoul(s, NULL, 10);
    tm_buf = localtime(&mytime);
    strftime(buf, BUFSIZ, format, tm_buf);
    retval = Malloc(sizeof(char)*(strlen(buf)+1));
    strcpy(retval, buf);
    return retval;
}

char *Qddb_FieldString(string, length)
    char			*string;
    size_t			length;
{
    size_t			str_len;
    int				i;
    char			*buf;

    str_len = strlen(string);
    buf = Malloc(length+1);
    if (length > str_len) {
	for (i = 0; i < length-str_len; i++)
	    buf[i] = ' ';
	strncpy(buf+i, string, str_len);
	buf[length] = '\0';
    } else {
	strncpy(buf, string, length);
	buf[length] = '\0';
    }
    return buf;
}

int Qddb_TranslateExternalID(schema, id, attr, inst)
    Schema		*schema;
    char		*id;
    size_t		*attr;
    char		**inst;
{
    char		*tok;
    int			i, tmpattr;
    size_t		attr_int[MAXIMUM_NUMBER_OF_SCHEMA_LEVELS];
    size_t		inst_int[MAXIMUM_NUMBER_OF_SCHEMA_LEVELS];

    if (*id == '%') {
	id++;
    } else {
	return -1;
    }
    tok = strtok(id, ".");
    i = 0;
    while (tok != NULL) {
	attr_int[i] = atoi(tok);
	tok = strtok(NULL, ".");
	inst_int[i++] = atoi(tok);
	tok = strtok(NULL, ".");
    }
    if ((tmpattr = Qddb_FindAttributeNumber(schema, attr_int, i)) == -1)
	return -1;
    if ((*inst = InstanceToString(inst_int, (size_t)i)) == NULL)
	return -1;
    *attr = tmpattr;
    return 0;
}

char *Qddb_GetFile(fd)
    int				fd;
{
    char			*buf;
    size_t			bufsize, place, num_read;

    bufsize = BUFSIZ;
    place = 0;
    buf = Malloc(bufsize);
    while ((num_read = read(fd, buf+place, BUFSIZ)) > 0) {
	if (num_read >= BUFSIZ-1) {
	    bufsize += BUFSIZ;
	    buf = Realloc(buf, bufsize);
	}
	place += num_read;
    }
    buf[place] = '\0';
    buf = Realloc(buf, place+1);
    return buf;
}

static char   *qddb_setbuf_buffer;
static size_t qddb_setbuf_bufsiz;
static size_t qddb_setbuf_buflen;

void Qddb_SetBuffer(buf)
    char			*buf;
{
    qddb_setbuf_buffer = buf;
}

size_t Qddb_ReadBuffer(buf, n)
    char			*buf;
    size_t			n;
{
    int				i;

    for (i = 0; i < n-1 && *qddb_setbuf_buffer != '\0'; i++)
	*buf++ = *qddb_setbuf_buffer++;
    *buf = '\0';
    return i;
}

char *Qddb_GetBuffer(void)
{
    qddb_setbuf_buffer = Realloc(qddb_setbuf_buffer, qddb_setbuf_buflen+1);
    qddb_setbuf_bufsiz = qddb_setbuf_buflen+1;
    return qddb_setbuf_buffer;
}

void Qddb_InitBuffer(void)
{
    qddb_setbuf_buffer = Malloc(BUFSIZ*sizeof(char));
    qddb_setbuf_buffer[0] = '\0';
    qddb_setbuf_bufsiz = BUFSIZ;
    qddb_setbuf_buflen = 0;
}

void Qddb_ConcatBuffer(newbuf)
    char			*newbuf;
{
    size_t			len = strlen(newbuf);

    if (qddb_setbuf_buflen+len >= qddb_setbuf_bufsiz) {
	qddb_setbuf_bufsiz += MAX(len+10, BUFSIZ);
	qddb_setbuf_buffer = Realloc(qddb_setbuf_buffer, qddb_setbuf_bufsiz);
    }
    strcpy(qddb_setbuf_buffer+qddb_setbuf_buflen, newbuf);
    qddb_setbuf_buflen += len;
}

char *Qddb_PrintErrno(qerrno) 
    int				qerrno;
{
    switch (qerrno) {
    case QDDB_ERRNO_INVALID_ARG:
	return "Invalid argument";
    case QDDB_ERRNO_INVALID_REGEXP:
	return "Invalid regular expression";
    default:
	return "";
    }
}


char *Qddb_FindKey(string, len)
    char			*string;
    size_t			*len;
{
    int				state;
    size_t			nlen;

    state = 0;
    nlen = 0;
    for (; *string != '\0'; string++, nlen++) {
	if (*string == '\\') {
	    state = 1;
	} else if (state == 1) {
	    state = 0;
	} else if (*string == ' ') {
	    break;
	}
    }
    *len = nlen;
    return string;
}

char *Qddb_StripKey(string, len)
    char			*string;
    size_t			len;
{
    int				state;
    char			*retval, *ch;

    state = 0;
    retval = Malloc(len+1);
    for (ch = retval; *string != '\0'; string++, ch++) {
	if (state == 0 && *string == '\\') {
	    state = 1;
	    ch--;
	} else {
	    state = 0;
	    *ch = *string;
	}
    }
    *ch = '\0';
    return retval;
}
