
/* qddb/Lib/LibQddb/Search.c 
 *
 * Copyright (C) 1993, 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */


#include "Qddb.h"

/* EXPORTED:
 *	KeyList	*Qddb_Search(Schema *, Qddb_SearchArg *, size_t *, int)
 *	KeyList *Qddb_ParseKey _ANSI_ARGS_((char *, off_t *, Boolean, int));
 *	char *Qddb_ConvertAttributeNumberToName _ANSI_ARGS_((Schema *, int));
 *	int Qddb_ConvertAttributeNameToNumber _ANSI_ARGS_((Schema *, char *));
 */

static KeyList *RegexpSearch _ANSI_ARGS_((Schema *, char *, size_t *, int));
static KeyList *NumericSearch _ANSI_ARGS_((Schema *, double, size_t *, int));
static KeyList *NumericRangeSearch _ANSI_ARGS_((Schema *, NumericRange *, size_t *, int));
static KeyList *StringRangeSearch _ANSI_ARGS_((Schema *, StringRange *, size_t *, int));
static KeyList *Search _ANSI_ARGS_((Schema *, char *, size_t *, int));
static KeyList *SecondarySearch _ANSI_ARGS_((Schema *, size_t *, KeyList *, Qddb_SearchArg *, int));

static KeyList *IndexLinesToKeyList _ANSI_ARGS_((Schema *, Entry, size_t *, int));

/* Qddb_Search -- Main driver for searching a QDDB database.
 */
KeyList *Qddb_Search(schema, arg, number_read, attrnum)
    Schema		*schema;
    Qddb_SearchArg	*arg;
    size_t		*number_read;
    int			attrnum;
{
    KeyList		*retval;
    NumericRange	NumericRangeVar;
    StringRange		StringRangeVar;

    qddb_errno = 0;
    if (qddb_errmsg != NULL)
	Free(qddb_errmsg);
    qddb_errmsg = NULL;
    switch (arg->Type) {
    case REGEXP_V8:
	retval = RegexpSearch(schema, arg->Alpha, number_read, attrnum);
	break;
    case NUMERIC:
	retval = NumericSearch(schema, arg->Number, number_read, attrnum);
	break;
    case ALPHA:
	retval = Search(schema, arg->Alpha, number_read, attrnum);
	break;
    case NUMERICRANGE:
	NumericRangeVar.UpperExists = arg->UpperExists;
	NumericRangeVar.LowerExists = arg->LowerExists;
	NumericRangeVar.Lower = arg->LowerNum;	
	NumericRangeVar.Upper = arg->UpperNum;	
	retval = NumericRangeSearch(schema, &NumericRangeVar, number_read, attrnum);
	break;
    case ALPHARANGE:
	StringRangeVar.UpperExists = arg->UpperExists;
	StringRangeVar.LowerExists = arg->LowerExists;
	StringRangeVar.Lower = arg->LowerStr;	
	StringRangeVar.Upper = arg->UpperStr;	
	retval = StringRangeSearch(schema, &StringRangeVar, number_read, attrnum);
	break;
    default:
	qddb_errno = QDDB_ERRNO_INVALID_ARG;
	return NULL;
    }
    return retval;
}

#ifdef V8_REGEXP
static KeyList *RegexpSearch(schema, Regexp, NumberRead, attrnum)
    Schema		*schema;
    char        	*Regexp;
    size_t	    	*NumberRead;
    int			attrnum;
{
    char		*RelationFN = schema->RelationName;
    char		KeyFN[MAXFILENAMELEN], IndexFN[MAXFILENAMELEN];
    int			KeyFile, IndexFile;
    regexp		*RegexpVal;
    char		*Chunk = NULL, *ch;
    size_t		Size, NumberOfChunks;
    int			NumberOfBytes;
    KeyList		*ReturnValue = NULL, *TmpKeyList, *last;
    int			i, j;
    Qddb_SearchArg	Arg;
    
#ifdef DEBUGREGEXP
    fprintf(stderr, "RegexpSearch(%s, %s) attrnum(%d)\n", RelationFN, Regexp, attrnum);
    fflush(stderr);
#endif
    *NumberRead = 0;
    strcpy(KeyFN, RelationFN);
    strcat(KeyFN, "/KeyIndex");
    KeyFile = open(KeyFN, O_RDONLY);
    strcpy(IndexFN, RelationFN);
    strcat(IndexFN, "/Index");
    IndexFile = open(IndexFN, O_RDONLY);
    ch = Regexp;
    while (*ch) {
	if (_isupper(*ch)) {
	    *ch = _tolower(*ch);
	}
	ch++;
    }
    RegexpVal = regcomp(Regexp);
    if (RegexpVal == NULL)
	return NULL;
    if (KeyFile == -1 || IndexFile == -1 || SizeOfFile(KeyFile) == 0 || SizeOfFile(IndexFile) == 0)
	goto secondary_search;
    Chunk = Malloc(CHUNKSIZE+1);
    /* Read in a chunk at a time, parse it up into newline separated
     * pieces and compare the regular expression to each word.
     */
    Size = SizeOfFile(KeyFile);
    NumberOfChunks = (Size/CHUNKSIZE);
    if (Size % CHUNKSIZE != 0)
	NumberOfChunks++;
#ifdef DEBUGREGEXP
    fprintf(stderr, "RegexpSearch: %d chunks\n", NumberOfChunks);
    fflush(stderr);
#endif
    for (i = 0; i < NumberOfChunks; i++) {
#ifdef DEBUGREGEXP
	fprintf(stderr, "RegexpSearch: %dth chunk (top of loop)\n", i);
	fflush(stderr);
#endif
	if ((NumberOfBytes = read(KeyFile, Chunk, CHUNKSIZE)) == -1)
	    PANIC("RegexpSearch on read");
	else {
	    char		*lastnewline;
	    int			tmpi;
	    
#ifdef DEBUGREGEXP
    fprintf(stderr, "RegexpSearch: %dth chunk\n", i);
    fflush(stderr);
#endif
	    j = 0;
	    /* rewind to last newline */
	    tmpi = NumberOfBytes;
	    lastnewline = Chunk+NumberOfBytes-1;
	    while (NumberOfBytes-1 > 0 && *lastnewline != '\n')
#ifdef DEBUGREGEXP
	    {
		fprintf(stderr, "backing up...\n");
#endif
		NumberOfBytes--, lastnewline--;
#ifdef DEBUGREGEXP
	    }
#endif
	    if (NumberOfBytes != 1)
		/* wastes a system call when tmpi == NumberOfBytes */
		lseek(KeyFile, (off_t)NumberOfBytes-tmpi, 1); /* move to beginning of last line */
	    else
		NumberOfBytes = tmpi+1; /* add the newline at the end */
	    if (Chunk[j] == '\n')
		j++;
	    Chunk[NumberOfBytes-1] = '\n';	/* guarantee the newline at the end of file */
#ifdef DEBUGREGEXP
	    fprintf(stderr, "NumberOfBytes %d\n", NumberOfBytes);
	    fflush(stderr);
#endif
	    while (j < NumberOfBytes) { /* guaranteed to end with newline */
		char		*strstart, *strend, *newstring;
		size_t		len;
		
		strstart = Chunk+j;
		strend = Qddb_FindKey(strstart, &len);
		*strend = '\0';
		newstring = Qddb_StripKey(strstart, len);
		if (regexec(RegexpVal, newstring) == 1 &&
		    newstring == RegexpVal->startp[0] && 
		    (newstring + strlen(newstring)) == RegexpVal->endp[0]) {
		    char		*stroffset, *strlength;
		    off_t		offset;
		    size_t		length;
		    char		*IndexBuffer;

		    *strend = ' ';
		    strend++;
		    /* Grab the offset:length of the KeyList */
		    /* offset */
		    stroffset = strend;
		    while (*strend != ' ')	/* hit a space */
			strend++;
		    *strend = '\0';
		    offset = atoi(stroffset);
		    *strend++ = ' ';
		    /* length */
		    strlength = strend;
		    while (*strend != '\n')	/* hit the newline */
			strend++;
		    *strend = '\0';
		    length = atoi(strlength);
		    *strend++ = '\n';
		    /* read in the Index entry */
		    lseek(IndexFile, offset, SEEK_SET);
		    IndexBuffer = Malloc(length+1);
		    Read(IndexFile, IndexBuffer, length);
		    IndexBuffer[length] = '\0';
		    /* Parse it into a KeyList */
		    TmpKeyList = Qddb_ParseKey(schema, IndexBuffer, NumberRead, True, attrnum);
		    Free(IndexBuffer);
 		    if (ReturnValue == NULL) {
			ReturnValue = TmpKeyList;
			last = NULL;
		    } else {
			ReturnValue = Qddb_KeyListConcat(ReturnValue, TmpKeyList, &last);
		    }
		} else if (qddb_errno != 0) {
		    Free(newstring);
		    return NULL;
		}
		Free(newstring);
		while (*strstart != '\n')
		    j++, strstart++;
		j++;
	    } 
	}
    }
secondary_search:
#ifdef DEBUGREGEXP
    fprintf(stderr, "Secondary Search, RegexpSearch(%s, %s) attrnum(%d)\n", RelationFN, Regexp, attrnum);
    fflush(stderr);
#endif
    Arg.Type = REGEXP_V8;
    Arg.Reg = RegexpVal;
    TmpKeyList = SecondarySearch(schema, NumberRead, ReturnValue, &Arg, attrnum);
    Free(RegexpVal);
    if (Chunk != NULL)
	Free(Chunk);
    close(KeyFile); 
    close(IndexFile);
    return TmpKeyList;
}
#endif

#ifdef SQL
/* Numeric , numeric range, and string range searching.
 */

static void Chunking_GetChunk _ANSI_ARGS_((int, char *, off_t, size_t));
static void Chunking_AdjustBounds _ANSI_ARGS_((char **, ChunkBounds *, size_t, size_t));
static ChunkBounds *Chunking_Search _ANSI_ARGS_((int, Qddb_SearchArg *, ChunkBounds *, size_t));
static void Chunking_FixBounds _ANSI_ARGS_((int, ChunkBounds *, size_t, size_t, Qddb_SearchArg *));
static char *Chunking_Ending _ANSI_ARGS_((char *, size_t));
static char *Chunking_Beginning _ANSI_ARGS_((char *, size_t));
static void Chunking_Build _ANSI_ARGS_((int, size_t, ChunkBounds **, size_t *));

static void Chunking_Build(fd, chunk_size, bounds_ptr, num_ptr)
    int			fd;
    size_t		chunk_size;
    ChunkBounds		**bounds_ptr;
    size_t		*num_ptr;
{
    int			i;
    size_t		fd_size, num;

    fd_size = SizeOfFile(fd);
    *num_ptr = fd_size / chunk_size;
    if ((fd_size % chunk_size) != 0)
	(*num_ptr)++;
    *bounds_ptr = (ChunkBounds *)Malloc(sizeof(ChunkBounds)*(*num_ptr));
    num = *num_ptr;
    for (i = 0; i < num-1; i++) {
	(*bounds_ptr)[i].start  = i*chunk_size;
	(*bounds_ptr)[i].length = chunk_size;
	(*bounds_ptr)[i].used = False;
    }
    (*bounds_ptr)[i].start = i*chunk_size;
    (*bounds_ptr)[i].length = fd_size - i*chunk_size;
}

static char *Chunking_Beginning(buf, len)
    char		*buf;
    size_t		len;
{
    size_t		newlen;
    char		*retval, *ch;

#if defined(DIAGNOSTIC)
    if (buf[0] != '\n')
	fprintf(stderr, "Chunking_Beginning: buf[0] != '\\n'");
#endif
    ch = Qddb_FindKey(buf+1, &newlen);
    *ch = '\0';
    retval = Qddb_StripKey(buf+1, newlen);
    *ch = ' ';
#if defined(notdef)
    for (i = 1; i < len; i++) {
	if (buf[i] == ' ')
	    break;
    }
    retval = Malloc((size_t)i);
    strncpy(retval, buf+1, (size_t)i-2);
#endif
    return retval;
}

static char *Chunking_Ending(buf, len)
    char		*buf;
    size_t		len;
{
    size_t		last = len-1, first, newlen;
    int			i;
    char		*retval, *ch;

#if defined(DIAGNOSTIC)
    if (buf[last] != '\n')
	fprintf(stderr, "Chunking_Ending: buf[%d] != '\\n'", (int)last);
#endif
    for (i = last-1; i > 0; i--) {
	if (buf[i] == '\n')
	    break;
    }
    i++;
    first = i;
    ch = Qddb_FindKey(buf+i, &newlen);
    *ch = '\0';
    retval = Qddb_StripKey(buf+i, newlen);
    *ch = ' ';
#if defined(notdef)
    retval = Malloc((size_t)(last-i));
    for (; i < last; i++) {
	if (buf[i] == ' ')
	    break;
    }
    strncpy(retval, buf+first, i-first);
#endif
    return retval;
}

static void Chunking_GetChunk(fd, buf, start, length)
    int			fd;
    char 		*buf;
    off_t		start;
    size_t		length;
{
    lseek(fd, start, SEEK_SET);
    if (read(fd, buf, length) != length) {
	perror("read");
	fprintf(stderr, "Chunking_GetChunk: read error (%d, %d)\n", (int)start, (int)length);
    }
}

static void Chunking_AdjustBounds(obuf, bounds, mid, num)
    char		**obuf;
    ChunkBounds		*bounds;
    size_t		mid, num;
{
    char		*buf, *endbuf;
    size_t		length;

    length = bounds[mid].length;
    buf = *obuf;
    endbuf = buf + length - 1; /* last char in buf */
    /* Adjust bounds in mid-1 and mid+1 to account for newlines.
     */
    if (buf[0] != '\n' || buf[length-1] != '\n') {
	while (*buf != '\n') {
	    bounds[mid].start++;
	    bounds[mid].length--;
	    if (mid != 0) {
		bounds[mid-1].length++;
	    }
	    buf++;
	}
	while (*endbuf != '\n') {
	    bounds[mid].length--;
	    if (mid != num-1) {
		bounds[mid+1].start--;
		bounds[mid+1].length++;
	    }
	    endbuf--;
	}
    }
    *obuf = buf;
}

static ChunkBounds *Chunking_Search(fd, search_arg, bounds, num)
    int			fd;
    Qddb_SearchArg	*search_arg;
    ChunkBounds		*bounds;
    size_t		num;
{
    size_t		mid = num/2;
    off_t		start;
    size_t		length;
    char		*buf, *ending, *beginning;
    double		end_num, begin_num;
    ChunkBounds		*retval;

    if (num <= 0)
	return NULL;
    start = bounds[mid].start;
    length = bounds[mid].length;
    buf = alloca(length);
    Chunking_GetChunk(fd, buf, start, length);
    Chunking_AdjustBounds(&buf, bounds, mid, num);
    length = bounds[mid].length;
    /* Check the extremities to see if the desired number matches either
     * endpoint or is contained in the middle somewhere.
     */
    beginning = Chunking_Beginning(buf, length);
    ending = Chunking_Ending(buf, length);
    if (search_arg->Type == NUMERIC) {
	begin_num = strtod(beginning, NULL);
	end_num = strtod(ending, NULL);
	if (search_arg->Number < begin_num) {
	    retval = Chunking_Search(fd, search_arg, bounds, mid);
	    if (retval == NULL)
		retval = bounds+mid;
	} else if (search_arg->Number > end_num) {
	    retval = Chunking_Search(fd, search_arg, bounds+mid+1, num-mid-1);
	    if (retval == NULL)
		retval = bounds+mid;
	} else {
	    /* Might be either totally contained in this chunk, or it
	     * might span chunks.
	     */
	    retval = bounds+mid;
	}
    } else {
	if (strcmp(search_arg->Alpha, beginning) < 0) {
	    retval = Chunking_Search(fd, search_arg, bounds, mid);
	    if (retval == NULL)
		retval = bounds+mid;
	} else if (strcmp(search_arg->Alpha, ending) > 0) {
	    retval = Chunking_Search(fd, search_arg, bounds+mid+1, num-mid-1);
	    if (retval == NULL)
		retval = bounds+mid;
	} else {
	    /* Might be either totally contained in this chunk, or it
	     * might span chunks.
	     */
	    retval = bounds+mid;
	}	
    }
    Free(beginning);
    Free(ending);
    return retval;
}

static void Chunking_FixBounds(fd, bounds, idx, num, arg)
    int			fd;
    ChunkBounds		*bounds;
    size_t		idx, num;
    Qddb_SearchArg	*arg;
{
    char		*beginning, *ending, *buf, *obuf;
    int			i;

    for (i = idx-1; i >= 0; i--) {
	obuf = buf = Malloc(bounds[i].length);
	Chunking_GetChunk(fd, buf, bounds[i].start, bounds[i].length);
	Chunking_AdjustBounds(&buf, bounds,  (size_t)i, num);
	ending = Chunking_Ending(buf, bounds[i].length);
	if (arg->Type == NUMERIC) {
	    if (arg->Number > strtod(ending, NULL)) {
		Free(ending);
		Free(obuf);
		break;
	    }
	    bounds[i].used = True;
	} else {
	    if (strcmp(arg->Alpha, ending) > 0) {
		Free(ending);
		Free(obuf);
		break;
	    }
	    bounds[i].used = True;
	}
	Free(ending);
	Free(obuf);
    }
    for (i = idx+1; i < num; i++) {
	obuf = buf = Malloc(bounds[i].length);
	Chunking_GetChunk(fd, buf, bounds[i].start, bounds[i].length);
	Chunking_AdjustBounds(&buf, bounds, (size_t)i, num);
	beginning = Chunking_Beginning(buf, bounds[i].length);
	if (arg->Type == NUMERIC) {
	    if (arg->Number < strtod(beginning, NULL)) {
		Free(beginning);
		Free(obuf);
		break;
	    }
	    bounds[i].used = True;
	} else {
	    if (strcmp(arg->Alpha, beginning) < 0) {
		Free(beginning);
		Free(obuf);
		break;
	    }
	    bounds[i].used = True;
	}
	Free(beginning);
	Free(obuf);
    }
}

/* IndexLinesToKeyList -- Convert an Entry full of *Index lines
 * to a single KeyList.   This routine destroys the Entry 'lines.'
 */
static KeyList *IndexLinesToKeyList(schema, lines, num_read, attrnum)
    Schema		*schema;
    Entry		lines;
    size_t		*num_read;
    int			attrnum;
{
    char		idx_path[BUFSIZ], *buf;
    int			fd;
    KeyList		*retval, *tmplist, *last = NULL;
    int			start, length;

    if (lines == NULL)
	return NULL;
    retval = NULL;
    strcpy(idx_path, schema->RelationName);
    strcat(idx_path, "/Index");
    if ((fd = open(idx_path, O_RDONLY)) < 0) {
	fprintf(stderr, "cannot open %s\n", idx_path);
	return NULL;
    }
    if (*lines != NULL && (*lines)[0] == '\0')
	lines++;
    while (*lines != NULL) {
	char		*ch = *lines;
	int		state;
	char		*start_ch, *length_ch;

	state = 0;
	while (state != 0 || *ch != ' ') {
	    if (state == 0 && *ch == '\\') {
		state = 1;
	    } else {
		state = 0;
	    }
	    ch++;
	}
	ch++;
	start_ch = ch;
	while (*ch != ' ')
	    ch++;
	*ch++ = '\0';
	length_ch = ch;
	start = atoi(start_ch);
	length = atoi(length_ch);
	buf = Malloc((size_t)length+1);
	buf[length] = '\0';
	if (lseek(fd, (off_t)start, SEEK_SET) < 0) 
	    perror("lseek");
	if (read(fd, buf, (size_t)length) != length) {
	    fprintf(stderr, "IndexLinesToKeyList: read failed\n");
	    break;
	}
	tmplist = Qddb_ParseKey(schema, buf, num_read, True, attrnum);
	Free(buf);
	if (retval == NULL) {
	    retval = tmplist;
	    last = NULL;
	} else if (tmplist != NULL)
	    retval = Qddb_KeyListConcat(retval, tmplist, &last);
	lines++;
    }
    close(fd);
    retval = Qddb_KeyListProcess(schema, retval, NULL, QDDB_KEYLIST_PROC_SORT, 0);
    return retval;
}

/* PruneLines -- Prune the outer edges of the chunk.   Free the old
 * entry before returning.
 */
static Entry PruneLines(lines, lower, upper)
    Entry		lines;
    Qddb_SearchArg	*lower, *upper;
{
    Entry		tmp, top, retval, oretval;
    int			i;

#if defined(DIAGNOSTIC)
    if (lines == NULL || *lines == NULL)
	return NULL;
#endif
    if (lower == NULL && upper == NULL)
	return lines;
    tmp = lines;
    while ((*tmp)[0] == '\0') {
	Free(*tmp);
	tmp++;
    }
    if (tmp == NULL) {
	Free(lines);
	return NULL;
    }
    top = NULL;
    i = 0;
    if ((lower != NULL && lower->Type == NUMERIC) || (upper != NULL && upper->Type == NUMERIC)) {
	/* Throw-aways */
	if (lower != NULL) {
	    while (*tmp != NULL) {
		if (lower->Number <= strtod(*tmp, NULL))
		    break;
		Free(*tmp);
		tmp++;
	    }
	}
	/* Keepers */
	top = tmp;
	while (*tmp != NULL) {
	    i++;
	    if (upper != NULL && upper->Number < strtod(*tmp, NULL))
		break;
	    tmp++;
	}
	i++;
    } else if ((lower != NULL && lower->Type == ALPHA) || (upper != NULL && upper->Type == ALPHA)) {
	char		*ch, *tmpch;
	size_t		newlen;

	/* Throw-aways */
	if (lower != NULL) {
	    while (*tmp != NULL) {
		ch = Qddb_FindKey(*tmp, &newlen);
		*ch = '\0';
		tmpch = Qddb_StripKey(*tmp, newlen);
		*ch = ' ';
		if (strcmp(lower->Alpha, tmpch) <= 0) {
		    Free(tmpch);
		    break;
		}
		Free(tmpch);
		Free(*tmp);
		tmp++;
	    }
	}
	/* Keepers */
	top = tmp;
	while (*tmp != NULL) {
	    ch = Qddb_FindKey(*tmp, &newlen);
	    *ch = '\0';
	    tmpch = Qddb_StripKey(*tmp, newlen);
	    *ch = ' ';
	    i++;
	    if (upper != NULL && strcmp(upper->Alpha, tmpch) < 0) {
		Free(tmpch);
		break;
	    }
	    Free(tmpch);
	    tmp++;
	}
	i++;
    }
    /* Throw-aways */
    while (*tmp != NULL) {
	Free(*tmp);
	*tmp = NULL;
	tmp++;
    }
    if (i == 0) {
	Free(lines);
	return NULL;
    }
    oretval = retval = (Entry)Malloc(sizeof(char *)*i);
    while (*top != NULL) {
	*oretval = *top;
	oretval++;
	top++;
    }
    *oretval = NULL;
    Free(lines);
    return retval;
}


/* NumericSearch -- Search for all occurances of a single number in 'NumericIndex' 
 * and return the union of keylists for all such occurances.
 */
static KeyList *NumericSearch(schema, number, number_read, attrnum)
    Schema		*schema;
    double		number;
    size_t		*number_read;
    int			attrnum;
{
    int			numidx_fd;
    char		*numidx_path = alloca(strlen(schema->RelationName)+14), *buf, *obuf;
    ChunkBounds		*chunk_bounds, *find_bounds;
    int			idx, minidx, maxidx, i;
    off_t		buf_start;
    size_t		numchunks, buf_len;
    Entry		lines;
    KeyList		*retval = NULL;
    Qddb_SearchArg	search_arg;
    
    strcpy(numidx_path, schema->RelationName);
    strcat(numidx_path, "/NumericIndex");
    search_arg.Type = NUMERIC;
    search_arg.Number = number;
    if ((numidx_fd = open(numidx_path, O_RDONLY)) < 0 || SizeOfFile(numidx_fd) == 0)
	goto secondary_search;
    Chunking_Build(numidx_fd, CHUNK_SIZE, &chunk_bounds, &numchunks);
    find_bounds = Chunking_Search(numidx_fd, &search_arg, chunk_bounds, numchunks);
    idx = find_bounds - chunk_bounds;
    chunk_bounds[idx].used = True;
    Chunking_FixBounds(numidx_fd, chunk_bounds, (size_t)idx, numchunks, &search_arg);
    minidx = maxidx = idx;
    while (minidx-1 >= 0 && chunk_bounds[minidx-1].used == True)
	minidx--;
    while (maxidx+1 < numchunks && chunk_bounds[maxidx+1].used == True)
	maxidx++;
    buf_start = chunk_bounds[minidx].start;
    buf_len = chunk_bounds[minidx].length;
    for (i = minidx+1; i <= maxidx; i++) {
	buf_len += chunk_bounds[i].length;
    }
    Free(chunk_bounds);
    obuf = buf = Malloc(buf_len);
    Chunking_GetChunk(numidx_fd, buf, buf_start, buf_len);
    lines = NULL;
    Qddb_SplitBufferIntoLines(&lines, buf, buf_len);
    lines = PruneLines(lines, &search_arg, &search_arg);
    retval = IndexLinesToKeyList(schema, lines, number_read, attrnum);
    Qddb_Free(QDDB_TYPE_ENTRY, lines);
    Free(obuf);
    /* buf contains each row with 'number' in it. Now we need to parse up 
     * the rows, read the 'Index' entries, and build the KeyLists.
     */
secondary_search:
    retval = SecondarySearch(schema, number_read, retval, &search_arg, attrnum);
    close(numidx_fd);
    return retval;
}

/* NumericRangeSearch -- Search for a range of numbers in 'NumericIndex'
 * and return the corresponding union of the keylists for each number.
 * Each row in 'NumericIndex' corresponds to a single row in 'Index.'
 * Note that multiple textual numbers may be equivalent to a single
 * number.
 */
static KeyList *NumericRangeSearch(schema, nr_var, number_read, attrnum)
    Schema		*schema;
    NumericRange	*nr_var;
    size_t	 	*number_read;
    int			attrnum;
{
    int			numidx_fd;
    char		*numidx_path = alloca(strlen(schema->RelationName)+14), *buf, *obuf;
    ChunkBounds		*chunk_bounds, *find_bounds;
    int			idx = 0, idx2 = 0, minidx, maxidx, i;
    off_t		buf_start;
    size_t		numchunks, buf_len;
    Entry		lines;
    KeyList		*retval = NULL;
    Qddb_SearchArg	search_arg, lower_arg, upper_arg;
    
    strcpy(numidx_path, schema->RelationName);
    strcat(numidx_path, "/NumericIndex");
    if ((numidx_fd = open(numidx_path, O_RDONLY)) < 0 || SizeOfFile(numidx_fd) == 0)
	goto secondary_search;
    Chunking_Build(numidx_fd, CHUNK_SIZE, &chunk_bounds, &numchunks);
    if (nr_var->LowerExists == True) {
	search_arg.Type = NUMERIC;
	search_arg.Number = nr_var->Lower;
	find_bounds = Chunking_Search(numidx_fd, &search_arg, chunk_bounds, numchunks);
	idx = find_bounds - chunk_bounds;
	chunk_bounds[idx].used = True;
	Chunking_FixBounds(numidx_fd, chunk_bounds, (size_t)idx, numchunks, &search_arg);
    }
    if (nr_var->UpperExists == True) {
	search_arg.Type = NUMERIC;
	search_arg.Number = nr_var->Upper;
	find_bounds = Chunking_Search(numidx_fd, &search_arg, chunk_bounds, numchunks);
	idx2 = find_bounds - chunk_bounds;
	chunk_bounds[idx2].used = True;
	Chunking_FixBounds(numidx_fd, chunk_bounds, (size_t)idx2, numchunks, &search_arg);
    }
    if (nr_var->LowerExists == True && nr_var->UpperExists == True) {
	maxidx = MAX(idx,idx2);
	for (i = MIN(idx,idx2)+1; i < maxidx; i++)
	    chunk_bounds[i].used = True;
    } else if (nr_var->LowerExists == True) {
	/* !UpperExists */
	for (i = idx; i < numchunks; i++)
	    chunk_bounds[i].used = True;
    } else if (nr_var->UpperExists == True) {
	/* !LowerExists */
	for (i = idx2; i >= 0; i--)
	    chunk_bounds[i].used = True;
	idx = idx2;
    } else {
	/* !LowerExists && !UpperExists */
	for (i = 0; i < numchunks; i++)
	    chunk_bounds[i].used = True;
	idx = 0;
    }
    minidx = maxidx = idx;
    while (minidx-1 >= 0 && chunk_bounds[minidx-1].used == True)
	minidx--;
    while (maxidx+1 < numchunks && chunk_bounds[maxidx+1].used == True)
	maxidx++;
    buf_start = chunk_bounds[minidx].start;
    buf_len = chunk_bounds[minidx].length;
    for (i = minidx+1; i <= maxidx; i++) {
	buf_len += chunk_bounds[i].length;
    }
    Free(chunk_bounds);
    obuf = buf = Malloc(buf_len);
    Chunking_GetChunk(numidx_fd, buf, buf_start, buf_len);
    lines = NULL;
    Qddb_SplitBufferIntoLines(&lines, buf, buf_len);
    lower_arg.Type = NUMERIC;
    lower_arg.Number = nr_var->Lower;
    upper_arg.Type = NUMERIC;
    upper_arg.Number = nr_var->Upper;
    lines = PruneLines(lines, nr_var->LowerExists == True? &lower_arg:NULL, 
		       nr_var->UpperExists == True? &upper_arg:NULL);
    retval = IndexLinesToKeyList(schema, lines, number_read, attrnum);
    Qddb_Free(QDDB_TYPE_ENTRY, lines);
    /* buf contains each row with 'number' in it. Now we need to parse up 
     * the rows, read the 'Index' entries, and build the KeyLists.
     */
secondary_search:
    search_arg.Type = NUMERICRANGE;
    search_arg.UpperNum = nr_var->Upper;
    search_arg.LowerNum = nr_var->Lower;
    search_arg.UpperExists = nr_var->UpperExists;
    search_arg.LowerExists = nr_var->LowerExists;
    retval = SecondarySearch(schema, number_read, retval, &search_arg, attrnum);
    close(numidx_fd);
    return retval;
}


/* StringRangeSearch -- Search for a range of strings (lexicographically)
 * in 'KeyIndex' and return the union of the keylists for each string.
 * Each entry in 'KeyIndex' corresponds to a single row in 'Index'.
 */
static KeyList *StringRangeSearch(schema, sr_var, number_read, attrnum)
    Schema		*schema;
    StringRange		*sr_var;
    size_t	 	*number_read;
    int			attrnum;
{
    int			stridx_fd;
    char		*stridx_path = alloca(strlen(schema->RelationName)+10), *buf, *obuf, *ch;
    ChunkBounds		*chunk_bounds, *find_bounds;
    int			idx = 0, idx2 = 0, minidx, maxidx, i;
    off_t		buf_start;
    size_t		numchunks, buf_len;
    Entry		lines;
    KeyList		*retval = NULL;
    Qddb_SearchArg	search_arg, lower_arg, upper_arg;
    
    strcpy(stridx_path, schema->RelationName);
    strcat(stridx_path, "/KeyIndex");
    if ((stridx_fd = open(stridx_path, O_RDONLY)) < 0 || SizeOfFile(stridx_fd) == 0)
	goto secondary_search;
    Chunking_Build(stridx_fd, CHUNK_SIZE, &chunk_bounds, &numchunks);
    if (sr_var->LowerExists == True) {
	for (ch = sr_var->Lower; *ch; ch++)
	    *ch = tolower(*ch);
	search_arg.Type = ALPHA;
	search_arg.Alpha = sr_var->Lower;
	find_bounds = Chunking_Search(stridx_fd, &search_arg, chunk_bounds, numchunks);
	idx = find_bounds - chunk_bounds;
	chunk_bounds[idx].used = True;
	Chunking_FixBounds(stridx_fd, chunk_bounds, (size_t)idx, numchunks, &search_arg);
    }
    if (sr_var->UpperExists == True) {
	for (ch = sr_var->Upper; *ch; ch++)
	    *ch = tolower(*ch);
	search_arg.Type = ALPHA;
	search_arg.Alpha = sr_var->Upper;
	find_bounds = Chunking_Search(stridx_fd, &search_arg, chunk_bounds, numchunks);
	idx2 = find_bounds - chunk_bounds;
	chunk_bounds[idx2].used = True;
	Chunking_FixBounds(stridx_fd, chunk_bounds, (size_t)idx2, numchunks, &search_arg);
    }
    if (sr_var->LowerExists == True && sr_var->UpperExists == True) {
	maxidx = MAX(idx,idx2);
	for (i = MIN(idx,idx2)+1; i < maxidx; i++)
	    chunk_bounds[i].used = True;
    } else if (sr_var->LowerExists == True) {
	/* !UpperExists */
	for (i = idx; i < numchunks; i++)
	    chunk_bounds[i].used = True;
    } else if (sr_var->UpperExists == True) {
	/* !LowerExists */
	for (i = idx2; i >= 0; i--)
	    chunk_bounds[i].used = True;
	idx = idx2;
    } else {
	/* !LowerExists && !UpperExists */
	for (i = 0; i < numchunks; i++)
	    chunk_bounds[i].used = True;
	idx = 0;
    }
    minidx = maxidx = idx;
    while (minidx-1 >= 0 && chunk_bounds[minidx-1].used == True)
	minidx--;
    while (maxidx+1 < numchunks && chunk_bounds[maxidx+1].used == True)
	maxidx++;
    buf_start = chunk_bounds[minidx].start;
    buf_len = chunk_bounds[minidx].length;
    for (i = minidx+1; i <= maxidx; i++) {
	buf_len += chunk_bounds[i].length;
    }
    Free(chunk_bounds);
    obuf = buf = Malloc(buf_len);
    Chunking_GetChunk(stridx_fd, buf, buf_start, buf_len);
    lines = NULL;
    Qddb_SplitBufferIntoLines(&lines, buf, buf_len);
    lower_arg.Type = ALPHA;
    lower_arg.Alpha = sr_var->Lower;
    upper_arg.Type = ALPHA;
    upper_arg.Alpha = sr_var->Upper;
    lines = PruneLines(lines, sr_var->LowerExists == True? &lower_arg:NULL, 
		       sr_var->UpperExists == True? &upper_arg:NULL);
    retval = IndexLinesToKeyList(schema, lines, number_read, attrnum);
    Qddb_Free(QDDB_TYPE_ENTRY, lines);
secondary_search:
    search_arg.Type = ALPHARANGE;
    search_arg.UpperStr = sr_var->Upper;
    search_arg.LowerStr = sr_var->Lower;
    search_arg.UpperExists = sr_var->UpperExists;
    search_arg.LowerExists = sr_var->LowerExists;
    retval = SecondarySearch(schema, number_read, retval, &search_arg, attrnum);
    close(stridx_fd);
    return retval;
}
   
#endif /* SQL */

static KeyList *Search(schema, String, NumberRead, attrnum)
    Schema		*schema;
    char        	*String;
    size_t	    	*NumberRead;
    int			attrnum;
{
    char		*RelationFN = schema->RelationName;
    char        	KeyFN[MAXFILENAMELEN]; 
    int        		KeyFile;
    char        	**KeyEntry;
    char        	*Buffer, *teststr;
    int        		HashVal, EntryNumber = 0;
    int        		i, len, last;
    size_t		Length;
    KeyList        	*ReturnValue = NULL;
    char        	*StringPtr = String, *tok_str;
    Qddb_SearchArg	Arg;
    
#ifdef DEBUG
fprintf(stderr, "Search()\n");
#endif

    while (*StringPtr != '\0') {
	if (_isupper(*StringPtr))
	    *StringPtr = _tolower(*StringPtr);
	StringPtr++;
    }
    /* split string if necessary */
    len = strlen(String);
    teststr = (char *)Malloc(strlen(String)+1);
    strcpy(teststr, String);
    *NumberRead = 0;
    if (attrnum == -1) {
	last = 0;
	for (i = 0; i < len; i++) {
	    if (ispunct(teststr[i]) || isspace(teststr[i])) {
		teststr[i] = '\0';
		if (strlen(teststr+last) == 0) {
		    continue;
		}
		if (ReturnValue == NULL) {
		    ReturnValue = Search(schema, teststr+last, &Length, attrnum);
		    *NumberRead += Length;
		} else {
		    ReturnValue = Qddb_KeyListOp(schema, 
						 Search(schema, teststr+last, &Length, attrnum), ReturnValue,
						 QDDB_KEYLIST_OP_INTERSECTION, 0);
		    *NumberRead += Length;
		}
		last = i+1;
	    }
	}
    } else {
	size_t			place_in_string;
	size_t			teststr_len = strlen(teststr);

	tok_str = strtok(teststr, schema->Entries[attrnum].Separators);
	place_in_string = 0;
	if (tok_str != NULL && strlen(tok_str) < strlen(String)) {
	    while (tok_str != NULL) {
		place_in_string += strlen(tok_str) + 1;
		if (ReturnValue == NULL) {
		    ReturnValue = Search(schema, tok_str, &Length, attrnum);
		    *NumberRead += Length;
		} else {
		    ReturnValue = Qddb_KeyListOp(schema, 
						 Search(schema, tok_str, &Length, attrnum), 
						 ReturnValue, QDDB_KEYLIST_OP_INTERSECTION, 0);
		    *NumberRead += Length;
		}
		if (place_in_string >= teststr_len)
		    break;
		tok_str = strtok(teststr+place_in_string, schema->Entries[attrnum].Separators);
	    }
	}
    }
    Free(teststr);
    if (ReturnValue != NULL) {
	return ReturnValue;
    }
    *NumberRead = 0;
    strcpy(KeyFN, RelationFN);
    strcat(KeyFN, "/Index");
    KeyFile = open(KeyFN, O_RDONLY);
    if (KeyFile == -1 || SizeOfFile(KeyFile) == 0)
	goto secondary_search;
    HashVal = Qddb_GetHashTableEntry(schema, (size_t)Qddb_HashValue(schema, String));
    if (HashVal == -1 || schema->HashTable[HashVal].num == 0)
	goto secondary_search;
    KeyEntry = (char **)Malloc(sizeof(char *)*(schema->HashTable[HashVal].num+1));
    Buffer = Malloc(schema->HashTable[HashVal].len);
    lseek(KeyFile, schema->HashTable[HashVal].pos, 0);
    Read(KeyFile, Buffer, schema->HashTable[HashVal].len);
    KeyEntry[0] = Buffer;
    /* Parse the hash bucket into newline terminated lines.
     */
    for (i = 0; i < schema->HashTable[HashVal].len; i++) {
        if (Buffer[i] == '\n') {
            Buffer[i] = '\0';
            if (EntryNumber >= schema->HashTable[HashVal].num)
                break;
            KeyEntry[++EntryNumber] = Buffer+i+1;
        } 
    }
    /* Search the hash bucket for the desired key.
     */
    for (i = 0; i < EntryNumber; i++) {
	char				*ch, *tmpch;
	size_t				len;

	ch = Qddb_FindKey(KeyEntry[i], &len);
	*ch = '\0';
	tmpch = Qddb_StripKey(KeyEntry[i], len);
        if (strcmp(String, tmpch) == 0) {
            /* The key was found, so get the list of all entries
             * in the database that contain the key.
             */
	    *ch = ' ';
	    Free(tmpch);
            ReturnValue = Qddb_ParseKey(schema, KeyEntry[i], NumberRead, True, attrnum);
            break;
        }
	*ch = ' ';
	Free(tmpch);
    }
    Free(Buffer);
    Free(KeyEntry);
secondary_search:
    close(KeyFile);
    Arg.Type = ALPHA;
    Arg.Alpha = String;
    ReturnValue = SecondarySearch(schema, NumberRead, ReturnValue, &Arg, attrnum);
    return ReturnValue;
}

KeyList *Qddb_ParseKey(schema, String, NumberRead, NormalIndex, attrnum)
    Schema		*schema;
    char        	*String;
    size_t	    	*NumberRead;
    Boolean 		NormalIndex;
    int			attrnum;
{
    int        		i, j, num, NUM;
    size_t		attribute_number;
    char        	*place, *word[4];
    KeyList        	*new, *list = NULL;
    size_t		len;

    place = Qddb_FindKey(String, &len) + 1;
    i = place - String;
    /* Build the key list */
    if (schema->UseReducedAttributeIdentifiers == True) {
	NUM = 3;
    } else {
	NUM = 4;
    }
    for (j = i+1, num = 0, word[0] = place; *place != '\0'; j++) {
        if (String[j] == ' ' || String[j] == '\n') {
            num++;
	    if (num < NUM) {
		word[num] = String + j + 1;
	    }
	    String[j] = '\0';
	}
	if (num == NUM) {
	    if (schema->UseReducedAttributeIdentifiers == True) {
		int 			idx;
		size_t			len;
		char			*ch;
		
		idx = Qddb_ReducedAttrToIndex(word[2]);
		len = strlen(schema->ReducedAttributeTable[idx]);
		ch = Malloc(len+2);
		ch[0] = '%', ch[1] = '\0';
		strcat(ch, schema->ReducedAttributeTable[idx]);
		if (Qddb_TranslateExternalID(schema, ch, &attribute_number, &word[3]) == -1) {
		    fprintf(stderr, "Cannot translate reduced attribute '%s' to full attribute\n", word[2]);
		    fprintf(stderr, "Found full attribute '%s'\n", schema->ReducedAttributeTable[idx]);
		    PANIC("INTERNAL ERROR");
		}
		Free(ch);
	    } else {
		attribute_number = atoi(word[2]);
	    }
	    if (attrnum != -1 && attrnum != attribute_number) {
		place = String + j + 1;
		word[0] = place;
                num = 0;
		if (schema->UseReducedAttributeIdentifiers == True) {
		    Free(word[3]);
		}
		continue;
	    }
            num = 0;
            if (list == NULL) {
                list = (KeyList *)Calloc(sizeof(KeyList));
                list->next = NULL;
            } else {
                new = (KeyList *)Calloc(sizeof(KeyList));
                new->next = list;
                list = new;
            }
	    list->Start = atoi(word[0]);
	    list->Length = atoi(word[1]);
	    list->Attribute = attribute_number;
	    if (schema->UseReducedAttributeIdentifiers == True) {
		if (qddb_stabilizing == 1) {
		    list->Instance = (char *)Malloc(strlen(word[2])+1);
		    strcpy(list->Instance, word[2]);
		    Free(word[3]);
		} else {
		    list->Instance = word[3];
		}
	    } else {
		list->Instance = (char *)Malloc(strlen(word[3])+1);
		strcpy(list->Instance, word[3]);
	    }
            QDDB_KEYLIST_SET_TYPE(list, ORIGINAL);
            place = String + j + 1;
	    word[0] = place;
            (*NumberRead)++;
        }
    }
    return list;
}


static KeyList *SecondarySearch(schema, NumberRead, ExistingList, Arg, attrnum)
    Schema		*schema;
    size_t	    	*NumberRead;
    KeyList        	*ExistingList;
    Qddb_SearchArg	*Arg;
    int			attrnum;
{
    char        	*RelationFN = schema->RelationName;
    char        	ChangeDir[MAXFILENAMELEN], AddDir[MAXFILENAMELEN];
    char        	TmpDir[MAXFILENAMELEN], dirnam[MAXFILENAMELEN];
    char        	*Buffer, *Line, *Token;
    KeyList        	*NewList = NULL, *LastNode = NULL;
    int			dirlen;
    size_t		attr;
    char		*inst;
    size_t	    	AddCount = 0, bufsize = BUFSIZ;
    FILE        	*FilePtr;
    DIR        		*dp, *opendir();
    struct dirent    	*DirEntry;
    
#if defined(DEBUG)
    fprintf(stderr, "in SecondarySearch\n");
    fflush(stderr);
#endif
    Buffer = Line = (char *)Malloc(bufsize);
    strcpy(ChangeDir, RelationFN);
    strcat(ChangeDir, "/Changes/");
    strcpy(AddDir, RelationFN);
    strcat(AddDir, "/Additions/");
    /* Search through the changes directory.
     */
    if ((dp = opendir(ChangeDir)) == NULL)
        PANIC("Cannot open Changes/ directory");
    while ((DirEntry = readdir(dp)) != NULL) {
	dirlen = NLENGTH(DirEntry);
	bcopy(DirEntry->d_name, dirnam, (size_t)dirlen);
	dirnam[dirlen] = '\0';
	if (dirlen == 0 || dirnam[0] == '.')
		continue;
        sprintf(TmpDir, "%s%s", ChangeDir, dirnam);
        FilePtr = fopen(TmpDir, "r");
        if (FilePtr == NULL)
            PANIC("An existing file in Changes/ cannot be opened");
	fgets(Line, bufsize, FilePtr); /* get %0 */
        while (fgets(Line, bufsize, FilePtr) != NULL) {
            char        *StringPtr;

	    while (Line[strlen(Line)-2] == '\\' || Line[strlen(Line)-1] != '\n') {
		if (strlen(Line) >= bufsize/2) {
		    bufsize *= 2;
		    Buffer = Line = (char *)Realloc(Buffer, bufsize);
		}
		if (fgets(Line+strlen(Line), bufsize-strlen(Line), FilePtr) == NULL)
		    break;
	    }
	    StringPtr = Line;
	    while (*StringPtr != '\0') {
		if (_isupper(*StringPtr))
		    *StringPtr = _tolower(*StringPtr);
		StringPtr++;
	    }
	    Token = strtok(Line, " "); /* Skip id */
	    Line = Line+strlen(Token)+1;
	    if (Qddb_TranslateExternalID(schema, Token, &attr, &inst) != 0) {
		fprintf(stderr, "Warning: attribute %s malformed, skipping..\n", Token);
		continue;
	    }
            /* Take care of punctuation */
            while ((Token=strtok(Line, schema->Entries[attr].Separators)) != NULL) {
		Boolean		Add;
		
		Add = False;
		switch(Arg->Type) {
#ifdef V8_REGEXP
		case REGEXP_V8:
		    /* Regexp or strcmp() */
		    if (Arg->Reg == NULL ||
			(regexec(Arg->Reg, Token) != 1 ||
			 Token != Arg->Reg->startp[0] ||
			 (Token+strlen(Token)) != Arg->Reg->endp[0]))
			break;
		    Add = True;
		    break;
#endif
		case ALPHA: 
		    if (strcmp(Token, Arg->Alpha) != 0)
			break;
		    Add = True;
		    break;
	        case NUMERIC:
		    if (strtod(Token, NULL) != Arg->Number)
			break;
		    Add = True;
		    break;
		case NUMERICRANGE: {
		    double 	tmpd;
		    
		    tmpd = strtod(Token, NULL);
		    if (Arg->LowerExists == True && Arg->UpperExists == True) {
			if (tmpd >= Arg->LowerNum && tmpd <= Arg->UpperNum)
			    Add = True;
		    } else if (Arg->LowerExists == True) {
			if (tmpd >= Arg->LowerNum)
			    Add = True;
		    } else if (Arg->UpperExists == True) {
			if (tmpd <= Arg->UpperNum)
			    Add = True;
		    } else 
			Add = True;
		    break;
		}
		case ALPHARANGE:
		    if (Arg->LowerExists == True && Arg->UpperExists == True) {
			if (strcmp(Arg->LowerStr, Token) <= 0 && strcmp(Arg->UpperStr, Token) >= 0)
			    Add = True;
		    } else if (Arg->LowerExists == True) {
			if (strcmp(Arg->LowerStr, Token) <= 0)
			    Add = True;
		    } else if (Arg->UpperExists == True) {
			if (strcmp(Arg->UpperStr, Token) >= 0)
			    Add = True;
		    } else 
			Add = True;
		    break;
		default:
		    PANIC("SecondarySearch: bad Arg->Type");
		}
		if (Add == True) {
		    KeyList        	*ListPtr;

		    ListPtr = (KeyList *)Calloc(sizeof(KeyList));
		    QDDB_KEYLIST_SET_TYPE(ListPtr, CHANGE);
#ifdef SQL
		    ListPtr->Attribute = attr;
		    if (attrnum != -1 && ListPtr->Attribute != attrnum) {
			Free(ListPtr);
			ListPtr = NULL;
			break;
		    } else {
#ifdef INSTANCE_QUALIFICATION
			ListPtr->Instance = Malloc(strlen(inst)+1);
			strcpy(ListPtr->Instance, inst);
#endif
#endif
			ListPtr->Number = atoi(DirEntry->d_name);
			if (NewList == NULL)
			    LastNode = ListPtr;
			ListPtr->next = NewList;
			NewList = ListPtr;
			(*NumberRead)++;
			break;
		    }
		}
                Line = NULL; /* make 1st arg to strtok nil */
	    }
	    Free(inst);
            Line = Buffer;
        }
        fclose(FilePtr);
    }
    closedir(dp);

    /* Search through the additions directory.
     */
    Line = Buffer;
    if ((dp = opendir(AddDir)) == NULL)
        PANIC("Cannot open Additions/ directory");
    while ((DirEntry = readdir(dp)) != NULL) {
	dirlen = NLENGTH(DirEntry);
	bcopy(DirEntry->d_name, dirnam, (size_t)dirlen);
	dirnam[dirlen] = '\0';
	if (dirlen == 0 || dirnam[0] == '.' || dirnam[0] == 'N')
		continue;
        sprintf(TmpDir, "%s%s", AddDir, dirnam);
	AddCount = atoi(dirnam);
        FilePtr = fopen(TmpDir, "r");
        if (FilePtr == NULL)
            PANIC("An existing file in Additions cannot be opened");
	fgets(Line, bufsize, FilePtr); /* get %0 */
        while (fgets(Line, bufsize, FilePtr) != NULL) {
            char        *StringPtr;

	    while (Line[strlen(Line)-2] == '\\' || Line[strlen(Line)-1] != '\n') {
		if (strlen(Line) >= bufsize/2) {
		    bufsize *= 2;
		    Buffer = Line = (char *)Realloc(Buffer, bufsize);
		}
		if (fgets(Line+strlen(Line), bufsize-strlen(Line), FilePtr) == NULL)
		    break;
	    }
	    StringPtr = Line;
	    while (*StringPtr != '\0') {
		if (_isupper(*StringPtr))
		    *StringPtr = _tolower(*StringPtr);
		StringPtr++;
	    }
	    Token = strtok(Line, " "); /* Skip id */
	    Line = Line+strlen(Token)+1;
	    if (Qddb_TranslateExternalID(schema, Token, &attr, &inst) != 0) {
		fprintf(stderr, "Warning: attribute %s malformed, skipping..\n", Token);
		continue;
	    }
            /* Take care of punctuation */
            while ((Token=strtok(Line, schema->Entries[attr].Separators)) != NULL) {
		Boolean		Add;

		Add = False;
		switch(Arg->Type) {
#ifdef V8_REGEXP
		case REGEXP_V8:
		    /* Regexp or strcmp() */
		    if (Arg->Reg == NULL ||
			(regexec(Arg->Reg, Token) != 1 ||
			 Token != Arg->Reg->startp[0] ||
			 (Token+strlen(Token)) != Arg->Reg->endp[0]))
			break;
		    Add = True;
		    break;
#endif
		case ALPHA: 
		    if (strcmp(Token, Arg->Alpha) != 0)
			break;
		    Add = True;
		    break;
	        case NUMERIC:
		    if (strtod(Token, NULL) != Arg->Number)
			break;
		    Add = True;
		    break;
		case NUMERICRANGE: {
		    double 	tmpd;
		    
		    tmpd = strtod(Token, NULL);
		    if (tmpd >= Arg->LowerNum && tmpd <= Arg->UpperNum)
			Add = True;
		    break;
		}
		case ALPHARANGE:
		    if (strcmp(Arg->LowerStr, Token) <= 0 && strcmp(Arg->UpperStr, Token) >= 0)
			Add = True;
		    break;
		default:
		    PANIC("SecondarySearch: bad Arg->Type");
		}
		if (Add == True) {
                    KeyList        *ListPtr;

                    ListPtr = (KeyList *)Calloc(sizeof(KeyList));
                    QDDB_KEYLIST_SET_TYPE(ListPtr, ADDITION);
#ifdef SQL
		    ListPtr->Attribute = attr;
		    if (attrnum != -1 && ListPtr->Attribute != attrnum) {
			Free(ListPtr);
			ListPtr = NULL;
			break;
		    } else {
#ifdef INSTANCE_QUALIFICATION
			ListPtr->Instance = Malloc(strlen(inst)+1);
			strcpy(ListPtr->Instance, inst);
#endif
#endif
			ListPtr->Number = AddCount;
			if (NewList == NULL)
			    LastNode = ListPtr;
			ListPtr->next = NewList;
			NewList = ListPtr;
			(*NumberRead)++;
			break;
		    }
                }
                Line = NULL; /* make 1st arg to strtok nil */
            }
	    Free(inst);
            Line = Buffer;
        }
        fclose(FilePtr);
    }
    closedir(dp);
    if (LastNode != NULL)
        LastNode->next = ExistingList;
    else
        NewList = ExistingList;
    Free(Buffer);
    return NewList;
}

static int RecurseSchema(Entries, start_entry, last_entry, List, start_list, last_list)
    SchemaNode			*Entries;
    int				start_entry, last_entry;
    char			**List;
    int				start_list, last_list;
{
    int				i;

    if (start_list > last_list)
	return -1;
    for (i = start_entry; i <= last_entry; i++) {
	if (Entries[i].Level == start_list && strcmp(Entries[i].Name, List[start_list]) == 0) {
	    int			retval;
	    if (start_list == last_list) /* && Entries[i].IsLeaf != False) */
		return i;
	    retval = RecurseSchema(Entries, i, last_entry, List, start_list+1, last_list);
	    if (retval != -1)
		return retval;
	}
    }
    return -1;
}

int Qddb_ConvertAttributeNameToNumber(schema, attr)
    Schema			*schema;
    char			*attr;
{
    char			**attr_list, *nattr;
    size_t			attr_len;
    int				i, attr_top, at;

    attr_len = strlen(attr);
    nattr = Malloc(attr_len+1);
    strcpy(nattr, attr);
    attr_list = (char **)Malloc(sizeof(char *)*(attr_len+1));
    if (attr_list == NULL) {
	Free(nattr);
	return -1;
    }
    attr_list[1] = nattr;
    attr_top = 2;
    for (i = 1; i < attr_len; i++) {
	if (nattr[i] == '.') {
	    nattr[i] = '\0';
	    attr_list[attr_top] = nattr+i+1;
	    attr_top++;
	}
    }
    attr_top--;
    at = RecurseSchema(schema->Entries, 1, schema->NumberOfAttributes, attr_list, 1, attr_top);
    Free(nattr);
    Free(attr_list);
    return at;
}

char *Qddb_ConvertAttributeNumberToName(schema, num)
    Schema			*schema;
    int				num;
{
    int				i, current_level;
    int				number_entries = schema->NumberOfAttributes;
    SchemaNode			*entries = schema->Entries;
    char			buf[BUFSIZ*2], *retval;

    /* Convert the attribute number to a fully qualified number */
    current_level = 1;
    buf[0] = '\0';
    for (i = 1; i <= number_entries && current_level < entries[num].Level; i++)
	if (entries[i].Level == current_level &&
	    entries[i].Number == entries[num].AncestorNumber[current_level]) {
	    strcat(buf, entries[i].Name);
	    strcat(buf, ".");
	    current_level++;
	}
    if (current_level != entries[num].Level)
	return (char *)NULL;
    strcat(buf, entries[num].Name);
    retval = (char *)Malloc(sizeof(char)*(strlen(buf)+1));
    strcpy(retval, buf);
    return retval;
}

