
/* Lib/LibQddb/Memory.c -- Miscellaneous memory-handling functions.
 *
 * Copyright (C) 1993, 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"

#if defined(QDDB_MEM_DEBUG)
/* Qddb memory checking routines
 */
#define QDDB_MEM_DEBUG_MAX 200000
static void *mem_debug_array[QDDB_MEM_DEBUG_MAX];
static int  mem_debug_num = 0;

static void memdebug_addval(arg)
    void		*arg;
{
    if (arg == NULL)
	abort();
    if (mem_debug_num > QDDB_MEM_DEBUG_MAX)
	abort();
    mem_debug_array[mem_debug_num++] = arg;
}

static void memdebug_check(arg)
    void		*arg;
{
    int			i;

    if (arg == NULL)
	abort();
    for (i = 0; i < mem_debug_num; i++)
	if (arg == mem_debug_array[i])
	    break;
    if (i == mem_debug_num)
	abort();
}

static void memdebug_delval(arg)
    void		*arg;
{
    int			i;

    if (arg == NULL)
	abort();
    for (i = 0; i < mem_debug_num; i++)
	if (arg == mem_debug_array[i]) {
	    mem_debug_array[i] = NULL;
	    break;
	}
    if (i == mem_debug_num)
	abort();
}
#endif

static void FreeKeyList _ANSI_ARGS_((KeyList *));
static void FreeEntry _ANSI_ARGS_((Entry));
static void FreeRowList _ANSI_ARGS_((RowList *));
static void FreeSingleRow _ANSI_ARGS_((RowList *));
static void FreeInCoreEntry _ANSI_ARGS_((InCoreEntry *));
static void FreeSchema _ANSI_ARGS_((Schema *));
static void FreeDataTree _ANSI_ARGS_((DataTree **));
static void FreeSchemaTree _ANSI_ARGS_((SchemaTreeNode **));


#if !defined(USE_GNU_MALLOC)
size_t		qddb_malloc_bytes_used = 0;
#endif


#if defined(USE_MALLOC_FUNCTIONS)
char *Malloc(Size)
    size_t			Size;
{
    char			*ReturnValue;

    if (Size < QDDB_MIN_MALLOC_SIZE) {
#if !defined(USE_GNU_MALLOC)
	qddb_malloc_bytes_used += QDDB_MIN_MALLOC_SIZE;
#endif
	Size = QDDB_MIN_MALLOC_SIZE;
    }
    if ((ReturnValue = malloc(Size)) == NULL) {
    	char 		buf[BUFSIZ];

    	sprintf(buf, "Malloc(): core size == %u + %u", 
    		(unsigned int)sbrk(0), (unsigned int)Size);
    	perror(buf);
	PANIC("Malloc");
    }
#if !defined(USE_GNU_MALLOC)
    qddb_malloc_bytes_used += Size;
#endif
#if defined(MALLOC_DEBUG)
    fprintf(stderr, "Malloc: (size %d) %x\n", Size, ReturnValue);
#endif
#if defined(QDDB_MEM_DEBUG)
    memdebug_addval(ReturnValue);
#endif
    return ReturnValue;
}


char *Calloc(Size)
    size_t		Size;
{
    char		*ReturnValue; 

    if (Size <= QDDB_MIN_MALLOC_SIZE) {
#if !defined(USE_GNU_MALLOC)
	qddb_malloc_bytes_used += QDDB_MIN_MALLOC_SIZE;
#endif
	return calloc((size_t)QDDB_MIN_MALLOC_SIZE, (size_t)1);
    }
    if ((ReturnValue = calloc(Size, (size_t)1)) == NULL) {
    	char 		buf[BUFSIZ];

    	sprintf(buf, "Calloc(): core size == %u + %u", 
    		(unsigned int)sbrk(0), (unsigned int)Size);
    	perror(buf);
	PANIC("Calloc");
    }
#if !defined(USE_GNU_MALLOC)
    qddb_malloc_bytes_used += Size;
#endif
#if defined(MALLOC_DEBUG)
    fprintf(stderr, "Malloc: (size %d) %x\n", Size, ReturnValue);
#endif
#if defined(QDDB_MEM_DEBUG)
    memdebug_addval(ReturnValue);
#endif
    return ReturnValue;
}

char *Realloc(OldPtr, NewSize)
    void       		*OldPtr;
    size_t		NewSize;
{
    char		*ReturnValue;

#if defined(DIAGNOSTIC)
    if (NewSize == 0) {
	fprintf(stderr, "reallocing zero length\n");
	fflush(stderr);
	abort();
    }
#endif
    if (NewSize < QDDB_MIN_MALLOC_SIZE)
	NewSize = QDDB_MIN_MALLOC_SIZE;
    if (OldPtr == NULL) {
	if ((ReturnValue = malloc(NewSize)) == NULL) {
	    PANIC("Realloc");
	}
    } else if ((ReturnValue = realloc(OldPtr, NewSize)) == NULL) {
    	perror("Realloc():");
    	if (Malloc(NewSize) != NULL)
    		printf("Malloc worked though!\n");
	PANIC("Realloc");
    }
#if !defined(USE_GNU_MALLOC)
    qddb_malloc_bytes_used += NewSize;
#endif
#if defined(MALLOC_DEBUG)
    if (ReturnValue != OldPtr) {
	fprintf(stderr, "Malloc: (Size %d) %x\n", NewSize, ReturnValue);
	fprintf(stderr, "Free: %x\n", OldPtr);
    }
#endif
#if defined(QDDB_MEM_DEBUG)
    if (ReturnValue != OldPtr && OldPtr != NULL)
	memdebug_delval(OldPtr);
    memdebug_addval(ReturnValue);
#endif
    return ReturnValue;
}

void Free(ptr)
    void		*ptr;
{
#if defined(MALLOC_DEBUG)
    fprintf(stderr, "Free: %x\n", ptr);
#endif
#if defined(QDDB_MEM_DEBUG)
    memdebug_delval(ptr);
#endif
    free(ptr);
}

#endif /* USE_MALLOC_FUNCTIONS */

void Qddb_Free(type, ptr)
    int			type;
    void		*ptr;
{
    if (ptr == NULL)
	return;
    switch (type) {
    case QDDB_TYPE_GENERIC:
	Free((void *)ptr);
	break;
    case QDDB_TYPE_ENTRY:
	FreeEntry((Entry)ptr);
	break;
    case QDDB_TYPE_INCOREENTRY:
	FreeInCoreEntry((InCoreEntry *)ptr);
	break;
    case QDDB_TYPE_KEYLIST:
	FreeKeyList((KeyList *)ptr);
	break;
    case QDDB_TYPE_ROWLIST:
	FreeRowList((RowList *)ptr);
	break;
    case QDDB_TYPE_DATATREE:
	FreeDataTree((DataTree **)ptr);
	break;
    case QDDB_TYPE_SCHEMA:
	FreeSchema((Schema *)ptr);
	break;
    case QDDB_TYPE_SCHEMATREE:
	FreeSchemaTree((SchemaTreeNode **)ptr);
	break;
    default:
	printf("DEFAULT FREE\n");
	Free(ptr);	
    }

}

static void FreeSchema(schema)
    Schema		*schema;
{
    int			i;
    size_t		num;
    SchemaNode		*nodes;

#if defined(DIAGNOSTIC)
    if (schema == NULL)
	return;
#endif
    Free(schema->RelationName);
    if (schema->HashTable != NULL)
	Free(schema->HashTable);
    if (schema->ReducedAttributeTable != NULL) {
	Free(schema->ReducedAttributeTable[0]);
	Free(schema->ReducedAttributeTable);
    }
    num = schema->NumberOfAttributes;
    nodes = schema->Entries;
    for (i = 1; i <= num; i++) {
	if (nodes[i].Name != NULL)
	    Free(nodes[i].Name);
	if (nodes[i].VerboseName != NULL)
	    Free(nodes[i].VerboseName);
	if (nodes[i].Alias != NULL)
	    Free(nodes[i].Alias);
	if (nodes[i].Format != NULL)
	    Free(nodes[i].Format);
	if (nodes[i].Separators != NULL)
	    Free(nodes[i].Separators);
    }
    Free(nodes);
    FreeSchemaTree(schema->Tree);
    Free(schema->default_date_format);
    Free(schema);
}

static void FreeSchemaTree(tree)
    SchemaTreeNode	**tree;
{
    int			i;

#if defined(DIAGNOSTIC)
    if (tree == NULL)
	return;
#endif
    for (i = 0; tree[i] != NULL; i++) {
	FreeSchemaTree(tree[i]->children);
	Free(tree[i]);
    }
    Free(tree);
}

static void FreeDataTree(tree)
    DataTree		**tree;
{
    DataTree		**i, *j;

    i = tree;
    while (*i != NULL) {
	for (j = *i; j->datatree_type != DATATREE_END; j++) {
	    switch (j->datatree_type) {
	    case DATATREE_CHILDREN:
		if (j->datatree_children != NULL)
		    FreeDataTree(j->datatree_children);
		break;
	    case DATATREE_DATE:
		if (j->datatree_date != NULL)
		    Free(j->datatree_date);
		break;
	    case DATATREE_STRING:
		if (j->datatree_string != NULL)
		    Free(j->datatree_string);
		break;
	    case DATATREE_NOVAL:
		if (j->datatree_string != NULL)
		    Free(j->datatree_string);
		break;
	    default:
		;
	    }
	}
	Free(*i);
	i++;
    }
    Free(tree);
}

static void FreeKeyList(list)
    KeyList		*list;
{
    KeyList		*tmp;

    while (list != NULL) {
	tmp = list;
	list = list->next;
	Free(tmp->Instance);
	Free(tmp);
    }
}

static void FreeEntry(ThisEntry)
    Entry		ThisEntry;
{
    int			i;

    for (i = 0; ThisEntry[i] != NULL; i++)
	Free(ThisEntry[i]);
    Free(ThisEntry);
}

static void FreeSingleRow(Row)
    RowList		*Row;
{
    RowList		*r;

    while (Row != NULL) {
	r = Row;
	Row = Row->next;
	Free(r);
    }
}

static void FreeRowList(Row)
    RowList		*Row;
{
    RowList		*r;

    while (Row != NULL) {
	r = Row;
	Row = Row->down;
	FreeSingleRow(r);
    }
}

static void FreeInCoreEntry(entry)
    InCoreEntry		*entry;
{
    InCoreEntry		*tmp;

    if (entry == NULL)
	return;
    tmp = entry;
    while (tmp->SequenceNumber != 0) {
	Free(tmp->Data);
	tmp++;
    }
    Free(entry);
}
