
/* qddb/Lib/LibQddb/Index.c
 *
 * Copyright (C) 1993, 1994, 1995 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"

/* EXPORTED:
 * 	int Qddb_HashValue(char *)
 *	void Qddb_MakeIndex(char *)
 */

#ifdef SQL
static size_t		NumberOfKeys;
static KeyIndexIn	*KeyIndex;
static size_t		KeyIndexTop;

static size_t		NumberOfNumericKeys;
static NumericIndexIn	*NumericIndex;
static size_t	 	NumericIndexTop;
#endif

static KeyNode *HashEntryToKeyNodes _ANSI_ARGS_((Schema *, FILE *, Hash *));
static void PurgeHashList _ANSI_ARGS_((Bucket *, size_t));
static void PurgeHashBucket _ANSI_ARGS_((KeyNode *));
static void PurgePosNode _ANSI_ARGS_((PosNode *));
static KeyNode *IndexParse _ANSI_ARGS_((Schema *, Entry, int *));
static void AddToHashList _ANSI_ARGS_((Bucket *, KeyNode *, off_t, size_t));
static void WriteHashList _ANSI_ARGS_((Schema *, FILE *, FILE *, Bucket *, Boolean, size_t, size_t));
static void WriteKeyIndex _ANSI_ARGS_((FILE *, KeyIndexIn *));
static void SortKeyIndex _ANSI_ARGS_((size_t, size_t));
static void WriteNumericIndex _ANSI_ARGS_((FILE *, NumericIndexIn *));
static void SortNumericIndex _ANSI_ARGS_((size_t, size_t));

int Qddb_HashValue(schema, string)
    Schema		*schema;
    char		*string;
{
    int		RetVal, len;

    len = strlen(string);
    if (len > 0)
	RetVal = (int)string[0]*17576;
    else
	return 0;
    if (len > 1)
    	RetVal += (int)string[1]*676;
    else
    	return RetVal % schema->HashSize;
    if (len > 2)
    	RetVal += (int)string[2]*26;
    else 
    	return RetVal % schema->HashSize;
    if (len > 3)
    	RetVal += (int)string[3];
    return RetVal % schema->HashSize;
}

void Qddb_MakeIndex(schema)
    Schema		*schema;
{
    char		RelationFN[MAXFILENAMELEN];
    char		IndexFN[MAXFILENAMELEN];
    char		KeyFN[MAXFILENAMELEN], HashFN[MAXFILENAMELEN];
    int			IndexFile, DatabaseFile, KeyFile;
    FILE		*KeyFilePtr, *IndexFilePtr, *fdopen();
    FILE		**IndexFilePtrs;
#ifdef SQL
    char		KeyIndexFN[MAXFILENAMELEN], NumericIndexFN[MAXFILENAMELEN];
    int			KeyIndexFile, NumericIndexFile;
    FILE		*KeyIndexFilePtr, *NumericIndexFilePtr;
#endif
    FILE		**HashFilePtrs;
    FILE		*HashFilePtr;
    Bucket		*HashList;
    Entry		ThisEntry = NULL;
    int			Start, Length;
    int			i, j, Error;
    KeyNode		*List;
#if defined(MAXIMUM_MEMORY_USAGE)
    int			idxnum = 0;
#endif
    size_t		LastHash;
#if defined(USE_MALLOC_STATS)
#if defined(QDDB_STAB_STATS)
    int			recnum = 0;
#endif
#endif
#if defined(USE_GNU_MALLOC)
    struct mstats 	m;
    size_t		starting_bytes;
#endif

#ifdef SQL
    NumberOfKeys = 0;
    NumberOfNumericKeys = 0;
    NumericIndexTop = 0;
#endif
    KeyIndex = NULL;
    KeyIndexTop = 0;
    strcpy(RelationFN, schema->RelationName);
    strcpy(IndexFN, RelationFN);
    strcat(IndexFN, "/Index");
    strcpy(HashFN, RelationFN);
    strcat(HashFN, "/HashTable");
    strcpy(KeyFN, RelationFN);
    strcat(KeyFN, "/Database.key");
#ifdef SQL
    strcpy(KeyIndexFN, RelationFN);
    strcat(KeyIndexFN, "/KeyIndex");
    strcpy(NumericIndexFN, RelationFN);
    strcat(NumericIndexFN, "/NumericIndex");
#endif
    IndexFile = Open(IndexFN, O_WRONLY|O_CREAT|O_TRUNC, 0666);
    IndexFilePtr = fdopen(IndexFile, "w");
#ifdef SQL
    KeyIndexFile = Open(KeyIndexFN, O_RDWR|O_CREAT|O_TRUNC, 0666);
    KeyIndexFilePtr = fdopen(KeyIndexFile, "w");
    NumericIndexFile = Open(NumericIndexFN, O_RDWR|O_CREAT|O_TRUNC, 0666);
    NumericIndexFilePtr = fdopen(NumericIndexFile, "w");
#endif
    KeyFile   = Open(KeyFN, O_RDONLY, 0);
    KeyFilePtr = fdopen(KeyFile, "r");
    DatabaseFile = schema->database_fd;
    lseek(DatabaseFile, (off_t)0, 0);
    HashList = (Bucket *)Malloc(schema->HashSize*sizeof(Bucket));
#if defined(USE_GNU_MALLOC)
    m = mstats();
    starting_bytes = m.bytes_used;
#else
    qddb_malloc_bytes_used = 0;
#endif
    lseek(schema->hashtable_fd, (off_t)0, 0);
#if HAVE_FTRUNCATE
    ftruncate(schema->hashtable_fd, (off_t)0);
#else
    close(open(HashFN, O_RDWR|O_TRUNC));
#endif
    HashFilePtr = fdopen(schema->hashtable_fd, "w");
    if (HashFilePtr == NULL) {
    	fprintf(stderr, "fdopen of %s/HashTable failed\n", schema->RelationName);
    	exit(1);
    }
    for (i = 0; i < schema->HashSize; i++)
    	HashList[i] = NULL;
    while (fscanf(KeyFilePtr, "%d %d\n", &Start, &Length) != EOF) {
#if defined(USE_MALLOC_STATS)
	if (++recnum % 100 == 0) {
#if defined(QDDB_STAB_STATS)
	    printf("%d Records Processed\n", recnum);
#endif
	    m = mstats();
	    printf("Malloc stats: bytes_total(%u) chunks_used(%u) bytes_used(%u) \
chunks_free(%u) bytes_free(%u)\n", 
		   m.bytes_total, m.chunks_used, m.bytes_used, m.chunks_free, m.bytes_free);
	    fflush(stdout);
	}
#endif
#if defined(MAXIMUM_MEMORY_USAGE)
#if defined(USE_GNU_MALLOC)
	m = mstats();
	if (m.bytes_used > (size_t)(starting_bytes+MAXIMUM_MEMORY_USAGE*1024)) {
#else
	if (qddb_malloc_bytes_used > (size_t)(MAXIMUM_MEMORY_USAGE*1024)) {
#endif
	    FILE		*NewIndexFilePtr, *NewHashFilePtr;
	    char		buf[BUFSIZ];

	    sprintf(buf, "%s-%d", IndexFN, idxnum);
	    if ((NewIndexFilePtr = fopen(buf, "w")) == NULL)
		PANIC(buf);
	    sprintf(buf, "%s-%d", HashFN, idxnum);
	    if ((NewHashFilePtr = fopen(buf, "w")) == NULL)
		PANIC(buf);
	    idxnum++;
	    WriteHashList(schema, NewIndexFilePtr, NewHashFilePtr, 
			  HashList, False, (size_t)0, (size_t)schema->HashSize);
	    PurgeHashList(HashList, schema->HashSize);
#if !defined(USE_GNU_MALLOC)
	    qddb_malloc_bytes_used = 0;
#endif
	    fclose(NewIndexFilePtr);
	    fclose(NewHashFilePtr);
#if defined(DIAGNOSTIC) && defined(USE_GNU_MALLOC) && defined(USE_MALLOC_STATS)
	    m = mstats();
	    if (m.bytes_used != starting_bytes)
		fprintf(stdout, "After purge, starting bytes (%d) != bytes used (%d)\n",
			(int)starting_bytes, (int)m.bytes_used);
#endif
	}
#endif
	Qddb_ReadEntry(DatabaseFile, &ThisEntry, (off_t)Start, (size_t)Length, False);
    	List = IndexParse(schema, ThisEntry+1, &Error);
	if (Error) {
	    fprintf(stdout, "Cannot index the following tuple:\n%s\n", ThisEntry[0]);
	    fflush(stdout);
	    exit(1);
	}
    	while (List != NULL) {
	    int		Value;
	    KeyNode	*next;

	    Value = Qddb_HashValue(schema, List->s);
	    next = List->next;
	    AddToHashList(&HashList[Value], List, (off_t)Start, (size_t)Length); /* Frees up List */
	    List = next;
    	}
    }
    if (ThisEntry != NULL)
	Qddb_Free(QDDB_TYPE_ENTRY, ThisEntry);
#ifdef SQL
#if defined(DIAGNOSTIC)
    fprintf(stdout, "Allocating %d keys\n", (int)NumberOfKeys);
    if (NumberOfKeys == 0)
	return;
#endif
    KeyIndex = (KeyIndexIn *)Malloc(sizeof(KeyIndexIn)*NumberOfKeys);
    /* This wastes some space, but the computation necessary to correct it
     *	is large.  Trade space for time.
     */
    NumericIndex = (NumericIndexIn *)Malloc(sizeof(NumericIndexIn)*NumberOfKeys);
#endif
#if defined(MAXIMUM_MEMORY_USAGE)
    if (idxnum == 0) {
#endif
	WriteHashList(schema, IndexFilePtr, HashFilePtr, HashList, 
		      True, (size_t)0, (size_t)schema->HashSize);
#ifdef SQL
	WriteKeyIndex(KeyIndexFilePtr, KeyIndex);
	WriteNumericIndex(NumericIndexFilePtr, NumericIndex);
#endif
#if defined(MAXIMUM_MEMORY_USAGE)
    } else {
	FILE		*NewIndexFilePtr, *NewHashFilePtr;
	char		buf[BUFSIZ];

	sprintf(buf, "%s-%d", IndexFN, idxnum);
	if ((NewIndexFilePtr = fopen(buf, "w")) == NULL) {
	    fprintf(stderr, "panic on write:");
	    PANIC(buf);
	}
	sprintf(buf, "%s-%d", HashFN, idxnum);
	if ((NewHashFilePtr = fopen(buf, "w")) == NULL) {
	    fprintf(stderr, "panic on write:");
	    PANIC(buf);
	}
	idxnum++;
	WriteHashList(schema, NewIndexFilePtr, NewHashFilePtr, HashList, 
		      False, (size_t)0, (size_t)schema->HashSize);
	PurgeHashList(HashList, schema->HashSize);
	fclose(NewIndexFilePtr);
	fclose(NewHashFilePtr);
	/* Merge all (Index-*,HashTable-*) into single (Index,HashTable)
	 *
	 * Open all HashTable-* and Index-*
	 * foreach HashEntry across all HashTable-*
	 *     Read the HashEntries
	 *     Read the indices they point to
	 *     Parse indices into KeyNodes
	 *     Add them to the HashList
	 *     if too much memory is used then
	 *         Write the (Index,HashTable), saving Key/Numeric info.
	 *         Purge HashList
	 *     endif
	 * endforeach
	 * Write the (Index,HashTable), saving Key/Numeric info.
	 * Purge HashList
	 * Close all HashTable-* and Index-*
	 * Remove old (Index-*,HashTable-*)
	 */
	HashFilePtrs = (FILE **)Malloc(sizeof(FILE *)*idxnum);
	IndexFilePtrs = (FILE **)Malloc(sizeof(FILE *)*idxnum);
	for (i = 0; i < idxnum; i++) {
	    char		buf[BUFSIZ];
	    
	    sprintf(buf, "%s-%d", HashFN, i);
	    if ((HashFilePtrs[i] = fopen(buf, "r")) == NULL) {
		fprintf(stderr, "panic on read:");
		PANIC(buf);
	    }
	    sprintf(buf, "%s-%d", IndexFN, i);
	    if ((IndexFilePtrs[i] = fopen(buf, "r")) == NULL) {
		fprintf(stderr, "panic on read:");
		PANIC(buf);
	    }
	}
	LastHash = 0;
	for (i = 0; i < schema->HashSize; i++) { /* each HashEntry */
	    for (j = 0; j < idxnum; j++) { /* each (HashTable-*,Index-*) */
		Hash			hash;
		KeyNode			*key;
		char			*ch, *ch1, buf[BUFSIZ];

		/* 1. Read HashEntry
		 * 2. Read Index pointed to by HashEntry
		 * 3. Parse Index into KeyNodes
		 * 4. Add them to HashList
		 */
		fgets(buf, sizeof(buf), HashFilePtrs[j]);
		hash.num = (size_t)strtoul(buf, &ch, 10);
		hash.index = (size_t)strtoul(ch, &ch1, 10);
		hash.pos = (off_t)strtoul(ch1, &ch, 10);
		hash.len = (size_t)strtoul(ch, &ch1, 10);
		key = HashEntryToKeyNodes(schema, IndexFilePtrs[j], &hash);
		while (key != NULL) {
		    KeyNode	*next;
		    PosNode	*posnode;

		    posnode = key->pos;
#if defined(DIAGNOSTIC)
		    if (posnode->next != NULL)
			PANIC("MakeIndex: posnode");
#endif
		    key->pos = NULL;
		    next = key->next;
		    AddToHashList(&HashList[i], key, posnode->Start, posnode->Length); /* Frees up key */
		    Free(posnode);
		    key = next;
		}
	    }
#if defined(USE_GNU_MALLOC)
	    m = mstats();
#if defined(USE_MALLOC_STATS)
	    if (i % 100 == 0) {
		printf("Hash #%d: bytes_total(%u) chunks_used(%u) bytes_used(%u) \
chunks_free(%u) bytes_free(%u)\n", i,
		   m.bytes_total, m.chunks_used, m.bytes_used, m.chunks_free, m.bytes_free);
		fflush(stdout);
	    }
#endif
	    if (m.bytes_used > (size_t)(starting_bytes+MAXIMUM_MEMORY_USAGE*1024)) {
#else
	    if (qddb_malloc_bytes_used > (size_t)(MAXIMUM_MEMORY_USAGE*1024)) {
#endif
		WriteHashList(schema, IndexFilePtr, HashFilePtr, HashList, 
			      True, LastHash, (size_t)i+1);
		LastHash = i+1;
		PurgeHashList(HashList, schema->HashSize);
#if !defined(USE_GNU_MALLOC)
		qddb_malloc_bytes_used = 0;
#endif
	    }
	}
	for (i = 0; i < idxnum; i++) {
	    char		buf[BUFSIZ];
	    
	    fclose(HashFilePtrs[i]);
	    fclose(IndexFilePtrs[i]);
	    sprintf(buf, "%s-%d", HashFN, i);
	    if (unlink(buf) == -1) {
		fprintf(stderr, "Warning: failure to unlink %s\n", buf);
	    }
	    sprintf(buf, "%s-%d", IndexFN, i);
	    if (unlink(buf) == -1) {
		fprintf(stderr, "Warning: failure to unlink %s\n", buf);
	    }
	}
	Free(HashFilePtrs);
	Free(IndexFilePtrs);
	WriteHashList(schema, IndexFilePtr, HashFilePtr, HashList, 
		      True, LastHash, (size_t)schema->HashSize);
	PurgeHashList(HashList, schema->HashSize);
#ifdef SQL
	WriteKeyIndex(KeyIndexFilePtr, KeyIndex);
	WriteNumericIndex(NumericIndexFilePtr, NumericIndex);
#endif
    }
#endif /* MAXIMUM_MEMORY_USAGE */
}

static void PurgePosNode(List)
    PosNode		*List;
{
    PosNode		*newlist;

    while (List != NULL) {
	newlist = List->next;
	if (List->inst != NULL)
	    Free(List->inst);
	Free(List);
	List = newlist;
    }
}

static void PurgeHashBucket(List) 
    KeyNode		*List;
{
    KeyNode		*newlist;

    while (List != NULL) {
	newlist = List->next;
	PurgePosNode(List->pos);
	if (List->s != NULL)
	    Free(List->s);
#if defined(notdef)
	if (List->inst != NULL)
	    Free(List->inst);
#endif
	Free(List);
	List = newlist;
    }
}

static void PurgeHashList(List, Size)
    Bucket		*List;
    size_t		Size;
{
    int			i;

    for (i = 0; i < Size; i++) {
	PurgeHashBucket(List[i]);
	List[i] = NULL;
    }
}

static KeyNode *IndexParse(schema, ThisEntry, ErrorPtr)
    Schema		*schema;
    Entry		ThisEntry;
    int			*ErrorPtr;
{
    int			len;
    char		*inst, *token, *nextspot, *ch, *full_attr, *tmp;
    size_t		attr;
    KeyNode		*ptr, *List;

    List = NULL;
    *ErrorPtr = 0;
    while (*ThisEntry != NULL) {
	token = strtok(*ThisEntry, " "); /* Skip id */
#if defined(DIAGNOSTIC)
	if (token == NULL) {
	    PANIC("IndexParse: token is NULL\n");
	}
#endif
	nextspot = *ThisEntry + strlen(token) + 1;
	if (schema->UseReducedAttributeIdentifiers == True) {
	    inst = Malloc(strlen(token)+1);
	    strcpy(inst, token+1);
	    attr = Qddb_ReducedAttrToIndex(inst);
	    full_attr = Malloc(strlen(schema->ReducedAttributeTable[attr])+2);
	    full_attr[0] = '%', full_attr[1] = '\0';
	    strcat(full_attr, schema->ReducedAttributeTable[attr]);
	    if (Qddb_TranslateExternalID(schema, full_attr, &attr, &tmp) != 0) {
		fprintf(stderr, "Warning: attribute %s malformed, skipping attr %d..\n", 
			schema->ReducedAttributeTable[attr], attr);
		Free(inst);
		Free(full_attr);
		ThisEntry++;
		continue;
	    }
	    Free(full_attr);
	    Free(tmp);
	} else if (Qddb_TranslateExternalID(schema, token, &attr, &inst) != 0) {
	    fprintf(stderr, "Warning: attribute %s malformed, skipping..\n", token);
	    ThisEntry++;
	    continue;
	}
	if (schema->Entries[attr].Exclude == True) {
	    ThisEntry++;
	    Free(inst);
	    continue;
	}
	while ((token=strtok(nextspot, schema->Entries[attr].Separators)) != NULL) {
	    nextspot = NULL;
	    len = 0;
	    ch = token;
	    while (*ch) {
		if (_isupper(*ch))
		    *ch = _tolower(*ch);
		ch++;
		len++;
	    }
	    ptr = List;
	    List = (KeyNode *)Malloc(sizeof(KeyNode));
	    List->next = ptr;
	    List->s = Malloc((size_t)len+1);
	    strcpy(List->s, token);
	    List->pos = NULL;
	    List->len = len;
#ifdef SQL
	    List->attr = attr;
#ifdef INSTANCE_QUALIFICATION
	    List->inst = (char *)Malloc(strlen(inst)+1);
	    strcpy(List->inst, inst);
#endif
#endif
	}
	ThisEntry++;
	Free(inst);
    }
    return List;
}

static void AddToHashList(List, Node, Start, Length)
    Bucket		*List;
    KeyNode		*Node;
    off_t		Start;
    size_t		Length;
{
    KeyNode		*ptr;

    if (*List == NULL) {
#ifdef SQL
	NumberOfKeys++;
#endif
	*List = (KeyNode *)Malloc(sizeof(KeyNode));
	**List = *Node;
	(*List)->inst = NULL;
	(*List)->next = NULL;
	(*List)->pos  = (PosNode *)Malloc(sizeof(PosNode));
	(*List)->pos->Start = Start;
	(*List)->pos->Length = Length;
	(*List)->pos->next = NULL;
#ifdef SQL
	(*List)->pos->attr = Node->attr;
#ifdef INSTANCE_QUALIFICATION
	(*List)->pos->inst = Node->inst;
#endif
#endif
	Free(Node);
	return;
    }
    ptr = *List;
    while (ptr != NULL) {
	PosNode		*PosPtr;
	
	if (ptr->len == Node->len && strcmp(ptr->s, Node->s) == 0) {
	    PosPtr = (PosNode *)Malloc(sizeof(PosNode));
	    PosPtr->Start = Start;
	    PosPtr->Length = Length;
#ifdef SQL
	    PosPtr->attr = Node->attr;
#ifdef INSTANCE_QUALIFICATION
	    PosPtr->inst = Node->inst;
#endif
#endif
	    PosPtr->next = ptr->pos;
	    ptr->pos = PosPtr;
	    /* Free up the word and node. */
	    Free(Node->s);
	    Free(Node);
	    return;
	}
	ptr = ptr->next;
    }
#ifdef SQL
    NumberOfKeys++;
#endif
    ptr = *List;
    *List = (KeyNode *)Malloc(sizeof(KeyNode));
    **List = *Node;
    (*List)->inst = NULL;
    (*List)->next = ptr;
    (*List)->pos  = (PosNode *)Malloc(sizeof(PosNode));
    (*List)->pos->Start = Start;
    (*List)->pos->Length = Length;
#ifdef SQL
    (*List)->pos->attr = Node->attr;
#ifdef INSTANCE_QUALIFICATION
    (*List)->pos->inst = Node->inst;
#endif
#endif
    (*List)->pos->next = NULL;
    Free(Node);
}

static void WriteHashList(schema, IndexFilePtr, HashFilePtr, HashList, PassTwo, Begin, End)
    Schema		*schema;
    FILE		*IndexFilePtr;
    FILE		*HashFilePtr;
    Bucket		*HashList;
    Boolean		PassTwo; /* True iff on the final pass */
    size_t		Begin, End;
{
    KeyNode		*Key;
    int			Start, Length;
    int			i, NumberHashed; 
    Boolean		IsALine = False;
#ifdef SQL
    off_t		WhereAmI, WhereAmINow;
#endif
    
    for (i = Begin; i < End; i++) {
	Key = HashList[i];
	Start = lseek(fileno(IndexFilePtr), (off_t)0, 1);
	if (Start == -1)
	    PANIC("Cannot seek to Start");
	NumberHashed = 0;
	while (Key != NULL) {
	    PosNode		*ptr;
	    char		*outchar;
	    
	    NumberHashed++;
	    IsALine = True;
#ifdef SQL
	    /* Caution: This fflush(3) is needed for the lseek(2) to
	     *	find where we are now.
	     */
	    fflush(IndexFilePtr);
	    /* For KeyIndex:
	     * 	Need to keep each key and its locator (start,length)
	     *	to the Index file.
	     */
	    WhereAmI = lseek(fileno(IndexFilePtr), (off_t)0, 1);
	    if (Start == -1)
		PANIC("SQL: lseek failed");
#endif
	    outchar = Key->s;
	    while (*outchar != '\0') {
		if (*outchar == ' ') {
		    putc('\\', IndexFilePtr);
		    putc(' ', IndexFilePtr);
		} else if (*outchar == '\\') {
		    putc('\\', IndexFilePtr);
		    putc('\\', IndexFilePtr);
		} else {
		    putc(*outchar, IndexFilePtr);
		}
		outchar++;
	    }
	    putc(' ', IndexFilePtr);
	    ptr = Key->pos;
	    while (ptr != NULL) {
#ifdef SQL
#ifdef INSTANCE_QUALIFICATION
		if (schema->UseReducedAttributeIdentifiers == True) {
		    fprintf(IndexFilePtr, "%d %d %s ", (int)ptr->Start,
			    (int)ptr->Length, ptr->inst);
		} else {
		    fprintf(IndexFilePtr, "%d %d %d %s ", (int)ptr->Start,
			    (int)ptr->Length, (int)ptr->attr, ptr->inst);
		}
#else
		fprintf(IndexFilePtr, "%d %d %d ", (int)ptr->Start,
			(int)ptr->Length, (int)ptr->attr);
#endif
#else
 		fprintf(IndexFilePtr, "%d %d ", (int)ptr->Start, (int)ptr->Length);
#endif
		ptr = ptr->next;
	    }
#ifdef SQL
	    /* Caution: This fflush(3) is needed for the lseek(2) to
	     *	find where we are now.
	     */
	    fflush(IndexFilePtr);
	    WhereAmINow = lseek(fileno(IndexFilePtr), (off_t)0, 1);
	    if (PassTwo == True) {
		/* For KeyIndex:
		 *	Need the length now.... WhereAmINow-WhereAmI is a good
		 *	thing.
		 */
		KeyIndex[KeyIndexTop].s = Malloc((size_t)Key->len+1);
		strcpy(KeyIndex[KeyIndexTop].s, Key->s);
		KeyIndex[KeyIndexTop].Start = WhereAmI;
		KeyIndex[KeyIndexTop].Length = WhereAmINow - WhereAmI;
		KeyIndexTop++;
		/* For NumericIndex:
		 *	If the key is numeric, convert to floating point and
		 *	save it away for NumericIndex.
		 */
		if (IsNumericString(Key->s) == True) {
		    NumericIndex[NumericIndexTop].s = strtod(Key->s, NULL);
		    NumericIndex[NumericIndexTop].Start = WhereAmI;
		    NumericIndex[NumericIndexTop].Length = WhereAmINow - WhereAmI;
		    NumericIndexTop++;
		}
	    }
#endif
	    fprintf(IndexFilePtr, "\n");
	    fflush(IndexFilePtr);
	    Key = Key->next;
	}
	if (IsALine == True) {
	    Length = lseek(fileno(IndexFilePtr), (off_t)0, 1); 
	    if (Length == -1)
		PANIC("Cannot seek to Length");
	    Length -= Start;
	    if (Length != 0) {
		if (PassTwo == False || schema->UseCachedHashing == True)
		    fprintf(HashFilePtr, "%10d %10d %10d %10d\n", 
			    NumberHashed, i, Start, Length);
		else
		    fprintf(HashFilePtr, "%d %d %d %d\n", 
			    NumberHashed, i, Start, Length);
	    }
	    IsALine = False;
	} else if (PassTwo == False || schema->UseCachedHashing == True)
	    fprintf(HashFilePtr, "%10d %10d %10d %10d\n", 0, i, 0, 0);
    }
}

static void WriteKeyIndex(KeyIndexFilePtr, KeyIndex)
    FILE		*KeyIndexFilePtr;
    KeyIndexIn		*KeyIndex;
{
    int			i;
    char		*outchar;

    /* Sort the array by name (method: recursive quicksort).
     */
    if (KeyIndexTop > 1)
	SortKeyIndex(1, KeyIndexTop);
    fprintf(KeyIndexFilePtr, "\n");
    for (i = 0; i < KeyIndexTop; i++) {
	outchar = KeyIndex[i].s;
	while (*outchar != '\0') {
	    if (*outchar == ' ') {
		putc('\\', KeyIndexFilePtr);
		putc(' ', KeyIndexFilePtr);
	    } else if (*outchar == '\\') {
		putc('\\', KeyIndexFilePtr);
		putc('\\', KeyIndexFilePtr);
	    } else {
		putc(*outchar, KeyIndexFilePtr);
	    }
	    outchar++;
	}
	putc(' ', KeyIndexFilePtr);
	fprintf(KeyIndexFilePtr, "%d %d\n", (int)KeyIndex[i].Start, (int)KeyIndex[i].Length);
    }
}

static void SortKeyIndex(l, r)
    size_t		l, r;
{
    size_t		i, j;
    KeyIndexIn		x, w;

    i = l; j = r;
    x = KeyIndex[((l+r)/2)-1];
    do {
	while (strcmp(KeyIndex[i-1].s, x.s) < 0)
	    i++;
	while (strcmp(x.s, KeyIndex[j-1].s) < 0)
	    j--;
	if (i <= j) {
	    w = KeyIndex[i-1];
	    KeyIndex[i-1] = KeyIndex[j-1];
	    KeyIndex[j-1] = w;
	    i++, j--;
	}
    } while (j >= i);
    if (l < j)
	SortKeyIndex(l, j);
    if (i < r)
	SortKeyIndex(i, r);
}


static void WriteNumericIndex(NumericIndexFilePtr, NumericIndex)
    FILE		*NumericIndexFilePtr;
    NumericIndexIn	*NumericIndex;
{
    int			i;
    void		SortNumericIndex();

    /* Sort the array by numeric key (method: recursive quicksort).
     */
    if (NumericIndexTop > 1)
	SortNumericIndex(1, NumericIndexTop);
    fprintf(NumericIndexFilePtr, "\n");
    for (i = 0; i < NumericIndexTop; i++)
	fprintf(NumericIndexFilePtr, "%f %d %d\n", NumericIndex[i].s, 
		(int)NumericIndex[i].Start, (int)NumericIndex[i].Length);
}

static void SortNumericIndex(l, r)
    size_t		l, r;
{
    size_t		i, j;
    NumericIndexIn	x, w;

    i = l; j = r;
    x = NumericIndex[((l+r)/2)-1];
    do {
	while (NumericIndex[i-1].s < x.s)
	    i++;
	while (x.s < NumericIndex[j-1].s)
	    j--;
	if (i <= j) {
	    w = NumericIndex[i-1];
	    NumericIndex[i-1] = NumericIndex[j-1];
	    NumericIndex[j-1] = w;
	    i++, j--;
	}
    } while (j >= i);
    if (l < j)
	SortNumericIndex(l, j);
    if (i < r)
	SortNumericIndex(i, r);
}


static KeyNode *ConvertKeyListToKeyNode(string, keylist)
    char		*string;
    KeyList		*keylist;
{
    KeyList		*tmp;
    KeyNode		*retval = NULL, *newnode;
    size_t		len;

    len = strlen(string);
    while (keylist != NULL) {
	newnode = (KeyNode *)Malloc(sizeof(KeyNode));
	newnode->pos = (PosNode *)Malloc(sizeof(PosNode));
	newnode->s = Malloc(len+1);
	strcpy(newnode->s, string);
	newnode->len = len;
	newnode->attr = newnode->pos->attr = keylist->Attribute;
	newnode->inst = newnode->pos->inst = keylist->Instance;
	newnode->pos->Start = keylist->Start;
	newnode->pos->Length = keylist->Length;
	newnode->pos->next = NULL;
	newnode->next = retval;
	retval = newnode;
	tmp = keylist;
	keylist = keylist->next;
	Free(tmp);
    }
    return retval;
}

static KeyNode *HashEntryToKeyNodes(schema, IndexFilePtr, HashPtr)
    Schema		*schema;
    FILE		*IndexFilePtr;
    Hash		*HashPtr;
{
    char		**KeyEntry, *string, *ch, *Buffer;
    int			i, EntryNumber, KeyFile = fileno(IndexFilePtr);
    size_t		NumberRead;
    KeyList		*keylist;
    KeyNode		*keynode, *retval = NULL, *tmp;

    if (HashPtr->len == 0)
	return NULL;
    KeyEntry = (char **)Malloc(sizeof(char *)*(HashPtr->num+1));
    Buffer = Malloc(HashPtr->len+1);
    Buffer[HashPtr->len] = '\0';
    lseek(KeyFile, HashPtr->pos, 0);
    Read(KeyFile, Buffer, HashPtr->len);
    KeyEntry[0] = Buffer;
    EntryNumber = 1;
    /* Parse the hash bucket into newline terminated lines.
     */
    for (i = 0; i < HashPtr->len; i++) {
        if (Buffer[i] == '\n') {
            Buffer[i] = '\0';
            if (i == HashPtr->len-1)
                break;
            KeyEntry[EntryNumber++] = Buffer+i+1;
        }
    }
#if defined(DIAGNOSTIC)
    if (HashPtr->num != EntryNumber)
	fprintf(stderr, "HashEntry number (%d) != number found (%d)\n", (int)HashPtr->num,
		(int)EntryNumber);
#endif
    for (i = 0; i < EntryNumber; i++) {
	size_t			key_len;

	ch = Qddb_FindKey(KeyEntry[i], &key_len);
#if defined(DIAGNOSTIC)
	if (ch == NULL) /* EEEEK */
	    PANIC("EEEK");
#endif
	*ch = '\0';
	string = Qddb_StripKey(KeyEntry[i], key_len);
	*ch = ' ';
	keylist = Qddb_ParseKey(schema, KeyEntry[i], &NumberRead, True, -1);
	keynode = ConvertKeyListToKeyNode(string, keylist);
	Free(string);
	if (retval == NULL) {
	    retval = keynode;
	} else {
	    if (keynode != NULL) {
		tmp = keynode;
		while (tmp->next != NULL)
		    tmp = tmp->next;
		tmp->next = retval;
		retval = keynode;
	    }
	}
    }
    Free(Buffer);
    Free(KeyEntry);
    return retval;
}
