/*----------------------------------------------------------------------*
** File: pql_print.c
** 
** 
** Project:	PQL
** =====================================================================*
** Description:	PQL (Subset II) - pql printing operation (proj, ag, star)
**
**  Author:	Bjoern Lemke
**  E-Mail:	lemke@lf.net
**
** Copyright (c) 1994 Bjoern Lemke.
**----------------------------------------------------------------------*/

/* include this first, FD_SETSIZE must be defined */
#include "eng.h"

#include <string.h>
#include <stdio.h>
#include "pql_tree.h"

/***********************/
/* imported operations */
/***********************/
extern int evalcondition(Condition *cond);

/*********************/
/* public prototypes */
/*********************/
void print_all(QueryOp *query);
void print_projection(QueryOp *query);
void print_agregate(QueryOp *query);
int print_group(QueryOp *query);

/********************/
/* global variables */
/********************/
extern int actdepth;
extern int attnum;
extern ATable attable[];
extern char *getvar(char *var);

/*******************/
/* static routines */
/*******************/
int getgrpidx(AttrSpec *attrspec, GroupTable *gtable, int gcount);  
static char *print_expr(Expr *expr, char *pbuf);

/*******************/
/* public routines */
/*******************/
void print_all(QueryOp *query)
{
  int i, j, t, s, loffset, goffset;
  FromList *fromlist;
  TableSpec *tablespec;
  JFormat *entry;
  float floatbuf;
  long longbuf;
  double doublebuf;
  int anum;
  int bm=0; /* NULL value bitmap */
  int bmidx=0; /* attribute index for bitmap */
  int count=0;
  int printit, first, amode;
  char pbuf[MAXTUPLEN], rempbuf[MAXTUPLEN], *pbufptr, *rbufptr;
  
  fd_set mask;

  /* vertical and horizontal control */
  char horcntl[NAMELEN], vercntl[NAMELEN];
  /* null value field */
  char nfield[NAMELEN]; 

  strcpy(horcntl, getvar(VN_HORSEP));  expvar(horcntl);
  strcpy(vercntl, getvar(VN_VERSEP));  expvar(vercntl);
  strcpy(nfield, getvar(VN_NFIELD));  expvar(nfield);

  if (!strcmp(getvar(VN_ATTMODE), "on")) 
    amode=1;
  else 
    amode=0;
  
  rempbuf[0] = '\0';
  
  if (query->depth > 0) {
    /* for the subquery, we just store attribute and tuple count */
    query->sqbuf->anum=0;
    query->sqbuf->stored=0;
    if (query->cj != 0) {
      /* got attribute information from the first join buffer */
      query->sqbuf->anum = query->jbuf[0]->anum;
      for (i=0; i < query->cj; i++) 
	query->sqbuf->stored += query->jbuf[i]->stored;
      if (query->countit) {
	bcopy(query->sqbuf->sqbuf, &query->sqbuf->stored, sizeof(long));
	query->sqbuf->stored = 1;
	query->sqbuf->type = T_LONG;	
	query->sqbuf->size = sizeof(long);
      }
    }
  } else {
    for (i=0; i < query->cj; i++) { /* all joined conjunctions */
      goffset=0;
      anum = query->jbuf[i]->anum; /* to get it shorter */ 
      for (t=0; t < query->jbuf[i]->stored; t++) {
	fromlist = query->selcondptr->fromlistptr;
	printit=0;
	pbufptr=pbuf;
	while(fromlist != NULL) {
	  tablespec = fromlist->tablespecptr;
	  j=0; loffset=0;
	  first=1;
	  while(j < anum) {

	    entry = &query->jbuf[i]->entry[j];
	    if (entry->atype == DROP) {
	      loffset += entry->size;
	      bmidx++;
	      j++;
	    } else {  /* if SEP */
	      if (entry->atype ==  SEP
		  || ( !strcmp(tablespec->tablename,
			       query->jbuf[i]->tables[entry->tableindex])
		      && !strcmp(tablespec->tablealias,
				 query->jbuf[i]->talias[entry->tableindex]))) {
		
		/* attribute found */
		
		if (entry->atype == ORDY 
		    && FD_ISSET(bmidx + BMOFFSET, &mask)) {
		  sprintf(pbufptr, "%s", nfield);
		  pbufptr += strlen(pbufptr);
		  printit=1;
		} else { /* if not NULL field */
		  if (amode && entry->type != NULLMAP) {
		    sprintf(pbufptr, "%s : ", entry->attribute);
		    pbufptr += strlen(pbufptr);
		  }
		  switch(entry->type) {
		  case T_CHAR:
		    sprintf(pbufptr, "%s",
			   query->jbuf[i]->jbuf[query->jbuf[i]->actbuf]
			   + loffset + goffset);
		    pbufptr += strlen(pbufptr);

		    /* for (s=strlen(query->jbuf[i]->jbuf[query->jbuf[i]->actbuf]
		       + loffset + goffset); s < entry->size; s++)
		       printf(" "); */
		    printit=1;
		    break;
		  case T_FLOAT:
		    bcopy(query->jbuf[i]->jbuf[query->jbuf[i]->actbuf] 
			  + loffset + goffset, &floatbuf, sizeof(float));
		    sprintf(pbufptr, "%f", floatbuf);
		    pbufptr += strlen(pbufptr);
		    printit=1;
		    break;
		  case T_LONG:
		    bcopy(query->jbuf[i]->jbuf[query->jbuf[i]->actbuf] 
			  + loffset + goffset, &longbuf, sizeof(long));
		    sprintf(pbufptr, "%d", longbuf);
		    pbufptr += strlen(pbufptr);
		    printit=1;
		    break;
		  case T_DOUBLE:
		    bcopy(query->jbuf[i]->jbuf[query->jbuf[i]->actbuf] 
			  + loffset + goffset, &doublebuf, sizeof(double));
		    sprintf(pbufptr, "%f", doublebuf);
		    pbufptr += strlen(pbufptr);
		    printit=1;
		    break;
		  case NULLMAP:
		    /* ignore, just set up new bm */
		    bm=loffset;
		    bmidx=-1; /* will be set up to 0 */
		    bcopy(query->jbuf[i]->jbuf[query->jbuf[i]->actbuf] 
			  + goffset 
			  + bm, mask.fds_bits, sizeof(fd_set));
		    printit=0;
		    break;
		  default:
		    sprintf(pbufptr, "<unknown>");
		    pbufptr += strlen(pbufptr);
		    printit=1;
		    break;
		  }
		}
		
		if (printit) {
		  sprintf(pbufptr, "%s", horcntl);
		  rbufptr=pbufptr;
		  pbufptr += strlen(pbufptr);
		}
	      }
	      loffset += entry->size;
	      bmidx++;
	      j++;
	    }
	  }
	  fromlist = fromlist->next;
	}
	/* printf("%s", vercntl); */
	sprintf(rbufptr, "%s", vercntl);
	
	/* now print out pbuf */
	if (query->distinct) {
	  if (strcmp(pbuf, rempbuf)) {
	    if (!query->countit)
	      printf("%s", pbuf);
	    strcpy(rempbuf, pbuf);
	    count++;
	  }
	} else {
	  if (!query->countit)
	    printf("%s", pbuf);
	  count++;
	}
	goffset += query->jbuf[i]->tupsize;
      }
    }
    if (query->countit)
      printf("%d\n", count);
    fprintf(stderr, "%d tuples selected.\n", count);
  }
  return;
}

void print_projection(QueryOp *query)
{
  int i, j, k, t, s;
  ExprList *exprlist;
  int type;
  char expr[STRINGLEN];
  char *sqbufptr;
  float floatbuf, remfloatbuf;
  long longbuf, remlongbuf;
  double doublebuf, remdoublebuf;
  char pbuf[MAXTUPLEN], rempbuf[MAXTUPLEN], *pbufptr;

  int count=0;
  int offset;
  int amode;

  /* null field */
  char nfield[NAMELEN];   
  /* vertical and horizontal control */
  char horcntl[NAMELEN], vercntl[NAMELEN];

  strcpy(horcntl, getvar(VN_HORSEP));   expvar(horcntl);
  strcpy(vercntl, getvar(VN_VERSEP));   expvar(vercntl);
  strcpy(nfield, getvar(VN_NFIELD));    expvar(nfield);

  if (!strcmp(getvar(VN_ATTMODE), "on")) 
    amode=1;
  else 
    amode=0;

  rempbuf[0] = '\0';

  /* initialize in case of a subquery */
  if (query->depth > 0) {
    query->sqbuf->anum=0;
    query->sqbuf->stored=0;
    sqbufptr = query->sqbuf->sqbuf;

    /* we will store attributes in case of one-choice-select */
    exprlist=query->selectionptr->exprlistptr;
    while(exprlist != NULL) {
      query->sqbuf->anum++;
      exprlist=exprlist->next;
    }
    if (query->sqbuf->anum > 1)  {
      /* get number of stored tuples and return */
      for (i=0; i < query->cj; i++)
	query->sqbuf->stored += query->jbuf[i]->stored;
      return;    
    }
  }

  for (i=0; i < query->cj; i++) { /* all joined conjunctions */
    /* set up slots */
    for (j=0; j < attnum; j++) {
      /* search join specific base */
      offset=0;
      for (k=0; k < query->jbuf[i]->anum; k++) {
	/* yes, this is ugly */
	if (attable[j].depth == actdepth
	    && !strcmp(query->jbuf[i]->entry[k].attribute,
		    attable[j].attrspec->attrname)
	    && !strcmp(query->jbuf[i]->tables[query->jbuf[i]->entry[k].tableindex], attable[j].attrspec->tablename)
	    && !strcmp(query->jbuf[i]->talias[query->jbuf[i]->entry[k].tableindex], attable[j].attrspec->talias)) {
	  attable[j].attrspec->slot=
	    (void*)query->jbuf[i]->jbuf[query->jbuf[i]->actbuf]  
	      + offset;
	  attable[j].attrspec->bm=
	    (void*)query->jbuf[i]->jbuf[query->jbuf[i]->actbuf]
	      + attable[j].attrspec->bmbase;
	}
	offset += query->jbuf[i]->entry[k].size;
      } 
    }
    if (query->depth > 0) {

      /* no need of tracing through the exprlist, just one-choice-select */
      exprlist=query->selectionptr->exprlistptr;
     
      /* first time, we also get the type, a little perfomance hack */
      if (getexpr(exprlist->exprptr, &type, expr) == -1)
	return;
      query->sqbuf->type = type;
      
      /* size setup */
      switch (type) {
      case T_FLOAT:
	query->sqbuf->size = sizeof(float);
	break;
      case T_LONG:
	query->sqbuf->size = sizeof(long);
	break;
      case T_DOUBLE:
	query->sqbuf->size = sizeof(double);
	break;
      case T_CHAR:
	query->sqbuf->size = STRINGLEN; /* sorry for this limit */
	break;
      }

      for (t=0; t < query->jbuf[i]->stored; t++) {
	if (getexpr(exprlist->exprptr, &type, expr) == -1)
	  return;
	switch(type) {
	case T_CHAR:
	  if (query->distinct && strcmp(expr, rempbuf)) {
	    strcpy(sqbufptr, expr);
	    sqbufptr += query->sqbuf->size;
	    strcpy(rempbuf, expr);
	  } else {
	    strcpy(sqbufptr, expr);
	    sqbufptr += query->sqbuf->size;
	  }
	  break;
	case T_FLOAT:
	  if (query->distinct) {
	    bcopy(expr, &floatbuf, sizeof(float));
	    if (remfloatbuf != floatbuf) {
	      bcopy(expr, sqbufptr, sizeof(float));
	      sqbufptr += sizeof(float);
	    }
	  } else {
	    bcopy(expr, sqbufptr, sizeof(float));
	    sqbufptr += sizeof(float);
	  }
	  break;
	case T_DOUBLE:
	  if (query->distinct) {
	    bcopy(expr, &doublebuf, sizeof(double));
	    if (remdoublebuf != doublebuf) {
	      bcopy(expr, sqbufptr, sizeof(double));
	      sqbufptr += sizeof(double);
	    }
	  } else {
	    bcopy(expr, sqbufptr, sizeof(double));
	    sqbufptr += sizeof(double);
	  }
	  break;
	case T_LONG:
	  if (query->distinct) {
	    bcopy(expr, &longbuf, sizeof(long));
	    if (remlongbuf != longbuf) {
	      bcopy(expr, sqbufptr, sizeof(long));
	      sqbufptr += sizeof(long);
	    }
	  } else {
	    bcopy(expr, sqbufptr, sizeof(long));
	    sqbufptr += sizeof(long);
	  }
	  break;
	case NVAL:
	  /* ignore */
	  break;
	default:
	  fprintf(stderr, "PQL-ERROR: unknown type %d\n", type);
	  break;
	}	
	/* increase slot and bm */
	for (j=0; j < attnum; j++) {
	  if (attable[j].depth == actdepth) {
	    attable[j].attrspec->slot += query->jbuf[i]->tupsize;
	    attable[j].attrspec->bm += query->jbuf[i]->tupsize;
	  }
	}
	query->sqbuf->stored++;
      }
    } else {
      for (t=0; t < query->jbuf[i]->stored; t++) {
	pbufptr=pbuf;
	exprlist=query->selectionptr->exprlistptr;
	while(exprlist != NULL) {
	  if (getexpr(exprlist->exprptr, &type, expr) == -1)
	    return;
	  if (amode) {
	    pbufptr=print_expr(exprlist->exprptr, pbufptr);
	    sprintf(pbufptr, " : ");
	    pbufptr += strlen(pbufptr);
	  }
	  switch(type) {
	  case T_CHAR:
	    sprintf(pbufptr,"%s", expr);
	    pbufptr += strlen(pbufptr);
	    break;
	  case T_FLOAT:
	    bcopy(expr, &floatbuf, sizeof(float));
	    sprintf(pbufptr, "%f", floatbuf);
	    pbufptr += strlen(pbufptr);
	    break;
	  case T_DOUBLE:
	    bcopy(expr, &doublebuf, sizeof(double));
	    sprintf(pbufptr, "%f", doublebuf);
	    pbufptr += strlen(pbufptr);
	    break;
	  case T_LONG:
	    bcopy(expr, &longbuf, sizeof(long));
	    sprintf(pbufptr, "%d", longbuf);
	    pbufptr += strlen(pbufptr);
	    break;
	  case NVAL:
	    sprintf(pbufptr, "%s", nfield);
	    pbufptr += strlen(pbufptr);
	    break;
	  default:
	    fprintf(stderr, "PQL-ERROR: unknown type %d\n", type);
	    break;
	  }
	  exprlist = exprlist->next;
	  if (exprlist != NULL) {
	    sprintf(pbufptr, "%s", horcntl);
	    pbufptr += strlen(pbufptr);
	  }
	}
	/* increase slot and bm */
	for (j=0; j < attnum; j++) {
	  if (attable[j].depth == actdepth) {
	    attable[j].attrspec->slot += query->jbuf[i]->tupsize;
	    attable[j].attrspec->bm += query->jbuf[i]->tupsize;
	  }
	}
	sprintf(pbufptr, "%s", vercntl); 
	if (query->distinct) {
	  if (strcmp(pbuf, rempbuf)) {
	    if (!query->countit)
	      printf("%s", pbuf);
	    strcpy(rempbuf, pbuf);
	    count++;
	  }
	} else {
	  if (!query->countit)
	    printf("%s", pbuf);
	  count++;
	}
      }
    }
  }
  if (query->depth == 0) {
    if (query->countit)
      printf("%d\n", count);
    fprintf(stderr, "%d tuples selected.\n", count);
  }
}

void print_agregate(QueryOp *query)
{
  int i, j, k, t, s;
  ExprList *exprlist;
  Expr *fexpr;
  int type;
  char val[STRINGLEN], *aslot;
  float floatbuf, agfloatval[MAXAGREGATE], avgfloatval;
  double doublebuf, agdoubleval[MAXAGREGATE], avgdoubleval;
  long longbuf, aglongval[MAXAGREGATE];
  int agtype[MAXAGREGATE];
  int count=0;
  int offset;
  int agnum;

  /* vertical and horizontal control */
  char horcntl[NAMELEN], vercntl[NAMELEN];
  strcpy(horcntl, getvar(VN_HORSEP));  expvar(horcntl);
  strcpy(vercntl, getvar(VN_VERSEP));  expvar(vercntl);

  /* in case of a subquery, initialize */
  if (query->depth > 0) {
    query->sqbuf->anum=0;
    query->sqbuf->stored=0;
    
    /* just one-choice-agregation is valid */
    exprlist=query->selectionptr->exprlistptr;
    while(exprlist != NULL) {
      query->sqbuf->anum++;
      exprlist=exprlist->next;
    }
    if (query->sqbuf->anum != 1)  {
      /* get number of stored tuples and return */
      fprintf(stderr, "PQL-ERROR: invalid multi-choice-agregation in subquery\n");
      return;
    }
  }

  /* initialize agregate values */
  for (i=0; i<MAXAGREGATE; i++) {
    agfloatval[i] = 0.0;
    aglongval[i] = 0;
    agdoubleval[i] = 0.0;
  }

  for (i=0; i < query->cj; i++) { /* all joined conjunctions */
    /* set up slots */
    for (j=0; j < attnum; j++) {
      /* search join specific base */
      offset=0;
      for (k=0; k < query->jbuf[i]->anum; k++) {
	/* yes, this is ugly */
	if (attable[j].depth == actdepth 
	    && !strcmp(query->jbuf[i]->entry[k].attribute,
		    attable[j].attrspec->attrname)
	    && !strcmp(query->jbuf[i]->tables[query->jbuf[i]->entry[k].tableindex], attable[j].attrspec->tablename)
	    && !strcmp(query->jbuf[i]->talias[query->jbuf[i]->entry[k].tableindex], attable[j].attrspec->talias)) {
	  attable[j].attrspec->slot=
	    (void*)query->jbuf[i]->jbuf[query->jbuf[i]->actbuf]  
	      + offset;
	  attable[j].attrspec->bm=
	    (void*)query->jbuf[i]->jbuf[query->jbuf[i]->actbuf] 
	      + attable[j].attrspec->bmbase;
	}
	offset += query->jbuf[i]->entry[k].size;
      }
    }
    for (t=0; t < query->jbuf[i]->stored; t++) {
      exprlist=query->selectionptr->exprlistptr;
      agnum = 0;
      while(exprlist != NULL) {
	switch(exprlist->exprptr->functionptr->rule) {
	case FUNC_MIN:
	case FUNC_MAX:
	case FUNC_SUM:
	case FUNC_AVG:
	  fexpr = exprlist->exprptr->functionptr->exprptr;
	  
	  if (getexpr(fexpr, &type, val) == -1)
	    return;
	  aslot = val;

	  /* now calculate agregate value 
	     (just supported on few types) */

	  switch(type) {
	  case T_FLOAT:
	    bcopy(aslot, &floatbuf, sizeof(float));
	    switch (exprlist->exprptr->functionptr->rule) {
	    case FUNC_MIN:
	      if (agfloatval[agnum] > floatbuf || count == NULL)
		agfloatval[agnum] = floatbuf;
	      break;
	    case FUNC_MAX:
	      if (agfloatval[agnum] < floatbuf || count == NULL)
		agfloatval[agnum] = floatbuf;
	      break;
	    case FUNC_SUM:
	    case FUNC_AVG:
	      agfloatval[agnum] += floatbuf;
	      break;
	    }
	    agtype[agnum] = T_FLOAT;
	    break;
	  case T_DOUBLE:
	    bcopy(aslot, &doublebuf, sizeof(double));
	    switch (exprlist->exprptr->functionptr->rule) {
	    case FUNC_MIN:
	      if (agdoubleval[agnum] > doublebuf || count == NULL)
		agdoubleval[agnum] = doublebuf;
	      break;
	    case FUNC_MAX:
	      if (agdoubleval[agnum] < doublebuf || count == NULL)
		agdoubleval[agnum] = doublebuf;
	      break;
	    case FUNC_SUM:
	    case FUNC_AVG:
	      agdoubleval[agnum] += doublebuf;
	      break;
	    }
	    agtype[agnum] = T_DOUBLE;
	    break;
	  case T_LONG:
	    bcopy(aslot, &longbuf, sizeof(long));
	    switch (exprlist->exprptr->functionptr->rule) {
	    case FUNC_MIN:
	      if (aglongval[agnum] > longbuf || count == 0)
		aglongval[agnum] = longbuf;
	      break;
	    case FUNC_MAX:
	      if (aglongval[agnum] < longbuf || count == 0)
		aglongval[agnum] = longbuf;
	      break;
	    case FUNC_SUM:
	    case FUNC_AVG:
	      aglongval[agnum] += longbuf;
	      break;
	    }
	    agtype[agnum] = T_LONG;
	    break;
	  default: 
	    /* case NVAL: */
	    /* skipping this value */
	    break;
	  }
	  break;
	}
	exprlist = exprlist->next;
	agnum++;
      }
      /* increase slots and bm */
      for (j=0; j < attnum; j++) {
	if (attable[j].depth == actdepth) {
	  attable[j].attrspec->slot += query->jbuf[i]->tupsize;
	  attable[j].attrspec->bm += query->jbuf[i]->tupsize;
	}
      }
      count++;
    }
  }

  /* calculate average values and print out */
  exprlist=query->selectionptr->exprlistptr;
  agnum = 0;
  if (query->depth > 0) {
    switch (exprlist->exprptr->functionptr->rule) {
    case FUNC_SUM:
    case FUNC_MAX:
    case FUNC_MIN:
      switch (agtype[agnum]) {
      case T_FLOAT:
	bcopy(query->sqbuf->sqbuf, &agfloatval[agnum], sizeof(float));
	query->sqbuf->type = T_FLOAT;	
	query->sqbuf->size = sizeof(float);
	break;
      case T_DOUBLE:
	bcopy(query->sqbuf->sqbuf, &agdoubleval[agnum], sizeof(double));
	query->sqbuf->type = T_DOUBLE;	
	query->sqbuf->size = sizeof(double);
	break;
      case T_LONG:
	bcopy(query->sqbuf->sqbuf, &aglongval[agnum], sizeof(long));
	query->sqbuf->type = T_LONG;	
	query->sqbuf->size = sizeof(long);
	break;
      }
      break;
    case FUNC_AVG:
      switch (agtype[agnum]) {
      case T_FLOAT:
	avgfloatval = (float)aglongval[agnum] / (float)count;
	bcopy(query->sqbuf->sqbuf, &avgfloatval, sizeof(float));
	query->sqbuf->type = T_FLOAT;	
	query->sqbuf->size = sizeof(float);
	break;
      case T_DOUBLE:
	avgdoubleval = agdoubleval[agnum] / (double)count;
	bcopy(query->sqbuf->sqbuf, &avgdoubleval, sizeof(double));
	query->sqbuf->type = T_DOUBLE;
	query->sqbuf->size = sizeof(double);
	break;
      case T_LONG:
	avgfloatval = (float)aglongval[agnum] / (float)count;
	bcopy(query->sqbuf->sqbuf, &avgfloatval, sizeof(float));
	query->sqbuf->type = T_FLOAT;	
	query->sqbuf->size = sizeof(float);
	break;
      }
      break;
    }
    query->sqbuf->stored = 1;
  } else {
    while (exprlist != NULL && count > 0) {
      switch (exprlist->exprptr->functionptr->rule) {
      case FUNC_SUM:
      case FUNC_MAX:
      case FUNC_MIN:
	switch (agtype[agnum]) {
	case T_FLOAT:
	  printf("%f", agfloatval[agnum]);
	  break;
	case T_DOUBLE:
	  printf("%f", agdoubleval[agnum]);
	  break;
	case T_LONG:
	  printf("%d", aglongval[agnum]);
	  break;
	}
	break;
      case FUNC_AVG:
	switch (agtype[agnum]) {
	case T_FLOAT:
	  printf("%f", agfloatval[agnum] / (float)count);
	  break;
	case T_DOUBLE:
	  printf("%f", agdoubleval[agnum] / (float)count);
	  break;
	case T_LONG:
	  printf("%f", ((float)aglongval[agnum]) / ((float)count));
	  break;
	}
	break;
      }
      exprlist = exprlist->next;
      if (exprlist != NULL) printf("%s", horcntl);
      agnum++;
    }
    printf("%s", vercntl);
  }
}     
    
int print_group(QueryOp *query)
{
  int i, j, k, t, s;
  ExprList *exprlist;
  Expr *fexpr;
  int type;
  char val[STRINGLEN], *aslot;
  float floatbuf, agfloatval[MAXAGREGATE][MAXGROUPS], avgfloatval;
  long longbuf, aglongval[MAXAGREGATE][MAXGROUPS];
  double doublebuf, agdoubleval[MAXAGREGATE][MAXGROUPS], avgdoubleval;

  int gtype;
  int gidx, gcount=0;
  
  GroupTable gby[MAXGROUPS];

  int agtype[MAXAGREGATE];
  int count[MAXGROUPS];
  int offset;
  int agnum;

  /* vertical and horizontal control */
  char horcntl[NAMELEN], vercntl[NAMELEN];
  strcpy(horcntl, getvar(VN_HORSEP));  expvar(horcntl);
  strcpy(vercntl, getvar(VN_VERSEP));  expvar(vercntl);


  /* in case of a subquery, initialize */
  if (query->depth > 0) {
    query->sqbuf->anum=0;
    query->sqbuf->stored=0;
    
    /* just one-choice-agregation is valid */
    exprlist=query->selectionptr->exprlistptr;
    while(exprlist != NULL) {
      query->sqbuf->anum++;
      exprlist=exprlist->next;
    }
    if (query->sqbuf->anum != 1)  {
      /* get number of stored tuples and return */
      fprintf(stderr, "PQL-ERROR: invalid multi-choice-grouping in subquery\n");
      return;
    }
  }
  
  /* initialize tuples per group count */
  for (i=0; i<MAXGROUPS; i++)
    count[i] = 0;

    /* initialize agregate values */
  for (i=0; i<MAXAGREGATE; i++) {
    for (j=0; j<MAXGROUPS; j++) {
      agfloatval[i][j] = 0.0;
      aglongval[i][j] = 0;
      agdoubleval[i][j] =0.0;
    }
  }

  for (i=0; i < query->cj; i++) { /* all joined conjunctions */
    /* set up slots */
    for (j=0; j < attnum; j++) {
      /* search join specific base */
      offset=0;
      for (k=0; k < query->jbuf[i]->anum; k++) {
	/* yes, this is ugly */
	if (attable[j].depth == actdepth
	    && !strcmp(query->jbuf[i]->entry[k].attribute,
		    attable[j].attrspec->attrname)
	    && !strcmp(query->jbuf[i]->tables[query->jbuf[i]->entry[k].tableindex], attable[j].attrspec->tablename)
	    && !strcmp(query->jbuf[i]->talias[query->jbuf[i]->entry[k].tableindex], attable[j].attrspec->talias)) {
	  attable[j].attrspec->slot=
	    (void*)query->jbuf[i]->jbuf[query->jbuf[i]->actbuf]  
	      + offset;
	  attable[j].attrspec->bm=
	    (void*)query->jbuf[i]->jbuf[query->jbuf[i]->actbuf] 
	      + attable[j].attrspec->bmbase;
	}
	offset += query->jbuf[i]->entry[k].size;
      }
    }
    /* set up group type */
    gtype = query->selcondptr->groupcondptr->attrspecptr->type;
    
    for (t=0; t < query->jbuf[i]->stored; t++) {

      /* evaluate having conditon */
      if (query->selcondptr->groupcondptr->havingcondptr != NULL)
	if (!evalcondition(query->selcondptr->groupcondptr->havingcondptr)) {
	  /* increase slots and continue */
	  for (j=0; j < attnum; j++) {
	    if (attable[j].depth == actdepth) {
	      attable[j].attrspec->slot += query->jbuf[i]->tupsize;
	      attable[j].attrspec->bm += query->jbuf[i]->tupsize;
	    }
	  }
	  continue;
	}
      if((gidx = getgrpidx(query->selcondptr->groupcondptr->attrspecptr,
			   gby, gcount)) == -1)
	return(-1);

            
      if (gidx + 1 > gcount )
	gcount = gidx + 1;

      exprlist = query->selectionptr->exprlistptr;
      agnum = 0;
      while(exprlist != NULL) {
	switch(exprlist->exprptr->functionptr->rule) {
	case FUNC_MIN:
	case FUNC_MAX:
	case FUNC_SUM:
	case FUNC_AVG:
	  fexpr = exprlist->exprptr->functionptr->exprptr;

	  if (getexpr(fexpr, &type, val) == -1)
	    return;
	  aslot = val;

	  /* now calculate agregate value 
	     (just supported on few types) */

	  switch(type) {
	  case T_FLOAT:
	    bcopy(aslot, &floatbuf, sizeof(float));
	    switch (exprlist->exprptr->functionptr->rule) {
	    case FUNC_MIN:
	      if (agfloatval[agnum][gidx] > floatbuf || count[gidx] == NULL)
		agfloatval[agnum][gidx] = floatbuf;
	      break;
	    case FUNC_MAX:
	      if (agfloatval[agnum][gidx] < floatbuf || count[gidx] == NULL)
		agfloatval[agnum][gidx] = floatbuf;
	      break;
	    case FUNC_SUM:
	    case FUNC_AVG:
	      agfloatval[agnum][gidx] += floatbuf;
	      break;
	    }
	    agtype[agnum] = T_FLOAT;
	    break;
	  case T_DOUBLE:
	    bcopy(aslot, &doublebuf, sizeof(double));
	    switch (exprlist->exprptr->functionptr->rule) {
	    case FUNC_MIN:
	      if (agdoubleval[agnum][gidx] > doublebuf || count[gidx] == NULL)
		agdoubleval[agnum][gidx] = doublebuf;
	      break;
	    case FUNC_MAX:
	      if (agdoubleval[agnum][gidx] < doublebuf || count[gidx] == NULL)
		agdoubleval[agnum][gidx] = doublebuf;
	      break;
	    case FUNC_SUM:
	    case FUNC_AVG:
	      agdoubleval[agnum][gidx] += doublebuf;
	      break;
	    }
	    agtype[agnum] = T_DOUBLE;
	    break;
	  case T_LONG:
	    bcopy(aslot, &longbuf, sizeof(long));
	    switch (exprlist->exprptr->functionptr->rule) {
	    case FUNC_MIN:
	      if (aglongval[agnum][gidx] > longbuf || count[gidx] == 0)
		aglongval[agnum][gidx] = longbuf;
	      break;
	    case FUNC_MAX:
	      if (aglongval[agnum][gidx] < longbuf || count[gidx] == 0)
		aglongval[agnum][gidx] = longbuf;
	      break;
	    case FUNC_SUM:
	    case FUNC_AVG:
	      aglongval[agnum][gidx] += longbuf;
	      break;
	    }
	    agtype[agnum] = T_LONG;
	    break;
	  case NVAL:
	    /* ignore */
	    break;
	  }
	  break;
	}
	exprlist = exprlist->next;
	agnum++;
      }
      /* increase slot and bm */
      for (j=0; j < attnum; j++) {
	if (attable[j].depth == actdepth) {
	  attable[j].attrspec->slot += query->jbuf[i]->tupsize;
	  attable[j].attrspec->bm += query->jbuf[i]->tupsize;
	}
      }
      count[gidx]++;
    }
  }

  /* calculate average values and print out */

  for (gidx=0; gidx < gcount; gidx++) {
    
    if (query->depth == 0) {
      /* print out grouping attribute */
      switch (gtype) {
      case T_CHAR:
	printf("%s", gby[gidx].gbystr);
	break;
      case T_LONG:
	printf("%d", gby[gidx].gbylong);
	break;
      case T_FLOAT:
	printf("%f", gby[gidx].gbyfloat);
	break;
      case T_DOUBLE:
	printf("%f", gby[gidx].gbydouble);
	break;
      }
      printf("%s", horcntl);
    }

    exprlist=query->selectionptr->exprlistptr;
    agnum = 0;
    if (query->depth > 0) {
      switch (exprlist->exprptr->functionptr->rule) {
      case FUNC_SUM:
      case FUNC_MAX:
      case FUNC_MIN:
	switch (agtype[agnum]) {
	case T_FLOAT:
	  bcopy(query->sqbuf->sqbuf, &agfloatval[agnum][gidx], sizeof(float));
	  query->sqbuf->type = T_FLOAT;	
	  query->sqbuf->size = sizeof(float);
	  break;
	case T_DOUBLE:
	  bcopy(query->sqbuf->sqbuf, &agdoubleval[agnum][gidx], sizeof(double));
	  query->sqbuf->type = T_DOUBLE;	
	  query->sqbuf->size = sizeof(double);
	  break;
	case T_LONG:
	  bcopy(query->sqbuf->sqbuf, &aglongval[agnum][gidx], sizeof(long));
	  query->sqbuf->type = T_LONG;	
	  query->sqbuf->size = sizeof(long);
	  break;
	}
	break;
      case FUNC_AVG:
	switch (agtype[agnum]) {
	case T_FLOAT:
	  avgfloatval = (float)aglongval[agnum][gidx] / (float)count[gidx];
	  bcopy(query->sqbuf->sqbuf, &avgfloatval, sizeof(float));
	  query->sqbuf->type = T_FLOAT;	
	  query->sqbuf->size = sizeof(float);
	  break;
	case T_DOUBLE:
	  avgdoubleval = agdoubleval[agnum][gidx] / (double)count[gidx];
	  bcopy(query->sqbuf->sqbuf, &avgdoubleval, sizeof(double));
	  query->sqbuf->type = T_DOUBLE;
	  query->sqbuf->size = sizeof(double);
	  break;
	case T_LONG:
	  avgfloatval = (float)aglongval[agnum][gidx] / (float)count[gidx];
	  bcopy(query->sqbuf->sqbuf, &avgfloatval, sizeof(float));
	  query->sqbuf->type = T_FLOAT;	
	  query->sqbuf->size = sizeof(float);
	  break;
	}
      }
      query->sqbuf->stored = gcount;
    } else {
      while (exprlist != NULL) {
	switch(exprlist->exprptr->functionptr->rule) {
	case FUNC_MAX:
	case FUNC_MIN:
	case FUNC_SUM:
	  switch (agtype[agnum]) {
	  case T_FLOAT:
	    printf("%f", agfloatval[agnum][gidx]);
	    break;
	  case T_LONG:
	    printf("%d", aglongval[agnum][gidx]);
	    break;
	  case T_DOUBLE:
	    printf("%f", agdoubleval[agnum][gidx]);
	    break;
	  }
	  break;
	case FUNC_AVG:
	  switch (agtype[agnum]) {
	  case T_FLOAT:
	    printf("%f", 
		   agfloatval[agnum][gidx] / (float)count[gidx]);
	    break;
	  case T_LONG:
	    printf("%f", 
		   ((float)aglongval[agnum][gidx]) / ((float)count[gidx]));
	    break;
	  case T_DOUBLE:
	    printf("%lf", 
		   agdoubleval[agnum][gidx] / (double)count[gidx]);
	    break;
	  }
	}
	exprlist = exprlist->next;
	if (exprlist != NULL) printf("%s", horcntl);
	agnum++;
      }
      printf("%s", vercntl);
    }
  }
}

int getgrpidx(AttrSpec *attrspec, GroupTable *gtable, int gcount)
{
  int i;
  long longbuf;
  float floatbuf;
  double doublebuf;

  /* get buf values */
  switch(attrspec->type) {
  case T_CHAR: 
    /* no buffer retrieve needed */
    break;
  case T_FLOAT:
    bcopy(attrspec->slot, &floatbuf, sizeof(float));
    break;
  case T_DOUBLE:
    bcopy(attrspec->slot, &doublebuf, sizeof(double));
    break;
  case T_LONG:
    bcopy(attrspec->slot, &longbuf, sizeof(long));
    break;
  }
  
  for (i=0; i < gcount; i++) {
    switch(attrspec->type) {
    case T_CHAR: 
      if (!strcmp(gtable[i].gbystr, attrspec->slot))
	return(i);
      break;
    case T_FLOAT:
      if (floatbuf == gtable[i].gbyfloat)
	return(i);
      break;
    case T_DOUBLE:
      if (doublebuf == gtable[i].gbydouble)
	return(i);
      break;
    case T_LONG:
      if (longbuf == gtable[i].gbylong)
	return(i);
      break;
    }
  }

  /* group not found, create new one */
  if (gcount >= MAXGROUPS) {
    printf("PQL-ERROR: Too many groups required.\n");
    return(-1);
  }

  switch(attrspec->type) {
  case T_CHAR:
    strcpy(gtable[gcount].gbystr, attrspec->slot);
    break;
  case T_FLOAT:
    gtable[gcount].gbyfloat = floatbuf;
    break;
  case T_DOUBLE:
    gtable[gcount].gbydouble = doublebuf;
    break;
  case T_LONG:
    gtable[gcount].gbylong = longbuf;
    break;
  }
  return(gcount);
}


static char *print_expr(Expr *expr, char *pbuf)
{
  switch(expr->rule) {
  case EXPR_LIT:
    switch(expr->literalptr->rule) {
    case T_LONG:
      sprintf(pbuf, "%d", expr->literalptr->intval);
      pbuf += strlen(pbuf);
      break;
    case T_CHAR:
      sprintf(pbuf, "%s", expr->literalptr->strval);
      pbuf += strlen(pbuf);
      break;
    case T_FLOAT:
      sprintf(pbuf, "%f", expr->literalptr->floatval);
      pbuf += strlen(pbuf);
      break;
    case T_DOUBLE:
      sprintf(pbuf, "%lf", expr->literalptr->doubleval);
      pbuf += strlen(pbuf);
      break;
    }
    break;
  case EXPR_ATTSPEC:
    sprintf(pbuf, "%s", expr->attrspecptr->attrname);
    pbuf += strlen(pbuf);
    break;
  case EXPR_PLUS:
    pbuf = print_expr(expr->expr1ptr, pbuf);
    sprintf(pbuf, "+");
    pbuf += strlen(pbuf);
    pbuf = print_expr(expr->expr2ptr, pbuf);
    break;
  case EXPR_MINUS:
    pbuf = print_expr(expr->expr1ptr, pbuf);
    sprintf(pbuf, "-");
    pbuf += strlen(pbuf);
    pbuf = print_expr(expr->expr2ptr, pbuf);
    break;
  case EXPR_MUL:
    pbuf = print_expr(expr->expr1ptr, pbuf);
    sprintf(pbuf, "*");
    pbuf += strlen(pbuf);
    pbuf = print_expr(expr->expr2ptr, pbuf);
    break;
  case EXPR_DIV:
    pbuf = print_expr(expr->expr1ptr, pbuf);
    sprintf(pbuf, "/");
    pbuf += strlen(pbuf);
    pbuf = print_expr(expr->expr2ptr, pbuf);
    break;
  }
  return(pbuf);
}




