/*----------------------------------------------------------------------*
** File: pql_insert.c
** 
** 
** Project:	PQL
** =====================================================================*
** Description:	PQL (Subset II) - pql insert operation and friends
**
**  Author:	Bjoern Lemke
**  E-Mail:	lemke@lf.net
**
** Copyright (c) 1994 Bjoern Lemke.
**----------------------------------------------------------------------*/

/* include this first, FD_SETSIZE must be defined */
#include "eng.h"

#include <string.h>
#include <stdio.h>
#include "pql_tree.h"

#define MAXBUFLEN 1000
#define MAXTBLHASH 30

/***********************/
/* imported operations */
/***********************/
extern char *getvar(char *var);

/*********************/
/* public prototypes */
/*********************/
int exec_begin(char *tablename);
int exec_commit(char *tablename);
int exec_abort(char *tablename);
int exec_insert(InsertOp *insert);
int exec_update(UpdateOp *update);
int exec_delete(DeleteOp *delete);
int exec_invoke(char *tablename);
int exec_stats(char *tablename);
int evalcondition(Condition *cond);
int init_htable();

/********************/
/* Global Variables */
/********************/
extern ATable attable[];
extern int attnum; /* number of scanned attributes */
extern int attidx; /* number of stored attributes */
static char tpath[NAMELEN];
static struct Hashtable {
  char tablename[MAXTNLEN];
  Relation *rel;
} htable[MAXTBLHASH];

/**********************/
/* private prototypes */
/**********************/
static int puttable(char *tablename, Relation *rel);
static Relation *gettable(char *tablename);

/*******************/
/* public routines */
/*******************/
int exec_invoke(char *tablename)
{
  Relation *rel;
  int i, acount;

  /* horizontal control */
  char horcntl[NAMELEN];
  char *pathvar=getvar(VN_DBPATH);
  strcpy(horcntl, getvar(VN_HORSEP));  expvar(horcntl);

  strcpy(tpath, pathvar);
  strcat(tpath, tablename);
  if ((rel=eng_begin(tpath, READ)) == NULL) {
    fprintf(stderr, "PQL-ERROR: cannot access table %s\n", tablename);
    return(-1);
  }
  acount=0;
  for (i=0; i < rel->maxidx; i++) {
    if (rel->atype[i] == PKEY) printf("*");
    if (rel->atype[i] != DROP && rel->atype[i] != SEP) {
      acount++;
      switch (rel->type[i]) {
      case T_CHAR:
	printf("%s<STR[%d]>", rel->name[i], rel->start[i+1] - rel->start[i] );
	break;
      case T_LONG:
	printf("%s<LNG>", rel->name[i]);
	break;
      case T_FLOAT:
	printf("%s<FLT>", rel->name[i]);
	break;
      case T_DOUBLE:
	printf("%s<DBL>", rel->name[i]);
	break;
      case NULLMAP:
	break;
      default:
	printf("<unknown> ");
	break;
      }
      if (acount < rel->num_f)
	printf("%s", horcntl);
    }
  }
  printf("\n");
  if (eng_commit(rel) == -1) {
    /* just a reader's commit */
    fprintf(stderr, "PQL-ERROR: readers commit for %s failed\n", tablename);
    return(-1);
  }
  return(0);
}

int exec_stats(char *tablename)
{
  Relation *rel;
  int i;

  /* horizontal control */
  char horcntl[NAMELEN];
  char *pathvar=getvar(VN_DBPATH);
  strcpy(horcntl, getvar(VN_HORSEP));  expvar(horcntl);

  strcpy(tpath, pathvar);
  strcat(tpath, tablename);
  if ((rel=eng_begin(tpath, READ)) == NULL) {
    fprintf(stderr, "PQL-ERROR: cannot access table %s\n", tablename);
    return(-1);
  }

  /* print stats here */
  printf("Modification Level : %d\n", rel->level-1);
  printf("TA Log             : ");
  for (i=0; i<rel->tasize; i++) {
    printf("%d|", rel->tastats[i]);
  }
  printf("\n");

  if (eng_commit(rel) == -1) {
    /* just a reader's commit */
    fprintf(stderr, "PQL-ERROR: readers commit for %s failed\n", tablename);
    return(-1);
  }
  return(0);
}


int exec_begin(char *tablename)
{
  Relation *rel;
  int i, acount;

  /* horizontal control */
  char horcntl[NAMELEN];
  char *pathvar=getvar(VN_DBPATH);
  strcpy(horcntl, getvar(VN_HORSEP));  expvar(horcntl);

  strcpy(tpath, pathvar);
  strcat(tpath, tablename);
  if ((rel=eng_begin(tpath, WRITE)) == NULL) {
    fprintf(stderr, "PQL-ERROR: begin for table %s failed\n", tablename);
    return(-1);
  }
  if (puttable(tablename, rel) == -1) {
    fprintf(stderr, "PQL-ERROR: begin for table %s failed\n", tablename);
    return(-1);
  }
  return(0);
}

int exec_commit(char *tablename)
{
  Relation *rel;
  int i, acount;

  if ((rel = gettable(tablename)) == NULL) {
    fprintf(stderr, "PQL-ERROR: commit for table %s failed\n", tablename);
    return(-1);
  }
  
  if (puttable(tablename, NULL)) { /* reinitialize */
    fprintf(stderr, "PQL-ERROR: commit for table %s failed\n", tablename);
    return(-1);
  }

  if (eng_commit(rel) == -1) {
    fprintf(stderr, "PQL-ERROR: commit for table %s failed\n", tablename);
    return(-1);
  }
  return(0);
}

int exec_abort(char *tablename)
{
  Relation *rel;
  int i, acount;

  if ((rel = gettable(tablename)) == NULL) {
    fprintf(stderr, "PQL-ERROR: abort for table %s failed\n", tablename);
    return(-1);
  }
  
  if (puttable(tablename, NULL)) { /* reinitialize */
    fprintf(stderr, "PQL-ERROR: abort for table %s failed\n", tablename);
    return(-1);
  }

  if (eng_abort(rel) == -1) {
    fprintf(stderr, "PQL-ERROR: abort for table %s failed\n", tablename);
    return(-1);
  }
  return(0);
}

int exec_insert(InsertOp *insert)
{
  int i, acount;
  Relation *rel;
  InsertList *inlist;
  ConstantList *constlist;
  char pkibuf[MAXBUFLEN], attibuf[MAXBUFLEN], *ibuf;
  int pkposptr, attposptr, *posptr;
  fd_set mask;
  int bmidx;

  if ((rel=gettable(insert->tablename)) == NULL) {
    fprintf(stderr, "PQL-ERROR: cannot access table %s\n", insert->tablename);
    return(-1);
  }

  inlist=insert->insertlistptr;
  
  if (inlist->rule == 1) {
    fprintf(stderr, "PQL-ERROR: query insert operation not supported yet\n");
    return;
  } else if (inlist->rule == 2) {
    
    /* file insert buffer with constants */
    constlist = inlist->constantlistptr;
    i=0; acount=0;
    pkposptr=0; 
    attposptr=0;

    FD_ZERO(&mask);
    
    while (constlist != NULL) {
      
      /* find out the atype of the attribute andselect the corresponding
	 input buffer, skip over dropped entries and over seperator */
      
      while (rel->atype[i] == DROP) i++;
      if (rel->atype[i] == SEP) {
	i++;
	bmidx=-1; /* start with attribute bitmap index now */
      }

      if (rel->atype[i] == PKEY) {
	posptr = &pkposptr;
	ibuf = pkibuf;
      } else if (rel->atype[i] == ORDY) {
	posptr = &attposptr;
	ibuf = attibuf;
	bmidx++;
      }

      /* type checking */
      if (constlist->literalptr->rule != NVAL 
	  && constlist->literalptr->rule != rel->type[i]) {
	fprintf(stderr, "PQL-ERROR: mismatched types\n");
	eng_commit(rel);	
	return(-1);
      }      

      /* store constant */
      switch (constlist->literalptr->rule) {
      case T_CHAR:
	*posptr = rel->start[i];
	if (strlen(constlist->literalptr->strval) 
	    >= rel->start[i+1] - rel->start[i]) {
	  fprintf(stderr, "PQL-ERROR: insert string longer than attribute size\n");
	  return(-1);
	}
	strcpy(ibuf+(*posptr), constlist->literalptr->strval);
	break;
      case T_LONG:
	*posptr = rel->start[i];
	bcopy(&(constlist->literalptr->intval), ibuf+(*posptr), sizeof(long));
	break;
      case T_FLOAT:
	*posptr = rel->start[i];
	bcopy(&(constlist->literalptr->floatval), ibuf+(*posptr), sizeof(float));
	break;
      case T_DOUBLE:
	*posptr = rel->start[i];
	bcopy(&(constlist->literalptr->doubleval), ibuf+ (*posptr), sizeof(double));
	break;
      case NVAL:
	if (rel->atype[i] != PKEY) {
	  /* set up bitmask */
	  FD_SET(bmidx + BMOFFSET, &mask);
	} else {
	  fprintf(stderr, "PQL-ERROR: NULL value on primary key attribute not allowed\n");
	  /* eng_unl(rel) */
	  return(-1);
	}
	break;
      }
      constlist=constlist->next;
      i++; acount++;
    }
  } else {
    fprintf(stderr, "PQL-ERROR: unknown insert rule\n");
    return(-1);
  }
  if (acount != rel->num_f) {
    fprintf(stderr, "PQL-ERROR: mismatched attribute count\n");
    return(-1);
  }

  /* put the mask */
  bcopy(mask.fds_bits, attibuf, sizeof(fd_set));

  if (eng_add(rel, pkibuf, attibuf) != 0) {
    fprintf(stderr, "PQL-ERROR: cannot add tuple\n");
    return(-1);
  }
  return(0);
}

int exec_update(UpdateOp *update)
{
  int i,j, tuples, updatecount=0, ret;
  Relation *rel;
  char opkupdatebuf[MAXTUPLEN];
  char pkupdatebuf[MAXTUPLEN];
  char attupdatebuf[MAXTUPLEN];
  char *updatebuf;
  SetList *setlist;
  fd_set mask;
  int bmidx;

  if ((rel=gettable(update->tablename)) == NULL) {
    fprintf(stderr, "PQL-ERROR: cannot access table %s\n", 
	    update->tablename);
    return(-1);
  }
  
  /* set up attribute pointers */
  for (i=0; i < attnum; i++) {
    for (j=0; j < rel->maxidx; j++) {
      if (!strcmp(rel->name[j], attable[i].attrspec->attrname)) {
	if (rel->atype[j] == PKEY)
	  attable[i].attrspec->slot = 
	    pkupdatebuf + rel->start[j];
	if (rel->atype[j] == ORDY) {
	  attable[i].attrspec->slot = 
	    attupdatebuf + rel->start[j];
	}
	attable[i].attrspec->type =
	  rel->type[j];
      }
    }
  }

  if (eng_fetch(rel, pkupdatebuf, attupdatebuf, FIRST)
      != MORETUP) {
    printf("PQL-ERROR: cannot fetch from table %s\n", update->tablename);
    return(-1);
  }
  
  do {
    
    /* evaluate condition */
    ret=0;
    if (update->conditionptr != (Condition*)0 ) {
      ret=evalcondition(update->conditionptr);
      if (ret == -1) return(-1);
    }

    if (ret || update->conditionptr == (Condition*)0) {
      
      setlist=update->setlistptr;
      
      /* save the old primary key */
      bcopy(pkupdatebuf, opkupdatebuf, rel->pksize);
      /* get the current NULL value bitmap */
      bcopy(attupdatebuf, mask.fds_bits, sizeof(fd_set));  

      while(setlist != NULL) {
	for (j=0; j < rel->maxidx; j++) {

	  if (rel->atype[j] == SEP) {
	    bmidx=-1; /* start with attribute bitmap index now */
	  }
	  
	  if (rel->atype[j] != DROP 
	      && !strcmp(setlist->assignmentptr->attrspecptr->attrname,
			 rel->name[j])) {
	    
	    /* unset the NULL value on default */
	    
	    /* set up updatebuf */
	    if (rel->atype[j] == PKEY) 
	      updatebuf=pkupdatebuf;
	    else if (rel->atype[j] == ORDY) {
	      updatebuf=attupdatebuf;
	      FD_CLR(bmidx + BMOFFSET , &mask);
	    }
	    
	    /* type checking */
	    if (setlist->assignmentptr->literalptr->rule == NVAL
		|| setlist->assignmentptr->literalptr->rule == rel->type[j]) {
	      switch (setlist->assignmentptr->literalptr->rule) {
	      case T_CHAR:
		strcpy(updatebuf + rel->start[j],
		       setlist->assignmentptr->literalptr->strval);
		break;
	      case T_LONG:
		bcopy(&(setlist->assignmentptr->literalptr->intval),
		      updatebuf + rel->start[j], sizeof(long));
		break;
	      case T_FLOAT:
		bcopy(&(setlist->assignmentptr->literalptr->floatval),
		      updatebuf + rel->start[j], sizeof(float));
		break;
	      case T_DOUBLE:
		bcopy(&(setlist->assignmentptr->literalptr->doubleval),
		      updatebuf + rel->start[j], sizeof(double));
		break;
	      case NVAL:
		if (rel->atype[j] != PKEY) {
		  /* set up bitmask */
		  FD_SET(bmidx + BMOFFSET , &mask);
		} else {
		  fprintf(stderr, "PQL-ERROR: NULL value on primary key attribute not allowed\n");
		  /* eng_unl(rel) */
		  return(-1);
		}
		break;		
	      }
	    } else {
	      fprintf(stderr, "PQL-ERROR: mismatched types\n");
	      /* eng_unl(rel); */
	      return(-1);
	    }
	  }
	  bmidx++;
	}
	setlist=setlist->next;
      }
      /* update operation comes here */
      bcopy(mask.fds_bits, attupdatebuf, sizeof(fd_set));
      if (eng_upd(rel, opkupdatebuf, pkupdatebuf, attupdatebuf)
	  != 0) {
	fprintf(stderr, "PQL-ERROR: cannot update table %s\n", update->tablename);
	/* eng_unl(rel); */
	return(-1);
      }
      updatecount++;
    }
  } while (eng_fetch(rel, pkupdatebuf, attupdatebuf, NEXT) != ENDOFTUP);
  fprintf(stderr, "%d tuples updated\n", updatecount);
  return(0);
}

int exec_delete(DeleteOp *delete)
{
  int i,j, tuples, deletecount=0, ret;
  Relation *rel;
  char pkdeletebuf[MAXTUPLEN], attdeletebuf[MAXTUPLEN];

  if ((rel=gettable(delete->tablename)) == NULL) {
    fprintf(stderr, "PQL-ERROR: cannot access table %s\n", 
	    delete->tablename);
    return(-1);
  }  
  
  /* set up attribute pointers */
  for (i=0; i < attnum; i++) {
    for (j=0; j < rel->maxidx; j++) {
      if (!strcmp(rel->name[j], attable[i].attrspec->attrname)) {
	if (rel->atype[j] == PKEY)
	  attable[i].attrspec->slot = 
	    pkdeletebuf + rel->start[j];
	if (rel->atype[j] == ORDY) {
	  attable[i].attrspec->slot = 
	    attdeletebuf + rel->start[j];
	}
	attable[i].attrspec->type =
	  rel->type[j];
      }
    }
  }
  
  if (eng_fetch(rel, pkdeletebuf, attdeletebuf, FIRST) != MORETUP) {
    fprintf(stderr, "PQL-ERROR: cannot fetch from table %s\n", delete->tablename);
    return(-1);
  }
  
  do {
    /* evaluate condition */

    ret=0;
    if (delete->conditionptr != (Condition*)0 ) {
      ret=evalcondition(delete->conditionptr);
      if (ret == -1) return(-1);
    }
    
    if (ret || delete->conditionptr == (Condition*)0) {
      
      /* delete operation comes here */
      if (eng_del(rel, pkdeletebuf)
	  != 0) {
	fprintf(stderr, "PQL-ERROR: cannot delete from table %s\n", delete->tablename);
	return(-1);
      }
      deletecount++;
    }
  } while (eng_fetch(rel, pkdeletebuf, attdeletebuf, NEXT) != ENDOFTUP); 

  fprintf(stderr, "%d tuples deleted\n", deletecount);
  return(0);
}

/* special condition evaluation for a single base table */
int evalcondition(Condition *cond)
{  
  int ret1, ret2;

  switch(cond->rule) {
  case COND_PREDICATE:
    return(evalpredicate(cond->predicateptr));
  case COND_NOT:
    ret1=evalcondition(cond->cond1ptr);
    if (ret1 == -1)
      return(-1);
    return(!ret1);
  case COND_AND:
    ret1=evalcondition(cond->cond1ptr);
    ret2=evalcondition(cond->cond2ptr);
    if (ret1 == -1 || ret2 == -1)
      return(-1);
    return(ret1 && ret2);
  case COND_OR:
    ret1=evalcondition(cond->cond1ptr);
    ret2=evalcondition(cond->cond2ptr);
    if (ret1 == -1 || ret2 == -1)
      return(-1);
    return(ret1 || ret2);
  default:
    fprintf(stderr, "PQL-ERROR: unknown condition rule %d\n",
	   cond->rule);
    return(-1);
  }
}

int init_htable()
{
  int i;
  for (i=0; i< MAXTBLHASH; i++) {
    htable[i].tablename[0]='\0';
  }
}

/********************/
/* private routines */
/********************/

static int puttable(char *tablename, Relation *rel)
{
  int index=0, count=0;
  char *tptr=tablename;

  while (*tptr) {
    index = (index + *tptr) % MAXTBLHASH;
    tptr++;
  }
  while (htable[index].tablename[0] != '\0' 
	 && strcmp(tablename, htable[index].tablename)
	 && count < MAXTBLHASH) {
    index = (index + 1) % MAXTBLHASH;
    count++;
  }
  if (count == MAXTBLHASH)  {
    fprintf(stderr, "PQL-ERROR: relation hashtable full\n");
    return(-1);
  }
  /* make the entry */
  strcpy(htable[index].tablename, tablename);
  htable[index].rel = rel;
  return(0);
}

static Relation *gettable(char *tablename)
{
  int index=0, count=0;
  char *tptr=tablename;

  while (*tptr) {
    index = (index + *tptr) % MAXTBLHASH;
    tptr++;
  }
  while (strcmp(htable[index].tablename, tablename)
	 && count < MAXTBLHASH) {
    index = (index + 1) % MAXTBLHASH;
    count++;
  }
  if (count == MAXTBLHASH || htable[index].rel == NULL)  {
    fprintf(stderr, "PQL-ERROR: relation not in hashtable\n");
    return(NULL);
  }
  return(htable[index].rel);
}



