/*-------------------------------------------------------------------------
 *
 * lsyscache.c--
 *    Routines to access information within system caches	
 *
 * Copyright (c) 1994, Regents of the University of California
 *
 *
 * IDENTIFICATION
 *    /usr/local/devel/pglite/cvs/src/backend/utils/cache/lsyscache.c,v 1.8 1995/03/03 07:36:01 andrew Exp
 *
 * NOTES
 *    Eventually, the index information should go through here, too.
 *    
 *    Most of these routines call SearchSysCacheStruct() and thus simply
 *    (1) allocate some space for the return struct and (2) call it.
 *    
 *-------------------------------------------------------------------------
 */
#include <string.h>
#include "postgres.h"

#include "nodes/pg_list.h"
#include "utils/syscache.h"
#include "utils/lsyscache.h"
#include "access/tupmacs.h"
#include "utils/rel.h"
#include "utils/palloc.h"
#include "utils/elog.h"
#include "access/attnum.h"

#include "catalog/pg_amop.h"
#include "catalog/pg_type.h"

/*    		---------- AMOP CACHES ----------			 */

/*    
 * op_class -
 *    
 *    	Return t iff operator 'opno' is in operator class 'opclass'.
 *    
 */
bool
op_class(Oid opno, int32 opclass)
{
    FormData_pg_amop amoptup;

    if (SearchSysCacheStruct(AMOPOPID, (char *) &amoptup,
			     (char *) ObjectIdGetDatum(opclass),
			     (char *) ObjectIdGetDatum(opno),
			     (char *) NULL, (char *) NULL))
	return(true);
    else
	return(false);
}

/*    		---------- ATTRIBUTE CACHES ----------			 */

/*    
 * get_attname -
 *    
 *    	Given the relation id and the attribute number,
 *    	return the "attname" field from the attribute relation.
 *    
 */
Name
get_attname(Oid relid, AttrNumber attnum)
{
    FormData_pg_attribute att_tup;
    Name retval = (Name)NULL;

    if (SearchSysCacheStruct(ATTNUM, (char *) &att_tup,
			     (char *) ObjectIdGetDatum(relid),
			     (char *) UInt16GetDatum(attnum),
			     (char *) NULL, (char *) NULL)) {
	retval = (Name)palloc(sizeof(att_tup.attname));
	memmove(retval, &(att_tup.attname), sizeof(att_tup.attname));
	return(retval);
    }
    else
      return((Name)NULL);
}

/*    
 * get_attnum -
 *    
 *    	Given the relation id and the attribute name,
 *    	return the "attnum" field from the attribute relation.
 *    
 */
AttrNumber
get_attnum(Oid relid, Name attname)
{
    FormData_pg_attribute att_tup;

    if (SearchSysCacheStruct(ATTNAME, (char *) &att_tup,
			     (char *) ObjectIdGetDatum(relid),
			     (char *) attname,
			     (char *) NULL, (char *) NULL))
	return(att_tup.attnum);
    else
	return(InvalidAttrNumber);
}

/*    
 * get_atttype -
 *    
 *    	Given the relation OID and the attribute number with the relation,
 *    	return the attribute type OID.
 *    
 */
Oid
get_atttype(Oid relid, AttrNumber attnum)
{
    AttributeTupleForm att_tup = (AttributeTupleForm)palloc(sizeof(*att_tup));

    if (SearchSysCacheStruct(ATTNUM, 
			     (char *) att_tup,
			     (char *) ObjectIdGetDatum(relid),
			     (char *) UInt16GetDatum(attnum),
			     (char *) NULL, (char *) NULL))
	return(att_tup->atttypid);
    else
	return((Oid)NULL);
}

/* This routine uses the attname instead of the attnum because it
 * replaces the routine find_atttype, which is called sometimes when
 * only the attname, not the attno, is available.
 */
bool
get_attisset(Oid relid, Name attname)
{
    HeapTuple htup;
    AttrNumber attno;
    AttributeTupleForm att_tup;

    attno = get_attnum(relid, attname);
     
    htup = SearchSysCacheTuple(ATTNAME, (char *) ObjectIdGetDatum(relid),
			       (char *) attname, (char *) NULL,
			       (char *) NULL);
    if (!HeapTupleIsValid(htup))
	elog(WARN, "get_attisset: no attribute %.16s in relation %d",
	     attname->data, relid);
    if (heap_attisnull(htup, attno))
	return(false);
    else {
	att_tup = (AttributeTupleForm)GETSTRUCT(htup);
	return(att_tup->attisset);
    }
}

/*    		---------- INDEX CACHE ----------			 */

/*    	watch this space...
 */

/*    		---------- OPERATOR CACHE ----------			 */

/*    
 * get_opcode -
 *    
 *    	Returns the regproc id of the routine used to implement an
 *    	operator given the operator uid.
 *    
 */
RegProcedure
get_opcode(Oid opno)
{
    FormData_pg_operator optup;

    if (SearchSysCacheStruct(OPROID, (char *) &optup,
			     (char *) ObjectIdGetDatum(opno),
			     (char *) NULL, (char *) NULL, (char *) NULL))
	return(optup.oprcode);
    else
	return((RegProcedure)NULL);
}

/*
 * get_opname -
 *    returns the name of the operator with the given opno
 *
 * Note: return the struct so that it gets copied.
 */
NameData
get_opname(Oid opno)
{
    FormData_pg_operator optup;

    if (SearchSysCacheStruct(OPROID, (char *) &optup,
			     (char *) ObjectIdGetDatum(opno),
			     (char *) NULL, (char *) NULL, (char *) NULL))
	return (optup.oprname);
    else
	elog(WARN, "can't look up operator %d\n", opno);
}

/*    
 * op_mergesortable -
 *    
 *    	Returns the left and right sort operators and types corresponding to a
 *    	mergesortable operator, or nil if the operator is not mergesortable.
 *    
 */
bool
op_mergesortable(Oid opno, Oid ltype, Oid rtype, Oid *leftOp, Oid *rightOp) 
{
    FormData_pg_operator optup;

    if (SearchSysCacheStruct(OPROID, (char *) &optup,
			     (char *) ObjectIdGetDatum(opno),
			     (char *) NULL, (char *) NULL, (char *) NULL) &&
	optup.oprlsortop &&
	optup.oprrsortop && 
	optup.oprleft == ltype &&
	optup.oprright == rtype) {

	*leftOp = ObjectIdGetDatum(optup.oprlsortop);
	*rightOp = ObjectIdGetDatum(optup.oprrsortop);
	return TRUE;
    } else {
	return FALSE;
    }
}

/*    
 * op_hashjoinable--
 *    
 * Returns the hash operator corresponding to a hashjoinable operator, 
 * or nil if the operator is not hashjoinable.
 *    
 */
Oid
op_hashjoinable(Oid opno, Oid ltype, Oid rtype)
{
    FormData_pg_operator optup;

    if (SearchSysCacheStruct(OPROID, (char *) &optup,
			     (char *) ObjectIdGetDatum(opno),
			     (char *) NULL, (char *) NULL, (char *) NULL) &&
	optup.oprcanhash  &&
	optup.oprleft == ltype &&
	optup.oprright == rtype) 
	return(opno);
    else
	return(InvalidOid);
}

/*    
 * get_commutator -
 *    
 *    	Returns the corresponding commutator of an operator.
 *    
 */
Oid
get_commutator(Oid opno)
{
    FormData_pg_operator optup;

    if (SearchSysCacheStruct(OPROID, (char *) &optup,
			     (char *) ObjectIdGetDatum(opno),
			     (char *) NULL, (char *) NULL, (char *) NULL))
	return(optup.oprcom);
    else
	return((Oid)NULL);
}

HeapTuple
get_operator_tuple(Oid opno)
{
    HeapTuple optup;

    if ((optup = SearchSysCacheTuple(OPROID, (char *) ObjectIdGetDatum(opno),
				     (char *) NULL, (char *) NULL,
				     (char *) NULL)))
	return(optup);
    else
	return((HeapTuple)NULL);
}

/*    
 * get_negator -
 *    
 *    	Returns the corresponding negator of an operator.
 *    
 */
Oid
get_negator(Oid opno)
{
    FormData_pg_operator optup;

    if (SearchSysCacheStruct(OPROID, (char *) &optup,
			     (char *) ObjectIdGetDatum(opno),
			     (char *) NULL, (char *) NULL, (char *) NULL))
	return(optup.oprnegate);
    else
	return((Oid)NULL);
}

/*    
 * get_oprrest -
 *    
 *    	Returns procedure id for computing selectivity of an operator.
 *    
 */
RegProcedure
get_oprrest(Oid opno)
{
    FormData_pg_operator optup;

    if (SearchSysCacheStruct(OPROID, (char *) &optup,
			     (char *) ObjectIdGetDatum(opno),
			     (char *) NULL, (char *) NULL, (char *) NULL))
	return(optup.oprrest );
    else
	return((RegProcedure) NULL);
}

/*    
 * get_oprjoin -
 *    
 *    	Returns procedure id for computing selectivity of a join.
 *    
 */
RegProcedure
get_oprjoin(Oid opno)
{
    FormData_pg_operator optup;

    if (SearchSysCacheStruct(OPROID, (char *) &optup,
			     (char *) ObjectIdGetDatum(opno),
			     (char *) NULL, (char *) NULL, (char *) NULL))
	return(optup.oprjoin);
    else
	return((RegProcedure)NULL);
}

/*    		---------- RELATION CACHE ----------			 */

/*    
 * get_relnatts -
 *    
 *    	Returns the number of attributes for a given relation.
 *    
 */
int
get_relnatts(Oid relid)
{
    FormData_pg_class reltup;

    if (SearchSysCacheStruct(RELOID, (char *) &reltup,
			     (char *) ObjectIdGetDatum(relid),
			     (char *) NULL, (char *) NULL, (char *) NULL))
	return(reltup.relnatts);
    else
    	return(InvalidAttrNumber);
}

/*    
 * get_rel_name -
 *    
 *    	Returns the name of a given relation.
 *    
 */
Name
get_rel_name(Oid relid)
{
    FormData_pg_class reltup;
    Name retval = (Name)NULL;

    if ((SearchSysCacheStruct(RELOID, (char *) &reltup,
			      (char *) ObjectIdGetDatum(relid),
			      (char *) NULL, (char *) NULL, (char *) NULL))) {
	retval = (Name)palloc(sizeof(reltup.relname));
	memmove(retval, &(reltup.relname), sizeof(reltup.relname));
	return (retval);
    } else
	return((Name)NULL);
}

/*    		---------- TYPE CACHE ----------			 */

/*    
 * get_typlen -
 *    
 *    	Given the type OID, return the length of the type.
 *    
 */
int16
get_typlen(Oid typid)
{
    TypeTupleFormData typtup;

    if (SearchSysCacheStruct(TYPOID, (char *) &typtup,
			     (char *) ObjectIdGetDatum(typid),
			     (char *) NULL, (char *) NULL, (char *) NULL))
	return(typtup.typlen);
    else
	return((int16)NULL);
}

/*    
 * get_typbyval -
 *    
 *    	Given the type OID, determine whether the type is returned by value or
 *    	not.  Returns 1 if by value, 0 if by reference.
 *    
 */
bool
get_typbyval(Oid typid)
{
    TypeTupleFormData typtup;

    if (SearchSysCacheStruct(TYPOID, (char *) &typtup,
			     (char *) ObjectIdGetDatum(typid),
			     (char *) NULL, (char *) NULL, (char *) NULL))
	return((bool)typtup.typbyval);
    else
	return(false);
}

/*    
 * get_typbyval -
 *    
 *    	Given the type OID, determine whether the type is returned by value or
 *    	not.  Returns 1 if by value, 0 if by reference.
 *    
 */
char
get_typalign(Oid typid)
{
    TypeTupleFormData typtup;

    if (SearchSysCacheStruct(TYPOID, (char *) &typtup,
			     (char *) ObjectIdGetDatum(typid),
			     (char *) NULL, (char *) NULL, (char *) NULL))
	return(typtup.typalign);
    else
	return ('i');
}

/*    
 * get_typdefault - 
 *    
 *    	Given the type OID, return the default value of the ADT.
 *    
 */
struct varlena *
get_typdefault(Oid typid)
{
    struct varlena *typdefault = 
	(struct varlena *)TypeDefaultRetrieve (typid);
    return(typdefault);
}

/*    
 * get_typtype -
 *    
 *    	Given the type OID, find if it is a basic type, a named relation
 *	or the generic type 'relation'.
 *	It returns the null char if the cache lookup fails...
 *    
 */
char
get_typtype(Oid typid)
{
    TypeTupleFormData typtup;

    if (SearchSysCacheStruct(TYPOID, (char *) &typtup,
			     (char *) ObjectIdGetDatum(typid),
			     (char *) NULL, (char *) NULL, (char *) NULL)) {
	return(typtup.typtype);
    } else {
	return('\0');
    }
}

