/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME: 	lincks.c
 *
 * SCCSINFO:		@(#)lincks.c	1.14 6/7/94
 *
 * ORIGINAL AUTHOR(S):  ???
 *
 * MODIFICATIONS:
 *      1994-03-11 Martin Sjlin. Remote host supports
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <rpc/rpc.h>
#include <sys/wait.h>
#include <netdb.h>			/* for MAXHOSTNAMELEN */
#ifndef MAXHOSTNAMELEN			/* seems like SCO have here */
#include <sys/socket.h>
#endif /* n MAXHOSTNAMELEN */

#include "lincks.h"
#include "xconfig.h"
#include "libshared.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
int main P_(( int argc, char **argv ));

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_rpc.h"
#include "f_xdr.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern int errno;
extern char *optarg;
extern int optind;
extern int opterr;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define BUFSIZE 1024
#define LOCKMSG "Another monitor or cutoff is already running - "
#ifndef RSH    
#define RSH "/usr/ucb/rsh"
#endif /* n RSH */
#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif /* MAXHOSTNAMELEN */

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int check_logging P_(( void ));
static void print_version P_(( void ));
static int remote P_(( char *host, char *dbid ));

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
static int debug = 0;		/* Debug flag */

/*  */
/**********************************************************************
 * Function: int main(int argc, char **argv)
 * 
 *  This is a small interface program that can send kill or restart
 *  commands to the monitor and also start/stop the entire system.
 *
 * Modifications:
 *      Wed Jan  5 15:20:07 1994 Martin Sjlin. Added check for
 *      already running monitor when trying to start up a new one
 *      <list mods with name and date>
 */
int main(argc, argv)
  int argc;
  char **argv;
{
  L_ENTRY largs;
  enum clnt_stat err;
  char host[MAXPATHLEN];
  char rhost[MAXPATHLEN];
  char monitorfn[MAXPATHLEN];
  char lockfile[MAXPATHLEN];      /* lockfile name and path */
  u_long lincksport = 0;
  int errflag = 0;
  int c;
  int len;
  int lockfd;
  int killflag = 0, restartflag = 0, startflag = 0;

#ifdef HAVE_SETLINEBUF
  (void)setlinebuf(stdout);
  (void)setlinebuf(stderr);
#else
  setvbuf(stdout, NULL, _IOLBF, 0);
  setvbuf(stderr, NULL, _IOLBF, 0);
#endif 	/* n HAVE_SETLINEBUF */
  print_version();

  /* Get default host name */
  (void)gethostname(host, MAXHOSTNAMELEN);
  (void)strcpy(rhost, host);

  /* Get arguments */
  while ((c = getopt(argc, argv, "h:krsd")) != EOF) {
    switch (c) {
    case 'd':
      debug = !debug;
      break;
    case 'h':
      (void)strcpy(rhost, optarg);
      break;
    case 'k':
      killflag = 1;
      break;
    case 'r':
      restartflag = 1;
      break;
    case 's':
      startflag = 1;
      break;
    default:
      ++errflag;
    }
  }

  if (argc - optind != 1) {
    ++errflag;
  } else {
    if (!configuration(argv[optind])) {
      (void)fprintf(stderr, "Configuration failed: likely bad directory\n");
      exit(-1);
    }
    lincksport = TCPIPNO;
  }

  /* Error condition */
  if (errflag || (killflag + restartflag + startflag) != 1) {
    (void)fprintf(stderr,
		  "usage: %s [-d] [-h hostname] -k|-r|-s dbid\n",
		  argv[0]);
    exit(-1);
  }
  /* Set up for command execution */
  largs.l_uid = SUPERUSER;

  /* get minimal length of host names (if we are running with resolve) */
  {
     int len_h = strlen(host);
     int len_H = strlen(HOST);
     len   = len_h > len_H ? len_H : len_h;
   }

  if (startflag) {
    /* Are we on the right host ? */
    if (strcmp(host, rhost) != 0) {
      exit(remote(rhost, DBDIR));
    }
    /* compare host given in configuration file with this host,
     * if different print out a message that we that we try
     * start LINCKS on the "real" host
     */ 
    if (strncmp(host, HOST, len) != 0) {
      (void) fprintf(stderr,
     "warning: current host, %s, is not host in configuration file, %s.\n",
		     rhost, HOST);
      (void)strcpy(rhost, HOST);
      exit(remote(rhost, DBDIR));
    }

    /* check if monitor is already running */
    (void) sprintf(lockfile, "%s/%s", DBDIR, INDEXLOCK);
    if ((lockfd = open(lockfile, O_RDONLY, 0644)) >= 0) {
      char buf[BUFSIZE];
      if (read(lockfd, buf, (IOSIZE_T)BUFSIZE) <= 0)
	buf[0] = '\0';
      (void)fprintf(stderr, "%s%s\n", LOCKMSG, buf);
      exit(1);
    }
    (void)close(lockfd);

    /* Create path to monitor */
    (void)sprintf(monitorfn, "%s/%s", BINARIES, MONITOR);

    if (check_logging() == 0) {
      (void)fprintf(stderr, "%s %s %s\n%s\n", "Logging to",
			    LOGDIR, "will not work.", 
			    "Logging will be done in /tmp/lincks.log.");
    }
    if (chdir(DBDIR) < 0) {
      (void)fprintf(stderr, "%s: Could not chdir to %s\n", argv[0], DBDIR);
      exit(1);
    }
    /* must print stdout to get right status after remote exec */
    /* since remote except only a \n with a correct return */
    (void)fprintf(stdout, "Starting monitor %s.\n", monitorfn);

    if (!debug) {
      /* Fork and start monitor */
      switch (fork()) {
      case -1:			/* Error */
	(void)fprintf(stderr, "%s: Fork failed: unix error %d\n",
		      argv[0], errno);
	exit(1);
      case 0:			/* Child */
	if (execl(monitorfn, MONITOR, DBDIR, (char *)NULL) == -1)
	  perror("lincks: failed to execl monitor");
	(void)fprintf(stderr, "%s: Could not start %s\n", argv[0], MONITOR);
	exit(1);
      default:			/* Parent */
	sleep(2);
	(void)fprintf(stderr, "\n");
	exit(0);
      }
    } else {
      if (execl(monitorfn, MONITOR, "-d", DBDIR, (char *)NULL) == -1)
	perror("lincks: failed to execl monitor");
      (void)fprintf(stderr, "%s: Could not start %s\n", argv[0], MONITOR);
      exit(1);
    }
  }
  if (killflag) {
    if (strncmp(host, HOST, len) != 0) {
      (void) fprintf(stderr,
     "warning: current host, %s, is not host in configuration file, %s.\n",
		       host, HOST);
      (void)strcpy(rhost, HOST);
    }
    if ((err = callrpc(rhost, (u_long)MONITORPROGNO, lincksport, KILL,
		xdr_entry, (char *)&largs, xdr_void, (char *)NULL))) {
      (void) clnt_perrno(err);
      (void)fprintf(stderr, "%s: %s  %s\n", argv[0], "RPC-call failed.",
		    "System already halted or running on other machine.");
      exit(1);
    } else {
      (void)fprintf(stderr, "System halted.\n");
      exit(0);
    }
  }
  if (restartflag) {
    if (strcmp(host, HOST) != 0) {
      (void) fprintf(stderr,
     "Warning: current host, %s, is not host in configuration file, %s\n",
		     host, rhost);
      (void)strcpy(rhost, HOST);
    }
    if ((err = callrpc(rhost, (u_long)MONITORPROGNO, lincksport, RESTART,
		xdr_entry, (char *)&largs, xdr_void, (char *)NULL))) {
      (void) clnt_perrno(err);
      (void)fprintf(stderr, "\n%s: %s  %s\n", argv[0], "RPC-call failed.",
		    "Cannot restart netserv.");
      exit(1);
    } else {
      (void)fprintf(stdout, "netserv restarted.\n");
      exit(0);
    }
  }
  return(0);
}

/*  */
/**********************************************************************
 * Function: static int remote(char *host, char *dbid)
 * 
 *  call lincks on remote host, returns 0 on success
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int remote(host, dbid)
  char *host;
  char *dbid;
{
  char lincksfn[MAXPATHLEN];
  int status = 0;
  int c, rsh;
  int pfd[2];
  FILE *err;

  /* Build command path */
  (void)sprintf(lincksfn, "%s/%s", BINARIES, LINCKS);

  /* Create pipe for stderr */
  if (pipe(pfd) < 0)
    return -1;

  switch (rsh = fork()) {
  case -1:			/* Error */
    return -1;
  case 0:			/* Child */
    /* Pipe from childs stderr  */
    (void)close(pfd[0]);
    if (dup2(pfd[1], fileno(stderr)) < 0)
      _exit(1);
    if (execl(RSH,host,"-n",lincksfn,!debug ?"-s":"-ds",dbid,(char *)NULL)==-1)
      perror("remote: failed to execl rsh");
    _exit(2);			/* Error */

  default:			/* Parent */
    /* Receive pipe */
    (void)close(pfd[1]);
    if ((err = fdopen(pfd[0], "r")) == NULL)
      return -1;
/*
 * Copy stderr; Anything but a newline here means remote failiure.
 * It's done like this beacause rsh does not return the value of
 * the remote command.
 */
    if ((c = getc(err)) != '\n') {
      status = 1;
      do {
	(void)putc(c, stderr);
      } while ((c = getc(err)) != EOF);
    }
/*
 * rsh doesn't return as long as it has stderr open to the pipe above.
 * But, since we know it's done and ready; just zap it.
 *
 * I don't like this, but what is the alternative ?	/Paul
 */
    (void)kill(rsh, SIGHUP);

    /* Clean up and return */
    (void)fclose(err);
    return status;
  }
}
/*  */
/**********************************************************************
 * Function: static int check_logging()
 * 
 * Check to see if logging is working ok.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int check_logging()
{
  struct stat stat_buffer;
  char filename[MAXPATHLEN];

  (void)sprintf(filename, "%s", LOGDIR);
  if (stat(filename, &stat_buffer) == -1) {
    (void)fprintf(stderr, "Cannot open LOGDIR %s.\n", LOGDIR);
    return 0;
  }
  return 1;
}

#define VERSION "1.3"
#define DATE "1994-06-01"
#define VERSION_INFO "\n"
/*  */
/**********************************************************************
 * Function: static void print_version()
 * 
 * print out the version # of lincks and pertinent info
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void print_version()
{
  (void)fprintf(stdout, "\tlincks version %s (%s)\n%s", VERSION, DATE,
  		VERSION_INFO);
}
